/****************************************************************************
 * assist_colr.c
 * Author Joel Welling
 * Copyright 1990, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This module provides support for renderers which do not have the
capability to manage all necessary attributes.  It provides a simple
mechanism for simulating direct color on an indexed color device,
and utilities for color manipulation.
*/
#include "alisp.h"
#include "ge_error.h"
#include "p3d.h"
#include "assist.h"

/*
 * The mechanism for direct color simulation is to initialize the device
 * color table (via a renderer-supplied function) with a uniform 
 * distribution of colors in RGB space.  A second function then
 * provides the index whose color is closest to that requested by
 * the renderer. 
 */

static int monochrome, offset, r_size, g_size, b_size;

static void set_sizes( num_indices )
int num_indices;
/* This routine picks pivot values for the uniformly distributed colors */
{
  int i;

  /* first try for a cube root; b_size gets result */
  for (i=1; i*i*i <= num_indices; ++i);
  --i;
  b_size= i;

  /* g_size gets raised higher if possible */
  while (i*i*b_size <= num_indices) ++i;
  --i;
  g_size= i;
  
  /* r_size gets raised above g_size if possible */
  while (i*g_size*b_size <= num_indices) ++i;
  --i;
  r_size= i;
}

void ast_ctb_init( first_index, num_indices, clr_set_fun )
int first_index, num_indices;
void (*clr_set_fun)();
/* 
 * This routine initializes the color table.  Its arguments are:
 *
 * first_index: first color table index to use
 * num_indices: max total indices to use ( table size less skipped indices )
 * clr_set_fun: a function which sets one color index, declared like:
 *
 *    void clr_set_fun( index, r, g, b )
 *    int index;
 *    float r, g, b;
 *
 */
{
  int i, j, k, index;
  float rval, gval, bval;

  ger_debug("ast_ctb_init: using %d indices starting at %d",
	    num_indices,first_index);

  offset= first_index;

  if (num_indices<2) 
    ger_fatal("ast_ctb_init: %d is not enough color table indices!",
	      num_indices);

  if (num_indices>2) { /* avoid monochrome case */
    monochrome= 0;
    set_sizes(num_indices);

    index= first_index;
    for (i=0; i<b_size; i++) {
      bval= (float)i / (float)(b_size-1);
      for (j=0; j<g_size; j++) {
	gval= (float)j / (float)(g_size-1);
	for (k=0; k<r_size; k++) {
	  rval= (float)k / (float)(r_size-1);
	  (*clr_set_fun)( index++, rval, gval, bval );
	}
      }
    }
  }
  else { /* monochrome case */
    monochrome= 1;
    (*clr_set_fun)(offset, 0.0, 0.0, 0.0);
    (*clr_set_fun)(offset+1, 1.0, 1.0, 1.0);
  }
}

int ast_ctb_bestindex( r, g, b )
float r, g, b;
/* This routine provides the closest index to the requested color */
{
  int index;

  /* no debugging; probably called too often */

  if (monochrome) index= offset+1;
  else {
    /* Clip values, since out of range values produce addressing errors */
    if (r>1.0) r= 1.0;
    if (r<0.0) r= 0.0;
    if (g>1.0) g= 1.0;
    if (g<0.0) g= 0.0;
    if (b>1.0) b= 1.0;
    if (b<0.0) b= 0.0;
    
    index= offset
          + (int)( r*(r_size-1) + 0.5 )
	  + r_size * (int)( g*(g_size-1) + 0.5 )
	  + r_size * g_size * (int)( b*(b_size-1) + 0.5 );
  }

  return( index );
}

void ast_vlist_ave_color(vlist, count, red, green, blue, alpha)
Vertex_list vlist;
int count;
float *red, *green, *blue, *alpha;
/* This routine traverses a vertex list, and returns an average color for
 * its vertices.  Any (or all) vertices having no color do not contribute
 * to the average.
 */
{
  register float r= 0.0, g= 0.0, b= 0.0, a= 0.0;
  register Vertex thisvertex;
  register Color thiscolor;
  register int colors_found= 0;

  while ( count-- ) {
    thisvertex= first_vertex( vlist );
    vlist= rest_of_vertices( vlist );
    if ( !null( thiscolor= vertex_color(thisvertex) ) ) {
      r += color_red( thiscolor );
      g += color_green( thiscolor );
      b += color_blue( thiscolor );
      a += color_alpha( thiscolor );
      colors_found++;
    }
  }

  if (colors_found) {
    r= r/colors_found;
    g= g/colors_found;
    b= b/colors_found;
    a= a/colors_found;
  }

  *red= r;
  *green= g;
  *blue= b;
  *alpha= a;
}
