/****************************************************************************
 * mouse_ui.c
 * Author Joel Welling
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This module provides a 'mouse user interface' which lets the user rotate
and examine models with a point-and-click interface.  The actual point-and-
click interface for a given user interface is separate.
*/

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "alisp.h"
#include "p3d.h"
#include "assist.h"
#include "ge_error.h"
#include "matrix_ops.h"
#include "control.h"
#include "ren.h"
#include "ui.h"
#include "mi.h"

/* Notes-
   -trace files which continue across loading of a new P3D file or even
   moving on to a new model in the same file are going to have breaks 
   in their tracked position.
*/

/* Place to keep the command line argument information */
static int argc_save;
static char **argv_save;

/* Buffer to hold name of file to load */
#define MAXNAMELENGTH 128
static char filebuff[MAXNAMELENGTH];

/* Name of file to hold trace, file pointer for file which holds the
 *  trace, flag to indicate that tracing is enabled,
 *  and a flag for the trace routine to use to determine if a new
 *  view must be saved.
 */
#define TRACE_FILE_NAME "path_trace.dat"
static FILE *tracefile;
static int trace_on= 0;
static int new_camera= 0; /* save first view of trace */

/* crystal ball radius (1.0 is full width of viewport) */
#define CRYSTAL_BALL_RADIUS 0.8

/* symbol to access P3D id string */
static Symbol p3d_id_symbol;

/* flags for needed updates */
static int needs_redraw= 0, needs_recenter= 0, done_flag= 0, exit_flag= 0;

/* Current model gob, attribute list, and centering transformation */
static Gob current_gob;
static Attribute_list current_attr;
static Transformation current_centering_trans;

/* Current light data */
static Gob lightgob;
static Attribute_list lightattr;

/* Current camera data*/
static Point lookfrom, lookatpt, orig_lookat;
static Vector up;
static float fovea, hither, yon;
static Color background;

/* 
Flags to indicate that user interface is initialized, and that it should
not be shut down (because a load is in progress).
*/
static int initialized= 0, no_shutdown= 0;

static void align_up()
/* 
This routine removes components of the up vector which are parallel to
the viewing direction, and renormalizes the up vector.
*/
{
  float vx, vy, vz, ux, uy, uz, unorm, vnorm, dot;

  ger_debug("align_up:");

  vx= point_x(lookfrom) - point_x(lookatpt);
  vy= point_y(lookfrom) - point_y(lookatpt);
  vz= point_z(lookfrom) - point_z(lookatpt);
  vnorm= vx*vx + vy*vy + vz*vz;
  if (vnorm==0.0) {
    ger_error("align_up: viewing distance is zero! continueing.");
    return;
  }

  ux= vector_x( up );
  uy= vector_y( up );
  uz= vector_z( up );

  dot= ux*vx + uy*vy + uz*vz;

  ux= ux - dot * (vx/vnorm);
  uy= uy - dot * (vy/vnorm);
  uz= uz - dot * (vz/vnorm);
  unorm= ux*ux + uy*uy + uz*uz;
  if (unorm==0.0) {
    ger_error("align_up: up vector vanishes, using y direction");
    ux= 0.0; uy= 1.0; uz= 0.0; unorm= 1.0;
  }

  unorm= sqrt(unorm);
  ux= ux/unorm;
  uy= uy/unorm;
  uz= uz/unorm;

  set_vector_x( up, ux );
  set_vector_y( up, uy );
  set_vector_z( up, uz );

  return;
}

static Transformation center_model()
/* 
This routine translates the lookat point to the origin.  It modifies
the camera information accordingly, and returns a transformation to move
the model and lights appropriately.
*/
{
  float *center;
  Transformation centering_trans;

  ger_debug("center_model:");

  center= make_translate_c( -point_x(lookatpt), -point_y(lookatpt), 
	-point_z(lookatpt) );
  centering_trans= array2d_to_lisp( center );
  grab(centering_trans);
  free( (char *)center );

  set_point_x( lookfrom, point_x(lookfrom) - point_x(lookatpt) );
  set_point_y( lookfrom, point_y(lookfrom) - point_y(lookatpt) );
  set_point_z( lookfrom, point_z(lookfrom) - point_z(lookatpt) );
  set_point_x( lookatpt,0.0 );
  set_point_y( lookatpt,0.0 );
  set_point_z( lookatpt,0.0 );

  return( centering_trans );
}

static Transformation recenter_model(oldtrans)
Transformation oldtrans;
/* 
This routine retranslates the lookat point to the origin.  It modifies
the camera information accordingly, and returns a transformation to move
the model and lights appropriately.
*/
{
  float *center;
  Transformation change, centering_trans;

  ger_debug("recenter_model:");

  center= make_translate_c( -point_x(lookatpt), -point_y(lookatpt), 
	-point_z(lookatpt) );
  change= array2d_to_lisp( center );
  grab(change); /* grab followed by free assures garbage collection */
  centering_trans= matrix_mult( change, oldtrans );
  free_transformation( oldtrans );
  free_transformation( change );
  grab(centering_trans);
  free( (char *)center );

  set_point_x( lookfrom, point_x(lookfrom) - point_x(lookatpt) );
  set_point_y( lookfrom, point_y(lookfrom) - point_y(lookatpt) );
  set_point_z( lookfrom, point_z(lookfrom) - point_z(lookatpt) );
  set_point_x( lookatpt,0.0 );
  set_point_y( lookatpt,0.0 );
  set_point_z( lookatpt,0.0 );

  needs_recenter= 0;

  return( centering_trans );
}

static void report()
/* This routine enters the current info in the trace file   */
{
  static int stepcount= 0;
  float fromx, fromy, fromz, tox, toy, toz;

  if (trace_on && new_camera) {
    fromx= point_x(lookfrom) + point_x(orig_lookat);
    fromy= point_y(lookfrom) + point_y(orig_lookat);
    fromz= point_z(lookfrom) + point_z(orig_lookat);

    tox = point_x(orig_lookat);
    toy = point_y(orig_lookat);
    toz = point_z(orig_lookat);

    fprintf(tracefile,
	    "----------------------------------------------------\n");
    fprintf(tracefile,"%d\n",stepcount++);
    fprintf(tracefile,"lookfrom= %f %f %f;", fromx, fromy, fromz);
    fprintf(tracefile," lookat= %f %f %f\n", tox, toy, toz);
    fprintf(tracefile,"up= %f %f %f\n", point_x(up), point_y(up), point_z(up));
    new_camera= 0;
  }
}

void miu_toggle_trace()
/* 
This routine activates or deactivates the writing of view data to
the trace file.
*/
{
  static int trace_initialized= 0;

  ger_debug("miu_toggle_trace");

  if (!trace_initialized) {
    if ( !(tracefile= fopen(TRACE_FILE_NAME,"w")) ) {
      ger_error("Unable to open trace file %s!", TRACE_FILE_NAME);
    }
    else {
      fprintf(tracefile,"fovea= %f, hither= %f, yon=%f\n", fovea, hither, yon);
      trace_initialized= 1;
      new_camera= 1;
    }
  }

  if (trace_initialized) trace_on= !trace_on;
  if (trace_on) report();
}

void miu_set_viewpoint( thisviewpoint )
Viewpoint *thisviewpoint;
/* This routine sets the viewpoint information to that given. */
{
  ger_debug("miu_set_viewpoint");

  /* Recentering fixes up the subtraction in lookfrom and lookatpt */
  set_point_x( lookfrom, thisviewpoint->from.x - point_x(orig_lookat) );
  set_point_y( lookfrom, thisviewpoint->from.y - point_y(orig_lookat) );
  set_point_z( lookfrom, thisviewpoint->from.z - point_z(orig_lookat) );

  set_point_x( lookatpt, thisviewpoint->at.x - point_x(orig_lookat) );
  set_point_y( lookatpt, thisviewpoint->at.y - point_y(orig_lookat) );
  set_point_z( lookatpt, thisviewpoint->at.z - point_z(orig_lookat) );
  set_point_x( orig_lookat, thisviewpoint->at.x );
  set_point_y( orig_lookat, thisviewpoint->at.y );
  set_point_z( orig_lookat, thisviewpoint->at.z );

  set_vector_x( up, thisviewpoint->up.x );
  set_vector_y( up, thisviewpoint->up.y );
  set_vector_z( up, thisviewpoint->up.z );

  fovea= thisviewpoint->fovea;
  hither= thisviewpoint->hither;
  yon= thisviewpoint->yon;

  needs_recenter= 1;
  needs_redraw= 1;
  new_camera= 1;

  /* Give the user interface the news that things have changed */
  mil_update();
}

Viewpoint *miu_viewpoint()
/* This routine returns the current viewpoint */
{
  Viewpoint *thisviewpoint;

  ger_debug("miu_viewpoint");

  if ( !( thisviewpoint= (Viewpoint *)malloc(sizeof(Viewpoint)) ) )
    ger_fatal("Unable to allocate %d bytes for viewpoint!",sizeof(Viewpoint));

  thisviewpoint->from.x= point_x(lookfrom) + point_x(orig_lookat);
  thisviewpoint->from.y= point_y(lookfrom) + point_y(orig_lookat);
  thisviewpoint->from.z= point_z(lookfrom) + point_z(orig_lookat);
  thisviewpoint->at.x= point_x(orig_lookat);
  thisviewpoint->at.y= point_y(orig_lookat);
  thisviewpoint->at.z= point_z(orig_lookat);
  thisviewpoint->up.x= vector_x(up);
  thisviewpoint->up.y= vector_y(up);
  thisviewpoint->up.z= vector_z(up);
  thisviewpoint->fovea= fovea;
  thisviewpoint->hither= hither;
  thisviewpoint->yon= yon;
  thisviewpoint->next= (Viewpoint *)0;

  return(thisviewpoint);
}

double miu_viewing_distance()
/* 
This routine returns the distance from the lookfrom point to the
lookat point.
*/
{
  float vx, vy, vz;

  vx= point_x(lookatpt) - point_x(lookfrom);
  vy= point_y(lookatpt) - point_y(lookfrom);
  vz= point_z(lookatpt) - point_z(lookfrom);
  return( sqrt( vx*vx + vy*vy + vz*vz ) ); 
}

main(argc, argv)
int argc;
char *argv[];
{
  ger_debug("main in mouse_ui:");

  argc_save= argc;
  argv_save= argv;
  strncpy( filebuff, argv[argc-1], MAXNAMELENGTH );
  filebuff[MAXNAMELENGTH-1]= '\0';

  while (1) {  /* exit via 'quit' button */
    SESSION_END= 0; /* alisp global variable */
    alisp(argc, argv); 
    argv[argc-1]= filebuff;
  }
}

char *miu_filename()
/* This routine returns a pointer to the current file name.  Note that
 * the string is statically allocated, so editing it or freeing it is a
 * bad idea.
 */
{
  ger_debug("miu_filename: returning <%s>",filebuff);
  return(filebuff);
}

void miu_load(fname)
char *fname;
/* 
This routine causes a new P3D file to be loaded (forgetting the
geometry in the current file).  Loading occurs on the next return
from mil_event, not immediately.
*/
{
  ger_debug("miu_load: setting up to load file <%s>",fname);

  strncpy( filebuff, fname, MAXNAMELENGTH );
  filebuff[MAXNAMELENGTH-1]= '\0';
  SESSION_END= 1; /* alisp global variable signalling end of eval loop */
  done_flag= 1; /* to cause exit from event loop */
  no_shutdown= 1; /* to prevent automatic shutdown of UI */
}

void miu_exit()
/*
This routine causes the current program to exit.  Exit happens on the
next return from mil_event, not immediately.
*/
{
  ger_debug("miu_exit: setting up to exit program");

  exit_flag= 1;
}

void miu_done()
/*  
Calling this routine indicates that at the next return from
mil_event, interpretation should proceed to the next model
in the P3D file.
*/
{
  ger_debug("miu_done");

  done_flag= 1;
  no_shutdown= 1; /* in case we run off the end of the metafile */
}

void miu_redraw()
/*
Calling this routine indicates that at the next return from mil_event,
the model should be redrawn.
*/
{
  ger_debug("miu_redraw");

  needs_redraw= 1;
}

void miu_prompt_redraw()
/*
Calling this routine causes the model to be redrawn immediately.
*/
{
  ger_debug("miu_prompt_redraw");

  if (needs_recenter) {
    current_centering_trans= recenter_model(current_centering_trans);
    ren_traverselights( lightgob, current_centering_trans, lightattr );      
  }

  if (needs_redraw) {
    ren_camera( lookatpt, lookfrom, up, fovea, hither, yon, background );
    needs_redraw= 0;
  }

  /* Save trace if needed, then render */
  report();
  ren_render( current_gob, current_centering_trans, current_attr );
}

void ui_setup( renderer, device, outfile, hints, controller, startframe, 
	      framecopies )
char *renderer, *device, *controller, *outfile;
int startframe, framecopies;
Attribute_list hints;
/* This routine initializes everything. */
{
  ger_debug(
    "ui_setup: initializing renderer %s, device %s, controller now \"none\" ",
	    renderer, device);

  if (!initialized) {
    /* Initialize the toolkit interface, controller and renderer */
    p3d_id_symbol= create_symbol("p3d-id-string");
    mil_setup( symbol_string(p3d_id_symbol), argc_save, argv_save );
    ctrl_setup( "none", startframe, framecopies );
    ren_setup( renderer, device, 0, outfile, hints );
    lookatpt= create_point();
    lookfrom= create_point();
    orig_lookat= create_point();
    up= create_vector();
    background= create_color();
    initialized= 1;
  }
  else {
    ger_debug(
	 "ui_setup: called twice; this call interpreted as a hard reset.");
    ui_reset( 1, startframe, framecopies );
  }
}

void ui_reset( hard, startframe, framecopies )
int hard, startframe, framecopies;
/* This routine resets everything */
{
  ger_debug("ui_reset: resetting renderer and controller; hard= %d",
	    hard);

  /* Hard resets require recreation of lisp-side variables */
  if (hard) {
    p3d_id_symbol= create_symbol("p3d-id-string");
    lookatpt= create_point();
    lookfrom= create_point();
    orig_lookat= create_point();
    up= create_vector();
    background= create_color();
    current_gob= (Gob)0;
    current_attr= (Attribute_list)0;
    lightgob= (Gob)0;
    lightattr= (Attribute_list)0;
    initialized= 1;
  }

  ast_text_reset( hard );
  ast_prim_reset( hard );
  ast_spln_reset( hard );
  ctrl_reset( hard, startframe, framecopies );
  ren_reset( hard );
}

void ui_shutdown()
/* 
This routine shuts everything down, unless the no_shutdown flag is
set.  That implies that we are in a load operation, and the UI should be
left open.
*/
{
  ger_debug("ui_shutdown: no_shutdown= %d.",no_shutdown);
  
  if ( no_shutdown ) {
    no_shutdown= 0;
  }
  else {
    ren_shutdown();
    mil_shutdown();
    ctrl_end();
  }

  /* **NOTE** We intentionally leave 'initialized' set, so that if ui_setup
   * is called again we will know that the needed behavior is a hard reset.
   */
}

void ui_camera(cam_lookat, cam_lookfrom, cam_up, cam_fovea, cam_hither, 
	       cam_yon, cam_background)
Point cam_lookat, cam_lookfrom;
Vector cam_up;
float cam_fovea, cam_hither, cam_yon;
Color cam_background;
/* This routine handles camera setting */
{
  ger_debug("ui_camera: fovea= %f, hither= %f, yon= %f", cam_fovea,
	    cam_hither, cam_yon);

  /* Save camera values */
  set_point_x( lookatpt, point_x(cam_lookat) );
  set_point_y( lookatpt, point_y(cam_lookat) );
  set_point_z( lookatpt, point_z(cam_lookat) );
  set_point_x( orig_lookat, point_x(cam_lookat) );
  set_point_y( orig_lookat, point_y(cam_lookat) );
  set_point_z( orig_lookat, point_z(cam_lookat) );
  set_point_x( lookfrom, point_x(cam_lookfrom) );
  set_point_y( lookfrom, point_y(cam_lookfrom) );
  set_point_z( lookfrom, point_z(cam_lookfrom) );
  set_vector_x( up, vector_x(cam_up) );
  set_vector_y( up, vector_y(cam_up) );
  set_vector_z( up, vector_z(cam_up) );
  fovea= cam_fovea;
  hither= cam_hither;
  yon= cam_yon;
  set_color_red( background, color_red(cam_background) );
  set_color_green( background, color_green(cam_background) );
  set_color_blue( background, color_blue(cam_background) );
  set_color_alpha( background, color_alpha(cam_background) );

  /* remove parallel component of up vector */
  align_up();

  /* Set camera */
  ren_camera( cam_lookat, cam_lookfrom, cam_up, cam_fovea, 
	     cam_hither, cam_yon, cam_background );

  /* Give the toolkit interface a chance to deal with changes */
  mil_update();
}

void miu_approach_model( scale )
float scale;
/* This routine translates the camera in or out by the given scale. */
{

  ger_debug("miu_approach_model: scale= %f",scale);

  if (scale<0.01) scale= 0.01; /* to keep from sliding into origin */
  set_point_x( lookfrom, scale*point_x(lookfrom) );
  set_point_y( lookfrom, scale*point_y(lookfrom) );
  set_point_z( lookfrom, scale*point_z(lookfrom) );

  needs_redraw= 1;
  new_camera= 1;
}

static void mouse_delta( mouse, dx, dy )
MousePosition mouse;
float *dx, *dy;
/* 
This routine calculates mouse coordinates as floating point relative
to the viewport center 
*/
{
  ger_debug("mouse_delta:");
  *dx= (float)( mouse.x - (mouse.maxx/2) )/(float)(mouse.maxx/2);
  *dy= (float)( mouse.y - (mouse.maxy/2) )/(float)(mouse.maxy/2);
}

static void mouse_to_3d( mouse, vx, vy, vz )
MousePosition mouse;
float *vx, *vy, *vz;
/* This routine translates mouse coords to 3D crystal ball coords */
{
  float x, y, radius_2d, radius_3d= CRYSTAL_BALL_RADIUS;

  ger_debug("mouse_to_3d:");

  mouse_delta( mouse, &x, &y );
  radius_2d= sqrt( x*x + y*y );
  if (radius_2d > CRYSTAL_BALL_RADIUS) {
    x= x*CRYSTAL_BALL_RADIUS/radius_2d;
    y= y*CRYSTAL_BALL_RADIUS/radius_2d;
  }
  *vx= x;
  *vy= y;
  if ( radius_2d < CRYSTAL_BALL_RADIUS )  /* avoid round-off errors */
    *vz= sqrt( CRYSTAL_BALL_RADIUS*CRYSTAL_BALL_RADIUS-radius_2d*radius_2d );
  else *vz= 0.0;
}

static float *get_normal_component( v1x, v1y, v1z, v2x, v2y, v2z )
float v1x, v1y, v1z, v2x, v2y, v2z;
/* This routine returns a 4-vector holding the perpendicular component
 * of v1 with respect to v2, or (0, 1, 0, 1) if the two vectors are
 * parallel.
 */
{
  float *perpvec;
  float dot, normsqr;

  if ( !(perpvec= (float *)malloc( 4*sizeof(float) ) ) )
    ger_fatal("get_normal_component: unable to allocate 4 floats!\n");

  dot= v1x*v2x + v1y*v2y + v1z*v2z;
  normsqr= v2x*v2x + v2y*v2y + v2z*v2z;

  perpvec[0]= v1x - v2x*dot/normsqr;
  perpvec[1]= v1y - v2y*dot/normsqr;
  perpvec[2]= v1z - v2z*dot/normsqr;
  perpvec[3]= 1.0;

  if ( (perpvec[0]==0.0) && (perpvec[1]==0.0) && (perpvec[2]==0.0) ) {
    ger_error(
      "mouse_ui: get_normal_component: vectors align; using Y.");
    perpvec[1]= 1.0;
    }

  return( perpvec );
}

static float *crystal_ball( mousedown, mouseup )
MousePosition mousedown, mouseup;
/* 
This routine returns a transformation matrix appropriate for a crystal
ball rotation as specified by the given mouse motion.
*/
{
  float *rot, *urot, *viewmatrix, *upvec, *rup, *rv1, *rv2, *result;
  float dx, dy, dz, upx, upy, upz;
  float v1[4], v2[4];

  ger_debug("crystal_ball:");

  /* 
  Create inverse viewing transformation by composing a matrix which
  aligns the viewing direction with the z axis with one which aligns the
  transformed up direction with the -y axis.  Finally, take the transpose
  to get the inverse.
  */
  dx= point_x(lookfrom);
  dy= point_y(lookfrom);
  dz= point_z(lookfrom);
  upx= vector_x(up);
  upy= vector_y(up);
  upz= vector_z(up);
  rot= make_aligning_rotation( dx, dy, dz, 0.0, 0.0, 1.0 );
  upvec= get_normal_component( upx, upy, upz, dx, dy, dz ); 
  rup= matrix_vector_c( rot, upvec );
  urot= make_aligning_rotation(rup[0], rup[1], rup[2], 0.0, 1.0, 0.0);
  viewmatrix= matrix_mult_c( urot, rot );
  (void)transpose( viewmatrix );


  /* Convert to 3D coords on crystal ball surface */
  mouse_to_3d( mousedown, &v1[0], &v1[1], &v1[2] );
  v1[3]= 1.0;
  mouse_to_3d( mouseup, &v2[0], &v2[1], &v2[2] );
  v2[3]= 1.0;

  /* Align vectors to world coordinates */
  rv1= matrix_vector_c( viewmatrix, v1 );
  rv2= matrix_vector_c( viewmatrix, v2 );

  /* Create transformation which aligns original radius vec with final one */
  result=
    make_aligning_rotation( rv1[0], rv1[1], rv1[2], rv2[0], rv2[1], rv2[2] );
  (void)transpose( result );

  /* clean up */
  free( (char *)rot );
  free( (char *)urot );
  free( (char *)viewmatrix );
  free( (char *)upvec );
  free( (char *)rup );
  free( (char *)rv1 );
  free( (char *)rv2 );

  return(result);
}

void miu_rotate_model( mousedown, mouseup )
MousePosition mousedown, mouseup;
/* This routine implements a crystal ball rotation of the model */
{
  float *matrix, *new_from, from_vec[4], *new_up, up_vec[4];

  ger_debug("miu_rotate_model:");

  from_vec[0]= point_x(lookfrom);
  from_vec[1]= point_y(lookfrom);
  from_vec[2]= point_z(lookfrom);
  from_vec[3]= 0.0;

  up_vec[0]= vector_x(up);
  up_vec[1]= vector_y(up);
  up_vec[2]= vector_z(up);
  up_vec[3]= 0.0;

  matrix= crystal_ball( mousedown, mouseup );
  new_from= matrix_vector_c( matrix, from_vec );
  new_up= matrix_vector_c( matrix, up_vec );

  set_point_x( lookfrom, new_from[0] );
  set_point_y( lookfrom, new_from[1] );
  set_point_z( lookfrom, new_from[2] );

  set_vector_x( up, new_up[0] );
  set_vector_y( up, new_up[1] );
  set_vector_z( up, new_up[2] );

  free( (char *)new_from );
  free( (char *)new_up );
  free( (char *)matrix );

  needs_redraw= 1;
  new_camera= 1;
}

void miu_translate_model( mousedown, mouseup )
MousePosition mousedown, mouseup;
/* 
This routine implements a translation of the model in the view plane.
The model needs to be recentered afterwards.
*/
{
  float view_vec[3], up_vec[3], right_vec[3], view_dist, screen_width,
        x1, y1, x2, y2, dx, dy;

  ger_debug("miu_translate_model:");

  view_vec[0]= -point_x(lookfrom); /* lookatpt is the origin */
  view_vec[1]= -point_y(lookfrom);
  view_vec[2]= -point_z(lookfrom);
  view_dist= sqrt( view_vec[0]*view_vec[0] + view_vec[1]*view_vec[1]
                  + view_vec[2]*view_vec[2] );
  if (view_dist != 0.0) {
    view_vec[0]= view_vec[0]/view_dist;
    view_vec[1]= view_vec[1]/view_dist;
    view_vec[2]= view_vec[2]/view_dist;
  }

  up_vec[0]= vector_x(up);
  up_vec[1]= vector_y(up);
  up_vec[2]= vector_z(up);

  right_vec[0]= view_vec[1]*up_vec[2] - view_vec[2]*up_vec[1];
  right_vec[1]= view_vec[2]*up_vec[0] - view_vec[0]*up_vec[2];
  right_vec[2]= view_vec[0]*up_vec[1] - view_vec[1]*up_vec[0];

  screen_width= view_dist * sin( DegtoRad * fovea/2.0 );

  mouse_delta( mousedown, &x1, &y1 );
  mouse_delta( mouseup, &x2, &y2 );
  dx= x2-x1;
  dy= y2-y1;

  set_point_x( lookatpt, 
    ( point_x(lookatpt) - screen_width*(dx*right_vec[0]+dy*up_vec[0]) ) ); 
  set_point_y( lookatpt, 
    ( point_y(lookatpt) - screen_width*(dx*right_vec[1]+dy*up_vec[1]) ) ); 
  set_point_z( lookatpt, 
    ( point_z(lookatpt) - screen_width*(dx*right_vec[2]+dy*up_vec[2]) ) ); 

  /* Move the orig_lookat point so that traces will come out correct */
  set_point_x( orig_lookat, point_x(orig_lookat) + point_x(lookatpt) );
  set_point_y( orig_lookat, point_y(orig_lookat) + point_y(lookatpt) );
  set_point_z( orig_lookat, point_z(orig_lookat) + point_z(lookatpt) );

  needs_recenter= 1;
  needs_redraw= 1;
  new_camera= 1;
}

void ui_render(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* 
This routine handles actual rendering of the gob. It implements the loop
that watches for keyboard input.
*/
{
  static int first_pass= 1;

  ger_debug("ui_render:");

  /* Grab model data */
  if ( current_gob && !null(current_gob) ) ungrab(current_gob);
  grab( current_gob= thisgob );
  if ( current_attr && !null(current_attr) ) ungrab(current_attr);
  grab( current_attr= thisattrlist );

  /* Calculate and implement centering transformation */
  current_centering_trans= center_model();
  ren_traverselights( lightgob, current_centering_trans, lightattr );      

  /* Draw the model, if the initial expose event won't do so */
  if ( !first_pass )
    ren_render( thisgob, thistrans, thisattrlist );
  else first_pass= 0;

  /* The following loop handles user interaction */
  needs_redraw= 0;
  needs_recenter= 0;
  done_flag= 0;
  exit_flag= 0;
  align_up(); /* align up vector */
  while (!done_flag) {
    mil_event();

    if (needs_recenter) {
      current_centering_trans= recenter_model(current_centering_trans);
      ren_traverselights( lightgob, current_centering_trans, lightattr );      
    }

    if (needs_redraw) {
      ren_camera( lookatpt, lookfrom, up, fovea, hither, yon, background );
      /* Save trace if needed, then render */
      report();
      ren_render( thisgob, current_centering_trans, thisattrlist );
      needs_redraw= 0;
    }

    if (exit_flag) exit(0);
  }

  free_transformation(current_centering_trans);
}

void ui_traverselights(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* This routine handles lighting traversal. */
{
  ger_debug("ui_traverselights");

  if ( lightgob && !null(lightgob) ) ungrab(lightgob);
  grab( lightgob= thisgob );
  if ( lightattr && !null(lightattr) ) ungrab(lightattr);
  grab( lightattr= thisattrlist );
  ren_traverselights( thisgob, thistrans, thisattrlist );
}

void ui_free(thisgob)
Gob thisgob;
/* This routine causes the renderer to free storage associated with the gob. */
{
  ger_debug("ui_free:");

  ren_free(thisgob);
}
