/****************************************************************************
 * scan_token.c
 * Author Phil Andrews
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/* 
This module extracts floating point or int values from tokens.  Radices
other than 10 are supported for scanning of ints.
*/
/* Modification log:

	10/23/89 Joel Welling 
			- changed from char **inptr to char *inptr
			- removed superfluous ret from scan_int
			- changed from indexing into 'digits' array
			  to subtraction of '0' to get base 10 digit values

*/

static int get_decimal();
static int get_based();

/* scan an input string for a CGM clear text integer */
scan_int(inptr, outval)
char *inptr;
int *outval;
{
register int is_neg, first_int;
#define NUMBER_SIGN '#'
#define MIN_BASE 2
#define MAX_BASE 16

	/* skip spaces */
	while ((*inptr == ' ') && (*inptr)) ++inptr;
	if (!*inptr) return(0);	/* empty string */

	/* is it signed ? */
	if (*inptr == '-') {
	    is_neg = 1;
	    ++inptr;
	} else 	if (*inptr == '+') {
	    ++inptr;
	    is_neg = 0;
	} else is_neg = 0;

	if ((*inptr<'0') || (*inptr>'9')) return(0);	/* no digits */
	/* now get the first (only ?) integer */
	first_int = (is_neg) ? -get_decimal(&inptr) : get_decimal(&inptr);

	/* do we have a base ? */
	if ((*inptr == NUMBER_SIGN) && (first_int <= MAX_BASE) &&
	    (first_int >= MIN_BASE)) {	/* based integer */
	    ++inptr;
	    return(get_based(&inptr, first_int, outval));
	} else {
	    *outval = first_int;
	    return(1);
	}
}

/* grab a decimal, optimised, assumed setup O.K., steps pointer */
/* Note that this assumes the integers are consecutive in the character
   encoding- this may be an ASCII dependency! */
static get_decimal(inptr)
char **inptr;
{
register int ret;

	while (**inptr == '0') ++*inptr;	/* skip zeros */
	ret = 0;
	while ((**inptr>='0') && (**inptr<='9')) {
	    ret = 10 * ret + (int)( **inptr - '0' );
	    ++*inptr;
	}
	return(ret);
}

/* grab a based integer, steps pointer */
static get_based(inptr, inbase, outval)
char **inptr;
int inbase, *outval;
{
#define max_extend 16
static char ext_digits[max_extend] = {'0', '1', '2', '3', '4', '5', '6', '7',
    '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};
register int ret, i, is_neg;
	
	/* is it signed ? */
	if (**inptr == '-') {
	    is_neg = 1;
	    ++*inptr;
	} else 	if (**inptr == '+') {
	    ++*inptr;
	    is_neg = 0;
	} else is_neg = 0;

	/* quick check that we have a legit number */
	for (i=0; (i<inbase) && (ext_digits[i] != **inptr); ++i);
	if (i>=inbase) return(0);
	
	ret = 0;
	while (1) {
	    for (i=0; (i<inbase) && (ext_digits[i] != **inptr); ++i);
	    if (i>=inbase) {
		*outval = (is_neg) ? -ret: ret;
		return(1);
	    } else {
		ret = inbase * ret + i;
		++*inptr;
	    }
	}
}

/* get a floating point value */
/* Note that this routine depends on the digits lieing consecutively in the
   character table-  this may be an ASCII dependency. */
scan_float(inptr, outval)
char *inptr;
float *outval;
{
register int i, is_neg, first_int, exponent, neg_exp;
register float fract, pwr10;;
	
	/* is it signed ? */
	if (*inptr == '-') {
	    is_neg = 1;
	    ++inptr;
	} else 	if (*inptr == '+') {
	    ++inptr;
	    is_neg = 0;
	} else is_neg = 0;
	
	/* do we have a legit number ? */
	if (((*inptr > '9') || (*inptr < '0')) && (*inptr != '.'))
	    return(0);

	if ((*inptr <='9') && (*inptr >= '0')) {	/* a number here */
	    first_int = get_decimal(&inptr);
	} else first_int = 0;


	if (*inptr == '.') {	/* explicit point number or scaled real */
	    ++inptr;
	    pwr10 = 10.0;
	    fract = 0.0;
	    while ((*inptr>='0') && (*inptr<='9')) {
	    	fract += ((float)(*inptr - '0')) / pwr10;
		pwr10 *= 10.0;
		++inptr;
	    }
	    if (*inptr == 'e') {	/* scaled real */
		++inptr;
		/* is exponent signed ? */
		if (*inptr == '-') {
	    	    neg_exp = 1;
		    ++inptr;
		} else 	if (*inptr == '+') {
	    	    ++inptr;
	    	    neg_exp = 0;
		} else neg_exp = 0;
		exponent = get_decimal(&inptr);
		pwr10 = 1.0;
		if (neg_exp) {
		    for (i=0; i<exponent; ++i) pwr10 /= 10.0;
		} else {
		    for (i=0; i<exponent; ++i) pwr10 *= 10.0;
		}
	        *outval = (is_neg) ? - (first_int + fract) * pwr10 : 
		    (first_int + fract) * pwr10;
	        return(1);
	    } else {			/* explicit point */
	        *outval = (is_neg) ? -(first_int + fract) : first_int + fract;
	        return(1);
	    }
	} else if (*inptr == 'e') {	/* scaled real number, no fract */
	    ++inptr;
	    /* is exponent signed ? */
	    if (*inptr == '-') {
		neg_exp = 1;
	        ++inptr;
	    } else 	if (*inptr == '+') {
	    	++inptr;
	    	neg_exp = 0;
	    } else neg_exp = 0;
	    exponent = get_decimal(&inptr);
	    pwr10 = 1.0;
	    if (neg_exp) {
	        for (i=0; i<exponent; ++i) pwr10 /= 10.0;
	    } else {
	        for (i=0; i<exponent; ++i) pwr10 *= 10.0;
	    }
	    *outval = (is_neg) ? - first_int * pwr10 : first_int * pwr10;
	    return(1);
	} else {	/* just an integer */
	    *outval = (float) (is_neg) ? -first_int : first_int;
	    return(1);
	}
	    
}
