 /*
  * Khoros: $Id: imgcache.c,v 1.2 1991/07/15 06:06:37 khoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: imgcache.c,v 1.2 1991/07/15 06:06:37 khoros Exp $";
#endif

 /*
  * $Log: imgcache.c,v $
 * Revision 1.2  1991/07/15  06:06:37  khoros
 * HellPatch1
 *
  */ 

/*
 *----------------------------------------------------------------------
 *
 * Copyright 1990, University of New Mexico.  All rights reserved.

 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as too the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

/*
    IMGCACHE.C - Support routines for the KHOROS Shared Memory Image Cache

    Written: Scott Wilson
    Date: January 1990

    Note: These routines are not for consumption by typical applications
          programs (like IPL routines). They are for internal use as
          needed by readimage(), readheader(), and writeimage(), with
          some use by the cache support programs.

    Modified: 30-May-91 Scott Wilson - Add checks for "-" and "#" in
                        find_image_in_cache and allocate_cache_slot
*/

#include "unmcopyright.h"	

#ifdef CACHE

#include "vinclude.h"	
#include "imgcache.h"

extern int prevent_unlock;

int 
find_image_in_cache(name)
char *name;
  {
   /* Try to find the image in the Khoros shared memory cache. If it is 
      there, return the slot number in the image directory */
    int id,i,fd;
    char *data;
    struct shmcache *sch;
    struct cacheimg *ci;
    int size;

    /* Make sure we haven't got a STDIN or STDOUT specification! */
    if (strcmp(name,"-") == 0 || strcmp(name,"#") == 0) return(-1);

    /* Try to get ahold of the shm seg if it already exists. If it doesn't
       it will be created. It is created with RW permissions for everyone. */
    size = khoros_cache_size;

    id = shmget(KREGION_KEY,size,00666 | IPC_CREAT);
    if (id == -1)
      {
        fprintf(stderr,"find_image_in_cache:Cannot obtain shared memory id!\n");
        fprintf(stderr,"Make sure that the amount of shared memory on\n");
        fprintf(stderr,"this machine is large enough to contain a segment\n");
        fprintf(stderr,"of the size specified in your KHOROS_CACHE_SIZE environment\n");
        fprintf(stderr,"variable. \n");
        fprintf(stderr,"  To find out how much shared memory is on your\n");
        fprintf(stderr,"machine, you will probably have to see your system\n");
        fprintf(stderr,"manager because this is specified when the kernel\n");
        fprintf(stderr,"is compiled. You will be looking for a parameter\n");
        fprintf(stderr,"called SMMAX (for non-SYSV) or SHMMAX (for SYSV).\n");
        fprintf(stderr,"\nperror() prints out:");
        perror(NULL);
        exit(1);
      }

    make_xvcachedir(id,&sch,&data);      /* Be sure directory is valid */

    /* Get pointers to image list and block list list */
    ci = xvcache_img_list(sch);

    /* Scan the directory to see if the image is in there */
    fd = -1;
    for (i=0; i<sch->nimgs; i++)
      {
        if (strcmp(ci[i].imgname,name) == 0)
          {
            fd = i;
            break;
          }
      }

    /* Detach from the region */
    shmdt(data);
    return(fd);
  }

att_cache_hdr(sch,data)
struct shmcache **sch;
char **data;
  {
    int id;
    int size;

    /* Try to get ahold of the shm seg if it already exists. If it doesn't
       it will be created. It is created with RW permissions for everyone. */
    size = khoros_cache_size;

    id = shmget(KREGION_KEY,size,00666 | IPC_CREAT);
    if (id == -1)
      {
        if (khoros_cache) {
        fprintf(stderr,"att_cache_hdr: Cannot obtain shared memory id!\n");
        fprintf(stderr,"Make sure that the amount of shared memory on\n");
        fprintf(stderr,"this machine is large enough to contain a segment\n");
        fprintf(stderr,"of the size specified in your KHOROS_CACHE_SIZE environment\n");
        fprintf(stderr,"variable. \n");
        fprintf(stderr,"  To find out how much shared memory is on your\n");
        fprintf(stderr,"machine, you will probably have to see your system\n");
        fprintf(stderr,"manager because this is specified when the kernel\n");
        fprintf(stderr,"is compiled. You will be looking for a parameter\n");
        fprintf(stderr,"called SMMAX (for non-SYSV) or SHMMAX (for SYSV).\n");
        fprintf(stderr,"\nperror() prints out:");
        perror(NULL);
        exit(1);
        }
        else
        {
        fprintf(stderr,"Unable to access Khoros Cache.\n");
        fprintf(stderr,"KHOROS_CACHE environment variable not set!\n");
        exit(1);
        }
      }

    make_xvcachedir(id,sch,data);          /* Be sure directory is valid */
  }

cache_write(fd,data,count)
int fd,count;
char *data;
  {
    /* Write the data to the cache image slot encoded in the file
       descriptor; slot # = -(1000+fd) */
    int i,n,n1,nc,size;
    char *addr,*dest;
    struct cacheblk *p;
    struct shmcache *sch;
    struct cacheimg *ci;

    fd = -(1000+fd);
    nc = count;

    att_cache_hdr(&sch,&addr);
    ci = xvcache_img_list(sch);

    ci[fd].last_access = time(NULL);

    if (ci[fd].blkptr == NULL)
      { /* New image - must allocate a block for it */
        ci[fd].blkptr = sch->freelist;
        sch->freelist = sch->freelist->next;
        ci[fd].blkptr->next = NULL;
        ci[fd].curblk = ci[fd].blkptr;
        ci[fd].bytes_remaining_in_blk = BLKSIZE;
        sch->remaining_space -= BLKSIZE;
      }

    if (ci[fd].bytes_remaining_in_blk != 0 )
      {
        /* Have a partially filled current block, so write as much
           as we can into it, trying to fill it */
        if (count <= ci[fd].bytes_remaining_in_blk)
          { /* Not enough to fill the block */
            dest = ci[fd].curblk->start+BLKSIZE-
                   ci[fd].bytes_remaining_in_blk;
            bcopy(data,dest,count);
            ci[fd].bytes_remaining_in_blk -= nc;
            nc = 0;
          }
        else
          { /* Fill up the current block */
            dest = ci[fd].curblk->start+BLKSIZE-
                   ci[fd].bytes_remaining_in_blk;
            bcopy(data,dest,ci[fd].bytes_remaining_in_blk);
            nc -= ci[fd].bytes_remaining_in_blk;
            data  += ci[fd].bytes_remaining_in_blk;
            ci[fd].bytes_remaining_in_blk = 0;
          }
      }

    if (nc > 0)
      { /* Still have data to be written, but now it will be whole
           blocks at a time */
        n = (nc+BLKSIZE-1)/BLKSIZE;    /* Number of blocks to write */
        for (i=0; i<n; i++)
          {
            /* Allocate another block, and attach it */
            p = sch->freelist;
            sch->freelist = sch->freelist->next;
            sch->remaining_space -= BLKSIZE;
            ci[fd].curblk->next = p;
            ci[fd].curblk = p; ci[fd].curblk->next = NULL; ci[fd].bytes_remaining_in_blk = 0; /* Zero : going to fill it! */

            /* Now copy the data into it */
            if (i == n-1) size = nc-i*BLKSIZE;
	    else size = BLKSIZE;

            bcopy(data,ci[fd].curblk->start, size);
            data += size;
          }
        /* If there wasn't really enough data to fill the last block, then
           go back and fix it */
        n1 = n*BLKSIZE-nc;
        if (n1 > 0) ci[fd].bytes_remaining_in_blk = n1;
      }
    shmdt(addr);                /* Release the region */
    return(count);
  }

int
allocate_cache_slot(size,filename)
unsigned int size;
char *filename;
  {
    /* Make space in the cache for an image of SIZE bytes.
       If a slot can be allocated then the slot number is
       returned as -(1000+offset), otherwise a -1 is returned. */
    int fd;
    struct shmcache *sch;
    char *addr,buf[512];
    struct cacheimg *ci;
    struct cacheblk *bp1,*bp2;
    struct xvimage *imageptr,*readimage();
    time_t t;
    int i,j;

    /* Check for "-" and "#" filenames */
    if (strcmp(filename,"-") == 0 || strcmp(filename,"#") == 0) return(-1);

    /* Look to see if we already have an image with the same name
       as the one we are about to write. If so, just free the
       resident image (without writing it back to disk). This is
       OK since 1) if the new image fits in cache, the other would
       be overwritten anyway (if it was on disk), and 2) if it
       doesn't fit in cache, the old version is lost along the
       way and only the new one should show up on disk. */
    if ((j = find_image_in_cache(filename)) != -1)
      {
        att_cache_hdr(&sch,&addr);
        ci = xvcache_img_list(sch);
        bp1 = ci[j].blkptr;
        while (bp1 != NULL)
          {
            bp2 = bp1->next;
            bp1->next = sch->freelist;
            sch->freelist = bp1;
            bp1=bp2;
            sch->remaining_space += BLKSIZE;
          }
        /* Now rearrange the cache image directory to make sure it
           remains contiguous. */
        if (j != sch->nimgs-1) 
          {
            /* Image directory entry was not the last in the list, so
               move all the rest of them down to fill the hole */
            for (i=j; i<sch->nimgs-1; i++)
                bcopy((char *)(&ci[i+1]),(char *)(&ci[i]),
                      sizeof(struct cacheimg));
                bzero(&ci[sch->nimgs-1],sizeof(struct cacheimg));
          }
        sch->nimgs--;
        shmdt(addr);
      }

    /* See if the amount of data can be stored at all */
    att_cache_hdr(&sch,&addr);
    if (size > sch->max_space)
      { /* Not enough space! */
        shmdt(addr);
        return(-1);
      }
    shmdt(addr);

    /* Now look to see if there is enough space to just add the
       image without throwing out one of the others */
    att_cache_hdr(&sch,&addr);
    ci = xvcache_img_list(sch);
    if (sch->remaining_space >= size)
      {
        fd = sch->nimgs;
        sch->nimgs += 1;
        strcpy(ci[fd].imgname,filename);
        ci[fd].blkptr = NULL;
        ci[fd].last_access = time(NULL);
        ci[fd].curblk = NULL;
        ci[fd].bytes_remaining_in_blk = 0;
        shmdt(addr);
        return(-(1000+fd));
      }
    else
      { 
        /* Must throw out some images to make space. Write them out
           in LRU order until enough space is obtained. */
        while (sch->remaining_space < size)
          {
            /* First, find the oldest image */
            t = ci[0].last_access;
            j = 0;
            for (i=1; i<sch->nimgs; i++)
              {
                if (ci[i].last_access < t)
                  {
                    t = ci[i].last_access;
                    j = i;
                  }
              }

            /* Use readimage() to get the data out of cache */
            strcpy(buf,ci[j].imgname);
            shmdt(addr);  /* Unhook from the regions so that readimage doesn't
                             get confused by the attached segment */
            prevent_unlock = 1;
            imageptr = readimage(buf);
            prevent_unlock = 0;
            att_cache_hdr(&sch,&addr); /* Re-attach to the region */
            ci = xvcache_img_list(sch);

            /* Turn of the cache for the writeimage() call. Otherwise the
               dang thing ends up back in the cache! */
            khoros_cache = 0;
            writeimage(buf,imageptr);
            khoros_cache = 1;

            /* Now go free up the block list for this image in the cache, then
               release the directory entry. */
            bp1 = ci[j].blkptr;
            while (bp1 != NULL)
              {
                bp2 = bp1->next;   /* Remember where next is */
                bp1->next = sch->freelist;
                sch->freelist = bp1;
                bp1 = bp2;
                sch->remaining_space += BLKSIZE;
              }

            /* Now rearrange the cache image directory to make sure it
               remains contiguous. */
            if (j != sch->nimgs-1) 
              {
                /* Image directory entry was not the last in the list, so
                   move all the rest of them down to fill the hole */
                for (i=j; i<sch->nimgs-1; i++)
                    bcopy((char *)(&ci[i+1]),(char *)(&ci[i]),
                          sizeof(struct cacheimg));
              }
            sch->nimgs--;
          }
        fd = sch->nimgs;
        sch->nimgs += 1;
        strcpy(ci[fd].imgname,filename);
        ci[fd].blkptr = NULL;
        ci[fd].last_access = time(NULL);
        ci[fd].curblk = NULL;
        ci[fd].bytes_remaining_in_blk = 0;
        shmdt(addr);
        return(-(1000+fd));
      }
  }

int
cache_read(fd,data,count)
int fd,count;
char *data;
  {
    /* Read data from the cached image at fd in the directory table at
       offset -(1000+fd). */
    int i,n,nc;
    struct shmcache *sch;
    struct cacheimg *ci;
    char *d,*addr;

    fd = -(1000+fd);
    nc = count;
    att_cache_hdr(&sch,&addr);
    ci = xvcache_img_list(sch);

    ci[fd].last_access = time(NULL);

    if (ci[fd].curblk == NULL)
      { /* Must start accessing the image's blocklist */
        ci[fd].curblk = ci[fd].blkptr;
        ci[fd].bytes_remaining_in_blk = BLKSIZE;
      }

    if (count <= ci[fd].bytes_remaining_in_blk)
      { /* Can satisfy the whole read request from this block */
        bcopy((ci[fd].curblk->start+(BLKSIZE-ci[fd].bytes_remaining_in_blk)),
              data,count);
        ci[fd].bytes_remaining_in_blk -= count;
        nc = count;
      }
    else
      {
        /* Read the remaining part of the current block, then go after the
           rest of the blocks */
        d = data;
        bcopy((ci[fd].curblk->start+(BLKSIZE-ci[fd].bytes_remaining_in_blk)),
              d,ci[fd].bytes_remaining_in_blk);
        d += ci[fd].bytes_remaining_in_blk;
        nc = ci[fd].bytes_remaining_in_blk;
        ci[fd].bytes_remaining_in_blk = 0;

        /* Get the remaining blocks */
        n = (count-nc+BLKSIZE-1)/BLKSIZE;
        for (i=0; i<n; i++)
          {
            /* Get at next block in list */
            ci[fd].curblk = ci[fd].curblk->next;
            ci[fd].bytes_remaining_in_blk = BLKSIZE;
            if (count-nc > BLKSIZE)
              { /* Read the whole block */
                bcopy(ci[fd].curblk->start,d,BLKSIZE);
                d += BLKSIZE;
                nc += BLKSIZE;
              }
            else
              { /* Read partial block to complete operation */
                bcopy(ci[fd].curblk->start,d,count-nc);
                ci[fd].bytes_remaining_in_blk = BLKSIZE-(count-nc);
              }
          }
      }
    shmdt(addr);
    return(count);
  }

make_xvcachedir(id,schdr,add)
int id;
struct shmcache **schdr;
char **add;
  {
    /* Check the shm segment to see if there is an image cache directory
       in it. If not, set one up. If so, just return. */
    int i;
    char *data,*d,*malloc();
    struct shmcache *sch;
    struct cacheimg *ci;
    struct cacheblk *cb;
    int size;

    size = khoros_cache_size;

    data = shmat(id,0,0);             /* Attach to the region */
    if ((int)data ==  -1)
      {
        perror("make_xvcachedir: shmat failed: ");
        exit(1);
      }
    *add = data;                     /* Set external */

    sch = (struct shmcache *)data;    /* Get at the cache header */
    *schdr = sch;                     /* Set external */

    if (strcmp(sch->region_name,KREGION_NAME) != 0)
      {
        /* Header not installed,  so install it and set it up */
        strcpy(sch->region_name,KREGION_NAME);
        sch->nimgs = 0;
        sch->max_space = NBLK(size)*BLKSIZE;
        sch->remaining_space = NBLK(size)*BLKSIZE;

        /* Install and set up the image directory */
        ci = xvcache_img_list(sch);
        bzero((char *)ci,NIMG(size)*sizeof(struct cacheimg));

        /* Install and set up the block list */
        cb = xvcache_blk_list(sch);
        sch->freelist = cb;
        d = (char *)cb + NBLK(size)*sizeof(struct cacheblk);
        for (i=0; i<NBLK(size)-1; i++) 
          {
            cb->start = d;
            d += BLKSIZE;
            cb->next = cb+1;
            cb++;
          }
        cb->start = d;
        cb->next = NULL;
      }
      else
	ci = xvcache_img_list(sch);
  }

struct cacheimg
*xvcache_img_list(sch)
struct shmcache *sch;
  {
    char *c;
    c = (char *)sch;
    return((struct cacheimg *)(c + sizeof(struct shmcache)));
  }


struct cacheblk
*xvcache_blk_list(sch)
struct shmcache *sch;
  {
    int size,k;
    char *c;
    c = (char *)sch;
    size = khoros_cache_size;

    k = sizeof(struct shmcache)+NIMG(size)*sizeof(struct cacheimg);
    return((struct cacheblk *)(c + k));
  }

xvcache_reset_ptrs(fd)
int fd;
  {
    struct shmcache *sch;
    char *addr;
    struct cacheimg *ci;
    int i;

    i = -(1000+fd);
    att_cache_hdr(&sch,&addr);
    ci = xvcache_img_list(sch);
    ci[i].curblk = ci[i].blkptr;
    ci[i].bytes_remaining_in_blk = BLKSIZE;
    shmdt(addr);
  }

xvcache_flush()
  {
    /* Flush all images in the image cache, leaving the cache empty, but
       with a valid directory structure */
    struct shmcache *sch;
    char *addr, filename[512];
    struct cacheimg *ci;
    struct cacheblk *bp1,*bp2;
    struct xvimage *imageptr,*readimage();
    int i,j,k;

    att_cache_hdr(&sch,&addr);
    ci = xvcache_img_list(sch);
    for (k=0; k<sch->nimgs; k++)
      {
        j = 0;

        /* Use readimage() to get the data out of cache */
	strcpy(filename,ci[j].imgname);

        shmdt(addr);  /* Unhook from the regions so that readimage doesn't
                         get confused by the attached segment */
        imageptr = readimage(filename);
        att_cache_hdr(&sch,&addr); /* Re-attach to the region */
        ci = xvcache_img_list(sch);

        /* Turn cacheing off while writing the image otherwise it ends
           up back in cache! */
        khoros_cache = 0;
        writeimage(filename,imageptr);
        khoros_cache = 1;

        /* Now go free up the block list for this image in the cache, then
           release the directory entry. */
        bp1 = ci[j].blkptr;
        while (bp1 != NULL)
          {
            bp2 = bp1->next;   /* Remember where next is */
            bp1->next = sch->freelist;
            sch->freelist = bp1;
            bp1 = bp2;
            sch->remaining_space += BLKSIZE;
          }

        /* Now rearrange the cache image directory to make sure it
           remains contiguous. */
        if (j != sch->nimgs-1) 
          {
            /* Image directory entry was not the last in the list, so
               move all the rest of them down to fill the hole */
            for (i=j; i<sch->nimgs-1; i++)
                bcopy((char *)(&ci[i+1]),(char *)(&ci[i]),
                      sizeof(struct cacheimg));
          }
      }
    sch->nimgs=0;
    shmdt(addr);
  }

int
xvcache_rm_file(name)
char *name;
  {
    /* Delete (rm) a file from the cache. The data blocks are simply
       returned to the freelist and the file entry is removed from
       the cache directory. */
    int id,fd,i;
    char *addr;
    struct shmcache *sch;
    struct cacheimg *ci;
    struct cacheblk *bp1,*bp2;

    fd = find_image_in_cache(name);
    if (fd == -1) return;

    att_cache_hdr(&sch,&addr);
    ci = xvcache_img_list(sch);

    ci[fd].last_access = time(NULL);

    /* Now go free up the block list for this image in the cache, then
       release the directory entry. */
    bp1 = ci[fd].blkptr;
    while (bp1 != NULL)
      {
        bp2 = bp1->next;   /* Remember where next is */
        bp1->next = sch->freelist;
        sch->freelist = bp1;
        bp1 = bp2;
        sch->remaining_space += BLKSIZE;
      }

    /* Now rearrange the cache image directory to make sure it
       remains contiguous. */
    if (fd != sch->nimgs-1) 
      {
        /* Image directory entry was not the last in the list, so
           move all the rest of them down to fill the hole */
        for (i=fd; i<sch->nimgs-1; i++)
            bcopy((char *)(&ci[i+1]),(char *)(&ci[i]),sizeof(struct cacheimg));
      }
    sch->nimgs--;

    shmdt(addr);                /* Release the region */
  }

xvcache_flush_file(name)
char *name;
  {
    /* Flush image "name" from the image cache. */
    struct shmcache *sch;
    char *addr, filename[512];
    struct cacheimg *ci;
    struct cacheblk *bp1,*bp2;
    struct xvimage *imageptr,*readimage();
    int i,fd;

    fd = find_image_in_cache(name);
    if (fd == -1) return;

    att_cache_hdr(&sch,&addr);
    ci = xvcache_img_list(sch);

    /* Use readimage() to get the data out of cache */
    strcpy(filename,name);

    shmdt(addr);  /* Unhook from the regions so that readimage doesn't
                     get confused by the attached segment */
    imageptr = readimage(filename);
    att_cache_hdr(&sch,&addr); /* Re-attach to the region */
    ci = xvcache_img_list(sch);

    /* Turn cacheing off while writing the image otherwise it ends
       up back in cache! */
    khoros_cache = 0;
    writeimage(filename,imageptr);
    khoros_cache = 1;
    
    /* Now go free up the block list for this image in the cache, then
       release the directory entry. */
    bp1 = ci[fd].blkptr;
    while (bp1 != NULL)
      {
        bp2 = bp1->next;   /* Remember where next is */
        bp1->next = sch->freelist;
        sch->freelist = bp1;
        bp1 = bp2;
        sch->remaining_space += BLKSIZE;
      }

    /* Now rearrange the cache image directory to make sure it
       remains contiguous. */
    if (fd != sch->nimgs-1) 
      {
        /* Image directory entry was not the last in the list, so
           move all the rest of them down to fill the hole */
        for (i=fd; i<sch->nimgs-1; i++)
            bcopy((char *)(&ci[i+1]),(char *)(&ci[i]), sizeof(struct cacheimg));
      }
    sch->nimgs--;
    shmdt(addr);
  }

/*
    holepath - REALLY obtain the whole path of a file. This
               is stored in the cache directory so that when
               a file gets tossed out it lands in its proper
               home rather than in the cwd of the process that
               causes the toss-out!

    Modified: 30-May-91 Scott Wilson - Ignore STDIN and STDOUT
*/
holepath(buf)
char *buf;
  {
    char wd[512];
    char buf2[512];

    if (strcmp(buf,"-") == 0 || strcmp(buf,"#") == 0) return;

    if (buf[0] != '/')
      {
        getcwd(wd,512);
        strcpy(buf2,buf);
        strcpy(buf,wd);
        strcat(buf,"/");
        strcat(buf,buf2);
      }
  }

/*
The following routine was used during debugging and is very handy
for that purpose. It is not called anywhere in the code at this
time - it is left here in case some poor soul gets
to hunt gremlins a little later...
*/
print_cache(sch,ci)
struct shmcache *sch;
struct cacheimg *ci;
  {
    int i;
    struct cacheblk *b;

    printf("Khoros image cache:\n");
    printf("Name: %s\n",sch->region_name);
    printf("Number of images: %d\n",sch->nimgs);
    printf("Max space: %d  Remaining space: %d\n",sch->max_space,sch->remaining_space);
    printf("Freelist pointer: %x\n",sch->freelist);

    printf("Images in cache:\n");
    for (i=0; i<sch->nimgs; i++)
      {
        printf("Image #%d: Name: %s\n",i,ci[i].imgname);
        fflush(stdout);
        printf("First block at: %x, currently at %x\n",ci[i].blkptr,ci[i].curblk);
        fflush(stdout);
        printf("Bytes remaining in curent block: %d\n",ci[i].bytes_remaining_in_blk);
        fflush(stdout);
      }
    b = sch->freelist;
    i = 1;
    while (b->next != NULL)
      {
        i++;
        b = b->next;
      }
    printf("Found %d blocks in freelist, total size: %d\n",i,i*BLKSIZE);
        fflush(stdout);
    printf("\n");
        fflush(stdout);
  }

#endif
