// This may look like C code, but it is really -*- C++ -*-
// 
// Copyright (C) 1988 University of Illinois, Urbana, Illinois
//
// written by Dirk Grunwald (grunwald@cs.uiuc.edu)
//
#include "HardwareContext.h"
#include "CpuMultiplexor.h"
#include "Thread.h"
#include "assert.h"
#include <stream.h>

//
//	Many machines have the same (or very similar) stack format.
//	The 68K & 32K are examples of such machines.
//
//	The const *registers* defines the number of additional longwords
//	needed on the stack past the last frame pointer. This value needs
//	to jibe with the one in HardwareContext-<arch>.s.
//

#ifdef mc68000
static const int registerQuads = (56 + 96)/4;
typedef long Quad;
#endif

#ifdef ns32000
static const int registerQuads = 15;
typedef long Quad;
#endif

const long MagicStackMarker = 0x464f4f20;	// this says 'FOO '

HardwareContext::HardwareContext (int check, unsigned size)
{
    checkStackLimits = check;

    saved_fp = 0;
    saved_sp = 0;
    stackBase = 0;
    stackEnd = 0;
    stackMax = 0;
    stackSize = 0;
    stackCheck = 0;
    stackMallocAt = 0;

    if (size > 0) {
	stackSize = size;
	stackMallocAt = new void *[ stackSize ];
	//
	// stackBase should point to the first writeable cell of the
	// new stack.
	//
	if (StackGrows == stackGrowsDown) {
	    stackEnd = stackMallocAt;
	    stackBase = &stackMallocAt[stackSize-1];
	}
	else {
	    stackBase = stackMallocAt;
	    stackEnd = &stackMallocAt[stackSize-1];
	}
    }
}

void
HardwareContext::switchContext(HardwareContext *to)
{
    if ( checkStackLimits ) {	// check old context
	checkStack();
    }

    magicSwitchTo(to);

    if ( checkStackLimits ) {	// check new context
	checkStack();
    }
}

void
HardwareContext::stackOverflow()
{
    register unsigned depth = stackBase - getSp();
    if (stackMax < depth) {
	stackMax = depth;
    }
    if ( stackMax >= stackSize ) {
	cerr << "\nStack overflow\n";
	cerr << " getSp() = " << hex(long(getSp()));
	cerr << " and stackBase = " << hex(long(stackBase)) << "\n";
	cerr << *this << "\n";
	cerr << "Current task is \n";
	cerr << *(CurrentThread());
	cerr << "\n";
	cerr.flush();
    }
    assert( stackMax < stackSize );
    assert( *stackCheck == MagicStackMarker );
}

#if  defined(mc68000) || defined(ns32000)

StackGrowthDirection StackGrows = stackGrowsDown;

void
HardwareContext::buildReturnFrame(void *returnThis, voidFuncP returnAddress)
{
//
//	To build a thread, we return to the first address of startOff,
//	which will use the current FP & SP to build a local context,
//	and then call the user main.
//
//	startOff needs to have a valid frame. The return address for this
//	frame is NULL, since we never exit procedure startOff.
//

    stackCheck = (long *) stackBase;
    Quad **stack = (Quad **) stackBase;
    *(stack--) = (Quad *) MagicStackMarker;
    *(stack--) = 0;			// return address
    Quad **startOffFp = stack;
    *(stack--) = 0;			// frame pointer

//
//	Construct the stack frame that will be used in procedure startOff.
//	startOff needs to know the value for *this*, which is passed in
//	as the first parameter. It also gets the back-link FP for the
//	last frame (build above).
//

    *(stack--) = (Quad *) returnThis;
    Quad **nullFp = stack;
    *(stack--) = (Quad *) startOffFp;		// FP for previous frame

//
//	Now build the stack frame that is used to return to startOff
//

    *(stack--) = (Quad *) returnAddress;

    saved_fp = (void **) stack;

    *(stack) = (Quad *) nullFp;
    stack -= registerQuads;	// number of Quadword registers we are saving
    saved_sp = (void **) stack;
}
#endif

void
HardwareContext::classPrintOn(ostream& s)
{
    s << "[HardwareContext] Stack spans " << hex(long(stackEnd));
    s << " to " << hex(long(stackBase));
    s << " used is  " << (stackMax) << " of " << stackSize << "\n";
    s << "[HardwareContext] fp = " << hex(long(saved_fp));
    s << " sp = " << hex(long(saved_sp));
    long p = *( (long *) saved_fp );
    s << " @fp = " << hex(p) << "\n";
}
