// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)
    based on code by Marc Shapiro (shapiro@sor.inria.fr)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

#ifdef __GNUG__
#pragma implementation
#endif
#include "LPmsgptr.XPlex.h"


LPmsgptrXPlex:: LPmsgptrXPlex()
{
  lo = fnc = 0;
  csize = DEFAULT_INITIAL_CAPACITY;
  LPmsgptr* data = new LPmsgptr[csize];
  set_cache(new LPmsgptrIChunk(data,  lo, lo, fnc, lo+csize));
  hd = ch;
}

LPmsgptrXPlex:: LPmsgptrXPlex(int chunksize)
{
  if (chunksize == 0) error("invalid constructor specification");
  lo = fnc = 0;
  if (chunksize > 0)
  {
    csize = chunksize;
    LPmsgptr* data = new LPmsgptr[csize];
    set_cache(new LPmsgptrIChunk(data,  lo, lo, fnc, csize));
    hd = ch;
  }
  else
  {
    csize = -chunksize;
    LPmsgptr* data = new LPmsgptr[csize];
    set_cache(new LPmsgptrIChunk(data,  chunksize, lo, fnc, fnc));
    hd = ch;
  }
}


LPmsgptrXPlex:: LPmsgptrXPlex(int l, int chunksize)
{
  if (chunksize == 0) error("invalid constructor specification");
  lo = fnc = l;
  if (chunksize > 0)
  {
    csize = chunksize;
    LPmsgptr* data = new LPmsgptr[csize];
    set_cache(new LPmsgptrIChunk(data,  lo, lo, fnc, csize+lo));
    hd = ch;
  }
  else
  {
    csize = -chunksize;
    LPmsgptr* data = new LPmsgptr[csize];
    set_cache(new LPmsgptrIChunk(data,  chunksize+lo, lo, fnc, fnc));
    hd = ch;
  }
}

void LPmsgptrXPlex::make_initial_chunks(int up)
{
  int need = fnc - lo;
  hd = 0;
  if (up)
  {
    int l = lo;
    do
    {
      int sz;
      if (need >= csize)
        sz = csize;
      else
        sz = need;
      LPmsgptr* data = new LPmsgptr [csize];
      LPmsgptrIChunk* h = new LPmsgptrIChunk(data,  l, l, l+sz, l+csize);
      if (hd != 0)
        h->link_to_next(hd);
      else
        hd = h;
      l += sz;
      need -= sz;
    } while (need > 0);
  }
  else
  {
    int hi = fnc;
    do
    {
      int sz;
      if (need >= csize)
        sz = csize;
      else
        sz = need;
      LPmsgptr* data = new LPmsgptr [csize];
      LPmsgptrIChunk* h = new LPmsgptrIChunk(data,  hi-csize, hi-sz, hi, hi);
      if (hd != 0)
        h->link_to_next(hd);
      hd = h;
      hi -= sz;
      need -= sz;
    } while (need > 0);
  }
  set_cache(hd);
}

LPmsgptrXPlex:: LPmsgptrXPlex(int l, int hi, const LPmsgptr  initval, int chunksize)
{
  lo = l;
  fnc = hi + 1;
  if (chunksize == 0)
  {
    csize = fnc - l;
    make_initial_chunks(1);
  }
  else if (chunksize < 0)
  {
    csize = -chunksize;
    make_initial_chunks(0);
  }
  else
  {
    csize = chunksize;
    make_initial_chunks(1);
  }
  fill(initval);
}

LPmsgptrXPlex::LPmsgptrXPlex(const LPmsgptrXPlex& a)
{
  lo = a.lo;
  fnc = a.fnc;
  csize = a.csize;
  make_initial_chunks();
  for (int i = a.low(); i < a.fence(); a.next(i)) (*this)[i] = a[i];
}

void LPmsgptrXPlex::operator= (const LPmsgptrXPlex& a)
{
  if (&a != this)
  {
    invalidate();
    lo = a.lo;
    fnc = a.fnc;
    csize = a.csize;
    make_initial_chunks();
    for (int i = a.low(); i < a.fence(); a.next(i)) (*this)[i] = a[i];
  }
}


void LPmsgptrXPlex::cache(int idx) const
{
  const LPmsgptrIChunk* tail = tl();
  const LPmsgptrIChunk* t = ch;
  while (idx >= t->fence_index())
  {
    if (t == tail) index_error();
    t = (t->next());
  }
  while (idx < t->low_index())
  {
    if (t == hd) index_error();
    t = (t->prev());
  }
  set_cache(t);
}


void LPmsgptrXPlex::cache(const LPmsgptr* p) const
{
  const LPmsgptrIChunk* old = ch;
  const LPmsgptrIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old) index_error();
  }
  set_cache(t);
}

int LPmsgptrXPlex::owns(Pix px) const
{
  LPmsgptr* p = (LPmsgptr*)px;
  const LPmsgptrIChunk* old = ch;
  const LPmsgptrIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old) { set_cache(t); return 0; }
  }
  set_cache(t);
  return 1;
}


LPmsgptr* LPmsgptrXPlex::dosucc(const LPmsgptr* p) const
{
  if (p == 0) return 0;
  const LPmsgptrIChunk* old = ch;
  const LPmsgptrIChunk* t = ch;
 
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old)  return 0;
  }
  int i = t->index_of(p) + 1;
  if (i >= fnc) return 0;
  if (i >= t->fence_index()) t = (t->next());
  set_cache(t);
  return (t->pointer_to(i));
}

LPmsgptr* LPmsgptrXPlex::dopred(const LPmsgptr* p) const
{
  if (p == 0) return 0;
  const LPmsgptrIChunk* old = ch;
  const LPmsgptrIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->prev());
    if (t == old) return 0;
  }
  int i = t->index_of(p) - 1;
  if (i < lo) return 0;
  if (i < t->low_index()) t = (t->prev());
  set_cache(t);
  return (t->pointer_to(i));
}


int LPmsgptrXPlex::add_high(const LPmsgptr  elem)
{
  LPmsgptrIChunk* t = tl();
  if (!t->can_grow_high())
  {
    if (t->LPmsgptrIChunk::empty() && one_chunk())
      t->clear(fnc);
    else
    {
      LPmsgptr* data = new LPmsgptr [csize];
      t = (new LPmsgptrIChunk(data,  fnc, fnc, fnc,fnc+csize));
      t->link_to_prev(tl());
    }
  }
  *((t->LPmsgptrIChunk::grow_high())) = elem;
  set_cache(t);
  return fnc++;
}

int LPmsgptrXPlex::del_high ()
{
  if (empty()) empty_error();
  LPmsgptrIChunk* t = tl();
  t->LPmsgptrIChunk::shrink_high();
  if (t->LPmsgptrIChunk::empty() && !one_chunk())
  {
    LPmsgptrIChunk* pred = t->prev();
    del_chunk(t);
    t = pred;
  }
  set_cache(t);
  return --fnc - 1;
}

int LPmsgptrXPlex::add_low (const LPmsgptr  elem)
{
  LPmsgptrIChunk* t = hd;
  if (!t->can_grow_low())
  {
    if (t->LPmsgptrIChunk::empty() && one_chunk())
      t->cleardown(lo);
    else
    {
      LPmsgptr* data = new LPmsgptr [csize];
      hd = new LPmsgptrIChunk(data,  lo-csize, lo, lo, lo);
      hd->link_to_next(t);
      t = hd;
    }
  }
  *((t->LPmsgptrIChunk::grow_low())) = elem;
  set_cache(t);
  return --lo;
}


int LPmsgptrXPlex::del_low ()
{
  if (empty()) empty_error();
  LPmsgptrIChunk* t = hd;
  t->LPmsgptrIChunk::shrink_low();
  if (t->LPmsgptrIChunk::empty() && !one_chunk())
  {
    hd = t->next();
    del_chunk(t);
    t = hd;
  }
  set_cache(t);
  return ++lo;
}

void LPmsgptrXPlex::append (const LPmsgptrXPlex& a)
{
  for (int i = a.low(); i < a.fence(); a.next(i)) add_high(a[i]);
}

void LPmsgptrXPlex::prepend (const LPmsgptrXPlex& a)
{
  for (int i = a.high(); i > a.ecnef(); a.prev(i)) add_low(a[i]);
}

void LPmsgptrXPlex::reverse()
{
  LPmsgptr tmp;
  int l = lo;
  int h = fnc - 1;
  LPmsgptrIChunk* loch = hd;
  LPmsgptrIChunk* hich = tl();
  while (l < h)
  {
    LPmsgptr* lptr = loch->pointer_to(l);
    LPmsgptr* hptr = hich->pointer_to(h);
    tmp = *lptr;
    *lptr = *hptr;
    *hptr = tmp;
    if (++l >= loch->fence_index()) loch = loch->next();
    if (--h < hich->low_index()) hich = hich->prev();
  }
}

void LPmsgptrXPlex::fill(const LPmsgptr  x)
{
  for (int i = lo; i < fnc; ++i) (*this)[i] = x;
}

void LPmsgptrXPlex::fill(const LPmsgptr  x, int l, int hi)
{
  for (int i = l; i <= hi; ++i) (*this)[i] = x;
}


void LPmsgptrXPlex::clear()
{
  if (fnc != lo)
  {
    LPmsgptrIChunk* t = tl();
    while (t != hd)
    {
      LPmsgptrIChunk* prv = t->prev();
      del_chunk(t);
      t = prv;
    }
    t->LPmsgptrIChunk::clear(lo);
    set_cache(t);
    fnc = lo;
  }
}


int LPmsgptrXPlex::OK () const
{
  int v = hd != 0 && ch != 0;     // at least one chunk

  v &= fnc == tl()->fence_index();// last chunk fence == plex fence
  v &= lo == ((hd))->LPmsgptrIChunk::low_index();    // first lo == plex lo

// loop for others:
  int found_ch = 0;                   // to make sure ch is in list;
  const LPmsgptrIChunk* t = (hd);
  for (;;)
  {
    if (t == ch) ++found_ch;
    v &= t->LPmsgptrIChunk::OK();              // each chunk is OK
    if (t == tl())
      break;
    else                              // and has indices contiguous to succ
    {
      v &= t->top_index() == t->next()->base_index();
      if (t != hd)                  // internal chunks full
      {
        v &= !t->empty();
        v &= !t->can_grow_low();
        v &= !t->can_grow_high();
      }
      t = t->next();
    }
  }
  v &= found_ch == 1;
  if (!v) error("invariant failure");
  return v;
}
