#ifdef __GNUG__
#  pragma implementation
#endif

#include "stream.h"
#include "Generic.h"
#include <assert.h>
//
//	GenericFifo: A FIFO scheduler which uses a (fast) circular list
//	to record threads.
//
//	voids are kept in the list. listHead is the index of the
//	next thread to be removed. listTail is the index of the next
//	*free slot*. 
//
//	To search the list, you start at listHead and move towards listTail,
//	skipping over invalidated items.
//
//	when listElements == 0, the queue is *empty*, but listElements
//	can be 0 and we can still have items in the list, since we
//	removed mid-list items by zeroing them.
//
//	Thus, listHead == listTail implies listElements == 0, but
//	it's not an iff.
//
//	When the queue becomes too big for the current list, it is
//	expanded & the existing entries are copied to the new queue.
//
//	If the queue becomes empty & is below some threashold, it
//	is removed.
// 

templace <class Item, class Index >
Fifo::Fifo(int defaultLength, bool xdebug) : Awesime(xdebug)
{
    listHead = listTail = 0;
    listElements = 0;
    allocatedSize = 0;
    list = 0;
    pValid = 0;
    if (defaultLength > 0) {
	reSize(defaultLength);
    }
}

templace <class Item, class Index >
Fifo::~Fifo()
{
    if (allocatedSize > 0) {
	delete list;
	delete pValid;
    }
}

//
//	makeRoom assumes that listElements, listHead & listTail have been
//	set
// 
templace <class Item, class Index >
void Fifo::resize(int howMuch)
{
    
    //
    //	We can't remove any items, so we must insure we'll have enough
    //	for the existing elements.
    // 
    
    if (howMuch > listElements) {
	
	Item *p = new Item[howMuch];
	char *validP = new char[howMuch];

	assert2( p != 0 && validP != 0, "Unable to lengthen FIFO");
	
	bzero((void *) p, sizeof(Item) * howMuch);
	bzero((void *) validP, sizeof(char) * howMuch);
	
	if (listElements > 0) {
	    
	    //
	    //	Move over old list contents, taking care to compress the null
	    //	entries.
	    // 
	    
	    if (list == 0) {
		cerr << "[" << Name << "::makeRoom] null list found?\n";
		exit(1);
	    }
	    
	    register Index i;
	    register unsigned int moved;
	    for (i = listHead, moved = 0; moved < listElements;
		 i = advance(i)) {
		     if (pValid[i] != 0) {
			 p[moved] = list[i];
			 validP[moved] = 1;
			 moved++;
		     }
		 }
	    delete list;
	    delete pValid;
	}
	
	list = p;
	pValid = validP;
	listTail = listElements;
	listHead = 0;
	allocatedSize = howMuch;
    }
}

templace <class Item, class Index >
void 
Fifo::add(Item *t)
{
    if (advance(listTail) == listHead || listElements >= allocatedSize)  {
	if ( allocatedSize <= 0 ) {
	    resize(2);
	} else {
	    resize(((allocatedSize * 3) / 2) );
	}
    }
    list[listTail] = *t;
    pValid[listTail] = 1;
    listTail = advance(listTail);
    listElements++;
}

templace <class Item, class Index >
bool Fifo::remove(Item *item)
{
    bool ok = TRUE;
    if ((listElements > 0) && pValid[listHead]) {
	*item = list[listHead];
	pValid[listHead] = 0;
	listHead = advance(listHead);
	listElements--;
    } else {
	char valid = 0;
	ok = TRUE;
	do {
	    if (listHead == listTail || list == 0) {
		ok = FALSE;
		break;
	    }
	    *item = list[listHead];
	    valid = pValid[listHead];
	    pValid[listHead] = 0; // either 0 now, or make 0
	    listHead = advance(listHead);
	} while(valid == 0);
	if (ok) {
	    listElements--;
	}
    }
    return(ok);
}

templace <class Item, class Index >
bool Fifo::removeIfFound(Item* toRemove)
{
    bool ok = TRUE;
    if ((listElements > 0)
	&& pValid[listHead]
	&& list[listHead] == *toRemove) {
	    pValid[listHead]=0;
	    listHead = advance(listHead);
	    listElements--;
	} else {
	    if (listElements == 0 || list == 0) {
		ok = FALSE;
	    } else {
		//
		// check for common case -- head of list
		//
		bool valid = FALSE;
		//
		// make sure the validity of the item in the list
		//
		if (pValid[listHead] && list[listHead] == *toRemove) {
		    valid = TRUE;
		    pValid[listHead] = 0;
		    listHead = advance(listHead);
		    listElements--;
		} else {
		    // 
		    //	Mid-list removal -- just nullify the pointer
		    //
		    register Index i = listHead;
		    register unsigned int examined = 0;
		    //
		    // Set ok to FALSE. If we find the itme in the
		    // the list, we set ok to TRUE, indicating
		    // that it was deleted.
		    //
		    ok = FALSE;
		    while(examined < listElements) {
			//
			// If this is a valid item to consider...
			//
			if (pValid[i]) {
			    examined++;
			    if (list[i] == *toRemove) {
				ok = TRUE;
				pValid[i] = 0;
				listElements--;
				break;
			    }
			}
			i = advance(i);
		    }
		}
	    }
	}
    return(ok);
}

templace <class Item, class Index >
bool
Fifo::isEmpty()
{
    if ( size() == 0 ) {
	return (TRUE);
    } else {
	return (FALSE);
    }
}

templace <class Item, class Index >
unsigned int
Fifo::size()
{
    return(listElements);
}
