// This may look like C code, but it is really -*- C++ -*-
// 
// Copyright (C) 1988 University of Illinois, Urbana, Illinois
//
// written by Dirk Grunwald (grunwald@cs.uiuc.edu)
//

#ifdef __GNUG__
#  pragma implementation
#endif

#include "MultiSimMux.h"
#include "CpuMuxP.h"
#include "SpinLock.h"
#include "SpinBarrier.h"
#include "SpinFetchAndOp.h"
#include "AwesimeHeap.h"
#include "Thread.h"

#include "SEvSplayPQ.h"

#include "ReserveByException.h"
#include <math.h>

static SpinBarrier CpuBarrier(1);

//
//	The CurrentEvents queue is private to each processor
//

static SEvPQ *CurrentEvents[_MAXIMUM_CPU_MUXS_];
static SpinLock CurrentEventsLock[_MAXIMUM_CPU_MUXS_];
static int CurrentEventsCounter[_MAXIMUM_CPU_MUXS_];

static inline void
AddToCpu(int cpu, SEv& whut)
{
    CERR_PRE;
    cerr << CpuMux::Cpu() -> name();
    cerr << " add " << (who.thread()) -> name() << " to pending at";
    cerr << who.time() << "\n";
    CERR_POST;

    CurrentEventsLock[cpu].reserve();
    CurrentEvents[cpu] -> add( whut );

    assert( CurrentEventsCounter[cpu] >= 0 );

    CurrentEventsCounter[cpu]++;
    GlobalCurrentEventsCounter.add(1);

    CurrentEventsLock[cpu].release();
}

//
// Possibly remove something from a given CPU queue; we err on the 
// side of optimization. A thread is only removed if it 
//
static inline Thread *
RemoveFromCpu(int cpu)
{
    Thread *x = 0;

    if ( CurrentEventsCounter[cpu] != 0 
	&& CurrentEventsLock[cpu].reserveNoBlock() ) {

	assert( CurrentEventsCounter[cpu] >= 0 );

	if ( CurrentEventsCounter[cpu] > 0 ) {
	    SEv& front = CurrentEvents[cpu] -> front();

	    if ( CurrentSimulatedTime <= front.time() ) {
		x = front.thread()
		CurrentEvents[cpu] -> del_front();

		CurrentEventsCounter[cpu]--;
		GlobalCurrentEventsCounter.add(-1);
	    }
	} 
	CurrentEventsLock[cpu].release();
    }
    return( x );
}

MultiSimMux::MultiSimMux(bool debug) : SimMux(debug)
{
    pNameTemplate = "SimMux";
    iYam = 0;
    sprintf(nameSpace, "[%s-%d] ", pNameTemplate, iYam);
    pName = nameSpace;
    CpuMux::Debug = debug;
}

void
MultiSimMux::allocateLocalEventStructures(int newIYam, int outOf)
{
    CERR_PRE;
    cerr << name() << "Allocate SimMux structures for " << newIYam << "\n";
    CERR_POST;
    
    iYam = newIYam;
    sprintf(nameSpace, "[%s-%d] ", pNameTemplate, iYam);
    pName = nameSpace;

    CurrentEventsCounter[iYam] = 0;
    CurrentEvents[iYam] = new SEvSplayPQ;
    
    myCurrentEvents = CurrentEvents[iYam];
    myCurrentEventsLock = &CurrentEventsLock[iYam];
    myCurrentEventsCounter = &CurrentEventsCounter[iYam];

    CpuMux::MuxsLock.reserve();
    CpuMux::Muxs = outOf;
    CpuMux::MuxsLock.release();

    GivingUpLock.reserve();
    if (GivingUpCounter >= outOf) {
	GivingUpGeneration++;
	GivingUpCounter = 0;
    }
    GivingUpLock.release();
}

void
MultiSimMux::deallocateEventStructures()
{
    assert( iYam != 0 );

    CERR_PRE;
    cerr << name() << "Deallocate CpuMux structures for " << iYam << "\n";
    CERR_POST;
    
    //
    // Move remaining events to another queue. We're not adding new events,
    // just moving them around, so we don't increase GlobalCurrentEventsCounter
    //

    myCurrentEventsLock -> reserve();

    while ( ! myCurrentEvents -> empty() ) {
	assert( iYam != 0 );
	SEv& item = myCurrentEvents -> deq();
	AddToCpu(0, item);
    }
    myCurrentEventsLock -> release();

    CpuMux::MuxsLock.reserve();
    CpuMux::Muxs--;

    GivingUpLock.reserve();
    if (GivingUpCounter >= CpuMux::Muxs) {
	GivingUpGeneration++;
	GivingUpCounter = 0;
    }
    GivingUpLock.release();
    
    CpuMux::MuxsLock.release();

    CERR_PRE;
    cerr << name() << "set CpuMux::Muxs to " << CpuMux::Muxs;
    cerr << " and trigger GivingUp\n";
    CERR_POST;


    delete myCurrentEvents;
    myCurrentEvents = 0;
    CurrentEvents[iYam] = 0;
    CurrentEventsCounter[iYam] = 0;
    myCurrentEventsLock -> release();
}

void
MultiSimMux::fireItUp( int cpus, unsigned shared)
{
    //
    // This is copied from MultiCpuMux.cc
    //

    assert(cpus > 0);
    
    if ( cpus > _MAXIMUM_CPU_MUXS_ ) {
	cpus = _MAXIMUM_CPU_MUXS_;
    }
    
    CERR_PRE;
    cerr << name() << "Allocate " << shared << " bytes of shared memory\n";
    CERR_POST;
    
    if ( cpus > 1 ) {
	extern void SharedMemoryInit( unsigned );
	SharedMemoryInit( shared );
    }
    
    //
    // Set the barrier height so everyone can rendezvous..
    //
    CpuBarrier.height(cpus);
    
    warmThePot(cpus);
    
    CERR_PRE;
    cerr << name() << " returns from warmThePot, join barrier\n";
    CERR_POST;
    
    CpuBarrier.rendezvous();
    
    stirItAround();
    coolItDown();
}

void
MultiSimMux::warmThePot(int cpus)
{
    //
    // copied from MultiCpuMux.cc
    //

    assert(cpus > 0);
    CpuMux::Muxs = cpus;
    enabled = 1;
    
    //
    //	Spawn the children, giving each a unique number from 0..(cpus-1).
    //  The first child gets id (cpus-1), and the original process gets 0.
    //
    
    iYam = 0;

    CERR_PRE;
    cerr << name() << "Allocate " << CpuMux::Muxs << " cpus\n";
    CERR_POST;
    
    for (int whoAmI = 1; whoAmI < CpuMux::Muxs; whoAmI++) {
	if (iYam == 0) {
	    int childPid = fork();
	    
	    if (childPid == 0) {	// child 
		
		CERR_PRE;
		cerr << getpid() << " is born,";
		CERR_POST;
		
		allocateLocalEventStructures(whoAmI, CpuMux::Muxs);
		break;
	    } else {
		
		CERR_PRE;
		cerr << name() << "Parent spawns child "<< childPid << "\n";
		CERR_POST;
		
		if ( pid == -1 ) {
		    cerr << "Error in spawn:\n";
		    perror("fork");
		    exit(99);
		}
	    }
	}
    }
    pid = getpid();
    
    CERR_PRE;
    cerr << name() << "I am now id " << iYam << " and pid " << pid <<" \n";

    //
    // give each child a distinct temporary directory for core dumps
    // when debugging.
    //

    char templateName[L_tmpnam];
    tmpnam(templateName);
    mkdir(templateName,0777);
    int xx = chdir(templateName);

    cerr << name() << "change dir to " << templateName;
    if ( xx >= 0 ) {
	cerr << " worked\n";
    } else {
	cerr << " did not work\n";
    }

    CERR_POST;
}

void
MultiSimMux::coolItDown()
{
    if (iYam > 0) {
	
	CERR_PRE;
	cerr << name() << "exit\n";
	CERR_POST;
	
	deallocateEventStructures();
	_exit(0);
    }
    else {
	//
	//	reap the dead children. This way we know they are all dead.
	//	The caller can then safely exit.
	//
	while (CpuMux::Muxs > 1) {
	    int pid = wait(0);
	    if (pid == -1) {
		perror("wait");
		break;
	    }
	}
	//
	//  In case of break in above loop
	//
	CpuMux::Muxs = 1;
    }
}

void
MultiCpuMux::add_(Thread *who)
{
    AddToCpu(iYam, SEv(who, CurrentSimulatedTime) );
}

void
MultiSimMux::addAt(Thread *who, double when)
{
    if (when <= CurrentSimulatedTime) {
	when = CurrentSimulatedTime;
    }

    AddToCpu(iYam, SEv(who, when) );
}

void
MultiSimMux::addAt(OEv *who, double when)
{
    if (when <= CurrentSimulatedTime) {
	when = CurrentSimulatedTime;
    }

    AddToCpu(iYam, SEv(who, when) );
}

void
MultiSimMux::await_(SimTimeUnit when)
{
    CERR_PRE;
    cerr << name() << " " << currentThread -> name();
    cerr << " awaits " << when << " at " << CurrentSimulatedTime << "\n";
    CERR_POST;
    
    if (when > CurrentSimulatedTime) {
	AddToCpu(iYam, SEv(currentThread, when) );
	CpuMux::Cpu() -> raise( &(CpuMux::Cpu() -> iveSuspendedException) );
    }
}

void
MultiSimMux::stirItAround()
{
    currentEvent = 0;
    
    if (!enabled) {
	cerr << "Need to initialize CpuMux before using it\n";
    }
    
    while( ! *terminated ) {
	while ( currentEvent == 0 ) {
	    
	    currentEvent = remove();
	    
	    //
	    // run if we got one
	    //
	    if (currentEvent != 0) break;

	    //
	    // reloop if some should exist
	    //
	    if ( GlobalCurrentEventsCounter.value() > 0 ) continue;
	    
	    CERR_PRE;
	    cerr << name() << "check if I should quit\n";
	    CERR_POST;
	    
	    GivingUpLock.reserve();
	    
	    GivingUpCounter++;
	    
	    CERR_PRE;
	    cerr << name() << GivingUpCounter;
	    cerr << " CPUs attempting to give up\n";
	    CERR_POST;
	    
	    assert( GivingUpCounter > 0);
	    assert( GivingUpCounter <= CpuMux::Muxs);
	    
	    if ( GivingUpCounter == CpuMux::Muxs 
		&& GlobalCurrentEventsCounter.value() == 0)
	    {
		
		GivingUpGeneration ++;
		GivingUpCounter = 0;
		GivingUpLock.release();
		
		CERR_PRE;
		cerr << name() << "give up\n";
		cerr << name() << " GCEC = " << GlobalCurrentEventsCounter.value() << "\n";
		cerr << name() << "my CEC = " << *myCurrentEventsCounter << "\n";
		CERR_POST;
		
		return;
	    }
	    else {
		
		//
		// Record the current generation of the the giving up
		// barrier -- we're going to give up only if every
		// one else agrees to give up and nothing new to do
		// comes along in the meantime.
		//
		
		int generation = GivingUpGeneration;
		VolatileInt *genp = &GivingUpGeneration;
		
		GivingUpLock.release();
		
		CERR_PRE;
		cerr << name() << " *genp = " << *genp << "\n";
		cerr << name() << " generation = " << generation << "\n";
		CERR_POST;

		while( generation == *genp
		      && GlobalCurrentEventsCounter.value() == 0
		      && !*terminated );
		

		GivingUpLock.reserve();
		if ( *genp != generation || *terminated ) {

		    CERR_PRE;
		    cerr << name() << " *genp = " << *genp << "\n";
		    cerr << name() << " generation = " << generation << "\n";
		    cerr << name() << " GCEC = " << GlobalCurrentEventsCounter.value() << "\n";
		    cerr << name() << " *terminated = " << *terminated << "\n";
		    cerr << name() << " giving up\n";
		    cerr << name() << "my CEC = " << *myCurrentEventsCounter << "\n";
		    CERR_POST;
		    
		    GivingUpLock.release();
		    return;
		}
		else {
		    
		    CERR_PRE;
		    cerr << name() << " bail out\n";
		    CERR_POST;

		    GivingUpCounter--;
		    assert(GivingUpCounter >= 0);
		    GivingUpLock.release();
		    
		    CERR_PRE;
		    cerr << name() << " check for something\n";
		    cerr << name() << " i have " << *myCurrentEventsCounter;
		    cerr << " events\n" ;
		    
		    cerr << name() << " out of " ;
		    cerr << GlobalCurrentEventsCounter.value() << "\n";
		    CERR_POST;
		}
	    }
	}
	
	CERR_PRE;
	cerr << name() << " switch to ";
	cerr << currentEvent->name() << "\n";
	CERR_POST;
	
#ifdef DEBUG_MALLOC
	assert( malloc_verify() );
#endif DEBUG_MALLOC
	systemContext.switchContext(&(currentEvent -> pContext));
#ifdef DEBUG_MALLOC
	assert( malloc_verify() );
#endif DEBUG_MALLOC
	
	assert(raisedBy != 0);
	raisedBy -> handleException();
	raisedBy = 0;
    }
}
