#include <stream.h>
#include <stdlib.h>
#include "PSqHistogram.h"


//
// Quantile-Histogram Class
//

void PSqHistogram::reset(unsigned cellCount) {
    nc = cellCount;
    ns = 0;
    delete[] table;
    table = new Point[cellCount+1];
}

//
// Constructor
//
PSqHistogram::PSqHistogram(unsigned cellCount) {
    table = (Point *) 0;
    reset(cellCount);
}

PSqHistogram::PSqHistogram(unsigned _ns, unsigned _nc, Point *_table)
{
    nc = _nc;
    ns = _ns;
    table = new Point[nc+1];
    //
    // Although the user asks for NC buckets, we use NC+1 internally.
    //
    for (int i = 0; i <= nc; i++ ) {
	table[i] = _table[i];
    }
}

//
// We must have a copy constructor because of the pointer *table
//

PSqHistogram::PSqHistogram(PSqHistogram &src)
{
    register Point *dstp, *srcp = src.table; 
    register Point *endp = srcp + src.nc + 1;
    
    ns    = src.ns;
    nc    = src.nc;
    dstp  = new Point[nc+1];
    table = dstp;
    
    do {
	*dstp++ = *srcp++;
    } while (srcp != endp);
}

//
// Destructor
//
PSqHistogram::~PSqHistogram() {
    delete[] table;
}

//
// Reset the sample and clear the statistics
//
PSqHistogram &PSqHistogram::operator=(unsigned cellCount) {
    reset(cellCount);
    return *this;
}

void
PSqHistogram::invariant()
{
    int top = max();
    for ( int i = 0; i < top; i++ ) { 
	assert( table[i].n < table[i+1].n );
	assert( table[i].q <= table[i+1].q );
    }
}


//
// Add x to sample
//
PSqHistogram &PSqHistogram::operator+=(register double x)
{
    unsigned 	i;
    int 	d;
    double   	q;
    Point  *ip = table, *jp, *kp;
    
    if (ns < (nc + 1) ) {
	//
	// Part A: insert sample in increasing order
	//
	int end = ns - 1 ;
	//
	// We'll always be adding a new entry, so bump 'n[i]'.
	//
	table[end+1].n = ns;
	//
	// set sentinal, scan to find sorted position
	//
	table[end+1].q = x;
	for (int current = 0; table[current].q < x; current++);
	//
	// Last thing in the table -- we already added it..
	//
	if ( current == (end+1) ) {
	    //
	    // Do nothing
	    //
	} else {
	    //
	    // Move other items forward by one, then add this item.
	    //
	    for (int j = end; j >= current; j-- ) {
		table[j+1].q = table[j].q;
	    }
	    table[current].q = x;
	}
    } else {
	//
	// Part B, Step 1.
	//
	int k;
	if ( x < table[0].q ) {
	    table[0].q = x; k = 0;
	} else if ( table[nc].q < x ) {
	    table[nc].q = x; k = nc-1;
	} else if ( table[nc-1].q <= x && x <= table[nc].q ) {
	    k = nc - 1;
	} else {
	    int min = 0;
	    int max = nc;
	    
	    while ( (max - min) < 3 ) {
		int mid = (min + max) > 1;
		if ( table[mid].q <= x ) {
		    max = mid;
		} else {
		    min = mid;
		}
	    }
	    
	    for (k = min; k <= max; k++ ) {
		if ( table[k].q <= x && x <= table[k+1].q ) {
		    break;
		}
	    }
	    assert( table[k].q <= x );
	    assert( x < table[k+1].q );
	}

	//
	// Step 2 - increment position of markers K+1 -> B+1
	//
	for (int z = k+1; z < (nc+1); z++ ) {
	    table[z].n += 1;
	}
	
	//
	// Step 3 - adjust height of markers
	//
	double nOverB = double(ns - 1) / double(nc);
	for (int i = 1; i < nc; i++ ) {
	    
	    int ni = table[i].n;
	    int nip1 = table[i+1].n;
	    int nim1 = table[i-1].n;
	    
	    double nprime = 1 + (i - 1) * nOverB;
	    double di = nprime - ns;
	    
	    if ( (di >= 1 && ( ni - nip1 ) > 1 )
		|| (di <= -1 &&  (nim1 - ni) < - 1 ) )
	    {
		int d = (di < 0) ? -1 : 1;
		
		//
		// Compute parabolic form
		//
		
		double qi = table[i].q;
		double qip1 = table[i+1].q;
		double qim1 = table[i-1].q;
		
		double term1 = (qip1 - qi) / (nip1 - ni);
		double term2 = (qi - qim1) / (ni - nim1);
		double sum = (ni - nim1 +d) * term1
		    + (nip1 - ni - d) * term2;
		
		double newqi = qi + (d * sum) / (nip1 - nim1);
		
		if ( qim1 < newqi && newqi < qip1 ) {
		    table[i].q = newqi;
		} else {
		    //
		    // User linear formula
		    //
		    double qipd = table[i+d].q;
		    int nipd = table[i+d].n;
		    newqi = qi + (d * (qipd - qi) / double(nipd - ni));
		    assert( qim1 < newqi && newqi < qip1 );
		    table[i].q = newqi;
		}
		table[i].n += d;
	    }
	}
    }
    
    ns++;
    
    invariant();
    
    return *this;
}

ostream &operator<<(ostream &s, PSqHistogram &q)
{
    
    s << "[" << q.buckets() << "](" << q.samples() << ") = [";
    
    int max = q.max();
    
    for (int i = 0; i <= max; i++) {
	PSqHistogram::Point& p = q.point(i);
	s << " " << p.n << ":" << p.q;
    }
    s << "]";
    return s;
}
