// This may look like C code, but it is really -*- C++ -*-
// 
// Copyright (C) 1988 University of Illinois, Urbana, Illinois
//
// written by Dirk Grunwald (grunwald@cs.uiuc.edu)
//

#ifdef __GNUG__
#  pragma implementation
#endif

#include "SingleSimMux.h"
#include "CpuMuxP.h"
#include "Thread.h"
#include "SEv.SplayPQ.h"
#include "OEv.h"
#include "ReserveByException.h"
#include <math.h>


#ifdef NDEBUG
#  define CERR_PRE if (0) {
#  define CERR_POST }
#else 
#  define CERR_PRE if (CpuMux::Debug) { CpuMux::CerrLock.reserve();
#  define CERR_ALWAYS_PRE { CpuMux::CerrLock.reserve();
#  define CERR_POST cerr.flush(); CpuMux::CerrLock.release();}
#endif

SingleSimMux::SingleSimMux(bool debug) : SimMux(debug)
{
    ThisSimMux = this;
    ThisCpu = this;
    
    pNameTemplate = "SingleSimMux";
    iYam = 0;
    sprintf(nameSpace, "[%s-%d] ", pNameTemplate, iYam);
    pName = nameSpace;
    
    myEvents = new SEvSplayPQ();
}

SingleSimMux::~SingleSimMux()
{
    // nada
}

void
SingleSimMux::fireItUp(int, unsigned)
{
    pid = getpid();
    enabled = 1;
    
    while( ! *terminated ) {
	
	if ( myEvents -> empty() ) {
	    return;
	} else {
	    SEv ev = myEvents -> deq();
	    CurrentSimulatedTime = ev.time();
	    
	    //
	    // if it's not a thread event, execute the OE
	    //
	    if ( ev.isOtherEvent() ) {
		currentThread = 0;
		ev.oev() -> executeEvent();
	    }
	    else {
		currentThread = ev.thread();
		
		CERR_PRE;
		cerr << CpuMux::Cpu() -> name();
		cerr << " switchTo to " << currentThread -> name();
		cerr << " at " << CurrentSimulatedTime << "\n";
		CERR_POST;
		
		systemTransfer( currentThread );

		raisedBy -> handleException();
		raisedBy = 0;
	    }
	}
    }
}

void
SingleSimMux::add_(Thread *who)
{
    myEvents -> enq( SEv(who, CurrentSimulatedTime) );
}

void
SingleSimMux::addAt_(Thread *who, SimTimeUnit when)
{
    if (when <= CurrentSimulatedTime) {
	when = CurrentSimulatedTime;
    }

    CERR_PRE;
    cerr << CpuMux::Cpu() -> name();
    cerr << " add " << who -> name() << " to pending\n";
    CERR_POST;

    //
    // Add them to pending events
    //
    myEvents->enq( SEv(who, when) );
}

void
SingleSimMux::addAt_(OEv *who, SimTimeUnit when)
{
    if (when <= CurrentSimulatedTime) {
	when = CurrentSimulatedTime;
    }

    CERR_PRE;
    cerr << CpuMux::Cpu() -> name();
    cerr << " add OEv to pending\n";
    CERR_POST;

    //
    // Add them to pending events
    //
    myEvents->enq( SEv(who, when) );
}


void
SingleSimMux::await_(SimTimeUnit when)
{
    CERR_PRE;
    cerr << name() << " " << currentThread -> name();
    cerr << " awaits " << when << " at " << CurrentSimulatedTime << "\n";
    CERR_POST;
    
    if (when > CurrentSimulatedTime) {
	
	if ( myEvents -> empty() ) {
	    CurrentSimulatedTime = when;
	} else {
	    
	    SEv& front = myEvents -> front();
	    //
	    // is the current one awaiting a time sooner than any other one?
	    //
	    
	    if ( when <= front.time() ) { 
		CurrentSimulatedTime = when;

		CERR_PRE;
		cerr << CpuMux::Cpu() -> name();
		cerr << " (await) bump time to " << when << "\n";
		CERR_POST;

	    } else {
		
		//
		// if it's an event, not a thread, suspend and let the
		// scheduler do it in the UNIX process context.
		//
		
		if  ( front.isOtherEvent() ) {
		    
		    myEvents -> enq( SEv(currentThread, when) );
		    CpuMux::Cpu() -> raise( &iveSuspendedException );

		} else { 
		    Thread *wuz = currentThread;
		    CurrentSimulatedTime = front.time();
		    currentThread = front.thread();
		    myEvents -> del_front();
		    
		    myEvents -> enq( SEv(wuz, when) );
		    
		    CERR_PRE;
		    cerr << CpuMux::Cpu() -> name();
		    cerr << "(await) transfer to " << currentThread -> name() << "\n";
		    CERR_POST;

		    threadTransfer(wuz, currentThread);
		}
	    }
	}
    }
}
