// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)
    based on code by Marc Shapiro (shapiro@sor.inria.fr)

This file is part of the GNU C++ Library.  This library is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This library is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef _ThreadPtrPlex_h
#ifdef __GNUG__
#pragma interface
#endif
#define _ThreadPtrPlex_h 1

#include <std.h>
#include <Pix.h>
#include "ThreadPtr.defs.h"

// Plexes are made out of ThreadPtrIChunks

#include <stddef.h>

class ThreadPtrIChunk
{
//public: // kludge until C++ `protected' policies settled
protected:      

  ThreadPtr*           data;           // data, from client

  int            base;           // lowest possible index
  int            low;            // lowest valid index
  int            fence;          // highest valid index + 1
  int            top;            // highest possible index + 1

  ThreadPtrIChunk*     nxt;            // circular links
  ThreadPtrIChunk*     prv;

public:

// constructors

                 ThreadPtrIChunk(ThreadPtr*     d,       // ptr to array of elements
                        int      base_idx, // initial indices
                        int      low_idx,  
                        int      fence_idx,
                        int      top_idx);

  virtual       ~ThreadPtrIChunk();

// status reports

  int            size() const;    // number of slots

  virtual int    empty() const ;
  virtual int    full() const ; 

  int            can_grow_high () const ;  // there is space to add data
  int            can_grow_low () const;        
  
  int            base_index() const;   // lowest possible index;
  int            low_index() const;    // lowest actual index;
  virtual int    first_index() const;  // lowest valid index or fence if none
  virtual int    last_index() const;   // highest valid index or low-1 if none
  int            fence_index() const;  // highest actual index + 1
  int            top_index() const;    // highest possible index + 1

// indexing conversion

  int            possible_index(int i) const; // i between base and top
  int            actual_index(int i) const;   // i between low and fence
  virtual int    valid_index(int i) const;    // i not deleted (mainly for mchunks)

  int            possible_pointer(const ThreadPtr* p) const; // same for ptr
  int            actual_pointer(const ThreadPtr* p) const; 
  virtual int    valid_pointer(const ThreadPtr* p) const; 

  ThreadPtr*           pointer_to(int i) const ;   // pointer to data indexed by i
                                      // caution: i is not checked for validity
  int            index_of(const ThreadPtr* p) const; // index of data pointed to by p
                                      // caution: p is not checked for validity

  virtual int    succ(int idx) const;     // next valid index or fence if none
  virtual int    pred(int idx) const;     // previous index or low - 1 if none

  virtual ThreadPtr*   first_pointer() const;   // pointer to first valid pos or 0
  virtual ThreadPtr*   last_pointer() const;    // pointer to first valid pos or 0
  virtual ThreadPtr*   succ(ThreadPtr*  p) const;     // next pointer or 0
  virtual ThreadPtr*   pred(ThreadPtr* p) const;     // previous pointer or 0

// modification

  virtual ThreadPtr*   grow_high ();      // return spot to add an element
  virtual ThreadPtr*   grow_low ();  

  virtual void   shrink_high ();    // logically delete top index
  virtual void   shrink_low ();     

  virtual void   clear(int lo);     // reset to empty ch with base = lo
  virtual void   cleardown(int hi); // reset to empty ch with top = hi
  void           re_index(int lo);  // re-index so lo is new low

// chunk traversal

  ThreadPtrIChunk*     next() const;
  ThreadPtrIChunk*     prev() const;

  void           link_to_prev(ThreadPtrIChunk* prev);
  void           link_to_next(ThreadPtrIChunk* next);
  void           unlink();

// state checks

  ThreadPtr*           invalidate();     // mark self as invalid; return data
                                   // for possible deletion

  virtual int    OK() const;             // representation invariant

  void   error(const char*) const;
  void   empty_error() const;
  void   full_error() const;
  void   index_error() const;
};

// ThreadPtrPlex is a partly `abstract' class: few of the virtuals
// are implemented at the Plex level, only in the subclasses

class ThreadPtrPlex
{
protected:      

  ThreadPtrIChunk*       hd;          // a chunk holding the data
  int              lo;          // lowest  index
  int              fnc;         // highest index + 1
  int              csize;       // size of the chunk

  void             invalidate();              // mark so OK() is false
  void             del_chunk(ThreadPtrIChunk*);        // delete a chunk

  ThreadPtrIChunk*       tl() const;                // last chunk;
  int              one_chunk() const;         // true if hd == tl()

public:

// constructors, etc.

                    ThreadPtrPlex();                  // no-op

  virtual           ~ThreadPtrPlex();

  
// Access functions 
    
  virtual ThreadPtr&      operator [] (int idx) = 0; // access by index;
  virtual ThreadPtr&      operator () (Pix p) = 0;   // access by Pix;

  virtual ThreadPtr&      high_element () = 0;      // access high element
  virtual ThreadPtr&      low_element () = 0;       // access low element

// read-only versions for const Plexes

  virtual const ThreadPtr& operator [] (int idx) const = 0; // access by index;
  virtual const ThreadPtr& operator () (Pix p) const = 0;   // access by Pix;

  virtual const ThreadPtr& high_element () const = 0;      // access high element
  virtual const ThreadPtr& low_element () const = 0;       // access low element


// Index functions

  virtual int       valid (int idx) const = 0;      // idx is an OK index

  virtual int       low() const = 0;         // lowest index or fence if none
  virtual int       high() const = 0;        // highest index or low-1 if none

  int               ecnef() const;         // low limit index (low-1)
  int               fence() const;         // high limit index (high+1)

  virtual void      prev(int& idx) const= 0; // set idx to preceding index
                                          // caution: pred may be out of bounds

  virtual void      next(int& idx) const = 0;       // set to next index
                                          // caution: succ may be out of bounds

  virtual Pix       first() const = 0;        // Pix to low element or 0
  virtual Pix       last() const = 0;         // Pix to high element or 0
  virtual void      prev(Pix& pix) const = 0;  // preceding pix or 0
  virtual void      next(Pix& pix) const = 0;  // next pix or 0
  virtual int       owns(Pix p) const = 0;     // p is an OK Pix

// index<->Pix 

  virtual int       Pix_to_index(Pix p) const = 0;   // get index via Pix
  virtual Pix       index_to_Pix(int idx) const = 0; // Pix via index

// Growth

  virtual int       add_high(const ThreadPtr  elem) =0;// add new element at high end
                                                // return new high

  virtual int       add_low(const ThreadPtr  elem) = 0;   // add new low element,
                                                // return new low

// Shrinkage

  virtual int       del_high() = 0;           // remove the element at high end
                                          // return new high
  virtual int       del_low() = 0;        // delete low element, return new lo

                                          // caution: del_low/high
                                          // does not necessarily 
                                          // immediately call ThreadPtr::~ThreadPtr


// operations on multiple elements

  virtual void      fill(const ThreadPtr  x);          // set all elements = x
  virtual void      fill(const ThreadPtr  x, int from, int to); // fill from to to
  virtual void      clear() = 0;                // reset to zero-sized Plex
  virtual int       reset_low(int newlow); // change low index,return old
  virtual void      reverse();                   // reverse in-place
  virtual void      append(const ThreadPtrPlex& a);    // concatenate a copy
  virtual void      prepend(const ThreadPtrPlex& a);   // prepend a copy

// status

  virtual int       can_add_high() const = 0;
  virtual int       can_add_low() const = 0;
  
  int               length () const;       // number of slots

  int               empty () const;        // is the plex empty?
  virtual int       full() const = 0;      // it it full?

  int               chunk_size() const;    // report chunk size;

  virtual int       OK() const = 0;        // representation invariant

  void		    error(const char* msg) const;
  void		    index_error() const;
  void		    empty_error() const;
  void		    full_error() const;
};


// ThreadPtrIChunk ops

inline int ThreadPtrIChunk:: size() const
{
  return top - base;
}


inline int ThreadPtrIChunk:: base_index() const
{
  return base;
}

inline  int ThreadPtrIChunk:: low_index() const
{
  return low;
}

inline  int  ThreadPtrIChunk:: fence_index() const
{
  return fence;
}

inline  int  ThreadPtrIChunk:: top_index() const
{
  return top;
}

inline  ThreadPtr* ThreadPtrIChunk:: pointer_to(int i) const
{
  return &(data[i-base]);
}

inline  int  ThreadPtrIChunk:: index_of(const ThreadPtr* p) const
{
  return ((int)p - (int)data) / sizeof(ThreadPtr) + base;
}

inline  int  ThreadPtrIChunk:: possible_index(int i) const
{
  return i >= base && i < top;
}

inline  int  ThreadPtrIChunk:: possible_pointer(const ThreadPtr* p) const
{
  return p >= data && p < &(data[top-base]);
}

inline  int  ThreadPtrIChunk:: actual_index(int i) const
{
  return i >= low && i < fence;
}

inline  int  ThreadPtrIChunk:: actual_pointer(const ThreadPtr* p) const
{
  return p >= data && p < &(data[fence-base]);
}

inline  int  ThreadPtrIChunk:: can_grow_high () const
{
  return fence < top;
}

inline  int  ThreadPtrIChunk:: can_grow_low () const
{
  return base < low;
}

inline  ThreadPtr* ThreadPtrIChunk:: invalidate()
{
  ThreadPtr* p = data;
  data = 0;
  return p;
}


inline ThreadPtrIChunk* ThreadPtrIChunk::prev() const
{
  return prv;
}

inline ThreadPtrIChunk* ThreadPtrIChunk::next() const
{
  return nxt;
}

inline void ThreadPtrIChunk::link_to_prev(ThreadPtrIChunk* prev)
{
  nxt = prev->nxt;
  prv = prev;
  nxt->prv = this;
  prv->nxt = this;
}

inline void ThreadPtrIChunk::link_to_next(ThreadPtrIChunk* next)
{
  prv = next->prv;
  nxt = next;
  nxt->prv = this;
  prv->nxt = this;
}

inline void ThreadPtrIChunk::unlink()
{
  ThreadPtrIChunk* n = nxt;
  ThreadPtrIChunk* p = prv;
  n->prv = p;
  p->nxt = n;
  prv = nxt = this;
}

inline  int ThreadPtrIChunk:: empty() const
{
  return low == fence;
}

inline  int  ThreadPtrIChunk:: full() const
{
  return top - base == fence - low;
}

inline int ThreadPtrIChunk:: first_index() const
{
  return (low == fence)? fence : low;
}

inline int ThreadPtrIChunk:: last_index() const
{
  return (low == fence)? low - 1 : fence - 1;
}

inline  int  ThreadPtrIChunk:: succ(int i) const
{
  return (i < low) ? low : i + 1;
}

inline  int  ThreadPtrIChunk:: pred(int i) const
{
  return (i > fence) ? (fence - 1) : i - 1;
}

inline  int  ThreadPtrIChunk:: valid_index(int i) const
{
  return i >= low && i < fence;
}

inline  int  ThreadPtrIChunk:: valid_pointer(const ThreadPtr* p) const
{
  return p >= &(data[low - base]) && p < &(data[fence - base]);
}

inline  ThreadPtr* ThreadPtrIChunk:: grow_high ()
{
  if (!can_grow_high()) full_error();
  return &(data[fence++ - base]);
}

inline  ThreadPtr* ThreadPtrIChunk:: grow_low ()
{
  if (!can_grow_low()) full_error();
  return &(data[--low - base]);
}

inline  void ThreadPtrIChunk:: shrink_high ()
{
  if (empty()) empty_error();
  --fence;
}

inline  void ThreadPtrIChunk:: shrink_low ()
{
  if (empty()) empty_error();
  ++low;
}

inline ThreadPtr* ThreadPtrIChunk::first_pointer() const
{
  return (low == fence)? 0 : &(data[low - base]);
}

inline ThreadPtr* ThreadPtrIChunk::last_pointer() const
{
  return (low == fence)? 0 : &(data[fence - base - 1]);
}

inline ThreadPtr* ThreadPtrIChunk::succ(ThreadPtr* p) const
{
  return ((p+1) <  &(data[low - base]) || (p+1) >= &(data[fence - base])) ? 
    0 : (p+1);
}

inline ThreadPtr* ThreadPtrIChunk::pred(ThreadPtr* p) const
{
  return ((p-1) <  &(data[low - base]) || (p-1) >= &(data[fence - base])) ? 
    0 : (p-1);
}


// generic Plex operations

inline ThreadPtrPlex::ThreadPtrPlex() {}

inline int ThreadPtrPlex::chunk_size() const
{
  return csize;
}

inline  int ThreadPtrPlex::ecnef () const
{
  return lo - 1;
}


inline  int ThreadPtrPlex::fence () const
{
  return fnc;
}

inline int ThreadPtrPlex::length () const
{
  return fnc - lo;
}

inline  int ThreadPtrPlex::empty () const
{
  return fnc == lo;
}

inline ThreadPtrIChunk* ThreadPtrPlex::tl() const
{
  return hd->prev();
}

inline int ThreadPtrPlex::one_chunk() const
{
  return hd == hd->prev();
}

#endif
