/*
   This is an emulation library for intel nx, allowing nx programs to 
   use COMM as an interface.  However, a few things must be changed:
   
   The program initialization must use the PICall mechanism

   Also, not all routines are implemented; just the ones that we needed
 */   

#include "tools.h"
#include "comm/comm.h"
#include "system/system.h"
extern void PIdmax();

/*D
    NX2COMM - Using Chameleon to run programs written using Intel NX

    Description:
    Chameleon provides a backward compatibility mode for the NX
    message-passing routines.  By linking with the appropriate file, many
    NX programs can be run with few changes.

    The only change that you must make to the source code is to make the 
    program hostless and use the PICall interface.  Normally, this involves 
    only a few simple changes to main program.

    The other change is to insert the appropriate compatibilty interface
    module in the link line ahead of the Chameleon libraries.  The name of
    the interface file is ``nx2comm$(COMM).a'', where $(COMM) is the usual
    COMM variable (equal either to "", "p4", or "pvm").  For example,
    this makefile fragment links the NX program in pi.f with Chameleon
$
$   LDIR = /usr/local/tools.core/libs/libsg/$(ARCH)
$   $(FLINKER) -g -o pi pi.f $(LDIR)/picl2comm$(COMM).a \
$                 $(LDIR)/tools$(COMM).a $(LDIR)/tools.a \
$                 $(LDIR)/system.a $(CLIB)
$
    As usual, symbol CLIB is defined by the appropriate bmake include and ARCH 
    is the architecture.

    Notes:
    Not all (or even much) of NX is supported.  The supported routines 
    include csend, crecv, isend, irecv, msgwait, gsync, and gdsum.

    Rs6000 limitations:
    Because of the design of the C and Fortran compilers on the IBM rs6000's, 
    it is impossible to generate a single interface file that works with both
    languages (the design of Chameleon allows this, but NX's design does 
    not).  By default, the interface file FOR THE RS6000s ONLY is built ONLY
    for Fortran.  To build the C version INSTEAD, define BUILDNXC when
    compiling the interface file.  That is, use

$    xlc -Drs6000 -DBUILDNXC -D_POSIX_SOURCE -g -Dp4 \
$                           -I/usr/local/tools.core nx2comm.c

    to build the p4 version for C programs.
D*/

/* The intel versions manually remove the ``forcetype'' types and switch to 
   the forcetype code.  We should provide an option to assume ``intel source''
   that ALWAYS checks for forcetypes and removes the type information

   Another problem is that the NX names are monocase, preventing their use
   from both C and Fortran on IBM RS6000's.  Possible solutions:

   Define a Cname and a Fortran name; then use a switch to build EITHER a
   Fortran version or a C version of this library.  The Fortran code must
   NOT call the C versions of the emulation routines (have them call a common
   version in that case).

 */

/* 
   If neither FORTRANUNDERSCORE or FORTRANCAPS is defined, 
   we can NOT distinguish between the Fortran and C versions,
   so we have to decide whether we are generating the Fortran or the C
   version.  So far, only the rs6000 requires this version.  In this case,
   we look for the ADDITIONAL define: BUILDPICLC .  In this case, 
   send0 etc will be Fortran versions; otherwise, they will be C versions.
*/
#if !defined(FORTRANUNDERSCORE) && !defined(FORTRANCAPS)
#ifndef BUILDNXC
#define NXFORTRANROUTINES
#else
#define NXCROUTINES
#endif /* BUILDNXC */
#else
#define NXCROUTINES
#define NXFORTRANROUTINES
#endif /* !defined(FORTRANUNDERSCORE) && !defined(FORTRANCAPS) */

#if defined(NXCROUTINES)
#define PCcsend      csend
#define PCcrecv      crecv 
#define PCprobe      probe
#define PCrecvinfo0  recvinfo0
#define PCgsync      gsync
#define PCclock      clock
#define PCgdmax      gdmax
#define PCgdsum      gdsum
#define PCmynode     mynode
#define PCmypid      mypid
#define PCmclock     mclock
#define PCgdlow      gdlow
#define PCgdhigh     gdhigh
#define PCkillproc   killproc
#endif
#if defined(NXFORTRANROUTINES)
#define PFcsend     csend_
#define PFcrecv     crecv_
#define PFprobe     probe_
#define PFgsync     gsync_
#define PFclock     clock_
#define PFgdsum     gdsum_
#define PFgdmax     gdmax_
#define PFmynode     mynode_
#define PFmypid      mypid_
#define PFmclock     mclock_
#define PFgdlow      gdlow_
#define PFgdhigh     gdhigh_
#endif

void PCcsend(a,b,c,d)
int b, c, d;
char *a;
{
PIbsend(c,a,b,d,MSG_OTHER);
}

void PCcrecv(a,b,c)
int b, c;
char *a;
{
PIbrecv(c,a,b,MSG_OTHER);
}

int PCprobe(a)
int a;
{
return PInprobe(a);
}

void PCrecvinfo0(a,b,c) 
int *a, *b, *c;
{
*a = RECVLEN(); *b = RECVTYPE(); *c = RECVFROM(); 
}

void PCgsync()
{
PIgsync(ALLPROCS);
}

void PCbcast(a,b,c,d) 
char *a;
int  b, c, d;
{
GSCATTERSRC(a,b,d,ALLPROCS,MSG_OTHER);
}

double PCclock()
{
return SYGetElapsedTime();
}

void PCgdsum(a,b,c,d,e)
void *a;
int  b, c, d, e;
{
}

/* 
   Fortran versions.  At this point, if the Fortran versions are being
   defined, then PFsend0 -> send0_.  See if we need to further define
   these (say by changing them to all caps or remove the trailing underscore)
 */ 
#if defined(FORTRANCAPS)
#define csend_      CSEND
#define crecv_      CRECV
#define probe_      PROBE
#define recvinfo_   RECVINFO
#define gsync_      GSYNC
#define clock_      CLOCK
#define gdsum_      GDSUM
#define gdmax_      GDMAX
#define mynode_     MYNODE
#define mypid_      MYPID
#define mclock_     MCLOCK
#define gdlow_      GDLOW
#define gdhigh_     GDHIGH
#elif !defined(FORTRANUNDERSCORE)
#define csend_      csend
#define crecv_      crecv
#define probe0_     probe
#define gsync_      gsync
#define clock_      clock
#define gdsum_      gdsum
#define gdmax0      gdmax
#define mynode_     mynode
#define mypid_      mypid
#define mclock_     mclock
#define gdlow_      gdlow
#define gdhigh_     gdhigh
#endif

void PFcsend(a,b,c,d)
int *b, *c, *d;
char *a;
{
PIbsend(*c,a,*b,*d,MSG_OTHER);
}

void PFcrecv(a,b,c)
int  *b, *c;
char *a;
{
PIbrecv(*c,a,*b,MSG_OTHER);
}

int PFprobe(a)
int *a;
{
return PInprobe(*a);
}

void PFrecvinfo0(a,b,c) 
int *a, *b, *c;
{*a = RECVLEN(); *b = RECVTYPE(); *c = RECVFROM(); 
}

void PFgsync()
{
PIgsync(ALLPROCS);
}

void PFbcast(a,b,c,d) 
char *a;
int  *b, *c, *d;
{
GSCATTER(a,*b,(*d)==MYPROCID,ALLPROCS,MSG_OTHER);
}

double PFclock()
{
return SYGetElapsedTime();
}

void PFgdsum(a,b,c,d,e)
void *a;
int  *b, *c, *d, *e;
{
PCgdsum(a,*b,*c,*d,*e);
}

void PFgdmax(a,b,c,d,e)
void *a;
int  *b, *c, *d, *e;
{
PCgdmax(a,*b,*c,*d,*e);
}

int PCmynode()
{
return PImytid;
}
int PCmypid()
{
return 0;
}
int PCmclock()
{
return (int)(SYGetCPUTime() * 1000.0);
}
void PCgdlow( val, n, buf )
double *val, *buf;
int    n;
{
GDMIN(val,n,buf,ALLPROCS);
}
void PCgdhigh( val, n, buf )    
double *val, *buf;
int    n;
{
GDMAX(val,n,buf,ALLPROCS);
}
int PFmynode()
{
return MYPROCID;
}
int PFmypid()
{
return 0;
}
int PFmclock()
{
return (int)(SYGetCPUTime() * 1000.0);
}
void PFgdlow( val, n, buf )
double *val, *buf;
int    *n;
{
GDMIN(val,*n,buf,ALLPROCS);
}
void PFgdhigh( val, n, buf )    
double *val, *buf;
int    *n;
{
GDMAX(val,*n,buf,ALLPROCS);
}
