#ifndef lint
static char SCCSid[] = "@(#) ./doc/doctext.c 07/23/93";
#endif

#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include "doc/doc.h"

/* Set this to 0 for LaTeX output */
static int IsMan = 1;
static int InVerbatim = 0;

/* This is designed to work with comments in C programs.  Comments to
   be processed by this routine have the form:
    / * <char> .... <char> * /
   where <char> is one of the characters in MATCH_CHARS.  The definitions
   of the characters follows
 */

/*D
doctext - program to extract man pages from source files

Input:
$ -mpath path
Sets the path where the man pages will be written
$ -ext n
Sets the extension (1-9,l, etc)
$ -I filename
filename contains the public includes needed by these routines; it will
automatically be added to the synopsis of the routines
$ -latex
Generate latex output rather than man (nroff) format files.
$filenames
Names the files from which documents are to be extracted

Notes:
The output file names are formed from a path, the name of the routine, and
an extension.  The format of the output is a man page, suitable for use
by man, nroff -man, or xman.

Typesetting Conventions:
A $<space> where $ is in column 1 turns into a new paragraph.  This allows the
insertion of line breaks into the man page (ordinarily, leading blanks
are deleted as is needed by n/troff).

A .<space> in the first column is used to indicate a routine argument; these
are set in a special way.  

There are some further restrictions if a "whatis" file is to be correctly
generated (this is necessary for man -k to work).  In this case, the first
few lines MUST look like

$    name - short description

$    sometext:

where sometext ends a line with a ":".  

Author: 
Bill Gropp
D*/
/* In the past, we said:

   Other .<non-space> commands are passed through, 
   allowing you to use any of the nroff commands.

   But I think that this is a bad idea.  I'd like to formalize most of
   the fields, and give them special processing (particularly for the
   TeX version)
 */
main( argc, argv )
int  argc;
char **argv;
{
char outfilename[MAX_FILE_SIZE];
char routine[MAX_ROUTINE_NAME];
char path[MAX_FILE_SIZE], extension[4];
char lextension[5];
char date[50];
char incfile[MAX_FILE_SIZE], userlevel[5];
char *infilename;
FILE *fd, *fout, *incfd;
int  nread;
char kind;

/* set the defaults */
strcpy( path, "./" );
strcpy( extension, "3" );
incfile[0] = '\0';
userlevel[0] = '\0';

/* check for option -extension and -mpath */
nread = HandleArgs( argc, argv, path, extension, incfile, userlevel );
argc -= nread;
argv += nread;

/* Open up the file of public includes */    
if (incfile[0])
    incfd = fopen( incfile, "r" );
else
    incfd = 0;

/* process all of the files */
while (argc--) {
    infilename = *argv++;
    /* Insert a routine to shorten file names here? */
    fd = fopen( infilename, "r" );
    if (!fd) {
    	fprintf( stderr, "Could not open file %s\n", infilename );
    	continue;
        }
    LastChangeToFile( infilename, date ); 
    /* At this point, we can trim the filename of any leading trash, 
       such as "./".  Later.... */
    while (FoundLeader( fd, routine, &kind )) {
	/* At this point, we could modify the routine name to make it
	   shorter */
	if (kind == INCLUDE) {
	    /* Eventually, cache the includes for this file */
	    continue;
	    }
	strcpy( lextension, extension );
	if (kind == FORTRAN) 
	    strcat( lextension, "f" );
    	sprintf( outfilename, "%s%s.%s", path, routine, lextension );
    	fout = fopen( outfilename, "w" );
    	if (!fout) {
    	    fprintf( stderr, "Could not open output file %s\n", outfilename );
    	    break;
    	    }
    	OutputManPage( fd, fout, routine, lextension, infilename, kind, date,
		       incfd, userlevel );
        fclose( fout );
	}
    fclose( fd );
    }
return 0;
}

/* 
   There are a number of things to watch for.  One is that leading blanks are
   considered significant; since the text is being formated, we usually don't
   agree with that. 
 */
OutputManPage( fin, fout, name, level, filename, kind, date, 
	       incfd, userlevel )
FILE *fin, *fout;
char *name, *level, *filename, kind, *date, *userlevel;
FILE *incfd;
{
int  i, c, nsp;
char token[1024];
int  has_synopsis;

has_synopsis = 0;
OutputText( fin, fout, name, level, filename, kind, &has_synopsis, date );

if (kind == ROUTINE && !has_synopsis) {
    /* This is a routine, so we can extract the exact usage */    
    /* Now, hunt for the first open brace.  Output with "obeylines".
       Note that we copy Synposis; some systems put strings into
       read-only pages */
    strcpy( token, "Synopsis" );
    OutputSection( fout, token );
    OutputVerbatimBegin( fout );
    /* include the includes that the user needs */
    if (incfd) {
	fseek( incfd, 0L, 0 );
	while ((c = getc( incfd )) != EOF) 
	    OutputChar( fout, (char)c );
	}
    /* We may also need to add some includes private to a file.  I haven't
       decided how to do that yet */

    SkipWhite( fin );
    while (1) {
	c = FindNextToken( fin, token, &nsp );
	if (c == EOF || token[0] == '{') break;
	/* Don't output "register" */
	if (strcmp( token, "register" ) != 0) {
	    for (i=0; i<nsp; i++) OutputChar( fout, ' ' );
	    OutputString( fout, token );
	    }
	}
    if (IsMan)
	OutputString( fout, "\n" );
    OutputVerbatimEnd( fout );
    }
    
/* Now add the filename where the routine or description is located */
if (filename) {
    /* Optionally, add the directory? */
    OutputLocation( fout, filename );
    }
OutputEndPage( fout );
}

/*
    This routine generates the text part of a text page.
    It has to do several things:
    Look for the end of the text (\n<white><kind> * /<white>\n)
    Process formating directives
        a: \n<white><string>:\n
        b: \n.<string>\n
        c: \n$<string>\n
    So the rule is:
    After seeing a newline, check for . and $
    if neither, skip white space.
    Copy rest of line into buffer.
    check for <kind> * / or :\n.
    Output line accordingly.
 */        
OutputText( fin, fout, name, level, filename, kind, has_synopsis, date )
FILE *fin, *fout;
char *name, *level, *filename;
char kind;
int  *has_synopsis;
char *date;
{
int  c;
char lineBuffer[MAX_LINE], *lp;
int  firsthyphen;
int  lastWasNl;
int  doing_synopsis = 0;
	
firsthyphen = 1;
/* After the name, if there IS a -, prefix it with a \.  This seems to be
   necessary for making the whatis file */
lastWasNl   = 1;
/* Note that the NAME field can't use a font changes */
OutputTitle( fout, name, level, date );
lineBuffer[0] = '+';   /* Sentinal on lineBuffer */
while (1) {
    lp = lineBuffer + 1;
    c  = getc( fin );
    if (c == EOF) break;
    if (doing_synopsis && (c == VERBATIM || c == ARGUMENT || c == '\n')) {
	OutputVerbatimEnd( fout );
	doing_synopsis = 0;
	}
    if (c == VERBATIM) {
	/* Raw mode output. */
	c = getc( fin );
	if (c == ' ' || c == '\t' || c == '\n') {
	    OutputLineBreak( lastWasNl, fout );
	    }
	else {
	    fprintf( fout, ".%c", (char)c );
	    }
	if (c != '\n') {
	    SkipWhite( fin );
	    while ( (c = getc( fin )) != EOF) {
		OutputChar( fout, c );
		if (c == '\n') { lastWasNl = 1; break; }
		}
	    }
        }
    else if (c == ARGUMENT) {
    	SkipWhite( fin );
	OutputArgBegin( fout );
        /* Get the name */
    	while ( (c = getc( fin )) != EOF) {
    	    OutputChar( fout, c );
    	    if (isspace(c)) { lastWasNl = 1; break; }
    	    }
	OutputArgDefn( fin, fout );
    	SkipWhite( fin );
    	while ( (c = getc( fin )) != EOF) {
    	    OutputChar( fout, c );
    	    if (c == '\n') { 
		int ns = 0;
		/* Check for a "continued" description, defined as:
		   next line is neither EMPTY nor starts with . or $ */
		/* Peek at next character */
		c = getc( fin );
		ungetc( (char)c, fin );
		/* Check for next character a formatting command or an
		   empty line */
		if (c == '.' || c == '$' || c == '\n' ||
		    c == '@' || c == 'M' || c == 'D') {
		    lastWasNl = 1; break;
		    }
		/* Check for a blank line, looking for lines of all blanks
		   as well as those containing only a newline. */
		if (c == ' ' || c == '\t') {
		    while (c == ' ' || c == '\t') {
			ns++;
			c = getc( fin );
			}
		    ungetc( (char)c, fin );
		    /* Really should check for <char> * /  */
		    if (c == '\n' || c == '@' || c == 'M' || c == 'D') {
			lastWasNl = 1; break;
			}
		    }
		}
    	    }
	OutputArgEnd( fout );
        }
    else if (c == '\n') { 
	lastWasNl = 1;
	OutputChar( fout, '\n' );
        }
    else {
	if (isspace(c) && c != '\n')
	    SkipWhite( fin );
	else {
            if (firsthyphen && c == '-') {
		if (IsMan) 
		    OutputRawChar( fout, '\\' );  /* *lp++ */
		else {
		    /* Turn it into an em-dash */
		    OutputChar( fout, c );
		    OutputChar( fout, c );
		    }
		}
            firsthyphen = 0;
	    *lp++ = c;
            }
    	/* Copy to end of line; do NOT include the EOL */
    	while ((c = getc( fin )) != EOF && c != '\n') 
    	    *lp++ = c;
    	lp--;
    	while (isspace(*lp)) lp--;
    	lp[1] = '\0';    /* Add the trailing null */
    	if (lineBuffer[1] == kind && strcmp(lineBuffer+2,"*/") == 0)
    	    break;
    	else if (lp[0] == ':') {
    	    *lp = '\0';
	    /* Using .SS confused the whatis builder */
	    OutputSection( fout, lineBuffer + 1 );
	    /* If there is a "manual" SYNOPSIS, don't generate one later */
	    if (MatchTokens( lineBuffer+1, "SYNOPSIS" ) == 0) {
		*has_synopsis = 1;
		doing_synopsis= 1;
		OutputVerbatimBegin( fout );
		/* Need to end the synopsis with .fi */
		}
    	    }
    	else {
	    OutputString( fout, lineBuffer + 1 );
	    OutputString( fout, "\n" );
    	    }
        }
    }
if (doing_synopsis) 
    OutputVerbatimEnd( fout );
}

/* Routine to process the arguments.  Returns the number read */
int HandleArgs( argc, argv, path, extension, incfile, userlevel )
int  argc;
char **argv;
char *path, *extension, *incfile, *userlevel;
{
int nread = 0, ln;
argc--; argv++; nread++; /* skip the program */

nread += 2*SYArgGetString( &argc, argv, 0, "-mpath", path, 1024 );
ln = strlen(path);
if (path[ln-1] != '/') {
    path[ln] = '/'; 
    path[ln+1] = '\0';
    }
nread += 2*SYArgGetString( &argc, argv, 0, "-ext",   extension, 3 );
nread += 2*SYArgGetString( &argc, argv, 0, "-I",     incfile,  1024 );
nread += 2*SYArgGetString( &argc, argv, 0, "-l",     userlevel, 10 );
if (SYArgHasName( &argc, argv, 0, "-latex" )) {
    nread += 1;
    IsMan = 0;
    }
return nread;
}
/*
  man macros that are available:

  .TH t s c n 
   Title t, Section s, commentary c (e.g., local), and manual name n

  .SH text
   Subhead text

  .SS text
   Sub-Subhead text
 
  .B text
   Bold text

  .I text
   Italics text

  .SM text
   Make text 1 point smaller

  .P 
   Begin paragraph

  .BP in
    Begin paragraph with hanging indent

  .PD v
    Set interparagraph spacing to v vertical spaces
 */

/*
   In order to maximize the flexibility of the formating, the output
   is run through these routines.  This will allow us to generate a 
   LaTeX reference manual (perhaps even LaTeXInfo!) with the same
   program.  

   Naturally, this is under construction
 */

OutputChar( fout, c )
FILE *fout;
char c;
{
/* The TeX version should look for things like $, ^, _, %, etc as well */
if (c == '\\') 
    fputs( "\\\\", fout );
else {
    if (!IsMan && !InVerbatim) {
	if (c == '<')
	    fputs( "$<$", fout );
	else if (c == '>') 
	    fputs( "$>$", fout );
	else if (c == '=')
	    fputs( "$=$", fout );
	else if (c == '^')
	    fputs( "{\\tt \\char`\\^}", fout );
	else if (c == '_')
	    fputs( "{\\tt \\char`\\_}", fout );
	else if (c == '#')
	    fputs( "{\\tt \\char`\\#}", fout );
	else if (c == '&')
	    fputs( "\\&", fout );
	else if (c == '$')
	    fputs( "\\$", fout );
	else if (c == '%')
	    fputs( "\\%", fout );
	else
	    putc( c, fout );
	}
    else
	putc( c, fout );
    }
}

OutputRawChar( fout, c )
FILE *fout;
char c;
{
putc( c, fout );
}

OutputString( fout, s )
FILE *fout;
char *s;
{
char *p, c, *rstr = 0;

/* look for special characters */
p = s;
while (*p) {
    c = *p;
    rstr = 0;
    if (c == '\\') 
	rstr = "\\\\";
    if (!IsMan && !InVerbatim) {
	if (c == '<')
	    rstr = "$<$";
	else if (c == '>') 
	    rstr = "$>$";
	else if (c == '=')
	    rstr = "$=$";
	else if (c == '^')
	    rstr = "{\\tt \\char`\\^}";
	else if (c == '_')
	    rstr = "{\\tt \\char`\\_}";
	else if (c == '#')
	    rstr = "{\\tt \\char`\\#}";
	else if (c == '&')
	    rstr = "\\&";
	else if (c == '%')
	    rstr - "\\%";
	else if (c == '$')
	    rstr = "\\$";
	}

    if (rstr) {
        *p = '\0';
        fputs( s, fout );
        fputs( rstr, fout );
	s = p + 1;
        }
    p++;
    }
if (*s) 
    fputs( s, fout );
}

OutputLocation( fout, s )
FILE *fout;
char *s;
{
if (IsMan)
    fprintf( fout, "\n.SH LOCATION\n %s\n", s );
else
    fprintf( fout, "\n\\location{%s}\n", s );
}

OutputTitle( fout, name, level, date )
FILE *fout;
char *name, *level, *date;
{
if (IsMan) 
    fprintf( fout, ".TH %s %s \"%s\" \" \" \"Tools\"\n.SH NAME\n%s ", 
 	     name, level, date, name );
else {
    char bname[256];
    strcpy( bname, name );
    /* We have to be prepared to correct the name if it contains a
       reserved character */
    fputs( "\\mantitle{", fout );
    OutputString( fout, name );
    fprintf( fout, "}{%s}{%s}\n\\manname{", level, date, name );
    OutputString( fout, bname );
    fputs( "}\n", fout );
    }
}

OutputSection( fout, name )
FILE *fout;
char *name;
{
if (IsMan) {
    UpperCase( name );
    fprintf( fout, ".SH %s\n", name );
    }
else
    fprintf( fout, "\\subhead{%s}\n", name );
}

OutputLineBreak( lastnl, fout )
int  lastnl;
FILE *fout;
{
if (IsMan) {
    if (lastnl) 
	fputs( ".br\n", fout );
    else
	fputs( "\n.br\n", fout );
    }
else 
    fputs( "\\nextline\n", fout );
}

OutputArgBegin( fout )
FILE *fout;
{
if (IsMan) 
    fputs( ".PD 0\n.TP\n.B ", fout );
else 
    fputs( "\\startarg{", fout );
}

OutputArgDefn( fin, fout )
FILE *fin, *fout;
{
int c;

SkipWhite( fin );
if (IsMan)
    fputs( "\n", fout );
else {
    fputs( "}{", fout );
    c = getc( fin );
    if (c != '-') 
	ungetc( (char)c, fin );
    else
	SkipWhite( fin );
    }
}

OutputArgEnd( fout )
FILE *fout;
{
if (IsMan) 
    fputs( ".PD 1\n", fout );
else 
    fputs( "}\n", fout );
}

OutputVerbatimBegin( fout )
FILE *fout;
{
if (IsMan)
    fputs( ".nf\n", fout );
else
    fputs( "\\startvb\\begin{verbatim}\n", fout );
InVerbatim = 1;
}

OutputVerbatimEnd( fout )
FILE *fout;
{
if (IsMan)
    fputs( ".fi\n", fout );
else
    fputs( "\\end{verbatim}\n\\endvb", fout );
InVerbatim = 0;
}

OutputEndPage( fout )
FILE *fout;
{
if (!IsMan)
    fputs( "\\endmanpage\n", fout );
}
