      subroutine dflow(n,x,fvec,fjac,ldfjac,task,r,nint)
      character*(*) task
      integer n,ldfjac,nint
      double precision r
      double precision x(n),fvec(n),fjac(ldfjac,n)
c     **********
c
c     Subroutine dflow
c
c     This subroutine computes the function and Jacobian matrix of the 
c     nonlinear system of equations obtained from a discretization of 
c     the Flow in a Channel problem by a k-stage collocation method.
c
c     The flow in a channel problem is modeled by the ordinary 
c     differential equation
c
c                     u'''' = R*[u'*u'' - u*u''']
c
c     with appropriate boundary conditions.  In this formulation R
c     is the Reynolds number, u is the potential function, and u' is
c     the tangential velocity of the fluid.
c
c     Additional information on this problem can be found in 
c
c     Brett M. Averick, Richard G. Carter, and Jorge J. More',
c     The MINPACK-2 Test Problem Collection (Preliminary Version),
c     Technical Report ANL/MCS-TM-150, Argonne National Laboratory,
c     Argonne, IL. May 1991, pp. 3-5.
c
c     The subroutine statement is:
c
c       subroutine dflow(n,x,fvec,fjac,ldfjac,task,r,nint)
c
c     where
c
c       n is an integer variable.
c         On entry n is the number of variables.
c            For the flow in a channel problem n must equal 8*nint.
c         On exit n is unchanged.
c
c       x is a double precision array of dimension n.
c         On entry x specifies the vector x if task = 'F', 'J', or 'FJ'.
c            Otherwise x need not be specified.
c         On exit x is unchanged if task = 'F', 'J', or 'FJ'. Otherwise
c            x is set according to task.
c
c       fvec is a double precision array of dimension n.
c         On entry fvec need not be specified.
c         On exit fvec contains the function evaluated at x if
c            task = 'F' or 'FJ'.
c
c       fjac is a double precision array of dimension (ldfjac,n).
c         On entry fjac need not be specified.
c         On exit fjac contains the Jacobian matrix evaluated at x if
c            task = 'J' or 'FJ'.
c
c       ldfjac is an integer variable.
c          On entry ldfjac is the leading dimension of fjac.
c          On exit ldfjac is unchanged.
c
c       task is a character variable.
c         On entry task specifies the action of the subroutine:
c
c            task               action
c            ----               ------
c             'F'     Evaluate the function at x.
c             'J'     Evaluate the Jacobian matrix at x.
c             'FJ'    Evaluate the function and the Jacobian at x.
c             'XS'    Set x to the standard starting point xs.
c
c         On exit task is unchanged.
c
c       r is a double precision variable.
c         On entry r is the Reynolds number.
c         On exit r is unchanged. 
c
c       nint is an integer variable.
c         On entry nint is the number of subintervals in the 
c            k-stage collocation.
c         On exit nint is unchanged.
c
c     MINPACK-2 Project. October 1991.
c     Argonne National Laboratory and University of Minnesota.
c     Brett M. Averick.
c
c     **********
      integer deg,cpts,bc,npi,dim
      parameter(deg=4,cpts=4,bc=2,dim=deg+cpts-1,npi=cpts+deg)
      double precision zero,one,two,three,six,twelve 
      parameter(zero=0.0d0,one=1.0d0,two=2.0d0,three=3.0d0,six=6.0d0,
     +          twelve=12.0d0)

      integer i,j,k,m,var,eqn,nsave
      double precision h,xt
      double precision rho(cpts),nf(0:dim),dw(deg+1,cpts+deg),
     +       rhnfhk(cpts,0:dim,0:dim,0:deg),w(deg+1)

      save rhnfhk,h,nsave

      data nsave /0/
      data (rho(i), i = 1, cpts)
     +     /0.694318413734436035D-1,0.330009490251541138D0,
     +      0.669990539550781250D0,0.930568158626556396D0/
      
c     Initialization on the first call to the subroutine with the
c     number of intervals = nint.

      if (nsave .ne. nint) then
         nsave = nint
         h = one/dble(nint)

c        Store vector of factorials 

         nf(0) = one
         do 10 i = 1, dim
            nf(i) = nf(i-1)*dble(i)
   10    continue

c        Store all possible combinations of rho, h, and n!.

         do 50 m = 0, deg
            do 40 k = 0, dim
               do 30 j = 0, dim
                  do 20 i = 1, cpts
                     rhnfhk(i,j,k,m) = (rho(i)**j)*(h**m)/nf(k)
   20             continue
   30          continue
   40       continue
   50    continue
      endif

c     Compute the standard starting point if task = 'XS'

      if (task .eq. 'XS') then

c        The standard starting point corresponds to the solution of the
c        flow in a channel problem with R = 0.

         xt = zero
         do 70 i = 1, nint
            var = (i - 1)*npi
            x(var+1) = three*xt**2 - two*xt**3
            x(var+2) = six*(xt - xt**2)
            x(var+3) = six*(one - two*xt)
            x(var+4) = -twelve
            do 60 j = 1, cpts
               x(var+deg+j) = zero
   60       continue
            xt = xt + h
   70    continue

         return

      endif

c     Evaluate the function if task = 'F', the Jacobian matrix if 
c     task = 'J', or both if task = 'FJ'.

c     Initialize arrays.

      do 90 j = 1, n
         if (task .eq. 'F' .or. task .eq. 'FJ') fvec(j) = zero
         if (task .eq. 'J' .or. task .eq. 'FJ') then
            do 80 i = 1, n
               fjac(i,j) = zero
  80        continue
         endif
  90  continue
      do 110 k = 1, npi
         do 100 j = 1, deg + 1
            dw(j,k) = zero
  100    continue
  110 continue

c     Set up the boundary equations at t = 0.  u(0) = 0, u'(0) = 0.

      if (task .eq. 'F' .or. task .eq. 'FJ') then
         fvec(1) = x(1)
         fvec(2) = x(2)
      endif
      if (task .eq. 'J' .or. task .eq. 'FJ') then
         fjac(1,1) = one
         fjac(2,2) = one
      endif

c     Set up the collocation equations.

       do 170 i = 1, nint
          var = (i - 1)*npi
          eqn = var + bc
          do 160 k = 1, cpts
             do 140 m = 1, deg + 1
                w(m) = zero
                do 120 j = m, deg
                   w(m) = w(m) + rhnfhk(k,j-m,j-m,j-m)*x(var+j)
                   dw(m,j) = rhnfhk(k,j-m,j-m,j-m)
  120           continue
                do 130 j = 1, cpts
                   w(m) = w(m) + rhnfhk(k,deg+j-m,deg+j-m,deg-m+1)
     +                                               *x(var+deg+j)
                   dw(m,deg+j) = rhnfhk(k,deg+j-m,deg+j-m,deg-m+1)
  130           continue
  140        continue
             if (task .eq. 'F' .or. task .eq. 'FJ') 
     +          fvec(eqn+k) = w(5) - r*(w(2)*w(3) - w(1)*w(4))
             if (task .eq. 'J' .or. task .eq. 'FJ') then
                do 150 j = 1, npi
                   fjac(eqn+k,var+j) = dw(5,j) - r*(dw(2,j)*w(3)
     +                                 + w(2)*dw(3,j) - dw(1,j)*w(4)
     +                                 - w(1)*dw(4,j)) 
  150           continue
             endif
  160    continue
  170 continue

c     Set up the continuity equations.

      do 230 i = 1, nint - 1
         var = (i - 1)*npi
         eqn = var + bc + cpts
         do 200 m = 1, deg
            w(m) = zero
            do 180 j = m, deg
               w(m) = w(m) + rhnfhk(1,0,j-m,j-m)*x(var+j)
               dw(m,j) = rhnfhk(1,0,j-m,j-m)
  180       continue
            do 190 j = 1, cpts
               w(m) = w(m) + rhnfhk(1,0,deg+j-m,deg-m+1)*x(var+deg+j)
               dw(m,deg+j) = rhnfhk(1,0,deg+j-m,deg-m+1)
  190       continue
  200    continue
         do 220 m = 1, deg
            if (task .eq. 'F' .or. task .eq. 'FJ') 
     +         fvec(eqn+m) = x(var+cpts+deg+m) - w(m)
            if (task .eq. 'J' .or. task .eq. 'FJ') then
               fjac(eqn+m,var+cpts+deg+m) = one
               do 210 j = 1, npi
                  fjac(eqn+m,var+j) = -dw(m,j)
  210          continue
            endif
  220    continue
  230 continue

c     Prepare for setting up the boundary conditions at t = 1.

      var = n - npi
      do 260 m = 1, deg + 1
         w(m) = zero
         do 240 j = m, deg 
            w(m) = w(m) + rhnfhk(1,0,j-m,j-m)*x(var+j)
            dw(m,j) = rhnfhk(1,0,j-m,j-m)
  240    continue
         do 250 j = 1, cpts
            w(m) = w(m) + rhnfhk(1,0,deg+j-m,deg-m+1)*x(var+deg+j)
            dw(m,deg+j) = rhnfhk(1,0,deg+j-m,deg-m+1)
  250    continue
  260 continue

c     Set up the boundary equations at t = 1.  u(1) = 1, u'(1) = 0.

      if (task .eq. 'F' .or. task .eq. 'FJ') then
         fvec(n-1) = w(1) - one
         fvec(n) = w(2)
      endif
      if (task .eq. 'J' .or. task .eq. 'FJ') then
         var = n - npi
         do 270 j = 1, npi
            fjac(n-1,var+j) = dw(1,j)
            fjac(n,var+j) = dw(2,j)
  270    continue
      endif

      return

      end
