#ifndef lint
static char SCCSid[] = "@(#) ./nonlin/newls/newls.c 07/23/93";
#endif

/*
     This implements Newton's Method with a line search approach
   for Nonlinear systems of equations.

     This is intended as a model implementation, it does not 
   necessarily have many of the bells and whistles of other 
   implementations. 

     It is intended to be DATA-STRUCTURE NEUTRAL and can be called
   RECURSIVELY.

     The Newton code has two context variables. 

     1) NLCntx *nlP -- The nonlinear solver context which is 
                       created by NLCreate(NLNEWTONLS);
     2) void *usrP - The user context, the user sets this pointer to
                     point to a user define structure which contains
                     stuff which is used to evaluate the function, solve 
                     systems which involve approximations to the 
                     Jacobian, etc.
*/
#include <math.h>
#include <stdio.h>
#include "tools.h"
#include "nonlin/nlall.h"

/*------------------------------------------------------------*/
/* Solves a nonlinear system of equations using Newton's      */
/* method and a line search scheme.                           */
/*------------------------------------------------------------*/
/*D
    NLNEWETONLS - This implements Newton's Method with a
                  line search approach.

    Calling seqeunce: 
.   nlP = NLCreate(NLNEWTONLS);
.   NLSet*()
.   NLSetUp()
.   NLSolve()
.   NLDestroy()

       This implements essentially a truncated Newton method with 
line search, by default it uses a cubic backtracking line search 
as described in Dennis and Schnabel. See the examples in 
nonlin/examples.

D*/
int NLNewtonLSSolve(nlP,usrP)
NLCntx *nlP;
void    *usrP;
{
  NLNewtonLSCntx     *neP = (NLNewtonLSCntx *) nlP->MethodPrivate;
  void               *x = nlP->vec_sol,*f,*y,*g,**work,*tmp, *w;
  int                N = nlP->max_it,i;
  double             fnorm,gnorm,gpnorm,xnorm;
  double             *res = nlP->residual_history, ynorm;
  FILE               *fp = nlP->fp;

  work = VGETVECS(nlP->vc,usrP,4); CHKPTRV(work,-1);
  f = work[0]; y = work[1]; g = work[2]; w = work[3];

  if (nlP->initial_guess) (*nlP->initial_guess)(nlP,usrP,x);  /* x <- x_0  */
  else VSET(nlP->vc,usrP,0.0,x);

  VNORM(nlP->vc,usrP,x,&xnorm);
   
  (*nlP->fun)(nlP,usrP,x,f);                  /* f <- function(x)   */
  VNORM(nlP->vc,usrP,f,&fnorm);               /* fnorm <- || f || */  
  if (res) *res++ = fnorm;

  if (nlP->usr_monitor) {
    (*nlP->usr_monitor)(nlP,usrP,x,f,fnorm);
  }

  nlP->nfunc++; nlP->nvectors += 2;
        
  for ( i=0; i<N; i++ ) {

    /* y <- -J\f  */
    if (nlP->stepSetup) (*nlP->stepSetup)(nlP,usrP,x); 
    (*nlP->stepCompute)(nlP,usrP,x,f,y,fnorm,neP->maxstep,
                                      nlP->trunctol,&gpnorm,&ynorm);  
    if (nlP->stepDestroy) (*nlP->stepDestroy)(nlP,usrP);
    /* line_search should be part of step compute */
    (*neP->line_search)(nlP,usrP,x,f,g,y,w,fnorm,&ynorm,&gnorm);

    if (fp) fprintf(fp,"%d f %g g %g ynorm %g ",i,fnorm,gnorm,ynorm);

    fnorm = gnorm;
    if (res) *res++ = fnorm;
    tmp = f; f = g; g = tmp;
    tmp = x; x = y; y = tmp;
    VNORM(nlP->vc,usrP,x,&xnorm);
    nlP->nsteps++; nlP->nvectors++;

    if (nlP->usr_monitor) {
      (*nlP->usr_monitor)(nlP,usrP,x,f,fnorm);
    }

    /* Test for convergence */
    if ((*nlP->converged)(nlP,usrP,xnorm,ynorm,fnorm)) {
      /* make sure solution is in corect location */
      if (x != nlP->vec_sol) VCOPY(nlP->vc,usrP,x,nlP->vec_sol);
      break;
    } 
  }

  VFREEVECS(nlP->vc,usrP,work,4);
  return i+1;
}

/* -------------------------------------------------------------*/
NLCntx *NLNewtonLSCreate()
{
  NLCntx            *nlP;
  NLNewtonLSCntx *neP;

  nlP                    = NEW(NLCntx); CHKPTRN(nlP);
  NLSetDefaults( nlP );

  nlP->method            = NLNEWTONLS;

  nlP->setup             = NLNewtonLSSetUp;
  nlP->solver            = NLNewtonLSSolve;
  nlP->destroy           = NLNewtonLSDestroy;

  nlP->trunctol          = 1.e-10;

  neP                    = NEW(NLNewtonLSCntx); CHKPTRN(neP);
  nlP->MethodPrivate     = (void *) neP;
  neP->line_search       = NLDefaultLineSearch;
  neP->alpha             = 1.e-4;
  neP->maxstep           = 1.e8;
  neP->steptol           = 1.e-12;
  return nlP;
}
/*------------------------------------------------------------*/
/*ARGSUSED*/
void NLNewtonLSSetUp(nlP,usrP)
NLCntx *nlP;
void   *usrP;
{
  if (!nlP->stepCompute) {
    fprintf(stderr,"NLNewtonLSSetUp needs stepCompute!\n");
    SETERR(1);
  }
  if (!nlP->fun) {
    fprintf(stderr,"NLNewtonLSSetUp needs func!\n");
    SETERR(1);
  }
  if (!nlP->vc) {
    fprintf(stderr,"NLNewtonLSSetUp needs vector ops!\n");
    SETERR(1);
  }
}
/*------------------------------------------------------------*/
void NLNewtonLSDestroy(nlP)
NLCntx *nlP;
{
  FREE(nlP->MethodPrivate);
  FREE(nlP);
}
/*------------------------------------------------------------*/
/* This line search is in not a line search at all, it simply */
/* uses the full Newton step                                  */
/*ARGSUSED*/
int NLSimpleLineSearch(nlP,usrP,x,f,g,y,w,fnorm,ynorm,gnorm)
void   *usrP, *x, *y, *w, *f;
NLCntx *nlP;
double fnorm,*gnorm,*ynorm;
{
  VNORM(nlP->vc,usrP,y,ynorm);
  VAXPY(nlP->vc,usrP,1.0,x,y);
  (*nlP->fun)(nlP,usrP,y,g);
  VNORM(nlP->vc,usrP,g,gnorm); 
  nlP->nfunc++; nlP->nvectors += 3;
  return 1;
}
/*------------------------------------------------------------*/
/* This line search is taken from "Numerical Methods for      */
/* Unconstrained Optimization and Nonlinear equations"        */
/* by Dennis and Schnabel Page 325                            */
/*                                                            */
/* x - present position                                       */
/* y - search direction,  will get new position               */
/* f - present value of function                              */
/* g - wil get new value of function                          */
/* w - work vector                                            */
/* fnorm - norm of f                                          */
/* gnorm - will get norm of f at new position                 */
/* ynorm - will get final search length                       */
int NLDefaultLineSearch(nlP,usrP,x,f,g,y,w,fnorm,ynorm,gnorm)
void   *usrP, *x, *y, *w, *f;
NLCntx *nlP;
double fnorm,*gnorm,*ynorm;
{
  double         alpha, maxstep, steptol;
  double         initslope, minlambda, lambda, lambdaprev,gnormprev,lambdatemp;
  double         a,b,d,t1,t2;
  int            count;
  FILE           *fp = nlP->fp;
  NLNewtonLSCntx *neP = (NLNewtonLSCntx *) nlP->MethodPrivate;
  
  alpha   = neP->alpha;
  maxstep = neP->maxstep;
  steptol = neP->steptol;
 
  VNORM(nlP->vc,usrP,y,ynorm);
  if (*ynorm > maxstep) {               /* step too big, so scale back */
    VSCALE(nlP->vc,usrP,maxstep/(*ynorm),y); 
    *ynorm = maxstep;
  }
  minlambda = steptol/(*ynorm);
  VDOT(nlP->vc,usrP,f,y,&initslope); 
  if (initslope > 0.0) initslope = -initslope;
  if (initslope == 0.0) initslope = -1.0;

  VCOPY(nlP->vc,usrP,y,w);
  VAXPY(nlP->vc,usrP,1.0,x,w);
  (*nlP->fun)(nlP,usrP,w,g);
  VNORM(nlP->vc,usrP,g,gnorm); 
  if (*gnorm <= fnorm + alpha*initslope) {      /* is reduction enough */
    if (fp) fprintf(fp,"Taking full newton step\n");
    VCOPY(nlP->vc,usrP,w,y);
    nlP->nfunc++; nlP->nvectors += 6;
    return 1;
  }

  /* fit points with quadratic */
  lambda = 1.0; count = 0;
  lambdatemp = -initslope/(2.0*(*gnorm - fnorm - initslope));
  lambdaprev = lambda;
  gnormprev = *gnorm;
  if (lambdatemp <= .1*lambda) lambda = .1*lambda;
  else lambda = lambdatemp;
  VCOPY(nlP->vc,usrP,x,w);
  VAXPY(nlP->vc,usrP,lambda,y,w);
  (*nlP->fun)(nlP,usrP,w,g);
  VNORM(nlP->vc,usrP,g,gnorm); 
  if (*gnorm <= fnorm + alpha*initslope) {      /* is reduction enough */
      if (fp) fprintf(fp,"Taking newton step from quadratic \n");
      VCOPY(nlP->vc,usrP,w,y);
      nlP->nfunc += 2; nlP->nvectors += 10;
      return 1;
  }

  /* fit points with cubic */
  count = 1;
  while (1) {
    if (lambda <= minlambda) { /* bad luck; use full step */
      fprintf(stderr,"Unable to find good step length! %d \n",count);
      fprintf(stderr,"f %g fnew %g ynorm %g lambda %g \n",fnorm,*gnorm, *ynorm,
                       lambda);
      VCOPY(nlP->vc,usrP,w,y);
      nlP->nfunc += 3 + count; nlP->nvectors += 12 + 3*count;
      return 0;
    }
    t1 = *gnorm - fnorm - lambda*initslope;
    t2 = gnormprev  - fnorm - lambdaprev*initslope;
    a = (t1/(lambda*lambda) - t2/(lambdaprev*lambdaprev))/(lambda-lambdaprev);
    b = (-lambdaprev*t1/(lambda*lambda) + 
                lambda*t2/(lambdaprev*lambdaprev))/(lambda-lambdaprev);
    d = b*b - 3*a*initslope;
    if (d < 0.0) d = 0.0;
    if (a == 0.0) lambdatemp = -initslope/(2.0*b);
    else lambdatemp = (-b + sqrt(d))/(3.0*a);
    if (lambdatemp > .5*lambda) lambdatemp = .5*lambda;
    lambdaprev = lambda;
    gnormprev = *gnorm;
    if (lambdatemp <= .1*lambda) lambda = .1*lambda;
    else lambda = lambdatemp;
    VCOPY(nlP->vc,usrP,x,w);
    VAXPY(nlP->vc,usrP,lambda,y,w);
    (*nlP->fun)(nlP,usrP,w,g);
    VNORM(nlP->vc,usrP,g,gnorm); 
    if (*gnorm <= fnorm + alpha*initslope) {      /* is reduction enough */
      if (fp) fprintf(fp,"Taking newton step from cubic %d\n",count);
      VCOPY(nlP->vc,usrP,w,y);
      nlP->nfunc += 2 + count; nlP->nvectors += 11 + 3*count;
      return 1;
    }
    count++;
  }
}
