#ifndef lint
static char SCCSid[] = "@(#) ./solvers/nullsp.c 07/23/93";
#endif

#include "tools.h"
#include "solvers/svctx.h"

/*
   This file contains routines for handling user-specified null-spaces
   for the solvers.

   The most common case, that of a constant null-vector, is optimized
   for.

   These routines were motivated by problems where a potential is solved
   for and only derivative of the potential have physical meaning.  An
   example is presure; only gradients of presure have any significance.
   Any linear PDE with exclusively Neumann boundary conditions will have
   the constant vector as a null vector.

   The approach used depends on whether an iterative or direct method is
   used.

   In the iterative approach, it is only (?citation?) necessary to insure that
   the preconditioner has the same null space as the matrix.  We do this
   by adding a post processing step where the null space is projected
   out of the solution (also note that this is not necessary if
   the preconditioner has the same null vector or the application of the
   preconditioner is immediately followed by an application of the matrix).

   In the direct approach, we modify the matrix by adding rows containing
   the null-vectors to rows of the matrix (note that this results in a
   non-symmetric matrix).  The choice is made by finding the first
   row FROM THE BOTTOM (to minimize fill) whose non-zero structure not
   linearly independent of the null vector.

   Since we define the solution in this case as the one with no component
   in the null space, this process amount to adding the equation that
   expresses that choice to one of the redundent equations.

   In the symmetric case, we would add the null vector to the corresponding
   COLUMN as well (since the right and left null spaces are the same
   for a symmetric matrix).
 */

void SViProjNullSpace( ctx, w )
SVctx  *ctx;
double *w;
{
int         i, j, n, nv;
SVnullSpace *ns = ctx->ns;
double      sum;

n = ctx->size;
if (ns->cnst_v) {
    sum = 0.0;
    for (i=0; i<n; i++) sum  += w[i];
    for (i=0; i<n; i++) w[i] -= sum;
    }
nv = ns->nv;
for (j=0; j<nv; j++) {
    DDOT(ns->v[j],w,n,sum);
    DAXPY(w,-sum,n,ns->v[j]);
    }
}

/*@
    SVSetNullSpace - Set the null space for a linear system

    Input Parameters:
.   ctx      - Solver context
.   has_cnst - Constant vector is a null vector
.   nv       - number of null vectors (excluding the constant vector)
.   v        - pointers to the vectors.  This routine does NOT copy
               the contents of this pointer.

    Error Conditions:
    Failure to allocate memory
@*/    
void SVSetNullSpace( ctx, has_cnst, nv, v )
SVctx  *ctx;
int    has_cnst, nv;
double **v;
{
SVnullSpace *ns;

ns = NEW(SVnullSpace);     CHKERR(1);
ns->nv      = nv;
ns->cnst_v  = has_cnst;
ns->v       = v;
ctx->ns     = ns;
}	

/* Private routine to free null-space */
SViFreeNullSpace( ctx )
SVctx *ctx;
{
FREE( ctx->ns );
}

#ifdef FOO
/* Update the matrix from the null space */
void SViUpdateMatrixFromNS( ctx )
SVctx *ctx;	
{
SVnullSpace *ns = ctx->ns;

if (ns->cnst_v) {
    /* Update the last row, since any non-zero row contains some elements */
    SpAddRow( ctx->mat, ctx->size-1, cols 0 to size-1, value of 1 );
    CHKERR(1);
    }
for (j=0; j<nv; j++) {
    /* Need to check for non-zero pattern */
    /* Also, compress the vector? */
    SpAddRow( ctx->mat, ctx->size-1-j, all cols, ns->v[j] );
    CHKERR(1);
    }    
}
#endif
