/* Local stuff for Overlapping Schwarz */

#ifndef SVSetOSMDomainsNumber

/*
   Overlapping Schwarz preconditioners.

   We need:

   The index set for the overlapping domain (ovnv,ovidx).
   The corresponding matrix (block)

   In addition, for multiplicative, we need

   The matrix corresponding to all columns for the rows in a (overlapping)
   domain (used to compute the residual on that overlapping domain)

   (Not yet implemented)
   When a global problem is provided, we need in addition an interpolation
   matrix on the subdomain (WITHOUT overlap) that corresponds to the 
   "coarse-grid" element.  If no such operators are provided, a
   piece-wise constant approach is used.  Further, we need a mapping from 
   the coarse grid to each subdomain.

   In order to aid in the exploration of this method, I've added a number
   of hooks to allow the action of each part of the preconditioner to be
   monitored.  These are:
   
   globalMonitor( ctx, nv, x, b, gctx, phase )
   where phase indicates where this was called; this is called in the
   global solve step

   localMonitor( ctx, nv, x, b, lctx, i, phase )
   is called for each subdomain solve.  
 */
   
typedef struct {
    SpMat *block;             /* Matrix for the (overlapping) subdomain */
    SpMat *submat;            /* Used to compute the local residual */
    SVctx *svc;

    /* These should all be index sets */
    int   nv,   *idx;         /* subdomain */
    int   ovnv, *ovidx;       /* subdomain with overlap */
    int   cnv,  *cidx;        /* coarse grid for subdomain */
    int   oinv, *oiidx;       /* overlap with "previous" domains; used
				 for the computation of the residual with
				 multiplicative Schwarz */
    int         *oividx;      /* mapping from oiidx to ovidx (size is oinv) */
    int         ovIsContig;   /* 1 if the overlap is contiguous elements 
			         (ovidx) */

    SVMETHOD    lmeth;        /* Method to use for local problem */
    
    } SVOSMBlock;

typedef struct {
  int          nd;            /* Number of blocks */
  int          overlap;       /* amount of overlap */
  int          is_mult;       /* True if the multiplicative version is to
				 be used */
  int          use_intersections;    
                              /* True if intersections of the domains
				 should be computed/used (for 
				 multiplicative only) */
  SVOSMBlock   *bb;
  SVMETHOD     defMeth;       /* Method to use on each block */
  ITMETHOD     defIt;
  int          itmax;

  double       *w1, *w2;      /* Work vectors */
  int          worksize;      /* size of work vectors */

  /* For the global problem */
  SpMat        *global;       /* Global matrix, if any */
  SVctx        *gctx;         /* Solver context for global problem */
  void         (*Interpolate)();
  void         (*Restrict)();
  void         *interpCtx, *restrictCtx;
  double       *cb, *cx;      /* work arrays for global problem */

                              /* Monitors for the preconditioner */
  void         (*globalMonitor)(), 
               (*localMonitor)(),
               *gmctx, *lmctx;

} SVOSMctx;

extern void SViSetupOSM();
extern int  SViSolveOSM();
extern void SViDestroyOSM();
extern void SViApplyAOSM();
extern void SViApplyMOSM();
extern void SViCreateOSM();
extern void SViSetOSMDecomp();
extern void SViSetOSMMethodDecomp();
extern void SViSetOSMSolverDecomp();
extern void SViAllocOSMDomains();

extern void SViDefaultInterpolateOSM();
extern void SViDefaultRestrictOSM();
extern void SViSetOSMCoarseDecomp();

extern void SVSetOSMRegularDomains2d();
extern void SVSetOSMRegularOverlap2d();

/* Some likely options to provide */

/*M
     SVSetOSMDomainsNumber - Sets the number of blocks

     Input Parameters:
.    ctx - solver context
.    n   - number of blocks

     Synopsis:
     void SVSetOSMDomainsNumber( ctx, n )
     SVctx *ctx;
     int   n;
M*/
#define SVSetOSMDomainsNumber(C,v) \
{if ((C)->type == SVOSM) {((SVOSMctx *)(C)->private)->nd = v;}}

/*M
     SVSetOSMDefaultMethod - Sets the default method for domain solves

     Input Parameters:
.    ctx - solver context
.    meth - solver method (i.e., SVILU)

     Synopsis:
     void SVSetOSMDefaultMethod( ctx, meth )
     SVctx *ctx;
     SVMETHOD meth;
M*/
#define SVSetOSMDefaultMethod(C,v) \
{if ((C)->type == SVOSM) {if (((SVOSMctx *)((C)->private))->bb) {\
SETERRC(1,"Must set default method before subdomains are defined");\
}else((SVOSMctx *)(C)->private)->defMeth = v;}}

#define SVSetOSMDefaultAccelerator(C,v) \
{if ((C)->type == SVOSM) {((SVOSMctx *)(C)->private)->defIt = v;}}

#define SVSetOSMDefaultMaxIt(C,v) \
{if ((C)->type == SVOSM) {((SVOSMctx *)(C)->private)->itmax = v;}}

#define SVSetOSMDefaultOverlap(C,v) \
{if ((C)->type == SVOSM) {((SVOSMctx *)(C)->private)->overlap = v;}}

/* Whether to use multiplicative OSM by default */
#define SVSetOSMDefaultMult(C,v) \
{if ((C)->type == SVOSM) {((SVOSMctx *)(C)->private)->is_mult = v;}}

#define SVSetOSMDefaultUseIntersections(C,v) \
{if ((C)->type == SVOSM) {((SVOSMctx *)(C)->private)->use_intersections = v;}}

/* define the indices to use for a particular subdomain (ie., define an
   alternate decomposition) */
#define SVSetOSMDecomp(C,i,idx,nv) \
             SViSetOSMDecomp( C, i, idx, nv );


#define SVSetOSMGlobalInterpolation( C, r, ictx ) \
    {if ((C)->type == SVOSM) {((SVOSMctx *)(C)->private)->Interpolate = r;\
			      ((SVOSMctx *)(C)->private)->interpCtx = ictx;}}
#define SVSetOSMGlobalRestriction( C, r, rctx ) \
    {if ((C)->type == SVOSM) {((SVOSMctx *)(C)->private)->Restrict = r;\
			      ((SVOSMctx *)(C)->private)->restrictCtx = rctx;}}

#define SVSetOSMGlobalMatrix( C, cmat ) \
    {if ((C)->type == SVOSM) {((SVOSMctx *)(C)->private)->global = cmat;}}

/* not right - specify solver METHOD, not context? */
#define SVSetOSMGlobalSolver( C, sv ) \
    {if ((C)->type == SVOSM) {((SVOSMctx *)(C)->private)->gctx = sv;}}

#define SVSetOSMGlobalDecomp( C, i, cidx, cn ) \
	SViSetOSMCoarseDecomp( C, i, cidx, cn )

/* These two are not yet supported */
/* Method for a particular subdomain */
#define SVSetOSMMethodDecomp(C,i,v) \
SViSetOSMMethodDecomp(C,i,v);

/* Solver context for a particular subdomain (with optional matrix) */
#define SVSetOSMSolverDecomp(C,i,v,mat) \
SViSetOSMSolverDecomp(C,i,v,mat);

/* Get the solver context for a domain */
#define SVGetOSMSolver(C,i,ctx) \
{if ((C)->type == SVOSM) {ctx=((SVOSMctx *)((C)->private))->bb[i].svc;}}

#endif

/* 
    Define the local and global monitors.
 */
#define SVSetOSMGlobalMonitor( C, r, ctx ) \
{if ((C)->type == SVOSM) {((SVOSMctx *)((C)->private))->globalMonitor = r;\
   ((SVOSMctx *)((C)->private))->gmctx = ctx;}}
#define SVSetOSMLocalMonitor( C, r, ctx ) \
{if ((C)->type == SVOSM) {((SVOSMctx *)((C)->private))->localMonitor = r;\
   ((SVOSMctx *)((C)->private))->lmctx = ctx;}}
