#ifndef LINT
static char SCCSid[] = "@(#) ./sparse/default/dffactor.c 07/23/93";
#endif

/*
    This file contains a simple sparse factor and solve routine,
    using the generic Sparse representation.
    Diagonal elements are stored as inverse (LDU storage).
 */

#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"
#include "inline/spops.h"

/*
   SpDfComputeFill - Find storage for factor of matrix B in BB 
                      (descriptors allocated but not set).

   Note: 
   Because fill propagates down, a single row effectively adds 
   all of the fill for the rows above.  This is crucial in reducing the
   cost of the fill algorithm.  The rule is

$   Let IM = # of non-zeros to include.
$   1. If, at row "prow", we add a row (row) that contains column prow, then
      set IM[row] = nz;
$   2. IM[prow] = nz of row.

 */
int SpDfComputeFill( B, BB )
SpMat       *B;
SpMatSplit *BB;
{
int          *fill, n, *im, err;

if (BB->factor->map)
    err = SpiDfComputeFillPerm( B, BB );
else {
    n    = B->rows;
    im   = (int *) SPAllocTemp( 2*(n+1)*sizeof(int) ); CHKPTRV(im,1);
    fill = im + n + 1;
    if (!im) return -(n+1);
    err = SpDfComputeFillBase( B, BB, fill, im );
    SPFreeTemp(im);
    }
return err;
}

int SpDfComputeFillBase( B, BB, fill, im )
SpMat       *B;
SpMatSplit *BB;
register int       *fill, *im;
{
int          prow, row, *xi, nz, nzi, n, fm, nnz, idx, m, nzf, 
             *nzl, err, dummy, *xxi, i;
double       *xxv;
SpMat        *BBf = BB->factor;


/* Nothing to do for the first row */
n          = B->rows;
xxi        = (int *)MALLOC( n * sizeof(int) ); CHKPTRV(xxi,-1);
xxv        = (double *)MALLOC( n * sizeof(int) ); CHKPTRV(xxv,-1);
for (i=0; i<n; i++) xxv[i] = 0.0;
BB->nzl[0] = 0;
nzl        = BB->nzl;
for (prow=0; prow<n; prow++) {
    /* Update row "prow" using all rows that are needed to eliminate
       entries on this row.  */
    SpScatterFromRow( B, prow, &nz, &xi, (double **)0 );
    row       = *xi++;
    /* Add the initial row (fill,rtmp) */
    nzf       = nz--;
    fm        = row;
    while (nz--) {
	idx       = *xi++;
	fill[fm]  = idx;
	fm        = idx;
	}
    fill[fm] = n;

    nzi = 0;
    /* Eliminate into (fill,rtmp) */
    while (row < prow) {
	SpScatterFromRow( BBf, row, &dummy, &xi, (double **)0 );
	nz         = nzl[row];
	/* Get the pivot row */
	xi         = xi + nz;
	if (*xi++ != row) 
	    {SETERRC(-(row+1),"Zero pivot encountered");
             return -(row+1);}     /* zero (missing) pivot */
	nz++;
	nnz        = im[row] - nz;
	fm         = row;
	while (nnz-- > 0) {
	    idx       = *xi++;
	    nz++;
	    if (idx == prow) im[row] = nz;
	    /* find fm such that fill[m] <= idx fill[fill[fm]] */
	    do {
		m  = fm;
		fm = fill[m];
		} while (fm < idx);
	    if (fm != idx) {
		/* insert */
		fill[m]   = idx;
		fill[idx] = fm;
		fm        = idx;
		nzf++;
		}
	    }
	row   = fill[row];
	nzi++;
	};
    /* Get next row */
    SpScatterFromRow( B, prow, &nz, &xi, (double **)0 );
    row      = xi[0];

    /* Save the number of non-zeros in the lower triangle */
    nzl[prow] = nzi;
    
    /* Store filled row */
    nnz = nzf;
    fm  = row;
    for (i=0; i<nnz; i++) {
	xxi[i] = fm;
	fm      = fill[fm];
	}
    SpGatherToRow( BBf, prow, nzf, xxi, xxv );
    if (GETERR) return -(prow+1);
    im[prow] = nzf;
    }
FREE(xxi);
return ERR_NONE;
}

/*
   SpDfComputeFactor - Factor a matrix.

   Description:
   Given a matrix B and a computed fill area BB, 
   find the numerical factor. BB should have be
   obtained previously by SpComputeFill() or
   SpComputeILUFill().   

   Input Parameters:
.  B    - matrix to factor
.  BB   - matrix to hold factor   
*/
int SpDfComputeFactor( B, BB )
SpMat       *B;
SpMatSplit *BB;
{
int          *fill, n, err;
double       *rtmp;

/* Allocate temporaries to compute fill in; row to compute values of fill in */
/* Check for mapping */
if (BB->factor->map) {
    err = SpDfComputeFactorPerm( B, BB );
    }
else {
    n    = B->rows;
    rtmp = (double *)SPAllocTemp( (n+1)*(sizeof(int) + sizeof(double)) );
    CHKPTRV(rtmp,-(n+1));
    fill = (int *)(rtmp + n + 1);
    err = SpDfComputeFactorBase( B, BB, rtmp, fill );
    SPFreeTemp(rtmp);
    }
return err;
}

int SpDfComputeFactorBase( B, BB, rtmp, fill )
SpMat       *B;
SpMatSplit *BB;
register double    *rtmp;
register int       *fill;
{
int          prow, row, *xi, nz, n, nnz, *yi, pnz, i;
double       multiplier, *pc, *xv;
SpMat *BBf = BB->factor;

n          = B->rows;
/* Nothing to do for the first row */
BB->nzl[0] = 0;
for (prow=0; prow<n; prow++) {
    /* Update row "prow" using all rows that are needed to eliminate
       entries on this row.  We simultaneously compute the fill and
       update the row. */
    SpScatterFromRow( BBf, prow, &nz, &xi, (double **)0 );
    yi        = xi;
    row       = *yi++;
    /* Set rtmp to 0 */
    while (nz--) rtmp[*xi++] = 0.0;
    /* Load in the initial row from B */
    SpScatterFromRow( B, prow, &nz, &xi, &xv );
    while (nz--) rtmp[*xi++] = *xv++;
    
    /* Eliminate into (fill,rtmp) */
    while (row < prow) {
	pc        = rtmp+row;
	if (*pc != 0.0) {
	    nz         = BB->nzl[row];
	    /* Get the pivot row */
	    SpScatterFromRow( BBf, row, &pnz, &xi, &xv );
	    xv         += nz;
	    xi         += nz;
	    if (*xi++ != row) 
		{SETERRC(-(row+1),"Zero pivot encountered"); 
                 return -(row+1);} /* zero (missing) pivot */
	    multiplier = *pc * *xv++;
	    *pc        = multiplier;
	    nnz        = pnz - nz - 1;
	    SPARSEDENSESMAXPY(rtmp,multiplier,xv,xi,nnz);
	    }
	row   = *yi++;
	};
    /* copy row into v */
    SpScatterFromRow( BBf, prow, &nz, &xi, &xv );
    /* Replace diagonal entry with inverse (pivot multiplier) */
    if (rtmp[prow] == 0.0) return -(prow+1);
    rtmp[prow] = 1.0 / rtmp[prow];
    for (i=0; i<nz; i++) {
	xv[i] = rtmp[xi[i]];
	}
    SpGatherToRow( BBf, prow, nz, xi, xv );
    }
return ERR_NONE;
}

/*+
   SpDfComputeFillPerm - Find storage for factor of matrix B in BB 
                          (descriptors allocated but not set).

   Note: 
   Because fill propagates down, a single row effectively adds 
   all of the fill for the rows above.  This is crucial in reducing the
   cost of the fill algorithm.  The rule is

$   Let IM = # of non-zeros to include.
$   1. If, at row "prow", we add a row (row) that contains column prow, then
      set IM[row] = nz;
$   2. IM[prow] = nz of row.

   This should be FillPermBase, and NOT allocate any storage.
 +*/
int SpiDfComputeFillPerm( B, BB )
SpMat        *B;
SpMatSplit   *BB;
{
int          prow, row, *fill, *xi, nz, nzi, n, fm, nnz, idx, m, nzf, 
             *nzl, err = 0, *im, fnz, d;
SpMat        *BBf = BB->factor;
int          *r = BBf->map->rowmap, *ic = BBf->map->icolmap;

/* Allocate temporaries to compute fill in; row to compute values of fill in */
BBf->nz = -1;   /* For now, don't keep track of the number of non-zeros in
                   the factored matrix.  We can use this value to provide
                   more options for allocating storage for the data */
n    = B->rows;
im   = (int *)MALLOC((n+1)*2*sizeof(int)); CHKPTRV(im,-(n+2));
fill = im + n + 1;

/* Nothing to do for the first row */
BB->nzl[0] = 0;
nzl        = BB->nzl;
for (prow=0; prow<n; prow++) {
    /* Update row "prow" using all rows that are needed to eliminate
       entries on this row.  */
    SpScatterFromRow( B, r[prow], &nz, &xi, (double **)0 );
    /* Add the initial row (fill) */
    nzf       = nz;
    /* May be inserted in any order.  This should probably try to do a 
       better job of finding the insert point (this amounts to an n^2
       bubble sort.  At the very least, remembering the previous insert
       location would be useful. */
    fill[n] = n;
    while (nz--) {
	fm        = n;
	idx       = ic[*xi++];
	do {
	    m = fm;
	    fm= fill[m];
	    } while (fm < idx);
	fill[m]  = idx;
	fill[idx] = fm;
	}

    nzi = 0;
    /* Eliminate into (fill) */
    row = fill[n];
    while (row < prow) {
	nz         = nzl[row];
	/* Get the pivot row */
	SpScatterFromRow( BBf, row, &d, &xi, (double **)0 );
	xi         += nz;
	if (*xi++ != row) 
	    {SETERRC(-(row+1),"Zero pivot encountered"); 
            return -(row+1);}      /* zero (missing) pivot */
	nz++;
	nnz     = im[row] - nz;
	fm      = row;
	while (nnz-- > 0) {
	    idx       = *xi++;
	    nz++;
	    if (idx == prow) im[row] = nz;
	    /* find fm such that fill[m] <= idx fill[fill[fm]] */
	    /* elements are ordered */
	    do {
		m  = fm;
		fm = fill[m];
		} while (fm < idx);
	    if (fm != idx) {
		/* insert */
		fill[m]   = idx;
		fill[idx] = fm;
		fm        = idx;
		nzf++;
		}
	    }
	row   = fill[row];
	nzi++;
	};

    /* Allocate elim_row */
#ifdef FOO
    row      = fill[n];
    elim_row = frs[prow];
    SPiMallocNVt(BBf,nzf,&elim_row->v,&elim_row->i,err); CHKERRV(1,-(row+1));
    elim_row->maxn = nzf;
    elim_row->nz   = nzf;

    /* Save the number of non-zeros in the lower triangle */
    nzl[prow] = nzi;
    
    /* Store filled row */
    nnz = nzf;
    xi  = elim_row->i;
    fm  = row;
    while (nnz--) {
	*xi++ = fm;
	fm    = fill[fm];
	}
#endif
    im[prow] = nzf;
    }

/* Recover temp used to find fill */
FREE( im );
return ERR_NONE;
}

/* Given a matrix B and a computed fill area BB, find the numerical 
   factor.  
 */
int SpDfComputeFactorPerm( B, BB )
SpMat       *B;
SpMatSplit *BB;
{
int          prow, row, *xi, nz, n, nnz, *yi, fnz;
double       multiplier, *pc, *xv, *rtmp;
SpMat        *BBf = BB->factor;
int          *r = BBf->map->rowmap, *ic = BBf->map->icolmap;
register int i;

/* Allocate temporaries to compute fill in; row to compute values of fill in */
n    = B->rows;
rtmp = (double *)MALLOC( (n+1) * sizeof(double) );
CHKPTRV(rtmp,-(n+2));

/* Nothing to do for the first row */
BB->nzl[0] = 0;
for (prow=0; prow<n; prow++) {
    /* Update row "prow" using all rows that are needed to eliminate
       entries on this row. */
    SpScatterFromRow( BBf, prow, &nz, &xi, (double **)0 );
    yi        = xi;
    row       = *yi++;
    /* Set rtmp to 0 */
    for (i=0; i<nz; i++) rtmp[xi[i]] = 0.0;

    /* Load in the initial row from B */
    SpScatterFromRow( B, r[prow], &nz, &xi, &xv );
    for (i=0; i<nz; i++) rtmp[ic[xi[i]]] = xv[i];
    
    /* Eliminate into (rtmp) */
    while (row < prow) {
	pc        = rtmp+row;
	if (*pc != 0.0) {
	    nz         = BB->nzl[row];
	    /* Get the pivot row */
	    SpScatterFromRow( BBf, row, &fnz, &xi, &xv );
	    xi += nz;
	    xv += nz;
	    if (*xi++ != row) 
		{SETERRC(-(row+1),"Zero pivot encountered"); 
                return -(row+1);}   /* zero (missing) pivot */
	    multiplier = *pc * *xv++;
	    *pc        = multiplier;
	    nnz        = fnz - nz - 1;
	    SPARSEDENSESMAXPY(rtmp,multiplier,xv,xi,nnz);
	    }
	row   = *yi++;
	};
    /* copy row into v */
#ifdef FOO
    xv  = elim_row->v;
    xi  = elim_row->i;
    nz  = elim_row->nz;
    /* Replace diagonal entry with inverse (pivot multiplier) */
    rtmp[prow] = 1.0 / rtmp[prow];
    GATHER(xv,xi,rtmp,nz);
#endif
    }

/* Recover temp used to find compute fill */
FREE( rtmp );
return ERR_NONE;
}
