/*
   A program to test blocked, sparse vector ops
 */

#include <stdio.h>
#include <math.h>
#include "tools.h"
#include "system/system.h"
#include "sparse/spmat.h"

extern SpMat *SpFBToBlk( );

main( argc, argv )
int  argc;
char **argv;
{
int        i,n, m, bsize, j, ops;
SpMat      *mat, *bmat;
double     *v, t1, t2, *vsol1, *vsol2, *vsol3, 
           tmat, tbmat, tbcmat, nrm, nrmc;

m     = 2;
SYArgGetInt( &argc, argv, 1, "-m", &m );
bsize = 1;
SYArgGetInt( &argc, argv, 1, "-b", &bsize );
n   = m * m;

/* Form a dense matrix as a sparse one */
mat = SpCreate( n, n, n );
for (j=0; j<n; j++) 
    for (i=0; i<n; i++) {
	SpAddValue( mat, (i == j) ? (double)n : 1.0/(i+ 1.0), i, j ); 
	CHKERR(1);
	}

/* Form the blocked version */
bmat = SpFBToBlk( mat, bsize );

#ifdef FOO
if (mat->rows < 5) {
    printf( "Block matrix:\n" );
    SpPrint( stdout, bmat );
    
    printf( "Sparse matrix:\n" );
    SpPrint( stdout, mat );
    }
#endif

TestMVProd( mat, bmat, bsize );

SpDestroy( bmat );

/* Factor the matrix, block the factor */
TestMVSol( mat, bsize );

/* Eventually, add factor the blocked matrix */
}

xerbla_( str, info, d )
char *str;
int  *info, d;
{
fprintf( stderr, 
	" ** On entry to %6s parameter number %2d had an illegal value\n",
	str, *info );
exit(1);
}

TestMVProd( mat, bmat, bsize )
SpMat *mat;
SpMat *bmat;
int   bsize;
{
int        i,n, m, j, ops;
double     *v, t1, t2, *vsol1, *vsol2, *vsol3, 
           tmat, tbmat, tbcmat, nrm, nrmc;

printf( "\nTesting blocked matrix-vector product:\n" );
n   = mat->rows;

v   = (double *)MALLOC( 4 * n * sizeof(double) );
vsol1 = v + n;
vsol2 = vsol1 + n;
vsol3 = vsol2 + n;
for (i=0; i<n; i++) v[i] = (double)i;

ops = 2*n*n;
/* Compare SpMults */
t2 = SYGetCPUTime();
SpMult( mat, v, vsol1 );
t1 = SYGetCPUTime();
tmat = t1 - t2;

t2 = SYGetCPUTime();
SpFBMult( bmat, v, vsol2 );
t1 = SYGetCPUTime();
tbmat = t1 - t2;

t2 = SYGetCPUTime();
SpFBMultContig( bmat, v, vsol3 );
t1 = SYGetCPUTime();
tbcmat = t1 - t2;

/* Check solutions */
DVaxpy( &n, -1.0, vsol1, vsol2 );
DVnorm( &n, vsol2, &nrm );
DVaxpy( &n, -1.0, vsol1, vsol3 );
DVnorm( &n, vsol3, &nrmc );

FREE( v );

fprintf( stderr, "Norm of difference between SpMult and SpFBMult is %e\n", 
	 nrm );
fprintf( stderr, "Norm of difference (contiguous) is %e\n", nrmc );

fprintf( stdout, "Regular mat*v in %12e (%f Mflops)\n", 
	          tmat, 1.0e-6 * (double)ops/tmat );
fprintf( stdout, "Blocked mat*v in %12e (%f Mflops)\n", 
	          tbmat, 1.0e-6 * (double)ops/tbmat );
fprintf( stdout, "Blocked mat*v (contiguous) in %12e (%f Mflops)\n", 
	          tbmat, 1.0e-6 * (double)ops/tbmat );
}

/* 
  Test block solve
 */  
TestMVSol( mat, bsize )
SpMat *mat;
int   bsize;
{
int        i,n, m, j, ops;
double     *v, t1, t2, *vsol1, *vsol2, *vsol3, 
           tmat, tbmat, tbcmat, nrm, nrmc;
SpMatSplit *smat;
SpMat      *bmat;

printf( "\nTesting blocked matrix-solves:\n" );
n   = mat->rows;

v   = (double *)MALLOC( 4 * n * sizeof(double) );
vsol1 = v + n;
vsol2 = vsol1 + n;
vsol3 = vsol2 + n;
for (i=0; i<n; i++) v[i] = (double)i;

smat = SpQuickFactor( mat, -1, -1, 0, PIVOT_NONE );
bmat = SpFBToBlk( smat->factor, bsize );

#ifdef FOO
if (mat->rows < 5) {
    printf( "Block matrix:\n" );
    SpPrint( stdout, bmat );
    
    printf( "Sparse matrix:\n" );
    SpPrint( stdout, smat->factor );
    }
#endif

ops = 2*n*n;
/* Compare SpMults */
t2 = SYGetCPUTime();
SpSolve( smat, v, vsol1 );
t1 = SYGetCPUTime();
tmat = t1 - t2;

SpDestroy( smat->factor );
smat->factor = bmat;
t2 = SYGetCPUTime();
SpFBSolve( smat, v, vsol2 );
t1 = SYGetCPUTime();
tbmat = t1 - t2;

/* Check solutions */
DVaxpy( &n, -1.0, vsol1, vsol2 );
DVnorm( &n, vsol2, &nrm );

FREE( v );
SpDestroy( bmat );

fprintf( stderr, "Norm of difference between SpSolve and SpFBSolve is %e\n", 
	 nrm );

fprintf( stdout, "Regular solve in %12e (%f Mflops)\n", 
	          tmat, 1.0e-6 * (double)ops/tmat );
fprintf( stdout, "Blocked solve in %12e (%f Mflops)\n", 
	          tbmat, 1.0e-6 * (double)ops/tbmat );
}
