#ifndef lint
static char SCCSid[] = "@(#) ./system/rlimit.c 07/23/93";
#endif

/* 
   This file contains a signal routine (timer interrupt) that
   insures that certain resource limits are not exceeded.  

   The resources that may be limited are:
   
   memory     (Mbytes)
   cputime    (seconds) 
   pagefaults (faults)
   exittime   (timeval structure) - program exits no later than this.
 */

#include "tools.h"
#include "system/system.h"

static void           (*RunDown)() = 0;
static void           *RunDownCtx  = 0;
static void           (*UserTimeRoutine)() = 0;

#if !defined(intelnx) && !defined(__MSDOS__) && !defined(cray)

#include <signal.h>
#include <sys/time.h>          /*I <sys/time.h> I*/
#include <sys/resource.h>
#if defined(solaris)
#include <sys/rusage>
#endif

/* 
   Note that these values are stored in the format that the resource
   limits are handled by the system; these are not necessarily the 
   units given by the users (for example, cputime is probably NOT
   in seconds) 
*/
#define SECONDS_BETWEEN_TESTS 60
static int            maxmem = 0, maxcpu = 0, maxpfaults = 0;
static struct timeval maxtime = { 0, 0 };
static struct timeval starttime;
static int            deltaToTest = SECONDS_BETWEEN_TESTS;
#if defined(tc2000)
static int            (*OldSigAlrm)() = 0;
#else
static void           (*OldSigAlrm)() = 0;
#endif

/*@
   SYSetResourceLimits - Set the limits on resource usage.

   Input Parameters:
.   mmem  - terminate the program if memory use exceeds mmem Mbytes, 
.   mcpu  - terminate the program if cpu use exceeds mcpu seconds,
.   mpf   - terminate the program if page faults exceeds mpf faults, 
.   mtime - terminate the program if the program runs past this time

    Notes:
    This routine actually establishes the limits that will be used.
    There are routines in ./system/resrc.c that help manage resource
    limits (setting defaults, getting values from command lines, etc.)
    As such, this routine is usually called by another routine that
    manages process startup, such as PICall.

    Values of 0 disable the test.
 @*/
void SYSetResourceLimits(mmem,mcpu,mpf,mtime)
int            mmem, mcpu, mpf;
struct timeval *mtime; 
{
#ifdef setrlimit_works
struct rlimit rlp;

rlp.rlim_cur = maxcpu;
rlp.rlim_max = RLIM_INFINITY;
setrlimit( RLIMIT_CPU,  &rlp );
rlp.rlim_cur = maxmem;
setrlimit( RLIMIT_DATA, &rlp );
#endif
struct itimerval value;
struct timeval   tval;

/* Establish the signal handler */
#ifdef rs6000
/* RS6000 always reports sizes in k instead of pages */
maxmem     = mmem * 1024;
#else
maxmem     = mmem * (1024*1024/getpagesize());
#endif
maxcpu     = mcpu;
maxpfaults = mpf;
if (mtime) 
    maxtime    = *mtime;
else {
    maxtime.tv_sec  = 0;
    maxtime.tv_usec = 0;
    }
/* Save the time when we started (in case we need to issue an error message) */
SYGetDayTime( &starttime );

/* Should use pass-through handler (get old handler and save it).  Note
   that this can not be handled with complete safety, since there is no
   estabilished way to safely insert and remove signal handlers (that is,
   each package of routines has to define its own approach; there is no
   standard method for removing a signal handler) */
OldSigAlrm = signal( SIGALRM, SYCheckLimits );

tval.tv_sec       = deltaToTest;
tval.tv_usec      = 0;
value.it_interval = tval;
value.it_value    = tval; 
setitimer( ITIMER_REAL, &value, 0 );
}

/*ARGSUSED*/
/*
   This routine is executed by the signal handler for the timer interrupt;
   it tests the limits against the stored values
 
   Note: rs6000 man pages say the signal handler has three parameters,
         but the include file prototype has listed one int parameter.
         Perhaps we should always use _NO_PROTO ??? 
 */
#if defined(rs6000) || defined(NeXT) || defined(fx2800)
void SYCheckLimits( sig )
int               sig;
#elif defined(tc2000)
int SYCheckLimits( sig, code, scp)
int               sig, code;
struct sigcontext *scp;
#else
void SYCheckLimits( sig, code, scp, addr )
int               sig, code;
struct sigcontext *scp;
char              *addr;
#endif
{
struct rusage ru;
struct timeval tv;
char buf[128];
int  sd;

/* Call a user routine.  Make sure that the user routine can NOT be
   called while this routine is running */
if (UserTimeRoutine) {
    void (*SaveUser)();
    SaveUser        = UserTimeRoutine;
    UserTimeRoutine = 0;
    (*SaveUser)();
    UserTimeRoutine = SaveUser;
    }

if (getrusage( RUSAGE_SELF, &ru ) == -1) {
    SYExit( "Could not get resource values", 0 );
    return;
    }
if (maxmem > 0 && ru.ru_maxrss > maxmem) {
    sd = getpagesize();
    sprintf( buf, "Exceeded memory resource: used %d MB, limit is %d MB", 
	     ((int)(ru.ru_maxrss) * sd)/(1024*1024), 
	     (maxmem * sd) / (1024*1024) );
    SYExit( buf, 1 );
    return;
    }
if (maxpfaults > 0 && ru.ru_majflt > maxpfaults) {
    sprintf( buf, "Exceeded page fault limits: used %d, limit is %d", 
	     (int)(ru.ru_majflt), maxpfaults );
    SYExit( buf, 1 );
    return;
    }
if (maxcpu > 0 && ru.ru_utime.tv_sec + ru.ru_stime.tv_sec > maxcpu) {
    sprintf( buf, "Exceeded cpu time limits: used %d secs, limit is %d secs", 
	     (int)(ru.ru_utime.tv_sec + ru.ru_stime.tv_sec), maxcpu );
    SYExit( buf, 1 );
    return;
    }
SYGetDayTime( &tv );
if (maxtime.tv_sec > 0 && tv.tv_sec > maxtime.tv_sec) {
    sprintf( buf, 
	     "Exceeded elapsed time limits: used %d secs, limit is %d secs", 
	     tv.tv_sec - starttime.tv_sec, maxtime.tv_sec - starttime.tv_sec );
    SYExit( buf, 1 );
    }

/* This is dangerous! The rs6000 man pages claim the signal 
   handler is reset after it is triggered and exited but that
   does not appear to be the case */
#if defined(rs6000)
OldSigAlrm = signal( SIGALRM, SYCheckLimits );
#endif
}

/*@
   SYSetResourceClockOff - Turns off clock which periodically signals to 
      check resource limits. 

   Note:
      This is used to stop the interupts in 
      certain delicate situations which cannot handle interupts. For 
      instance the recvfrom() system command on the RS6000 which is 
      used in PVM. Use  SYResourceClockOn() to resume the periodic 
      checking.

      The correct solution is to make sure all calls to routines that
      can return the error EINTR check for that error and correctly
      handle it.  This usually means re-calling the system routine.
      For an example of code that does this correctly, see p4.
@*/
void SYSetResourceClockOff()
{
OldSigAlrm = signal( SIGALRM, SIG_IGN );
}
/*@
   SYSetResourceClockOn - Turns on the clock which periodically signals to 
      check resource limits. 

   Note:
   Called after a call to SYSetResourceClockOff(). 
@*/
void SYSetResourceClockOn()
{
OldSigAlrm = signal( SIGALRM, SYCheckLimits );
}

/*@
   SYSetNice - Set the "nice" value.

   Input parameter:
.   n - nice value.  Must be positive unless running as root.

   Note:
   On some systems, this has no effect.   
@*/
void SYSetNice( n )
int n;
{
nice(n);
}

/*@
   SYSetUserTimerRoutine - set the user routine to be called on 
   a timer interrupt

   Input Parameter:
   routine - routine to call.  Has no arguments.

   Notes:
   Timer interrupts are scheduled when SYSetResourceLimits is called.
   Every minute (by default) the resources being used by the
   program are checked, and if any limits are exceeded, the program
   is terminated.  If this routine is called, the user-specified 
   routine is called BEFORE any resource checking is done.

   If the timer interrupt occurs again BEFORE the user routine returns,
   the usual termination tests proceed without executing the user's
   routine.
@*/
void SYSetUserTimerRoutine( routine )
void (*routine)();
{
UserTimeRoutine = routine;
}

#else
/* Intel nx and MSDOS version */
void SYSetResourceLimits( mmem, mcpu, mpf, mtime )
int  mmem, mcpu, mpf;
void *mtime; 
{}
void SYSetNice( n )
int n;
{}
#endif
/*@
    SYSetRundownRoutine - Set the routine to be called if a program exceeds
    some resource limit.

    Input parameters:
.   routine - routine to call.
.   ctx     - context to pass to routine

    The form of the routine is
$   routine( ctx, reason )
$   where ctx is the value passed to SYSetRundownRoutine, and reason is either
    the signal that terminated the program, or -1 for some resource
    exceeded.
@*/
void SYSetRundownRoutine( ctx, routine )
void *ctx, (*routine)();
{
RunDown    = routine;
RunDownCtx = ctx;
}    

/*
  Exit routine to be called in case of an error 
 */
void SYExit( str, parm )
char *str;
int  parm;
{
void (*RunDownSV)();
if (RunDown) {
    RunDownSV = RunDown;
    /* Avoid possible race-conditions. */
    RunDown   = 0;
    (*RunDownSV)( RunDownCtx, -1 );
    }	
SYexitall( str, parm );
}

