/* 
   This defines a structure that contains pointers to routines for
   manipulating vectors.  These are patterned after the Level-1
   Blas, but with some additions that have proved useful.
   These include routines to allocate and free vectors.

   Note that the routines that are normally thought of as returning a
   value (e.g., dot, norm) return their value through an argument.
   This allows these routines to be used with other datatype, such
   as float and dcomplex.
 */

#ifndef __VECTOR 
#define __VECTOR
typedef struct {
  void *(*create_vector)();   /* Routine returns a single vector */
  void (*destroy_vector)();   /* Free a single vector */
  void **(*obtain_vectors)(); /* Routine returns pointer to pointers */
  void (*release_vectors)();  /* of user vectors, the user may and */ 
  void (*dot)(),                   /* z = x^H * y */
       (*norm)(),                  /* z = sqrt(x^H * x) */
       (*max)(),                   /* z = max(|x|); idx = index of sup(|x|) */
       (*tdot)(),                  /* x'*y */
       (*scal)(),                  /*  x = alpha * x   */
       (*copy)(),                  /*  y = x */
       (*set)(),                   /*  y = alpha  */
       (*axpy)(),                  /*  y = y + alpha * x */
       (*aypx)(),                  /*  y = x + alpha * y */
       (*swap)(),                  /* exchange x and y */
       (*waxpy)(),                 /*  w = y + alpha * x */
       (*pmult)(),                 /*  w = x .* y */
       (*pdiv)();                  /*  w = x ./ y */
  void (*scatter)(),               /* y[ix[i]] = x[i] */
       (*scatteradd)(),            /* y[ix[i]] += x[i] */
       (*gather)();                /* y[i]     = x[ix[i]] */
  
  /* It is also useful to have some "group" operations, such as 
     a "multi-axpy" and "multi-dot".  These are particularly useful
     for block operations */
  
  /* Hanger for implementation-specific data (often null) */
  void *Private;
  } VECntx;

/* This is the default context; other vectors (such as parallel ones) may
   require additional entries (such as the subset of processors that hold
   a parallel vector) */
typedef struct {
    int n;
    } VEDefaultUsrCntx;

extern VECntx           *VECreate();
extern VEDefaultUsrCntx *VECreateDefaultUsrCntx();
extern void             VEDestroy();
extern void             VEDestroyDefaultUsrCntx();

/* Macros to simplify the use of these */
#define VDOT(vp,N,x,y,val)       (*vp->dot)(N,x,y,val)
#define VNORM(vp,N,x,val)        (*vp->norm)(N,x,val)
#define VTDOT(vp,N,x,y,val)      (*vp->tdot)(N,x,y,val)
#define VSCALE(vp,N,alpha,x)     (*vp->scal)(N,alpha,x) 
#define VCOPY(vp,N,x,y)          (*vp->copy)(N,x,y) 
#define VSET(vp,N,alpha,x)       (*vp->set)(N,alpha,x) 
#define VAXPY(vp,N,alpha,x,y)    (*vp->axpy)(N,alpha,x,y) 
#define VAYPX(vp,N,alpha,x,y)    (*vp->aypx)(N,alpha,x,y) 
#define VSWAP(vp,N,x,y)          (*vp->swap)(N,x,y)
#define VWAXPY(vp,N,alpha,x,y,w) (*vp->waxpy)(N,alpha,x,y,w) 
#define VPMULT(vp,N,x,y,w)       (*vp->pmult)(N,x,y,w) 
#define VPDIV(vp,N,x,y,w)        (*vp->pdiv)(N,x,y,w) 
#define VCREATE(vp,N)            (*vp->create_vector)(N)
#define VDESTROY(vp,v)           (*vp->destroy_vector)(v)
#define VGETVECS(vp,N,m)         (*vp->obtain_vectors)( N, m )
#define VFREEVECS(vp,N,vv,m)     (*vp->release_vectors)( N, vv, m )
#define VSCATTER(vp,N,x,ix,y)    (*vp->scatter)( N, x, ix, ni, y )
#define VSCATTERADD(vp,N,x,ix,y) (*vp->scatteradd)( N, x, ix, ni, y )
#define VGATHER(vp,N,x,ix,y)     (*vp->gather)( N, x, ix, ni, y )
#endif
