#ifndef lint
static char SCCSid[] = "@(#) ./vectors/zvec.c 07/23/93";
#endif

/*
   This file defines the vector operations in the simplest way possible
   (that still allows for general user contexts):  The first field in the
   user context contains size of the vector, and the actual vector type
   is no more than a pointer to the data 

   These routines are for complex double precision serial vectors.
 */

#include "tools.h"
#include "system/flog.h"
#include <math.h>
#include "vectors/vector.h"        /*I "vectors/vector.h" I*/
#include "vectors/zvec.h"          /*I "vectors/zvec.h" I*/

/*@
   ZVdot - Dot product for serial complex double precision vectors.
   This is the Hermitian dot product, i.e. it uses the complex 
   conjugate.

   Input Parameters:
.  N     - Vector context
.  x,y   - vectors to form dot product of

   Output Parameter:
.  z     - result
 @*/
void ZVdot( N, x, y, z )
VEDefaultUsrCntx *N;
register dcomplex *x, *y, *z;
{
register int n = N->n, i;
dcomplex sum;

sum.r = 0.0; 
sum.i = 0.0;
for (i=0; i<n; i++) {
    sum.r += x[i].r*y[i].r + x[i].i*y[i].i;
    sum.i += x[i].r*y[i].i - x[i].i*y[i].r;
    }
*z = sum;
}

/*@
   ZVtdot - Dot product for serial complex double precision vectors.
   This is the non-Hermitian dot product, i.e. it does not use the 
   complex conjugate.

   Input Parameters:
.  N     - Vector context
.  x,y   - vectors to form dot product of

   Output Parameter:
.  z     - result
 @*/
void ZVtdot( N, x, y, z )
VEDefaultUsrCntx *N;
register dcomplex *x, *y, *z;
{
register int n = N->n, i;
dcomplex sum;
sum.r = 0.0; 
sum.i = 0.0;
for (i=0; i<n; i++) {
    sum.r += x[i].r*y[i].r - x[i].i*y[i].i;
    sum.i += x[i].r*y[i].i + x[i].i*y[i].r;
    }
*z = sum;
}

/*@
   ZVnorm - Two norm for serial complex double precision vectors.

   Input Parameters:
.  N     - Vector context
.  x     - vector to form 2-norm of

   Output Parameter:
.  z     - result
 @*/
void ZVnorm( N, x, z )
VEDefaultUsrCntx *N;
register dcomplex *x;
double   *z;
{
register int n = N->n, i;
register double sum = 0;
for (i=0; i<n; i++) 
    sum += x[i].r*x[i].r + x[i].i*x[i].i;
*z = sqrt( sum );
}

/*@
   ZVmax - Max norm for serial double precision complex vectors.

   Input Parameters:
.  N     - Vector context
.  x     - vector to form max-norm of
.  idx   - index of maximum element.  If there are several elements with the
           maximum value, the index of one (probably but no necessarily the
	   first) is returned.

   Output Parameter:
.  z     - result
 @*/
void ZVmax( N, x, idx, z )
VEDefaultUsrCntx *N;
dcomplex         *x;
int              *idx;
double           *z;
{
register int i, j=0, n = N->n;
register double max = 0.0, tmp;
for (i=0; i<n; i++) {
    tmp = sqrt( x->r * x->r + x->i * x->i ); x++;
    if ( tmp > 0.0 ) { if (tmp > max) { j = i; max = tmp; } }
    else                      { if (-tmp > max) { j = i; max = -tmp; } }
    }
*z   = max;
if (idx)
    *idx = j;
}

/*@
   ZVscal - Multiply a serial complex double precision vector by a scalar.

   Input Parameters:
.  N     - Vector context
.  alpha - scale factor
.  x     - vectors to scale
 @*/
void ZVscal( N, alpha, x )
VEDefaultUsrCntx *N;
dcomplex alpha, *x;
{
register int n = N->n, i;
register double ai, ar;

for (i=0; i<n; i++) {
    ar     = x[i].r * alpha.r - x[i].i * alpha.i;
    ai     = x[i].r * alpha.i - x[i].i * alpha.r;
    x[i].r = ar;
    x[i].i = ai;
    }
}

/*@
   ZVcopy - Copy a serial complex double precision vector.

   Input Parameters:
.  N     - Vector context
.  x     - vector to copy

   Output Parameter:
.  y     - copy of x
 @*/
void ZVcopy( N, x, y )
VEDefaultUsrCntx *N;
register dcomplex *x, *y;
{
register int n = N->n, i;
for (i=0; i<n; i++) 
    y[i] = x[i];
}

/*@
   ZVswap - Swap two serial complex double precision vectors.

   Input Parameters:
.  N     - Vector context
.  x,y   - vectors to swap
 @*/
void ZVswap( N, x, y )
VEDefaultUsrCntx *N;
register dcomplex *x, *y;
{
register int n = N->n, i;
dcomplex t;
for (i=0; i<n; i++) {
    t    = x[i];
    x[i] = y[i];
    y[i] = t;
    }
}

/*@
   ZVset - Set the values in a serial complex double precision vector
   to a scalar.

   Input Parameters:
.  N     - Vector context
.  alpha - value to set every element in vector to
.  x     - vector to set
 @*/
void ZVset( N, alpha, x )
VEDefaultUsrCntx *N;
register dcomplex *x;
dcomplex alpha;
{
register int n = N->n, i;

for (i=0; i<n; i++) 
    x[i] = alpha;
}

/*@
   ZVaxpy -  y = alpha*x + y, where x and y are serial complex double
   precision vectors.

   Input Parameters:
.  N     - Vector context
.  alpha - multiplier
.  x,y   - vectors
 @*/
void ZVaxpy( N, alpha, x, y )
VEDefaultUsrCntx *N;
dcomplex alpha, *x, *y;
{
register int n = N->n, i;
register double ai, ar;

for (i=0; i<n; i++) {
    ar     = x[i].r * alpha.r - x[i].i * alpha.i;
    ai     = x[i].r * alpha.i - x[i].i * alpha.r;
    y[i].r += ar;
    y[i].i += ai;
    }
}

/*@
   ZVaypx -  y = x + alpha*y where x and y are serial complex double
   precision serial vectors.

   Input Parameters:
.  N     - Vector context
.  alpha - multiplier
.  x,y   - vectors
 @*/
void ZVaypx( N, alpha, x, y )
VEDefaultUsrCntx *N;
dcomplex alpha, *x, *y;
{
register int n = N->n, i;
register double ai, ar;

for (i=0; i<n; i++) {
    ar     = y[i].r * alpha.r - y[i].i * alpha.i;
    ai     = y[i].r * alpha.i - y[i].i * alpha.r;
    y[i].r = x[i].r + ar;
    y[i].i = x[i].i + ai;
    }
}

/*@
   ZVwaxpy -  w = alpha*x + y, where w, x and y are serial complex double
   precision vectors.

   Input Parameters:
.  N     - Vector context
.  alpha - multiplier
.  w,x,y   - vectors
 @*/
void ZVwaxpy( N, alpha, x, y, w )
VEDefaultUsrCntx *N;
dcomplex alpha, *x, *y, *w;
{
register int n = N->n, i;
register double ai, ar;

for (i=0; i<n; i++) {
    ar     = x[i].r * alpha.r - x[i].i * alpha.i;
    ai     = x[i].r * alpha.i - x[i].i * alpha.r;
    w[i].r += y[i].r + ar;
    w[i].i += y[i].i + ai;
    }
}

/*@
   ZVpmult - Multiply the elements of one vector by another.

   Input Parameters:
.  N     - Vector context
.  x,y   - vectors to multiply together

   Output Parameter:
.  w     - w(i) <- x(i) * y(i)
 @*/
void ZVpmult( N, x, y, w )
VEDefaultUsrCntx  *N;
register dcomplex *x, *y, *w;
{
register int n = N->n, i;
for (i=0; i<n; i++) {
    w[i].r = x[i].r * y[i].r - x[i].i * y[i].i;
    w[i].i = x[i].i * y[i].i + x[i].i * y[i].r;
    }
}

/*@
   ZVpdiv - Divide the elements on one vector by another.

   Input Parameters:
.  N     - Vector context
.  x,y   - vectors to divide

   Output Parameter:
.  w     - w(i) <- x(i) / y(i)
 @*/
void ZVpdiv( N, x, y, w )
VEDefaultUsrCntx *N;
register dcomplex *x, *y, *w;
{
register int n = N->n, i;
register double r;
for (i=0; i<n; i++) {
    r    = y[i].r * y[i].r + y[i].i * y[i].i;
    w[i].r = (x[i].r * y[i].r + x[i].i * y[i].i) / r;
    w[i].i = (x[i].i * y[i].r - x[i].r * y[i].i) / r;
    }
}

/*@
   ZVobtain_vectors -  Returns a pointer to a list of pointers
   which point to newly obtained serial complex double precision vectors.
   To obtain a single serial complex double precision vector use 
   ZVCreateVector().
 
   Input Parameters: 
.     N - user context
.     m - number of vectors requested
 @*/
void **ZVobtain_vectors( N, m )
VEDefaultUsrCntx *N;
int              m;
{
void **v;
int  i, n = N->n;

v = (void **) MALLOC( m * sizeof(void *) );
for (i=0; i<m; i++)
    v[i] = (void *)MALLOC( n * sizeof(dcomplex) );
return v;
}

/*@
   ZVrelease_vectors -  Frees a set of serial complex double precision
   vectors obtained by ZVobtain_vectors().
 
   Input Parameters: 
.     N  -  user context
.     v  - pointer to set of pointers 
.     m  - number of vectors
 @*/
void ZVrelease_vectors( N, v, m )
VEDefaultUsrCntx *N;
int              m;
void             **v;
{
int i;
for (i=0; i<m; i++)
    FREE( v[i] );
FREE( v );
}

/*@
   ZVCreateVector - Returns a pointer to a  newly created complex serial
   double precision vector. Use ZVobtain_vectors() if
   several vectors are needed.
 
   Input Parameters:
.     N  - user context, the first element must be the vector length
 @*/
void *ZVCreateVector( N )
VEDefaultUsrCntx *N;
{
return (void *)MALLOC( N->n * sizeof(dcomplex) );
}

/*@
   ZVDestroyVector - Frees a serial complex double precision vector
   obtained  with ZVCreateVector().
 
  Input Parameters:
.    v  -  pointer to vector
 @*/
void ZVDestroyVector( v )
void *v;
{
FREE(v);
}

