/*

    WINDESIGN.C

    This file contains routines to produce various window looks.  The most
    notable is a routine that will produce a 3-D like box that can be used
    to frame various windows and window items.

*/

#include <X11/Xlib.h>
#include "tools.h"
#include "xtools/basex11.h"
#include "xtools/vfp/windesign.h"     /*I "xtools/vfp/windesign.h" I*/

#ifndef DEBUGTBC
#define DEBUGTBC(c)
#endif


/* @
   XBInitTextBox - Initilizes the structure for a text box

   Input Parameters:
.  x, y   - coodinates of upper left corner of the box.
.  width  - the width of the box
.  height - the height of the box
.  text_pos - value specifing the horizontal justification of the text
.  string   - the character string to appear in the box
@ */
XBTextBoxStruct *XBInitTextBox( x, y, width, height, text_pos, string ) 
     int    x, y,
            width,
            height,
            text_pos ;
     char  *string ;
{
  XBTextBoxStruct  *text_info ;
  
  text_info = NEW(XBTextBoxStruct); CHKPTRN(text_info);
  text_info->x        = x ;
  text_info->y        = y ;
  text_info->width    = width ;
  text_info->height   = height ;
  text_info->text_pos = text_pos ;
  text_info->string   = string ;
  return (XBTextBoxStruct*)text_info ;
}


/* @
    XBFreeTextBox - Take care of any needed clean up for a text box

    Input Parameter:
.   info - structure containing the text box information

@ */ 
XBFreeTextBox( info ) 
     XBTextBoxStruct  *info ;
{
  FREE( info );
}



/* @
    XBInit3DBox - Initilizes structures need for the 3-D box

    Input Parameters:
.   x,y    - coodinates for the upper left corner of the drawing surface.
.   height - the overall height of the box.
.   width  - the overal width of the box.
.   depth  - the depth for the box (to give 3-D look).
.   out    - specifies inward or outward apperance.
.   pixmap - pixmap value for the surface of the box for b&w display.
.   pixval - pixvalue for surface to the box for color display.
    notes:
    The x and y coodinates specified indicate the x and y corner of the actual
    drawing surface of the box.  The border of an inward box will extend to the 
    left and upward past and a upward box will extend to the right and downward.
    Also note that the bottom and right edge of a inward box is not visible.  The
    width and heights of the previous mentioned areas is that specified by the depth
    parameter. 
@ */
XB3DBoxStruct* XBInit3DBox( x, y, height, width, depth, out, pixmap, pixval ) 
    int      x, y,
             height,
             width,
             depth,
             out;
     Pixmap  pixmap ;
     PixVal  pixval ;
{
  XB3DBoxStruct* Box ;

  Box = NEW(XB3DBoxStruct) ; CHKPTRN( Box ) ;
  Box->xStart   = x ;    
  Box->yStart   = y ;   
  Box->nHeight  = height ;
  Box->nWidth   = width ;  
  Box->nDepth   = depth ;   
  Box->out      = out ;
  Box->pix      = pixmap ;
  Box->pixval   = pixval ;
  return (XB3DBoxStruct*)Box ;
}


/* @
    XBFree3DBox - Take care of any needed clean up for a 3D box

    Input Parameter:
.   info - structure containing the 3D box information

@ */ 
XBFree3DBox( info ) 
     XB3DBoxStruct  *info ;
{
  FREE( info );
}





/* @
    XBDraw3DBox - Draw a 3-D like box

    Input Parameters:
.   XBWin     - Window
.   Box       - box info

@ */
XBDraw3DBox( XBWin, Box ) 
XBWindow      *XBWin ;
XB3DBoxStruct *Box ; 
{ 
  int i, 
      nLength,
      nTextWidth ;

  if( Box->out )
    { 
      /* clear area to draw box on */
      XSetForeground( XBWin->disp, XBWin->gc.set, XBWin->background ) ;
      XFillRectangle( XBWin->disp, XBWin->win, XBWin->gc.set, Box->xStart, Box->yStart,
          Box->nWidth, Box->nHeight) ;
      XSetForeground( XBWin->disp, XBWin->gc.set, XBWin->foreground ) ;
      XSetLineAttributes( XBWin->disp, XBWin->gc.set, 1, LineSolid, CapButt, JoinMiter ) ; 
      /* fill rectangle and draw outline for 3-D outward look */
      /* draw rect outline */
      XDrawRectangle( XBWin->disp, XBWin->win, XBWin->gc.set,  Box->xStart, 
	  Box->yStart, Box->nWidth, Box->nHeight ) ;
      if( XBWin->maxcolors > 2 )
	{
	  /* printf( " 1 Color display - using pixval\n" ) ; */
	  XSetForeground( XBWin->disp, XBWin->gc.set, Box->pixval ) ;
	}
      else
	{
	  /* printf( " 1 B&W display - using pixmap\n" ) ; */
	  XSetFillStyle( XBWin->disp, XBWin->gc.set, FillStippled );
	  XSetStipple( XBWin->disp, XBWin->gc.set, Box->pix );
	  XSetForeground( XBWin->disp, XBWin->gc.set, Box->pix ) ;
	}

      XFillRectangle( XBWin->disp, XBWin->win, XBWin->gc.set, Box->xStart+1, Box->yStart+1,
         Box->nWidth-1, Box->nHeight-1) ;
      XSetForeground( XBWin->disp, XBWin->gc.set, XBWin->foreground ) ;

      if( XBWin->maxcolors <= 2 )
	XSetFillStyle( XBWin->disp, XBWin->gc.set, FillSolid );

      for( i=1; i <= Box->nDepth; i++) 
	{ 
	  XDrawLine( XBWin->disp, XBWin->win, XBWin->gc.set, Box->xStart + i,
	     Box->yStart + Box->nHeight + i, Box->xStart + Box->nWidth + i,
	     Box->yStart + Box->nHeight + i ) ;
	  XDrawLine( XBWin->disp, XBWin->win, XBWin->gc.set, Box->xStart + Box->nWidth + i, 
	     Box->yStart + i, Box->xStart + Box->nWidth + i, Box->yStart + Box->nHeight + i ) ; 
	}
    }
  else
    {
      /* clear area to draw box on */
      XSetForeground( XBWin->disp, XBWin->gc.set, XBWin->background ) ;
      XFillRectangle( XBWin->disp, XBWin->win, XBWin->gc.set, Box->xStart - Box->nDepth, 
          Box->yStart - Box->nDepth, Box->nWidth, Box->nHeight) ;
      XSetForeground( XBWin->disp, XBWin->gc.set, XBWin->foreground ) ;

      XSetLineAttributes( XBWin->disp, XBWin->gc.set, 1, LineSolid, CapButt, JoinMiter ) ; 

      /* fill rectangle and draw outline for 3-D inward look */
      /* draw rect outline */
      XDrawRectangle( XBWin->disp, XBWin->win, XBWin->gc.set,  Box->xStart - Box->nDepth,
          Box->yStart - Box->nDepth, Box->nWidth, Box->nHeight ) ;

      if( XBWin->maxcolors > 2 )
	{
	  /* printf( " 2 Color display - using pixval\n" ) ; */
	  XSetForeground( XBWin->disp, XBWin->gc.set, Box->pixval ) ;
	}
      else
	{
	  /* printf( " 2 B&W display - using pixmap\n" ) ; */
	  XSetFillStyle( XBWin->disp, XBWin->gc.set, FillStippled );
	  XSetStipple( XBWin->disp, XBWin->gc.set, Box->pix );
	  XSetForeground( XBWin->disp, XBWin->gc.set, Box->pix ) ;
	}
      XFillRectangle( XBWin->disp, XBWin->win, XBWin->gc.set, Box->xStart+1, 
          Box->yStart+1, Box->nWidth-Box->nDepth-1, Box->nHeight-Box->nDepth-1) ;
      XSetForeground( XBWin->disp, XBWin->gc.set, XBWin->foreground ) ;

      if( XBWin->maxcolors <= 2 )
	XSetFillStyle( XBWin->disp, XBWin->gc.set, FillSolid );

      for( i=1; i <= Box->nDepth; i++) 
	{ 
	  XDrawLine( XBWin->disp, XBWin->win, XBWin->gc.set, 
	      Box->xStart - Box->nDepth + i + 1, Box->yStart - Box->nDepth + i, 
	      Box->xStart + Box->nWidth - Box->nDepth + 1, Box->yStart - Box->nDepth + i ) ;
	  XDrawLine( XBWin->disp, XBWin->win, XBWin->gc.set, 
	      Box->xStart - Box->nDepth + i, Box->yStart -Box->nDepth + i + 1, 
	      Box->xStart - Box->nDepth + i, Box->yStart + Box->nHeight - Box->nDepth + 1 ) ; 
	}
    }
  XFlush( XBWin->disp ) ;

}


/* @
    XBDrawTextBox - Draws box containg a given string.

    Input Parameters:
.   XBWin   - window
.   font    - font
.   info    - information on the box

    notes:
    This is currently a rather weak function that can draw a box containing
    a string.  The background is used to shade in the box behind the text.
    Currently only one line strings are supported, but can be justified as 
    either right, left, or center.
@ */

XBDrawTextBox( XBWin, font, info ) 
     XBWindow      *XBWin ;
     XBFont        *font ;
     XBTextBoxStruct *info ;
{
  int   length, 
        text_width,
        text_x,
        text_y ;

  XSetForeground( XBWin->disp, XBWin->gc.set, XBWin->background ) ;
  XFillRectangle( XBWin->disp, XBWin->win, XBWin->gc.set, info->x+1, info->y+1,
      info->width-1, info->height-1) ;
  XSetForeground( XBWin->disp, XBWin->gc.set, XBWin->foreground ) ;
  XDrawRectangle( XBWin->disp, XBWin->win, XBWin->gc.set,  info->x, info->y,
      info->width, info->height) ;

  /* truncate string if necessary to fit in button */
  length = strlen( info->string ) ;
  text_width =  length * font->font_w ;
  while( !( text_width < info->width ) )
  {
    info->string[length-1] = (char)0 ;
    length = strlen( info->string ) ; 
    text_width =  length * font->font_w ;
  }
  switch( info->text_pos )
    {
    case CENTER:
      text_x = info->x + (int)((info->width-text_width)/2) ;
      break;
    case RIGHT:
      text_x = info->x + info->width - text_width ;
      break;
    case LEFT:
      text_x = info->x ;  /* 1 character space given */
      break ;
    }
  text_y = (int)(info->y + .5*info->height + .5*font->font_h - font->font_descent) ;
  XDrawString( XBWin->disp, XBWin->win, XBWin->gc.set, text_x, text_y, 
      info->string, length ) ;
  XFlush( XBWin->disp ) ;
}






