#ifndef lint
static char SCCSid[] = "@(#) ./blkcm/compile/findown.c 05/18/93";
#endif

/*
   This file contains routines to determine the owners of given id's
   and to propagate that information to the processors that need it.
   The algorithm is:

   The OWNER is taken as the DESTINATION of any data (note that the
   id field is the id of the DESTINATION).
   Each routine determines how many non-local owners it has (note that
   a single owner may contribute multiple destination operations).
   This number is summed over all processors.
   
   Space is allocated for a map of size (all owners) on each processor.
   All processors exchange ALL of their owner data.
   Each processor then uses this data to find the owners that they need 
   to know about (sorting the data and using a binary search).
   The space can then be freed.

   This algorithm trades space for time; the assumption is that the
   size of any individual program is small, and that even over the
   entire array, the number of distinct id's is small.  Further,
   it is assumed to be faster to do a global collection of data than 
   to send individual queries out.

   id's are globally unique.  We need to identify the PARTNER processor,
   not just the owner.  Finally, we don't want to assume that the program
   is valid (error-free), since we need to do this step before the 
   program can be checked for correctness.
   Algorithm: 
   1. For each SOURCE, find the destination.  This is done by collecting a
      list of id's and their matching processors and distributing it
      to every processor.
   2. For each DESTINATION, find the source.  This is done by matching
      the phase/id tuple of each destination with a source.  
      Since this is potentially very large, we send the information
      directly to the processor that needs it.  Termination is handled
      by knowing how much data is will be sent (so that this will
      work correctly even if the program is not valid).
 */

#include "tools.h"
#include "blkcm/bc.h"
#include "blkcm/bcp.h"
#include <search.h>

#define VALID_TYPE2 54321 
#define OLD_SORT

static int BCdist_senders();   /* IRIX needs this here */

#ifdef DISTRIBUTED_MEMORY
static int BCget_local_id();

typedef struct { int id, phase, owner, pad; } ownid;

/* Routines to pass to qsort or bsearch for comparisions */
#ifdef OLD_SORT
static int int_compare( p1, p2 )
int *p1, *p2;
{
return *p1 - *p2;
}
#endif /* OLD_SORT */
static int owner_compare( p1, p2 )
ownid *p1, *p2;
{
return p1->id - p2->id;
}
static int phaseId_compare( p1, p2 )
BCentry *p1, *p2;
{
if (p1->phase != p2->phase)
    return p1->phase - p2->phase;
if (GET_MAJOR_MODE(p1) != GET_MAJOR_MODE(p2))
    return GET_MAJOR_MODE(p1) - GET_MAJOR_MODE(p2);
return p1->id - p2->id;
}

/*
  Routine to return the owner processor 
 */
static int FindOwner( owners, nowners, val )
ownid *owners;
int   nowners, val;
{
ownid key, *result;
key.id = val;
result = (ownid *)bsearch( (char*)&key, (char *)owners, (unsigned)nowners, 
			   sizeof(ownid), (int (*)())owner_compare );
return result ? result->owner : -1;
}

/*
   Find the owners of all the id's.  All programs must call this routine if
   any one does.  Returns 0 on success, != 0 on failure.
 */
int BCfind_owner( Program )
BCPGM *Program;
{
ownid        *owners;
int          *ids, nids, n, nowners, i, nc, mypid;
BCentry   *pgm;

/* find all of the unique DESTINATION ids */
if (BCget_local_id( Program, &ids, &nids )) return 1;

TRPUSH(BCTRID+3);
/* Globally sum the count */
nowners = nids;

GISUM( &nowners, 1, &n, 0 );

/* Do a global exchange of all the owner data. */
owners = (ownid *) MALLOC( (nowners + nids) * sizeof(ownid) );
if (!owners) return 1;

/* Set the local values */
mypid = MYPROCID;
for (i=0; i<nids; i++) {
    owners[i].id    = ids[i];
    owners[i].owner = mypid;
    /* printf( "[%d] has id %d\n", mypid, ids[i] ); */
    }
FREE( ids );
GCOL( (char *)owners,          nids*sizeof(ownid), 
      (char *)(owners + nids), nowners*sizeof(ownid), &nc, 0 ,MSG_INT);


/* Sort the list */
qsort( (char *)(owners + nids), nowners, sizeof( ownid ), 
       (int (*)())owner_compare );

/* Run through the program filling in the sources */
pgm = Program->pgm;
n   = Program->n;
while (n--) {
    if (IS_SRC(pgm) && pgm->processor < 0) 
	pgm->processor = FindOwner( owners+nids, nowners, pgm->id );
    NEXTLINE(pgm);
    }
FREE( owners );

/* Now, we need to get the sources for all of the destinations.  The approach
   is:
   determine the number of sources (termination criteria)
   send to all my recepients the source
   receive all sources
 */
BCdist_senders( Program );

TRPOP;
return BCNO_ERROR;
}

/*
   This routine returns an array that contains the local id's of all
   of the entries in the Program.  The list is unique, and the size
   of the list is also returned.
 */
static int BCget_local_id( Program, ids, nids )
BCPGM *Program;
int   **ids, *nids;
{
int        *sources, *p, *p2, ns, n;
BCentry *pgm;

TRPUSH(BCTRID+4);
/* find all of the unique DESTINATION ids */
sources = (int *) MALLOC(  (Program->n + 1) * sizeof(int) );
if (!sources) return 1;

pgm = Program->pgm;
n   = Program->n;
p   = sources;
ns  = 0;
while (n--) {
    if (IS_DEST(pgm)) {
	*p++    = pgm->id;
	ns++;
	}
    NEXTLINE(pgm);
    }
if (ns > 0) {
#ifdef OLD_SORT
    qsort( (char *)sources, ns, sizeof(int), (int (*)())int_compare );
#else
    SYIsort( ns, sources );
#endif
    /* Merge duplicates */
    p = sources;
    p2= p+1;
    ns--;
    while (ns) {
	while (*p2 == *p && ns) {p2++; ns--;}
	if (ns) {
	    *++p = *p2++;
	    ns--; 
	    }
	}
    /* Globally sum the count */
    ns      = 1 + (int)(p - sources);
    }

*ids = sources;
*nids= ns;
TRPOP;
return BCNO_ERROR;
}

typedef struct { int id, phase, proc, pad; } sender;

/*
   This routine distributes the senders to all of the receivers 
 */
static int BCdist_senders( Program )
BCPGM *Program;
{
int        nrecv, n, mypid = MYPROCID;
sender     *buf, buff;
BCentry *pgm, key, *match;

nrecv       = BCget_nrecv( Program, 0 );
n           = Program->n;
pgm         = Program->pgm;
buf = (sender *) MALLOC( n*sizeof(sender) ) ; CHKPTRN(buf);

  while (n--) {
    if (IS_SRC(pgm) && pgm->processor != mypid && pgm->processor >= 0) {
	  buf[n].id    = pgm->id;
      buf[n].proc    = mypid;
	  buf[n].phase = pgm->phase;
	  SENDASYNCNOMEM( VALID_TYPE2, buf+n, sizeof(sender), 
		       pgm->processor, MSG_INT, pgm->rc );
	}
    NEXTLINE(pgm);
  }

/* Now, receive the messages and put the info in the program */
qsort( (char *)Program->pgm, Program->n, sizeof(BCentry), 
       (int (*)())phaseId_compare );
key.type  = BLOCK_COMM_DEST;
while (nrecv--) {
    RECVSYNCNOMEM(VALID_TYPE2, &buff, sizeof(sender), MSG_INT );
    /* match this phase/id up and set the processor */
    key.id    = buff.id;
    key.phase = buff.phase;
    match = (BCentry *)bsearch( (char *)&key, (char *)(Program->pgm), 
			        (unsigned)(Program->n),
				   sizeof(BCentry), 
			        (int (*)())phaseId_compare );
    if (match) match->processor = buff.proc;
    }

/* make sure all messages have been sent */
  n           = Program->n;
  pgm         = Program->pgm;
  while (n--) {
    if (IS_SRC(pgm) && pgm->processor != mypid && pgm->processor >= 0) {
	  PIwsend( VALID_TYPE2, buf+n, sizeof(sender), 
		       pgm->processor, MSG_INT, pgm->rc );
	}
    NEXTLINE(pgm);
  }
FREE(buf);
return BCNO_ERROR;
}
#endif /* DISTRIBUTED_MEMORY */
