/*
    Processor subset structure 
 */
#ifndef ALLPROCS
#include "set/set.h"

typedef struct {
    /* Identify this processor and the total number of processors.
       These are the GLOBAL numbers */
    int myid, nump;

    /* Tree links */
    int l_child, r_child, parent, root, am_left;

    /* Ring links */
    int right, left;

    /* Mesh size and links */
    int nx, ny;
    int north, south, east, west;

    /* This is a context value that is used to separate actions
       on different processor groups.  It should be combined with
       name (see the discussion of GMSGTYPE in global.h). */
    int ctxoff;

    /* A unique name for this processor set (all processors in this set
       must agree on this name) */
    int name;

    /* actual membership info */
    int npset;          /* Number of nodes in processor set */
    int lidx;           /* Index of this processor in node_nums */
    int *node_nums;     /* list of actual nodes; may be null if not provided */
    int *node_inums;    /* This is the inverse of node_nums; given a node,
			   we can find the location in node_nums of it */
    /* pset ABSOLUTELY MUST be the last element in the procset, since
       we "fake" having a variable-length structure.  See PSCreate */
    Set pset;
    } ProcSet;

extern void    PSDestroy(), PSCompile(), PSAddMember();
extern ProcSet *PSCreate(), *PSPartition(), *PSUnion();
extern int     PSNbrMesh(), PSNbrRing(), PSNbrTree();
extern void    PSPrintProcset(), PSPrintProcsetDetails();

typedef enum { PS_LCHILD, PS_RCHILD, PS_PARENT } PS_Tree_t;

#define ALLPROCS (ProcSet *)0

/*M
     PSnumtids - Returns the number of nodes in a processor set

     Synopsis:
     int PSnumtids( procset )
     ProcSet *procset;
M*/
#define PSNUMNODES(a) ((a)?(a)->npset:NUMNODES)

/*M 
     PSmytid - Returns the relative processor number in a processor set

     Synopsis:
     int PSmytid( procset )
     ProcSet *procset;

     Note:
     The values returned by this routine are between 0 and 
     PSnumtids(procset)-1.
M*/
#define PSMYPROCID(a) ((a)?(a)->lidx:MYPROCID)     

/*M
     PStidFromRank - Returns the process id from the rank in a processor set

     Synopsis:
     int PStidFromRank( procset, rank )
     ProcSet *procset;
     int     rank;
M*/
#define PStidFromRank(p,r) ((p)?(p)->node_nums[r]:r)

/*M
     PSPROCLIST - return the processors in the processor set.  This is
     the ordering used by GCOLX.

     Synopsis:
     void PSPROCLIST(procset,list)
     ProcSet *procset;
     int     *list;
M*/
#define PSPROCLIST(procset,list) \
if (procset == 0) { int __i;\
    for (__i=0;__i<NUMNODES;__i++) { \
        (list)[__i] = __i; \
    } \
} else { int __i;\
    for (__i=0;__i<(procset)->npset;__i++) { \
        (list)[__i] = (procset)->node_nums[__i]; \
    } \
}

/*M
    PSISROOT - return 1 if this processor is the root of the procset, 0 
    otherwise.

    Synopsis:
    int PSISROOT(procset)
    ProcSet *procset;
M*/
#define PSISROOT(procset) \
((PSNbrTree(PS_PARENT,procset) == -1) ? (1) : (0))
/* Why isn't this the same as (procset)->lidx == 0? */

/*M
    PSROOT - return global id of the root processor of this processor set

    Synopsis:
    int PSROOT(procset)
    ProcSet *procset;
M*/
#define PSROOT(procset) \
((procset)?(procset)->node_nums[0]:0)

/*M
    PSMESHLOC - return the location of the processor in the mesh
    
    Synopsis:
    void PSMESHLOC( procset, i, j )
    ProcSet *procset;
    int     *i, *j;

    Output Parameters:
.   i - x position in the mesh
.   j - y position in the mesh    
M*/
#define PSMESHLOC( procset, i, j ) {\
 *(i) = ((procset)->lidx % (procset)->nx);\
 *(j) = (((procset)->lidx / (procset)->nx) % (procset)->ny);}

#endif
