#ifndef lint
static char SCCSid[] = "@(#) ./sparse/dense/dnclamp.c 07/23/93";
#endif

/*
   This file contains a routine to generate a sparse matrix from a dense
   one by discarding "small" elements.
 */

#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"
#include "sparse/dense/dense.h"
#include <math.h>

/*@
    SpDnClampToSparse - Forms a sparse matrix from a dense one.

    Input Parameters:
.   mat - dense matrix
.   rtol - include only those elements that satisfy |v|>rtol * norm(row)/nr
.   n    - include only the largest n elements in magnitude

    Returns:
    A sparse (row-format) matrix that contains some of the elements of
    mat.   This may be used to form a preconditioner when using an
    iterative method to solve a dense linear system.
@*/
SpMat *SpDnClampToSparse( mat, rtol, n )
SpMat  *mat;
double rtol;
int    n;
{
SpMat  *nmat;
int    nr, nc, nrd, *c, i, j, cnt;
SpMatDense *dmat;
double *p, val, *v, norm;

dmat = (SpMatDense *)(mat->data);
nr   = mat->rows;
nc   = mat->cols;
nrd  = dmat->decl_rows;
nmat = SpCreate( nr, nc, n );    CHKERRN(1);
c    = (int *)MALLOC( nc * sizeof(int) );       CHKPTRN(c);
v    = (double *)MALLOC( nc * sizeof(double) ); CHKPTRN(v);
for (i=0; i<nr; i++) {
    /* Compute norm */
    if (rtol > 0) {
	p    = dmat->p + i;
	norm = 0.0;
	for (j=0; j<nc; j++) {
	    val = fabs( *p );   p += nrd;
	    norm += val*val;
	    }
	/* Form relative norm */
	norm = rtol * sqrt(norm) / nc;
	}
    else
	norm = 0;

    /* Get the largest elements (and the diagonal) */
    p    = dmat->p + i;
    cnt  = 0;
    for (j=0; j<nc; j++) {
	val = fabs( *p );   p += nrd;
	if (val > norm || j == i) {
	    c[cnt]   = j;
	    v[cnt++] = val;
	    }
	}

    /* If there are too many of them, sort and take the top.  Always
       include the diagonal */
    if (cnt > n) {
	SpiNDSort( cnt, v, c, 0, cnt-1 );
	cnt  = n;
	for (j=0; j<cnt; j++) 
	    if (c[j] == i) break;
	if (j >= cnt) 
	    c[cnt-1] = i;
	}

    /* Add to nmat */
    p    = dmat->p + i;
    for (j=0; j<cnt; j++) 
	v[j] = p[c[j]*nrd];
    SpAddInRow( nmat, i, cnt, v, c );      CHKERRV(1,0);
    }

FREE( c );
FREE( v );
return nmat;
}
