#ifndef lint
static char SCCSid[] = "@(#) ./system/sbcnst.c 07/23/93";
#endif

#include <stdio.h>
#include "tools.h"
/* #include "system/sbcnst.h" */

/*
   This file contains routines for allocating a number of fixed-sized blocks.
   This is often a good way to improve the performance of dynamic-memory
   codes, at the expense of some additional space.  However, unlike a purely
   static allocation (a fixed maximum), this mechanism allows space to grow.

   The basic interface is

  sb = SBinit( blocksize, initialnumber, incrementnumber );
  ptr = SBalloc( sb );
  ...
  SBfree( sb, ptr );
  ...
  SBdestroy( sb );

  This is much like the "pool" system used in the sparse matrix code, except
  that it provides ONLY fixed sized blocks.  New blocks are allocated as
  required.

  In order to provide a simplified system for freeing data, 
  
  The chunck allocator (used by sparse/pool) calls this to get space.
 */

/* This is the allocation unit. */
typedef struct _sbialloc {
    struct _sbialloc *next;
    int              nbytes, nballoc;
    int              nbinuse;
    } SBiAlloc;

/* Blocks are linked together; they are (much) larger than this */
typedef struct {
    char *next;
    } SBblock;

/* Context for fixed-block allocator */
typedef struct {
    SBiAlloc *blocks;	         /* allocated storage */
    SBblock  *avail;             /* fixed blocks (of size sizeb) to provide */
    int     nbfree, nballoc,     /* blocks free and in use */
            sizeb,               /* sizes in bytes */
            sizeincr;            /* # of blocks to allocate when more needed */
    } SBHeader;

void SBiAllocate();

SBHeader *SBinit( bsize, nb, nbincr )
int bsize, nb, nbincr;
{
SBHeader *head;

/* Make sure that the blocksizes are multiples of pointer size */
if (bsize < sizeof(char *)) bsize = sizeof(char *);

head           = NEW(SBHeader);  CHKPTRN(head);
head->nbfree   = 0;
head->nballoc  = 0;
head->sizeb    = bsize;
head->sizeincr = nbincr;
head->avail    = 0;
head->blocks   = 0;
if (nb > 0) {
    SBiAllocate( head, bsize, nb );
    CHKERRNC(1,"Failed to allocate space");
    }

return head;
}

/* @
    SBfree - return a fixed-sized block to the allocator
@ */    
void SBfree( sb, ptr )
SBHeader *sb;
void     *ptr;
{
((SBblock *)ptr)->next = (char *)(sb->avail);
sb->avail              = (SBblock *)ptr;
sb->nbfree++;
sb->nballoc--;
}

/*
    Internal routine to allocate space
 */
void SBiAllocate( sb, bsize, nb )
SBHeader *sb;
int      bsize, nb;
{
char     *p, *p2;
int      i, headeroffset;
SBiAlloc *header;

/* printf( "Allocating %d blocks of size %d\n", nb, bsize ); */
/* Double-align block */
headeroffset    = (sizeof(SBiAlloc) + sizeof(double) - 1) / sizeof(double);
headeroffset    *= sizeof(double);

p               = (char *) MALLOC( bsize * nb + headeroffset );  CHKPTR(p);
header          = (SBiAlloc *)p;
/* Place at header for list of allocated blocks */
header->next    = sb->blocks;
sb->blocks      = header;
header->nbytes  = bsize * nb;
header->nballoc = nb;
header->nbinuse = nb;

/* Link the list together */
p2 = p + headeroffset;
for (i=0; i<nb-1; i++) {
    ((SBblock *)p2)->next = p2 + bsize;
    p2 += bsize;
    }
((SBblock *)p2)->next = (char *)sb->avail;
sb->avail  = (SBblock *)(p + headeroffset);
sb->nbfree += nb;
}

/* @
    SBalloc - Gets a block from the fixed-block allocator.

    Input Parameter:
.   sb - Block context (from SBinit)

    Returns:
    Address of new block.  Allocates more blocks if required.
@ */
void *SBalloc( sb )
SBHeader *sb;
{
SBblock *p;
	
if (!sb->avail) {
    SBiAllocate( sb, sb->sizeb, sb->sizeincr );   /* nbincr instead ? */
    CHKERRN(1);
    }
p         = sb->avail;
sb->avail = (SBblock *)(p->next);
sb->nballoc++;
sb->nbfree--;
/* printf( "Allocating a block at address %x\n", (char *)p ); */
return (void *)p;
}	

/* @
    SBPrealloc - Insure that at least nb blocks are available

    Input Parameters:
.   sb - Block header
.   nb - Number of blocks that should be preallocated

    Notes:
    This routine insures that nb blocks are available, not that an
    additional nb blocks are allocated.  This is appropriate for the common
    case where the preallocation is being used to insure that enough space
    is available for a new object (e.g., a sparse matrix), reusing any
    available blocks.
@ */
void SBPrealloc( sb, nb )
SBHeader *sb;
int      nb;
{
if (sb->nbfree < nb) {
    SBiAllocate( sb, sb->sizeb, nb - sb->nbfree );
    }	
}

/* @
    SBdestroy - Destroy a fixed-block allocation context

@ */
void SBdestroy( sb )
SBHeader *sb;
{
SBiAlloc *p, *pn;

p = sb->blocks;
while (p) {
    pn = p->next;
    FREE( p );
    p = pn;
    }
FREE( sb );
}

/* Decrement the use count for the block containing p */
void SBrelease( sb, ptr )
SBHeader *sb;
void     *ptr;
{
char *p = (char *)ptr;
SBiAlloc *b = sb->blocks;
char *first, *last;

/* printf( "Releasing a block at address %x\n", (char *)ptr ); */
while (b) {
    first = ((char *)b) + sizeof(SBiAlloc) - 1;
    last  = first + b->nbytes - 1;
    if (p >= first && p <= last) {
	b->nbinuse--;
	break;
	}
    b = b->next;
    }
}

/* Release any unused chuncks */
void SBFlush( sb )
SBHeader *sb;
{
SBiAlloc *b, *bnext, *bprev = 0;

b = sb->blocks;
while (b) {
    bnext = b->next;
    if (b->nbinuse == 0) {
	if (bprev) bprev->next = bnext;
	else       sb->blocks  = bnext;
	sb->nballoc -= b->nballoc;
	FREE( b );
	}
    else 
	bprev = b;
    b = bnext;
    }
}

/* Print the allocated blocks */
void SBDump( fp, sb )
FILE     *fp;
SBHeader *sb;
{
SBiAlloc *b = sb->blocks, *bnext, *bprev = 0;

while (b) {
    fprintf( fp, "Block %x of %d bytes and %d chuncks in use\n", 
	     (char *)b, b->nbytes, b->nbinuse );
    b = b->next;
    }
}

void SBReleaseAvail( sb )
SBHeader *sb;
{
SBblock *p, *pnext;
	
p         = sb->avail;
while (p) {
    pnext = (SBblock *)(p->next);
    sb->avail = pnext;
    sb->nbfree--;
    SBrelease( sb, p );
    p     = pnext;
    }
}
