#ifndef lint
static char SCCSid[] = "@(#) ./comm/hosts.c 07/23/93";
#endif

#include <stdio.h>
#include <time.h>
#if !defined(__MSDOS__)
#include <sys/param.h>
#endif
#include <string.h>

#ifdef ipsc2
#include <sys/types.h>
#endif

#include "tools.h"
#include "comm/hosts.h"

Host *PIiAcceptHost();
void PIiAddToTable();
void PIiCopyHostsToTable();

/* For debugging */
static int dbg = 0;

/* 
   This file contains routines to search a database of possible hosts 
   for a distributed program and identifies potential hosts.  

   Notes:
   The method is to search the table for machines that have the
   correct architecture and are available for running a program
   with the requested resource values (programs which need more
   resources are prohibited; this makes them more friendly on 
   a distributed cluster of workstations).

   The process is basically to look at each entry and see if the entry
   is acceptable or not.  The rules are (in this order):
   
$   if (architecture does not match)    reject
$   if (owner DOES match)               accept
$   if (timeofday does not match)       reject
$   if (cpulimit  does not match)       reject
$   if (memlimit  does not match)       reject
$   if (pagefaultlimits does not match) reject

   On an accept, 
$  if machine is a multiprocessor, take as many processors as needed.
$  skip through machines with the same name (note that a machine many
   be mentioned several times in the hosts file, once for each limit
   category.  Typically, there are entries for different times of day).
   
 */

/*+
    PIReadInHostTable - Build a list of all available machines from a "standard"
    file of available machines.

    Input parameters:
.    fin    - file pointer to file with description
.    mem    - amount of memory that the program may use
.    tim    - amount of time that the program may use
.    Dbg    - if 1, generate debugging output (explain why machines were
              rejected or accepted).

    Output parameters:
.    table - table of acceptable hosts

The format of the file is
$ hostname architecture owner hs:ms-he:me days memory time pagefaults nproc nice
$ (%s       %s          %s    %d %d %d %d %s   %d    %d   %d         %d     %d
$ Days are of the form daystart-dayend.  The value of days is
$   M - Monday
$   Tu- Tuesday
$   W - Wednesday
$   Th- Thursday
$   F - Friday
$   S - Saturday
$   Su- Sunday.
$ There may be no spaces in the specification.
+*/    
HostTable *PIReadInHostTable( fin, mem, cpu, pfaults, Dbg )
FILE   *fin;
int    mem, cpu, pfaults, Dbg;
{
int    hh, mm;
char   owner[12];
char   localhost[MAXHOSTNAMELEN];
char   lastname[MAXHOSTNAMELEN];
char   *datestr;
int    dayofweek;
int    locallen, otherlen;
time_t timer;
struct tm *tm;
HOSTDATA hinfo;
int    np = 0, ne = 0, maxarch = MAXARCHES;
HostTable *table;
Host      *head = 0, *last = 0, *list;

dbg = Dbg;

table = NEW(HostTable); CHKPTRN(table);
MEMSET(table,0,sizeof(HostTable));


SYGetUserName( owner, 12 );

SYGetHostName( localhost, MAXHOSTNAMELEN);
#if !defined(RELIABLEHOSTNAME)
if (strchr(localhost,'.')) {
  locallen = strlen(localhost) - strlen(strchr(localhost,'.'));
}
else 
#endif
{
  locallen = strlen(localhost);
}

/* Get the time */
timer     = time( 0 );
tm        = localtime( &timer );
hh        = tm->tm_hour;
mm        = tm->tm_min;
/* Use datestr to determine the day-of-the-week */
datestr   = ctime( &timer );
dayofweek = PIConvertSysDay( datestr );

if (dbg) fprintf(stdout,"--------Hosts from File---------\n");

strcpy( lastname, "??" );
while (1) {
    if (!PIiReadHosts( fin, &hinfo )) break;
    if (hinfo.np == 0) continue;
    /* note localhost returned by SYGetHostName may be only hilgard */
    /* while listing in hosts database maybe hilgard.math.ucla.edu */
    /* therefore we only check up to first . This will fail if you */
    /* are using two machines at different sites with same first name */
    /* We must also be careful that "spnode1" does NOT match "spnode10" */
#if !defined(RELIABLEHOSTNAME)
    if (strchr(hinfo.name,'.')) {
	otherlen = strlen(hinfo.name) - strlen(strchr(hinfo.name,'.'));
	}
    else {
	otherlen = strlen(hinfo.name);
	}  
    if (otherlen == locallen && !strncmp(hinfo.name,localhost,locallen) ) {
#else
    if (!strcmp(hinfo.name,localhost) ) {
#endif
       /* this is host machine therefore only use hinfo.np - 1 processors */
       if (hinfo.np <= 1) continue; 
       else hinfo.np--;
    }
    if (!strcmp( lastname, hinfo.name )) continue;
    if (PIiTestOwner( owner, &hinfo, dbg )) {
      list = PIiAcceptHost( &hinfo, dbg );
      if (head) {last->next = list; last = list;}
      else { head = last = list;}
      PIiAddToTable( &hinfo, list, table );
      np += hinfo.np;
      ne ++;
      /* Skip machines with the same name */
      strcpy( lastname, hinfo.name );
      continue;
    }

    if (PIiTestLim( mem, hinfo.mem,    "mem", &hinfo, dbg )) continue;
    if (PIiTestLim( cpu, hinfo.cpu,    "cpu", &hinfo, dbg )) continue;
    if (PIiTestLim( pfaults, hinfo.pf, "Page faults", &hinfo, dbg )) continue;
    if (PIiTestDay( datestr, dayofweek, &hinfo, dbg )) continue;
    if (PIiTestTime( hh, mm, &hinfo, dbg )) continue;

    list = PIiAcceptHost( &hinfo, dbg );
    if (head) {last->next = list; last = list;}
    else { head = last = list;}
    PIiAddToTable( &hinfo, list, table );
    np += hinfo.np; 
    ne ++; 
    strcpy( lastname, hinfo.name );
  }
   
  /* allocate space for the pointers to the table entries */
  PIiCopyHostsToTable( table, ne, head );

  table->np = np;
  table->ne = ne; 
  table->na = PIiAddToArchList( table, head, maxarch );

  if (dbg) fprintf(stdout,"--------------------------------------------\n");
  return table; 
}

Host *PIiAcceptHost( hinfo, dbug )
HOSTDATA *hinfo;
int      dbug;
{
  Host *list;
  list       = NEW(Host); CHKPTRN(list);
  list->np   = hinfo->np;
  list->nice = hinfo->niceval;
  list->arch = PIStringToArch(hinfo->arch);
  strncpy( list->name, hinfo->name, MAXHOSTNAMELEN );
  if (dbug) {
      fprintf( stderr, 
	       "Accepted %s \t(arch = %s)\n", hinfo->name, hinfo->arch );
      }
  
  return list;
}

/*+ 
    PIDestroyHostTable - frees the space taken by a host table.

   Input parameters:
.    table - 
+*/
void PIDestroyHostTable(table)
HostTable *table;
{
  int i,n;
  
  n = table->ne;
  if (table->hosts) {
    for ( i=0; i<n; i++ ) {
      FREE(table->hosts[i]);
    }
    FREE(table->hosts);
  }
  n = MAXARCHES;
  for ( i=0; i<n; i++ ) {
    if (table->archtable[i].hosts) FREE(table->archtable[i].hosts);
    if (table->archtable[i].fname) FREE(table->archtable[i].fname); 
  }
}

/*+ 
    PIStringToArch - converts a string containing the name of 
        an architecture to an integer.

   Input parameters:
.    name - name of architecture

   Returns:
.    integer indicating architecture
+*/ 
Arch PIStringToArch(name)
char *name;
{
  if (!strcmp(name,"sun4")) return ARCHsun4;
  else if (!strcmp(name,"rs6000")) return ARCHrs6000;
  else if (!strcmp(name,"intelnx")) return ARCHintelnx;
  else if (!strcmp(name,"NeXT")) return ARCHNeXT;
  else if (!strcmp(name,"IRIX")) return ARCHIRIX;
  else if (!strcmp(name,"tc2000")) return ARCHtc2000;
  else if (!strcmp(name,"dec5000")) return ARCHdec5000;
  else return ARCHunknown;
}

/* 
  Convert a string of the form arch:arch:...:arch to value Arch values 
 */
void PIArchPathToArch( archpath, Na, iarch )
char *archpath;
int  *Na;
Arch iarch[];
{
char *p, *cdir;
int  na = 0;

while(archpath) {
    /* parse the acceptable arches from archpath */
    cdir = archpath;
    p    = strchr( archpath, ':' );
    if (p) {
	*p  = 0;
	archpath = p + 1;
	}
    else
	archpath = 0; 
    iarch[na] = PIStringToArch(cdir);
    if (iarch[na] == ARCHunknown) {
	fprintf(stderr,"Parsing unknown arch %s \n",cdir);
	SETERR(2);
	return;
	}
    na++;
    }
*Na = na;
}

/* Get the name of the executable.  Return null if there is no one that
   matches this architecture */
char *PIExecutableName( bname, format, path, cwd, iarch, larch )
char *bname, *format, *path, *cwd;
Arch iarch, larch;
{
char *name;

name = (char *) MALLOC( MAXPATHLEN ); CHKPTRN(name);

if (larch == iarch) {
    if (PIFullPathFromBase(bname,-1,format,path,name,MAXPATHLEN,cwd)) 
	return name;
    }
if (PIFullPathFromBase(bname,iarch,format,path,name,MAXPATHLEN,cwd))
    return name;

FREE(name);
return 0;
}


/*+ 
    PIArchToString - converts an integer to the string name of the 
        architecture it represents.

   Input parameters:
.    architecture integer

   Output parameters:
.    name - name of architecture
+*/ 
void PIArchToString(name,arch)
char *name;
Arch arch;
{
  /* should use case */
  if      ( arch == ARCHsun4 ) {strcpy(name,"sun4"); }
  else if ( arch == ARCHrs6000 ) {strcpy(name,"rs6000"); }
  else if ( arch == ARCHIRIX ) {strcpy(name,"IRIX"); }
  else if ( arch == ARCHNeXT ) {strcpy(name,"NeXT"); }
  else if ( arch == ARCHintelnx ) {strcpy(name,"intelnx"); }
  else if ( arch == ARCHtc2000 ) {strcpy(name,"tc2000"); }
  else if ( arch == ARCHdec5000 ) {strcpy(name,"dec5000"); }
  else {strncpy(name,"unknown",7); }
  return;
}
/*+ 
    PIBuildHostTable - Given a complete hosttable obtained by 
      PIReadInHostTable(), forms the list of hosts to be actually used.

   Input parameters:
.    bname - base name of process
.    np - number of processors requested.
.    table - original host table
.    argc, argv -  the original command line arguments
.    path - path to look for executables
.    archpath - list of architectures to use
.    format - not yet used

   Returns:
.    utable - table of hosts to actually be used
+*/
HostTable *PIBuildHostTable(bname,np,table,argc,argv,path,archpath,
                            format,narch,cwd, debug)
char      ***argv,*path,*archpath,*format,*bname, *cwd;
int       *argc,np, debug;
HostTable *table;
Arch      narch;
{
  HostTable *utable;
  int       count;
  Arch      iarch;
  int       iarchi;
  char      *name;
  Arch      iarchs[MAXARCHES];
  int       i, ia, na, Np = np + 1;

  utable = NEW(HostTable); CHKPTRN(utable);
  MEMSET(utable,0,sizeof(HostTable));

  if (!archpath) {
    fprintf(stderr,"PIBuildHostTable called without any arches \n");
    SETERR(1);
    return 0;
  }

  if (debug) fprintf(stdout,"-----------Processors used ------------\n");

  PIArchPathToArch( archpath, &na, iarchs );     CHKERRV(2,0);

  for (ia = 0; ia < na; ia++) {
      iarch = iarchs[ia];
      iarchi = (int)iarch;

      if (dbg) {
	  char arname[20];
	  PIArchToString( arname, iarch );
	  fprintf( stderr, "Looking for architecture %s\n", arname );
	  }
      /* look for the executable */
      if ( table->archtable[iarchi].np > 0 ) {
	  name = PIExecutableName( bname, format, path, cwd, iarch, narch );
	  if (!name) continue;
	  
	  utable->archtable[iarchi].fname = name;
	  
	  /* loop over all entries for narch getting as many as needed */
	  count = 0;
	  utable->archtable[iarchi].hosts = (Host **)
	      MALLOC( table->archtable[iarchi].ne*sizeof(Host) ); 
	  CHKPTRN(utable->archtable[iarchi].hosts);
	  for ( i=0; i<table->archtable[iarchi].ne; i++ ) {
	      utable->archtable[iarchi].hosts[i] = 
		  table->archtable[iarchi].hosts[i];
	      if (debug) fprintf(stdout,"%s running %s\n",
				 utable->archtable[iarchi].hosts[i]->name,
				 utable->archtable[iarchi].fname);
	      if ( np < table->archtable[iarchi].hosts[i]->np ) {
		  count += np; 
		  utable->archtable[iarchi].hosts[i]->np = np; /* BAD; corrupts table*/
		  utable->archtable[iarchi].ne = i+1;
		  utable->archtable[iarchi].np = count;
		  np = 0;
		  return utable;
		  }
	      count += table->archtable[iarchi].hosts[i]->np;
	      np -= table->archtable[iarchi].hosts[i]->np;
	      if (!np) { 
		  utable->archtable[iarchi].ne = i+1;
		  utable->archtable[iarchi].np = count;
		  return utable;
		  }
	      } 
	  } 
      /* 
      else if (debug) {
	  fprintf(stderr,"There are no processors from %s \n",cdir);
	  }
       */
      utable->archtable[iarchi].ne = table->archtable[iarchi].ne;
      utable->archtable[iarchi].np = table->archtable[iarchi].np;
      } 
  if (debug) fprintf(stdout,"------------------------------------------\n");
  if (np) {
      fprintf(stderr,
   "Could not find enough acceptable processors (%d of %d found)\n", 
	      Np-np, Np );
      SETERR(1);
      return 0;
      }
return utable;
}

void PIPrintHostTable( utable, fp )
HostTable *utable;
FILE      *fp;
{
int       np, i, j;
int       maxarches = MAXARCHES;
char      *machine,*program;
char      localhost[MAXHOSTNAMELEN];

/* Always include the local machine */
SYGetHostName( localhost, MAXHOSTNAMELEN);
fprintf( fp, "%s %d %s\n", localhost, 1, " " );
for ( i=0; i<maxarches; i++ ) {
    if (utable->archtable[i].np) {
	for ( j=0; j<utable->archtable[i].ne; j++ ) {
	    machine = utable->archtable[i].hosts[j]->name;
	    np      = utable->archtable[i].hosts[j]->np;
	    program = utable->archtable[i].fname;
#if defined(DIFFERENTHOMEDIRECTORYNAMES)
	    SYRemoveHomeDir(program);
#endif
	    fprintf(fp, "%s %d %s \n",machine, np, program );
	    }   
	}
    }
}

extern char *getenv();
#include "comm/pgcomm/pgcomm.h"

/* This routine forms the host table, given a desired number and type of
   processors.

   The order of processing is as follows:
   If there is no generic hostslist file AND no host list, use the
   "hosts" file to find processors.

   else, use that list to specify the hosts (not yet implemented).

   Returns -1 on failure, 0 on success 
 */
int PIiFormHostTable( utable, table, np, mem, cpu, pf, arch, bname, cwd, 
		      Argc, Argv, hostsfile, defhostsfile )
HostTable **utable, **table;
int       np, mem, cpu, pf;
char      *arch, *bname, *cwd;
int       *Argc;
char      ***Argv;
char      *hostsfile, *defhostsfile;
{
char      *archpath,*path, hostfile[MAXPATHLEN], fname[MAXPATHLEN];
char      archstring[MAXPATHLEN];
Arch      narch;
FILE      *fd;
PGCommAll *pg = 0;

/* get environmental variables needed  */
SYGetwd( cwd, MAXPATHLEN );
archpath = getenv("TOOLSARCHES");
if (!archpath) {
    if (SYArgGetString( Argc, *Argv, 1, "-piarches", archstring, MAXPATHLEN ))
	archpath = archstring;
    if (!archpath) {
	archpath = arch;
	}
    }
path     = getenv("PATH"); if (!path) path = cwd;
narch    = PIStringToArch( arch );

dbg      = SYArgHasName( Argc, *Argv, 1, "-pidbg" );
if (SYArgGetString( Argc, *Argv, 1, "-pifile", fname, MAXPATHLEN )) {
    pg = PGFileToList( fname );
    }
else if (SYArgGetString( Argc, *Argv, 1, "-pihosts", fname, MAXPATHLEN )) {
    pg = PGStringToList( fname );
    }
if (pg) {
    PGtoHostTables( pg, utable, table,
		    bname, path, cwd );
    PGDestroy( pg );
    }
else {
    /* Get database of available processors */
    SYGetFileFromEnvironment( hostsfile, defhostsfile, "hosts", 
			      hostfile, 'r' ); 
    fd = fopen(hostfile,"r"); 
    if (!fd) {
	fprintf(stderr,"Could not open hosts database %s\n",hostfile);
	fprintf( stderr, "setenv %s to the hosts database\n", hostsfile );
	SETERR(1); return -1;
	}
    *table = PIReadInHostTable( fd, mem, cpu, pf, dbg ); fclose(fd); 
    if (!*table) {SETERRC(2,"Could not read in host table"); return -1;}
    fclose( fd );
    
    *utable = 
	PIBuildHostTable( bname, np, *table, Argc, Argv, path, archpath, 
			 (char *)0, narch, cwd, dbg ); 
    if (!*utable) {SETERRC(3,"Could not build host table"); return -1;}
    }
return 0;
}

void PIiAddToTable( hinfo, list, table )
HOSTDATA  *hinfo;
Host      *list;
HostTable *table;
{
int iarchi;
if ((int)(list->arch) >= 0 && (int)(list->arch) < MAXARCHES) {
    iarchi = (int)list->arch;
    table->archtable[iarchi].np += hinfo->np;
    table->archtable[iarchi].ne ++;
    }
}

/* Add hosts to the individual architecuture lists, return the number
   of architectures with at least one member */
int PIiAddToArchList( table, head, maxarch )
HostTable *table;
Host      *head;
int       maxarch;
{
int       i, j, k, na, ne;
Host      *last;
ArchTable *archtable;

na = 0;
ne = table->ne;
/* For each arch */
for ( j=0; j<maxarch; j++ ) {
    archtable = &table->archtable[j];
    if ( archtable->ne ) {
	na++;
	archtable->hosts = (Host **) MALLOC(archtable->ne*sizeof(Host *));
	CHKPTRN(archtable->hosts);
	last = head;
	k    = 0;
	for ( i=0; i<ne; i++ ) {
	    if ( (int)(last->arch) == j ) { archtable->hosts[k++] = last;}
	    last = last->next;
	    }
	}
    }
return na;
}

void PIiCopyHostsToTable( table, ne, head )
HostTable *table;
int       ne;
Host      *head;
{
int  i;
Host *last;

/* allocate space for the pointers to the table entries */
table->hosts = (Host **) MALLOC(ne*sizeof(Host *));  CHKPTR(table->hosts);
last = head;
for ( i=0; i<ne; i++ ) {
    table->hosts[i] = last;
    last = last->next;
    }
}

/* Copy a host table */
HostTable *PICopyHostTable( table )
HostTable *table;
{
HostTable *new;
int i, j;

new = NEW(HostTable); CHKPTRN(new);
MEMSET(new,0,sizeof(HostTable));

/* Copy host pointers */
new->ne    = table->ne;
new->np    = table->np;
new->na    = table->na;
for (i=0; i<MAXARCHES; i++) {
    new->archtable[i] = table->archtable[i];
    if (table->archtable[i].np > 0) {
	new->archtable[i].hosts = 
	    (Host **) MALLOC( table->archtable[i].ne*sizeof(Host) );
	CHKPTRN(new->archtable[i].hosts);
	for (j=0; j<table->archtable[i].ne; j++) {
	    new->archtable[i].hosts[j] = table->archtable[i].hosts[j];
	    }
	}
    }
return new;
}
