#ifndef lint
static char SCCSid[] = "@(#) ./comm/ialog.c 07/23/93";
#endif

/* This file contains routines to initialize the log files */
/*
  In previous versions, this was enabled only for the profiling version.
  However, that is rather restrictive.  Further, compile-time control,
  while useful in keeping the size of the image down (and removing 
  extra costs from the computation), does not permit easy debugging
  by runtime control (such as a command line parameter to enable 
  communication logging).

  In addition, it is useful to be able to control whether Chameleon operations
  are considered atomic or not.  For example, a gscatter or the setup code.

  Enhancements to consider:
  Pass the file and line to the event log routine so that the event file can
  contain the file and line number that generated the event.  The AIMS (Ames
  InstruMentation System does this, allowing you to click on an event and
  get the source code displayed).
 */

#include <string.h>
#include "tools.h"

#if !defined(LOGCOMMDISABLE) && !defined(LOGCOMMEVENTS) && !defined(LOGCOMMALL)
#define LOGCOMMEVENTS
#endif

#include "comm/comm.h"
/* global.h is needed to get a valid system message type */
#include "comm/global/global.h"
#include <stdio.h>

/* Define the logging level.  For now, all logging is disabled unless
   specifically enabled.  */
int        OnlyAtomic = 1, inAtomicOp = 0;
static int MergeLogFiles = 1;
int        cloglevel = 0;
static int initlog   = 0;
static int didinit   = 0;
FILE *_tracefile     = stdout;

/* By making the message buffer a static, we can print the last entry out
   on an exception if the trace option is enabled.  The same buffer is used
   for error messages */
static char buf[256];

/*@
  PISetMergeEventFiles - Control whether event files are merged

  Input Parameter:
. flag - 
@*/
void PISetMergeEventFiles( flag )
int flag;
{
MergeLogFiles = flag;
}

/*@
   PISetLogging - Set the logging level for communications

   Input Parameter:
.  level - level to use:
$             0 - no logging
$             1 - event logs (to a file for postmortum use)
$             2 - print out for each send and recv
$             4 - keep a summary of times and counts
$             8 - print out LAST send or recv on error exit (interrupt)
$   Or the bits together to get the combinations (ie, 3 is print + log).
@*/
void PISetLogging( level )
int level;
{
cloglevel = level;
if (cloglevel & 0x1) LOGENABLE;
else                 LOGDISABLE;
}
/*@
   PISetLoggingBit - Set/Clear the logging level for communications, for a 
                     single option

   Input Parameter:
.  bit - option to set
$             0 - no logging
$             1 - event logs (to a file for postmortum use)
$             2 - print out for each send and recv
$             4 - keep a summary of times and counts
$   Or the bits together to get the combinations (ie, 3 is print + log).
.  flag - whether to set or clear the option.  1 = set, 0 = clear
@*/
void PISetLoggingBit( bit, flag )
int bit, flag;
{
if (flag) {
    cloglevel |= bit;
    if (bit == 0x1) { LOGENABLE; initlog = 1; }
    }
else {
    cloglevel = cloglevel & (~bit);
    if (bit == 0x1) LOGDISABLE; /* ? initlog = 0; ? */
    }
}

/* 
   In the past, this routine was set up to either issue the logging events
   or not, defined at COMPILE time.  This made it impossible to select this
   at RUNTIME.  
 */
PIinitlog()
{
#if defined(LOGCOMMEVENTS) && !defined(LOGCOMMDISABLE)
if (initlog && !didinit) {
    didinit = 1;
    if (PImytid == 0) {
	/* printf( "I am the master logger\n" ); */
	/* Alog support has been removed */
	BLOG_MASTER(BLOG_TRUNCATE);
	LOGDEFINE(LOG_RECVWS,"Wait on recv");
	LOGDEFINE(LOG_RECVWE,"End recv wait");
	LOGDEFINE(LOG_RECVS,"Start recv");
	LOGDEFINE(LOG_RECVE,"End recv");
	LOGDEFINE(LOG_SENDS,"Start send");
	LOGDEFINE(LOG_SENDE,"End send");
	LOGDEFINE(LOG_GSUMS,"Start GSum");
	LOGDEFINE(LOG_GSUME,"End GSum");
	LOGDEFINE(LOG_GMAXS,"Start GMax");
	LOGDEFINE(LOG_GMAXE,"End GMax");
	LOGDEFINE(LOG_GMINS,"Start GMin");
	LOGDEFINE(LOG_GMINE,"End GMin");
	LOGDEFINE(LOG_GCOLS,"Start GCollection");
	LOGDEFINE(LOG_GCOLE,"End GCollection");
	LOGDEFINE(LOG_GSCATS,"Start GScatter");
	LOGDEFINE(LOG_GSCATE,"End GScatter");
	LOGDEFINE(LOG_GSYNCS,"Start GSync");
	LOGDEFINE(LOG_GSYNCE,"End GSync");
	LOGSTATE(LOG_RECVS,LOG_RECVE,"green","Receiving");
	LOGSTATE(LOG_SENDS,LOG_SENDE,"blue","Sending");
	LOGSTATE(LOG_GSYNCS,LOG_GSYNCE,"yellow","Sync");
	LOGSTATE(LOG_GSUMS,LOG_GSUME,"purple","GlobalSum");
	LOGSTATE(LOG_GCOLS,LOG_GCOLE,"magenta","GlobalCol");
	LOGSTATE(LOG_GMINS,LOG_GMINE,"firebrick","GlobalMin");
	LOGSTATE(LOG_GMAXS,LOG_GMAXE,"Olivedrab","GlobalMax");
	LOGSTATE(LOG_GSCATS,LOG_GSCATE,"cyan","Scatter");
	LOGSTATE(LOG_GTOKS,LOG_GTOKE,"white","Gtoken");
	LOGSTATE(LOG_RECVWS,LOG_RECVWE,"red","RecvWait");
	}
    else {
	/* printf( "Calling alog_setup with %d\n", PImytid ); */
	BLOG_SETUP(BLOG_TRUNCATE);
	}
    /* Add the information needed to adjust the clocks */
    BLOG_FIX_CLOCKS();
    }
#endif
}

PIendlog()
{
#if !defined(LOGCOMMDISABLE)
if (initlog && didinit) {
    /* Add the sync event at the end of the run */
    PIgsync((ProcSet *)0);
    LOGEVENT(EVENT_SYNC);
    
    if (MergeLogFiles) {
	xx_BLOG_adjusttimes();
	/* Merge replaces xx_BLOG_dump() */
	BLOGMergeSetup( BLOGIsPicl() ? "bl.trf" : "bl" );
	BLOGParallelMerge( (void (*)())0 );
	}
    else {
	BLOG_OUTPUT;
	}
    didinit = 0;
    }
/* This might be a nice place to generate some summary data */
if (cloglevel & 0x4) {
    int i;
    /* Turn off logging of the GTOKEN */
    cloglevel = 0;
    for (i=0; i<=NUMNODES; i++) {
	if (GTOKEN((ProcSet*)0,i)) {
	    fprintf( stdout, "Summary for node %d:\n", PImytid );
	    PILogSummary( stdout );
	    }
	}
    }
#endif
}

/*
   This routine generates a summary to the given output file for the 
   calling processor.
   So far, a reasonable way to use this might be:
   for (i=0; i<=NUMNODES; i++)
       if (GTOKEN((ProcSet*)0,i)) {
           fprintf( stdout, "Summary for node %d:\n", PImytid );
	   PILogSummary( stdout );
	   }

    An alternate approach could generate some kind of graphical summary,
    or several processors per line, like
 
    processor 0            1            2      ...
             time[#]<len>    ...

    Finally, communication rates and fits to models would be nice.  For
    example, for the simple latency + n*rate method, if we assume that
    the formula is accurate, and that there are m sends, then we have
    time = m * lat + rate * len .   We need one more equation.  For example,
    we could use
    sum(lat + rate*len[i]==t[i]) = time
    sum(t[i]/len[i]) = sum(lat/len[i] + rate) = m*rate + lat*sum(1/len[i])
    For this, we'd also have to save the sum of the inverses of the lengths, 
    and the sum of the times, scaled by the lengths.
 */
PILogSummary( fp )
FILE *fp;
{
double r;
	
/* Do the most detailed preferentially */
#if defined(LOGCOMMSUMMARY) && !defined(LOGCOMMDISABLE)
fprintf( fp, "Op:           calls      bytes    total time Average rate\n" );
if (_state_send.count > 0) {
    if (_state_send.time > 0) r = _state_send.data / _state_send.time;
    else 		      r = 0.0;
    fprintf( fp, "Send:    %10d %10d %13.4le %12.4le\n", 
	    _state_send.count, _state_send.data, _state_send.time, r );
    }	    
if (_state_recvwait.count > 0) {
    fprintf( fp, "Recvwait:%10d %10d %13.4le\n",
	     _state_recvwait.count, _state_recvwait.data, 
	     _state_recvwait.time );
    }
if (_state_recv.count > 0) {
    if (_state_recv.time > 0) r = _state_recv.data / _state_recv.time;
    else 		      r = 0.0;
    fprintf( fp, "Recv:    %10d %10d %13.4le %12.4le\n", 
	     _state_recv.count, _state_recv.data, _state_recv.time, r );
    }
if (_state_gop.count > 0) {
    if (_state_gop.time > 0) r = _state_gop.data / _state_gop.time;
    else 		     r = 0.0;
    fprintf( fp, "Global:  %10d %10d %13.4le %12.4le\n", 
	    _state_gop.count, _state_gop.data, _state_gop.time, r );
    }

#elif defined(LOGCOMMCOUNTS) && !defined(LOGCOMMDISABLE)
fprintf( fp, "Op:     calls    bytes\n" );
fprintf( fp, "Send:   %10d %10d %13.4le\n", _logsend, _logsendn );
fprintf( fp, "Recv:   %10d %10d %13.4le\n", _logrecv, _logrecvn );

#endif

/* A more advanced estimate of the state and times could be done as
   follows.  Save not only the length of the message n_i and the time t_i,
   but also n_i^2 and n_i t_i.
   Define

   N0 = \sum 1
   N1 = \sum n_i
   N2 = \sum n_i^2
   T0 = \sum t_i
   T1 = \sum n_i t_i

   Then, if we model communication by s + rn, we have the equations

   T0 = \sum(s + r n_i) = s N0 + r N1
   T1 = \sum(n_i(s+r n_i)) = s N1 + r N2

   and the resulting estimate for s

       N2 T0 - N1 T1
   s = -------------
       N0 N2 - N1 N1

   and for r

       N0 T1 - N1 T0
   r = -------------
       N0 N2 - N1 N1

   Note that if all the n_i's are the same, the system is singular.
   It can also be very poorly conditioned, though we can compute how
   badly from the singular values (just the eigenvalues, since the
   matrix is symmetric).  The eigenvalues are

   (1/2) ( N2 + N0 +/- sqrt( (N2-N0)^2 + 4N1^2 ) )
   
   The ratio of these is the condition number of the matrix.  This
   value can be computed (carefully so as to avoid overflow); if it is
   small enough, the solution of the system is well-behaved.  Whether
   the results are meaningful is another matter...
 */
}

/* 
   There are a number of places where we need to temporarily turn off 
   logging, then restore logging to its previous state.  This means that
   we need to remember the previous state and restore it (mostly, this
   happens around the global operations, but it would be nice if this
   worked correctly always
 */
#if !defined(LOGCOMMDISABLE)
#define MAX_STACK 25
static int logstatus[MAX_STACK+1];
static int logsp = 0;
void LOGpush()
{
if (logsp < MAX_STACK)
    logstatus[logsp++] = LOGSTATUS;
}
void LOGpop()
{
if (logsp > 0) logsp--;
LOGSTATUS = logstatus[logsp];
}
#else
/* Stubs incase these are used in their raw form (as they are in initp4) */
void LOGpush()
{}
void LOGpop()
{}
#endif 

#if !defined(LOGCOMMDISABLE)
/* This routine takes an event number and returns the name of the related op */
char *_PIEventName( n )
int n;
{
switch (n) {
    case LOG_GSUMS:
    case LOG_GSUME: return "GSUM";
    case LOG_GMAXS:
    case LOG_GMAXE: return "GMAX";
    case LOG_GMINS:
    case LOG_GMINE: return "GMIN";
    case LOG_GCOLS:
    case LOG_GCOLE: return "GCOL";
    case LOG_GSCATS:
    case LOG_GSCATE: return "GSCATTER";
    case LOG_GSYNCS:
    case LOG_GSYNCE: return "GSYNC";
    case LOG_GTOKS:
    case LOG_GTOKE: return "GTOKEN";
    }
return "Unknown";
}
#else
char *_PIEventName( n )
int n;
{
return "Unavailable event name";
}
#endif

/*@
   PISetTracefile - set the name of the file for tracing output

   Input Parameter:
.  name - name of trace file.  It may contain a "%d", in which case
          the %d is replaced with the value of PImytid.
@*/
void PISetTracefile( name )
char *name;
{
char filename[1024];

if (strchr( name, '%' )) {
    sprintf( filename, name, PImytid );
    _tracefile = fopen( filename, "w" );
    }
else
    _tracefile = fopen( name, "w" );

if (!_tracefile)
    _tracefile = stdout;
}

/*
   In order to simplify the collection and display of data on 
   communications, I'm introducing a few functions that 
   will be called instead of the current macros.

   These are

   PIiSendStart( from, to, len, type, stype, file, line )
   PIiSendEnd(   from, to, len, type, stype, file, line )

   PIiRecvWait( to, type, stype, file, line )
   PIiRecvReady( to, type, stype, file, line )

   PIiRecvStart( to, type, stype, file, line )
   PIiRecvEnd( from, to, len, type, stype, file, line )

   These routine will contain all of the logging, state management, 
   event tracing, and so on.  

   The production versions will continue to have none of these.

   The usual use is
   PIiSendStart(...) send() PIiSendEnd(...)

   PIiRecvWait(...) probe() PIiRecvReady(...) PIiRecvStart(...) recv()
       PIiRecvEnd(...)

   Disadvantages:
   If these routines are used, you can't control the amount of logging
   at compile time.  Still, I think that this may be a better way
   to go than the current version

   The "file" and "line" args should be __FILE__ and __LINE__ .
   These allow error messages to be directed to the point where the
   routines were called.

   Finally, since we are calling routines here, we'll always provide
   all of the services (the additional overhead is slight)
 */

/* Redefine SETERRC to use "line,file" for the line and file. */
#undef SETERRC
#define SETERRC(a,s)      {__terrno.err = a;DEBUGTBCF(line,file,s);}

/* vfp... is basically a hook for arbitrary additional processing.
   The code is intended to be used to for a x-graphics virtual-front-panel,
   but the default is null to allow applications to run without X.
   
   The routine PIVFPInit() should be called to set up such a virtual front
   panel.
 */
static void (*vfpdraw)() = 0;
static void *vfpctx    = 0;

void PIiSendStart( to, len, type, stype, file, line )
int  to, len, type, line;
char *stype, *file;
{
if (type < -1) { 
    sprintf( buf, "Invalid tag (%d) in send", type );
    SETERRC(1,buf);
    }
if (len < 0) {
    sprintf( buf, "Invalid (negative) length (%d) in send", len );
    SETERRC(1,buf);
    }
if (to < 0 || to >= NUMNODES) {
    sprintf( buf, "Invalid destination (%d)", to );
    SETERRC(1,buf);
    }

#if defined(LOGCOMMCOUNTS) && !defined(LOGCOMMDISABLE)
_logsend++ ;
_logsendn += len;
#endif

if((cloglevel & 0xA) && !((OnlyAtomic && inAtomicOp))) {
    sprintf( buf, "[%d] %s <Tag %d(%d) To %d> [%s:%d]\n", 
	    PImytid, stype, type, len, to, file, line );
    if (cloglevel & 0x2) {
        fputs( buf, _tracefile );
        fflush(_tracefile);
	}
    }
#if defined(LOGCOMMSUMMARY) && !defined(LOGCOMMDISABLE)
if(cloglevel & 0x4)
    SYStatePush(&_state_send,len);
#endif

#if defined(LOGCOMMEVENTS) && !defined(LOGCOMMDISABLE)
if (!(OnlyAtomic && inAtomicOp))
    BLOG_LOG4(LOG_SENDS,len,type,to);
#endif
if (vfpdraw)
    (*vfpdraw)( vfpctx, PImytid, PImytid, to, len, LOG_SENDS );
}

void PIiSendEnd( to, len, type, stype, file, line )
int  to, len, type, line;
char *stype, *file;
{
if (!(OnlyAtomic && inAtomicOp))
    LOGEVENT(LOG_SENDE);
#if defined(LOGCOMMSUMMARY) && !defined(LOGCOMMDISABLE)
if(cloglevel & 0x4)
    SYStatePop(&_state_send,0);
#endif
if (vfpdraw)
    (*vfpdraw)( vfpctx, PImytid, PImytid, to, len, LOG_SENDE );
}

/* stype should be something like "recvwait" */
void PIiRecvWait( type, stype, file, line )
int  type, line;
char *stype, *file;
{
#if defined(LOGCOMMSUMMARY) && !defined(LOGCOMMDISABLE)
if(cloglevel & 0x4)
    SYStatePush(&_state_recvwait,0);
#endif
if((cloglevel & 0xA) && !(OnlyAtomic && inAtomicOp)) {
    sprintf( buf, "[%d] %s <Tag %d> [%s:%d]\n", 
	     PImytid, stype, type, file, line );
    if (cloglevel & 0x2) {
        fputs( buf, _tracefile );
        fflush(_tracefile);
        }
    }
if (vfpdraw)
    (*vfpdraw)( vfpctx, PImytid, PImytid, -1, -1, LOG_RECVWS );
if (!(OnlyAtomic && inAtomicOp))
    LOGEVENT(LOG_RECVWS);
}

void PIiRecvReady( type, stype, file, line )
int  type, line;
char *stype, *file;
{
if (!(OnlyAtomic && inAtomicOp))
    LOGEVENT(LOG_RECVWE);
if (vfpdraw)
    (*vfpdraw)( vfpctx, PImytid, PImytid, -1, -1, LOG_RECVWE );
#if defined(LOGCOMMSUMMARY) && !defined(LOGCOMMDISABLE)
if(cloglevel & 0x4)
    SYStatePop(&_state_recvwait,0);
#endif
}
/* type is something like "recvstart" */
void PIiRecvStart( maxlen, type, stype, file, line )
int  maxlen, type, line;
char *stype, *file;
{
if (type < -1) {
    sprintf( buf, "Invalid tag (%d) in recv", type );
    SETERRC(1,buf);
    }
if (maxlen < 0) {
    sprintf( buf, "Invalid (negative) max length (%d) in recv", maxlen );
    SETERRC(1,buf);
    }
if((cloglevel & 0xA) && !(OnlyAtomic && inAtomicOp)) {
    sprintf( buf, "[%d] %s <Tag %d> [%s:%d]\n", 
	    PImytid, stype, type, file, line );
    if (cloglevel & 0x2) {
        fputs( buf, _tracefile );
        fflush(_tracefile);
	}
    }
#if defined(LOGCOMMSUMMARY) && !defined(LOGCOMMDISABLE)
if(cloglevel & 0x4)
    SYStatePush(&_state_recv,0);
#endif
if (vfpdraw)
    (*vfpdraw)( vfpctx, PImytid, PImytid, -1, -1, LOG_RECVS );
if (!(OnlyAtomic && inAtomicOp))
    LOGEVENT(LOG_RECVS);
}

void PIiRecvEnd( from, len, type, stype, file, line )
int  from, len, type, line;
char *stype, *file;
{
#if defined(LOGCOMMEVENTS) && !defined(LOGCOMMDISABLE)
if (!(OnlyAtomic && inAtomicOp))
    BLOG_LOG4(LOG_RECVE,len,type,from); 
#endif
#if defined(LOGCOMMSUMMARY) && !defined(LOGCOMMDISABLE)
if(cloglevel & 0x4)
    SYStatePop(&_state_recv,len);
#endif
#if defined(LOGCOMMCOUNTS) && !defined(LOGCOMMDISABLE)
_logrecv++; 
_logrecvn += len;
#endif
if((cloglevel & 0xA) && !(OnlyAtomic && inAtomicOp)) {
    sprintf( buf, "[%d] %s <Tag %d(%d) From %d> [%s:%d]\n", 
	    PImytid, stype, type, len, from, file, line );
    if (cloglevel & 0x2) {
        fputs( buf, _tracefile );
        fflush(_tracefile);
        }
    }

if (vfpdraw)
    (*vfpdraw)( vfpctx, PImytid, from, PImytid, len, LOG_RECVE );
}

void PISetVFP( vctx, vdraw )
void *vctx, (*vdraw)();
{
vfpctx  = vctx;
vfpdraw = vdraw;
}

/* This routine should be called by an error handler to print the
   last trace message that would have been generated by -trace
   I'm not sure yet how to do that (though establising a chain of
   error handlers isn't a bad idea) */
void PIPrintLastTraceMessage()
{
if(cloglevel & 0x8) {
    buf[255] = 0;  /* just in case */
    fputs( buf, _tracefile );
    }
}
