#ifndef lint
static char SCCSid[] = "@(#) ./comm/nbrs.c 07/23/93";
#endif

/*
   This file contains routines to find "neighbors".
   Currently, this contains "default" routines.  Eventually, topology-specific
   routines will be available.

   These will be consolidated in the very near future into ??? groups:
   Routines used by PSCompile to find immediate neighbors (These are set 
   to reflect a given machine topology).
       PINbr....
       PI<toplogy><virtualtopology>

   Routines used by users to get neighbors 
       PSNbr.... 

   We have the following problem.  For each topology, you really need
   to have a SEPARATE mapping of ALL of the nodes in the subset to the
   given topology.

   Note that the neighbor routines, when generating a mapping for one
   topology, may need to know something about the existing "virtual"
   topology.  For example, when imbedding a ring in a mesh that is smaller
   than the actual mesh, a different mapping may be used.  In the extreme
   case, all of the indices may be evaluated when chosing a mapping.
   Thus, we pass to all routines a procset structure which will be used
   for all EXCEPT the neighbor fields.  STILL, the return value is the 
   relative position in the set, not the actual machine index.
 */

#include "tools.h"
#include "comm/comm.h"
#include "comm/global/global.h"
#include "comm/nbr.h"

static int  (*_PINbrTree)()   = PIDefTree, 
            (*_PINbrRing)()   = PIDefRing, 
            (*_PINbrMesh2d)() = PIDefMesh2d;

/*@
  PSNbrRing - return the processor id's of neighbors in a ring 

  Input parameters:
. offset - relative location from this processor
. wrap   - 1 if should wrap, 0 if not
. procset - procset to use
@*/
int PSNbrRing( offset, wrap, procset )
int offset, wrap;
ProcSet *procset;
{
int loff;
if (!procset) 
    return (*_PINbrRing)( ALLPROCS, MYPROCID, NUMNODES, offset, wrap );

loff = (*_PINbrRing)( procset, procset->lidx, procset->npset, offset, wrap );
if (loff < 0) return -1;
return procset->node_nums[loff];
}

/*@
  PSNbrMesh - return the processor id's of neighbors in a mesh

  Input parameters:
. offx - relative location from this processor in x
. offy - relative location from this processor in y
. wrapx   - 1 if should wrap in x, 0 if not
. wrapy   - 1 if should wrap in y, 0 if not
. procset - procset to use
@*/
int PSNbrMesh( offx, offy, wrapx, wrapy, procset )
int offx, offy, wrapx, wrapy;
ProcSet *procset;
{
int loff, li;

if (!procset)
    return (*_PINbrMesh2d)( ALLPROCS, MYPROCID, NUMNODES, 0, 0, 
			    offx, offy, wrapx, wrapy );

loff = (*_PINbrMesh2d)( procset, procset->lidx, procset->npset, 
		        procset->nx, procset->ny, offx, offy, wrapx, wrapy );
if (loff < 0) return -1;
return procset->node_nums[loff];
}

/*@
   PSNbrTree - return the selected child or parent of this node

   Input parameters:
.  nbr - neighbor.  One of PS_LCHILD, PS_RCHILD, or PS_PARENT
.  procset - procset to use

   Note: returns -1 for no node
@*/
int PSNbrTree( nbr, procset )
PS_Tree_t nbr;
ProcSet   *procset;
{
if (!procset) 
    return (*_PINbrTree)( ALLPROCS, MYPROCID, NUMNODES, nbr );

switch (nbr) {
    case PS_LCHILD: return procset->l_child;
    case PS_RCHILD: return procset->r_child;
    case PS_PARENT: return procset->parent;
    }
return -1;   
}

/*
   External access to the selected routines
 */
int PINbrTree( procset, myid, np, nbr )
ProcSet *procset;
int     myid, np, nbr;
{
return (*_PINbrTree)( procset, myid, np, nbr );
}

int PINbrRing( procset, myid, np, offset, wrap )
ProcSet *procset;
int     myid, np, offset, wrap;
{
return (*_PINbrRing)( procset, myid, np, offset, wrap );
}

int PINbrMesh2d( procset, myid, np, nx, ny, offx, offy, wrapx, wrapy )
ProcSet *procset;
int     myid, np, nx, ny, offx, offy, wrapx, wrapy;
{
return (*_PINbrMesh2d)( procset, myid, np, nx, ny, offx, offy, wrapx, wrapy );
}

/*@
  PISetNbrRoutines - Set the routines used to compute the neighbors 

  Input Parameters:
. tree - routine to compute tree neighbors
. ring - routine to compute ring neighbors
. mesh2d - routine to compute 2d-mesh neighbors

  Description:
  This routine sets the routines that are used by PSNbrTree, PSNbrRing, and
  PSNbrMesh2d, as well as the PI versions of these routines.  In addition, 
  since PINbrTree, PINbrRing, and PINbrMesh2d are used by PSCompile, these
  routines control the neighbors returned by processor sets and the 
  neighbors used by the collective communications routines (like gsetop, used
  by GDSUM etc) in their computations.

  If an entry is null, don't change that routine.
@*/
void PISetNbrRoutines( tree, ring, mesh2d )
int (*tree)(), (*ring)(), (*mesh2d)();
{
if (tree)   _PINbrTree   = tree;
if (ring)   _PINbrRing   = ring;
if (mesh2d) _PINbrMesh2d = mesh2d;
}
/* 
   Here are the default routines for computing the neighbors in a 
   processor set.  The exact interfaces may evolve as needed.

   These routines must NOT know anything about the processor set structure,
   since they will be used to SET elements of the processor set.
   They must also function outside of the context of a processor set.
   Primarily, this means that the routines take the local index and
   the number of processors are arguments.

   Further, these routines return a single neighbor; often, they will be
   called several times in succession to generate several neighbors;
   it may be advantageous to cache any expensive and common computations.
 */

int PIDefTree( procset, myid, np, nbr )
ProcSet *procset;
int     myid, np, nbr;
{
int n;

switch (nbr) {
    case PS_LCHILD: n = 2 * myid + 1; break;
    case PS_RCHILD: n = 2 * myid + 2; break;
    case PS_PARENT: if (myid == 0) n = -1; else n = (myid-1) / 2; break;
    }
if (n >= np) n = -1;
return n;
}

int PIDefRing( procset, myid, np, offset, wrap )
ProcSet *procset;
int     myid, np, offset, wrap;
{
myid += offset;
if (!wrap && (myid < 0 || myid >= np)) 
    return -1;
if (myid < 0) myid += np;
else if (myid >= np) myid -= np;
return myid;
}

int PIDefMesh2d( procset, myid, np, nx, ny, offx, offy, wrapx, wrapy )
ProcSet *procset;
int     myid, np, nx, ny, offx, offy, wrapx, wrapy;
{
int myi, myj;

/* Note that if we wrapx and/or wrapy are set, we can avoid some 
   of this computation by simply adding (offx + offy * np) to myid
   and then taking that value mod np 
 */
/* printf( "[%d] starting def mesh 2d\n", MYPROCID ); fflush( stdout ); */
if (nx == 0 && procset) {
    /* printf( "[%d] getting dimensions\n", MYPROCID ); fflush( stdout ); */
    nx = procset->nx;
    ny = procset->ny;
    }
/* In case the processor set has no defined mesh size... */
if (nx == 0) {
    nx = NUMNODES;
    ny = 1;
    }
/* printf( "[%d] getting mesh indices\n", MYPROCID ); fflush( stdout ); */
myj = (myid / nx) + offy;
myi = (myid % nx) + offx;

if (!wrapx && (myi < 0 || myi >= nx)) return -1;
if (!wrapy && (myj < 0 || myj >= ny)) return -1;

if (myi < 0)   myi += nx;
if (myi >= nx) myi -= nx;
if (myj < 0)   myj += ny;
if (myj >= ny) myj -= ny;

/* printf( "[%d] exiting def mesh 2d\n", MYPROCID ); fflush( stdout ); */

return myi + myj * nx;
}

/* 
   These routines give all of the IMMEDIATE neighbors of a node
   The default is the HC topology.  Various systems may reset this 
   Eventually, these should migrate off into the various topology-specific
   files.
 */
static int (*_PIGetNbrs)() = PIGetNbrsHC;
static int _PINODEDIM = 0;
static int _PINX = 1, _PINY = 1;
/*
   Here are neighbor routines:

   nnbrs = PIGetNbrs( node, nbrs ) - nbrs of node
   PIRoute( from, to, *len, *nodes ) - route that a message might take

   These are all for the "current" topology.  Specific topologies
   are ....HC  (hypercube)
       ....T   (tree)
       ....R   (ring)
       ....MSH (mesh)

   To allow cross architecture use of these routines, the parameters
   of these architectures can be independently set.
*/   

void PISetMeshSize( nx, ny )
int nx, ny;
{
_PINX = nx;
_PINY = ny;
}

void PISetDim( dim )
int dim;
{
_PINODEDIM = dim;
}

int PIGetNbrsHC( myid, nbrs )
int myid, *nbrs;
{
int i, mask, np;

np   = NUMNODES;
mask = 0x1;
i    = 0;
while (mask < np) {
    nbrs[i++] = myid ^ mask;
    mask      <<= 1;
    }
return i;
}

int PIGetNbrsMSH( myid, nbrs )
int myid, *nbrs;
{
int i, nx, ny, np;

np   = NUMNODES;
i    = 0;
nx   = _PINX;
ny   = _PINY;
if (myid + 1 < np)   nbrs[i++] = myid + 1;
if (myid - 1 >= 0 )  nbrs[i++] = myid - 1;
if (myid + nx < np)  nbrs[i++] = myid + nx;
if (myid - nx >= 0)  nbrs[i++] = myid - nx;

return i;
}


void PISetNbrRoutine( r )
int (*r)();
{
_PIGetNbrs = r;
}

/*@
     PIGetNbrs - Return all of the immediate neighbors of a node

     Input Parameters:
.    myid - node to return neighbors of
.    nbrs - array to hold neighbor id's.  Must be large enough.

     Returns:
     Number of neighbors.
@*/
int PIGetNbrs( myid, nbrs )
int myid, *nbrs;
{
return (*_PIGetNbrs)( myid, nbrs );
}

/* Eventually a routine to get the interconnection type */
int PIInterconnectionType()
{
return 0;
}

/* Here is a version of tree that produces a faster tree on some systems.
   nbr indicates which nbr is desired.
 */
int PIDefFastTree( procset, myid, np, nbr )
ProcSet *procset;
int     myid, np, nbr;
{
int n, l_child, r_child, parent, am_left;

PISetTreeNodesFast( myid, np, &l_child, &r_child, &parent, &am_left );

switch (nbr) {
    case PS_LCHILD: n = l_child;; break;
    case PS_RCHILD: n = r_child;; break;
    case PS_PARENT: if (myid == 0) n = -1; else n = parent; break;
    }
if (n >= np) n = -1;
return n;
}
