/* bltin2.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1994  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "code.h"
#include "symbol.h"
#include "mem.h"
#include "list.h"
#include "btree.h"
#include "bltin.h"
#include "scop1.h"
#include "matop1.h"
#include "matop2.h"
#include "r_string.h"
#include "util.h"
#include "mathl.h"
#include "function.h"

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#ifdef THINK_C
char* getpref(char*);
#define getenv(env_name) getpref(env_name)
#endif

/* scan.l */
extern int new_file _PROTO ((char *file_name));

/* print.c */
extern FILE *get_file_ds _PROTO ((char *name, char *mode, int buffsize));
extern int close_file_ds _PROTO ((char *name));

#define rabs(x) ((x) >= 0 ? (x) : -(x))

#define TARG_DESTROY(arg, targ)   if (targ.u.ent != arg.u.ent) \
                                    remove_tmp_destroy (targ.u.ent);

/* **************************************************************
 * Set/Reset the current directory.
 * ************************************************************** */

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

void
CD (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  static char *cd_string;
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("cd: requires 1 argument", 0);

  /* Copy the argument string into env_string */
  arg = get_bltin_arg ("cd", d_arg, 1, STRING);

  if (cd_string != 0)
    FREE (cd_string);
  cd_string = cpstr (string_GetString (e_data (arg.u.ent)));

  if (chdir (cd_string))
  {
    switch (errno)
    {
    case EACCES:
      fprintf (stderr, "Search permission to: %sdenied\n", cd_string);
      errno = 0;
      break;

    case ENOTDIR:
      fprintf (stderr, "Part of the path is not a directory: %s\n", cd_string);
      errno = 0;
      break;

    case ENOENT:
      fprintf (stderr, "Part of the path does not exist: %s\n", cd_string);
      errno = 0;
      break;

    default:
      error_1 ("error during call to cd()", 0);
    }
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }
  *return_ptr = (VPTR) scalar_Create (1.0);
}

void
Exist (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("exist: requires 1 argument", 0);

  arg = d_arg[0];

  switch (arg.type)
  {
  case CONSTANT:
  case iCONSTANT:
    error_1 ("invalid argument to exist", 0);
  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case UNDEF:
      *return_ptr = (VPTR) scalar_Create (0.0);
      break;

    default:
      *return_ptr = (VPTR) scalar_Create (1.0);
      break;
    }
  }
  return;
}

#ifdef titan
extern double strtod ();
#endif

void
Strtod (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int i, nel;
  Datum arg, targ;
  Matrix *m, *new = 0;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("strtod: requires 1 argument", 0);

  arg = get_bltin_arg ("strtod", d_arg, 1, 0);
  targ = convert_all_to_matrix (arg);
  m = (Matrix *) e_data (targ.u.ent);

  switch (MTYPE (m))
  {
  case REAL:
  case COMPLEX:
    error_1 ("strtod: input must be string", matrix_GetName (m));
    break;
  case STRING:
    new = matrix_Create (MNR (m), MNC (m));
    nel = MNR (m) * MNC (m);
    for (i = 0; i < nel; i++)
      MATrv (new, i) = strtod (MATsv (m, i), (char **) 0);
    break;
  default:
    error_1 ("strtod: invalid matrix type", 0);
    break;
  }

  TARG_DESTROY (arg, targ);
  *return_ptr = (VPTR) new;
  return;
}

void
System (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  double retval;
  Datum arg;
  char *s;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("system() requires a STRING argument", (char *) 0);

  arg = get_bltin_arg ("system", d_arg, 1, STRING);
  s = string_GetString (e_data (arg.u.ent));

  retval = (double) system (s);
  *return_ptr = (VPTR) scalar_Create (retval);
}

/* **************************************************************
 * Casting functions
 * ************************************************************** */
void
Scalar_cast (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;
  Matrix *m;
  Scalar *s;

  /* Check n_args */
  if (n_args > 1)
    error_1 ("max of 1 arg allowed for scalar()", (char *) 0);

  if (n_args == 0)
  {
    *return_ptr = (VPTR) scalar_Create (0.0);
    return;
  }

  arg = get_bltin_arg ("scalar", d_arg, 1, 0);

  switch (arg.type)
  {
  case CONSTANT:
    *return_ptr = (VPTR) scalar_Create (arg.u.val);
    break;
  case iCONSTANT:
    *return_ptr = (VPTR) scalar_CreateC (0.0, arg.u.val);
    break;
  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      s = (Scalar *) e_data (arg.u.ent);
      *return_ptr = (VPTR) scalar_CreateC (SVALr (s), SVALi (s));
      break;
    case MATRIX:
      m = (Matrix *) e_data (arg.u.ent);
      if (MNR (m) == 1 && MNC (m) == 1)
      {
	if (MTYPE (m) == REAL)
	  *return_ptr = (VPTR) scalar_Create (MAT (m, 1, 1));
	else if (MTYPE (m) == COMPLEX)
	  *return_ptr = (VPTR) scalar_CreateC (MATr (m, 1, 1),
					       MATi (m, 1, 1));
	else if (MTYPE (m) == STRING)
	  *return_ptr = (VPTR) string_Create (MATs (m, 1, 1));
      }
      else
      {
	error_1 (matrix_GetName (m),
		 "cannot cast matrix to scalar when dim. are not 1x1");
      }
      break;
    case STRING:
      *return_ptr = (VPTR) string_Create (string_GetString (e_data (arg.u.ent)));
      break;
    default:
      error_1 (e_name (arg.u.ent), "invalid type for scalar()");
      break;
    }
    break;
  default:
    error_1 (e_name (arg.u.ent), "invalid type for scalar()");
    break;    
  }
}

void
Matrix_cast (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;
  Matrix *m;
  Scalar *s;
  String *str;

  /* Check n_args */
  if (n_args > 1)
    error_1 ("max of 1 arg allowed for matrix()", (char *) 0);

  if (n_args == 0)
  {
    *return_ptr = (VPTR) matrix_Create (0, 0);
    return;
  }

  arg = get_bltin_arg ("matrix", d_arg, 1, 0);

  switch (arg.type)
  {
  case CONSTANT:
    m = matrix_Create (1, 1);
    MAT (m, 1, 1) = arg.u.val;
    *return_ptr = (VPTR) m;
    break;
  case iCONSTANT:
    m = matrix_CreateC (1, 1);
    MATr (m, 1, 1) = 0.0;
    MATi (m, 1, 1) = arg.u.val;
    *return_ptr = (VPTR) m;
    break;
  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      s = (Scalar *) e_data (arg.u.ent);
      if (SVALi (s) == 0.0)
      {
	m = matrix_Create (1, 1);
	MAT (m, 1, 1) = SVALr (s);
      }
      else
      {
	m = matrix_CreateC (1, 1);
	MATr (m, 1, 1) = SVALr (s);
	MATi (m, 1, 1) = SVALi (s);
      }
      *return_ptr = (VPTR) m;
      break;
    case MATRIX:
      /* do nothing */
      *return_ptr = (VPTR) matrix_Copy (e_data (arg.u.ent));
      break;
    case STRING:
      str = (String *) e_data (arg.u.ent);
      m = matrix_CreateS (1, 1);
      MATs (m, 1, 1) = cpstr (string_GetString (str));
      *return_ptr = (VPTR) m;
      break;
    default:
      error_1 (e_name (arg.u.ent), "invalid type for matrix()");
      break;
    }
    break;
  default:
    error_1 (e_name (arg.u.ent), "invalid type for matrix()");
    break;    
  }
}

/* **************************************************************
 * Return the type of an object, REAL or COMPLEX.
 * ************************************************************** */
void
Type (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;
  Matrix *m;
  Scalar *s;
  String *str;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("1 arg allowed for type()", 0);

  arg = get_bltin_arg ("type", d_arg, 1, 0);

  switch (arg.type)
  {
  case CONSTANT:
    *return_ptr = (VPTR) string_Create (cpstr ("real"));
    break;

  case iCONSTANT:
    *return_ptr = (VPTR) string_Create (cpstr ("complex"));
    break;

  case ENTITY:
    switch (e_type (arg.u.ent))
    {
    case SCALAR:
      s = (Scalar *) e_data (arg.u.ent);
      if (SVALi (s) == 0.0)
	*return_ptr = (VPTR) string_Create (cpstr ("real"));
      else
	*return_ptr = (VPTR) string_Create (cpstr ("complex"));
      break;

    case MATRIX:
      m = (Matrix *) e_data (arg.u.ent);
      if (MTYPE (m) == REAL)
	*return_ptr = (VPTR) string_Create (cpstr ("real"));
      else if (MTYPE (m) == COMPLEX)
	*return_ptr = (VPTR) string_Create (cpstr ("complex"));
      else if (MTYPE (m) == STRING)
	*return_ptr = (VPTR) string_Create (cpstr ("string"));
      break;

    case STRING:
      str = (String *) e_data (arg.u.ent);
      *return_ptr = (VPTR) string_Create (cpstr ("string"));
      break;

    case BLTIN:
      str = (String *) e_data (arg.u.ent);
      *return_ptr = (VPTR) string_Create (cpstr ("builtin"));
      break;

    case U_FUNCTION:
      str = (String *) e_data (arg.u.ent);
      *return_ptr = (VPTR) string_Create (cpstr ("user"));
      break;

    case BTREE:
      {
	/* Try and report the contents of the type element */
	ListNode *type;

	if ((type = btree_FindNode (e_data (arg.u.ent), "type")))
	{
	  if (e_type (type) == STRING)
	  {
	    *return_ptr = (VPTR)
	      string_Create (cpstr (string_GetString (e_data (type))));
	  }
	  else
	    *return_ptr = (VPTR) string_Create (cpstr (""));
	}
	else
	{
	  *return_ptr = (VPTR) string_Create (cpstr (""));
	}
	break;
      }
    default:
      error_1 (e_name (arg.u.ent), "invalid type for type()");
      break;
    }
  }
}

/* **************************************************************
 * Return the size of an object in bytes.
 * ************************************************************** */

size_t ent_sizeof _PROTO ((ListNode *arg));
extern size_t btree_sizeof _PROTO ((Btree *btree));

void
Sizeof (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg;
  int size = 0;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("size: 1 argument required", 0);

  arg = get_bltin_arg ("size", d_arg, 1, 0);

  switch (arg.type)
  {
  case CONSTANT:
    size = sizeof (double);
    break;
  case iCONSTANT:
    size = sizeof (Complex);
    break;
  case ENTITY:
    size = ent_sizeof (arg.u.ent);
    break;
  default:
    error_1 ("size: invalide type", 0);
  }

  *return_ptr = (VPTR) scalar_Create ((double) size);
  return;
}

size_t
ent_sizeof (arg)
     ListNode *arg;
{
  Matrix *m;
  int i;
  size_t size = 0;
  
  switch (e_type (arg))
  {
  case SCALAR:
    size = sizeof (Complex);
    break;
    
  case MATRIX:
    m = (Matrix *) e_data (arg);
    switch (MTYPE (m))
    {
    case REAL:
      size = MNR (m) * MNC (m) * sizeof (double);
      break;
    case COMPLEX:
      size = MNR (m) * MNC (m) * sizeof (Complex);
      break;
    case STRING:
      size = MNR (m) * MNC (m) * sizeof (char *);
      for (i = 0; i < MNR (m) * MNC (m); i++)
	size += strlen (MATsv (m, i))*sizeof (char);
      break;
    }
    break;

  case STRING:
    size = sizeof (char) * string_GetLength (e_data (arg));
    break;
    
  case BLTIN:
    size = 0;
    break;
    
  case U_FUNCTION:
    size = function_GetCodeSize (e_data (arg)) * sizeof (Inst);
    break;
    
  case BTREE:
    size += btree_sizeof (e_data (arg));
    break;
    
  case UNDEF:
    size = 0;
    break;

  default:
    error_1 (e_name (arg), "invalid type for size");
    break;
  }
  return size;
}

void
Garbage (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  garbage_print ();
  *return_ptr = (VPTR) scalar_Create (0.0);
}

/* **************************************************************
 * Time related functions. Original versions contributed
 * by T. L. Kay. Modified to use time() instead of gettimeofday(),
 * since time() seems more reliable/portable, Ian Searle.
 * ************************************************************** */

/*
 * Start the timer.
 */

#ifdef HAVE_TIME_H
#include <time.h>
#endif

static time_t tictime;

void
Tic (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  /* Check n_args */
  if (n_args != 0)
    error_1 ("Wrong number of args to tic()", (char *) 0);

  tictime = time (0);
  if (tictime == -1)
  {
    errno = 0;
    error_1 ("ERROR in system time() call", (char *) 0);
  }
  *return_ptr = (VPTR) scalar_Create (0.);
  return;
}

/*
 * Report the elapsed time, in seconds,
 * since last call to tic().
 */

void
Toc (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  time_t toctime;
  double elapsed;

  /* Check n_args */
  if (n_args != 0)
    error_1 ("Wrong number of args to toc()", (char *) 0);

  if (tictime)
  {
    toctime = time (0);
    if (toctime == -1)
    {
      errno = 0;
      error_1 ("ERROR in system time() call", (char *) 0);
    }
#ifdef HAVE_DIFFTIME
    elapsed = difftime (toctime, tictime);
#else
    elapsed = toctime - tictime;
#endif
    *return_ptr = (VPTR) scalar_Create (elapsed);
  }
  else
    error_1 ("must call tic() 1st", (char *) 0);

  return;
}

/*
 * Return a matrix containing the indices of the
 * non-zero elements of the input matrix.
 */

void
Find (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg, targ;
  Matrix *m, *mfind;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("find: 1 arg allowed", 0);

  arg = get_bltin_arg ("find", d_arg, 1, 0);
  targ = convert_all_to_matrix (arg);
  m = (Matrix *) e_data (targ.u.ent);
  mfind = matrix_Find (m);
  TARG_DESTROY (arg, targ);
  *return_ptr = (VPTR) mfind;
}

/*
 * Generate am Inf.
 */

static const unsigned char __nan[8] = r__nan_bytes;
#define	R_NAN	(*(const double *) __nan)

static const unsigned char __inf_val[8] = r__inf_val_bytes;
#define	R_INF   (*(const double *) __inf_val)

void
Inf (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  /* Check n_args */
  if (n_args != 0)
    error_1 ("inf does not take any arguments", (char *) 0);

  *return_ptr = (VPTR) scalar_Create (R_INF);
}

/*
 * Generate a NaN.
 */

void
Nan (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  /* Check n_args */
  if (n_args != 0)
    error_1 ("nan does not take any arguments", (char *) 0);

  *return_ptr = (VPTR) scalar_Create (R_NAN);
}

/* **************************************************************
 * Function Variable Scope analyzer.
 * Go through a function's op-codes, looking for variable references.
 * When one is found, print out the scope of the variable, and
 * it's name.
 * ************************************************************** */

#include "function.h"
extern ListNode *static_lookup _PROTO((char *file_name, char *name));

void
FVScope (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *fname, *fstring, *name;
  int i, lineno, narg, ncode, nlocal, off;
  Datum arg, argf;
  Function *f;
  List *localv, *argv;
  ListNode *ltmp;
  FILE *fptr;
  Inst *pc;

  fstring = 0;   /* Initialize */

  if (n_args > 2)
    error_1 ("fvscope: at most 2 arguments allowed", 0);
  arg = get_bltin_arg ("fvscope", d_arg, 1, U_FUNCTION);

  fptr = stdout;      /* Default */
  if (n_args == 2)
  {
    /*
     * Extract the string ptr to the output file
     * or process.
     */

    argf = get_bltin_arg ("fvscope", d_arg, 2, 0);
    if (argf.type == ENTITY)
    {
      if (e_type (argf.u.ent) == STRING)
	fstring = string_GetString (e_data (argf.u.ent));
      else if (e_type (argf.u.ent) == MATRIX && 
	       MTYPE (e_data (argf.u.ent)) == STRING)
      {
	fstring = (char *) MATsv (e_data (argf.u.ent), 0);
      }
      else
	error_1 ("fvscope: 2nd arg must be a string", 0);
    }
    else
    {
      error_1 ("fvscope: 2nd arg must be string", 0);
    }
    if ((fptr = get_file_ds (fstring, "w", 0)) == 0)
    {
      warning_1 (fstring, "cannot open for write");
      *return_ptr = (VPTR) scalar_Create (0.0);
      return;
    }
  }
  
  /* 
   * Set up function searching defaults.
   */

  f = (Function *) e_data (arg.u.ent);	/* Function ptr */
  pc = function_GetCodePtr (f);	        /* Code List ptr */
  ncode = function_GetCodeSize (f);	/* # of instructions */
  argv = f->args;		        /* Arg List ptr */
  narg = f->n_args;		        /* # of arguments */
  localv = f->local;		        /* Local var list ptr */
  nlocal = f->n_local;		        /* # of local vars */
  lineno = 1;

  fname = pc[1].ptr;     		/* File name */

  fprintf (fptr, "\tFunction Variable SCOPE analysis for : %s\n",
	   e_name (arg.u.ent));
  fprintf (fptr, "\tFilename: %s\n\n", fname);
  fprintf (fptr, "\tline\tGLOBAL\t\t\tARG\t\tLOCAL\n\n");

  for (i = 0; i < ncode; i++)
  {
    switch ((*pc++).op_code)
    {
    case OP_PUSH_VAR:
      name = ((ListNode *) ((*pc++).ptr))->key;
      i++;
      if (static_lookup (fname, name))
	fprintf (fptr, "\t%4i\tStatic-Var: %s\n", lineno, name);
      else
	fprintf (fptr, "\t%4i\tGlobal-Var: %s\n", lineno, name);
      break;

    case OP_PUSH_ARG:
      off = (*pc++).op_code;
      i++;
      ltmp = list_GetNodeByPos (argv, (narg - off) + 1);
      name = ltmp->key;
      fprintf (fptr, "\t%4i\t\t\t\tArg-Var: %s\n", lineno, name);
      break;

    case OP_PUSH_LOCAL_VAR:
      off = (*pc++).op_code;
      i++;
      ltmp = list_GetNodeByPos (localv, (nlocal - off) + 1);
      name = ltmp->key;
      fprintf (fptr, "\t%4i\t\t\t\t\t\tLocal-Var: %s\n", lineno, name);
      break;

    case OP_ADD:
    case OP_SUB:
    case OP_MUL:
    case OP_DIV:
    case OP_LDIV:
    case OP_NEGATE:
    case OP_POWER:
    case OP_ASSIGN:
      break;

    case OP_FOR:
      pc += 2;
      i += 2;
      break;

    case OP_EL_MUL:
    case OP_EL_DIV:
    case OP_EL_LDIV:
    case OP_EL_POWER:
      break;

    case OP_PUSH_CONSTANT:
    case OP_PUSH_iCONSTANT:
      pc++;
      i++;
      break;

    case OP_PRINT:
    case OP_GT:
    case OP_LT:
    case OP_EQ:
    case OP_GE:
    case OP_LE:
    case OP_NE:
    case OP_AND:
    case OP_OR:
    case OP_NOT:
      break;

    case OP_IF:
      pc += 3;
      i += 3;
      break;

    case OP_WHILE:
      pc += 2;
      i += 2;
      break;

    case OP_SWAP:
    case OP_INC:
    case OP_DEC:
    case OP_POP:
    case OP_POP_CLEAN:
      break;

    case OP_VECTOR_CREATE:
      pc++;
      i++;
      break;

    case OP_VEC_APPEND:
    case OP_MATRIX_VEC_SUB:
    case OP_MATRIX_VEC_ASSIGN:
    case OP_MATRIX_CREATE:
    case OP_MATRIX_APPEND:
      break;

    case OP_MATRIX_ASSIGN:
    case OP_MATRIX_SUB:
    case OP_LIST_CREATE:
      pc++;
      i++;
      break;

    case OP_LIST_MEMB:
    case OP_LIST_ASSIGN:
      if (((*pc).op_code) == 1)
      {
	pc++;
	i++;
      }
      else
      {
	pc += 2;
	i += 2;
      }
      break;

    case OP_LIST_EL_CREATE:
    case OP_FUNCTION_CALL:
    case OP_FUNCTION_CALL_SELF:
      pc++;
      i++;
      break;

    case OP_FUNCTION_RETURN:
      break;

    case OP_DEF_FUNC_RET:
      i = ncode;
      break;

    case OP_TRANSPOSE:
      break;

    case OP_PUSH_STRING:
      pc++;
      i++;
      break;

    case OP_BREAK:
    case OP_CONTINUE:
    case OP_QUIT:
      break;

    case OP_LINE_NO:
      lineno = (*pc++).op_code - 100;
      i++;
      break;

    case OP_FILE_NAME:
    case OP_JMP:
      pc++;
      i++;
      break;

    case OP_EMPTY_MATRIX_CREATE:
    case OP_MATRIX_COL:
    case OP_EL_TRANSPOSE:
    case OP_RFILE:
      break;

    case OP_RFILE_NAME:
      pc++;
      i++;
      break;

    case OP_HELP:
      break;

    case OP_HELP_NAME:
      pc++;
      i++;
      break;

    case OP_PUSH_UNDEF:      
    case OP_EL_ADD:
    case OP_EL_SUB:
    case STOP:
      break;

    default:
      error_1 ("fvscope: error in function codes", 0);
      break;
    }
  }
  *return_ptr = (VPTR) scalar_Create (1.0);
}

/* **************************************************************
 * RLaB interface to system getenv() function.
 * ************************************************************** */

void
Getenv (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *retval;
  Datum arg;
  String *str, *sr;

  if (n_args != 1)
    error_1 ("getenv: requires 1 argument", 0);

  arg = get_bltin_arg ("getenv", d_arg, 1, STRING);
  str = (String *) e_data (arg.u.ent);

  retval = getenv (string_GetString (str));

  sr = string_Create (cpstr (retval));

  *return_ptr = (VPTR) sr;
  return;
}

/* **************************************************************
 * RLaB interface to ANSI-C function tmpnam().
 * ************************************************************** */

void
Tmpnam (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *fn;
  String *retf;

  if (n_args != 0)
    error_1 ("tmpnam: no arguments allowed", 0);

  fn = tmpnam (0);
  retf = string_Create (cpstr (fn));

  *return_ptr = (VPTR) retf;
  return;
}

char *eval_string;
extern void set_rlab_input _PROTO ((int type));    /* scan.l */
extern void set_rlab_input _PROTO ((int type));    /* scan.l */
extern int run_program_eval _PROTO ((void));       /* main.c */
extern Datum eval_ret;

int do_eval = 0;

void
Eval (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int retv;
  void *retd;
  Datum arg;
  Scalar *s = 0;
  String *str;

  if (n_args != 1)
    error_1 ("eval: 1 argument allowed", 0);

  arg = get_bltin_arg ("eval", d_arg, 1, STRING);
  str = (String *) e_data (arg.u.ent);

  eval_string = string_GetString (str);
  do_eval++;
  set_rlab_input (1);
  retv = run_program_eval ();
  if (--do_eval == 0)
    set_rlab_input (0);

  /* Now return eval_ret */

  if (retv == 0)
    *return_ptr = (VPTR) scalar_Create (0.0);
  else
    if (eval_ret.type != ENTITY)
    {
      /* Make an ENTITY out of it. */
      if (eval_ret.type == CONSTANT)
	s = scalar_Create (eval_ret.u.val);
      else if (eval_ret.type == iCONSTANT)
	s = scalar_CreateC (0.0, eval_ret.u.val);

      *return_ptr = (VPTR) s;
    }
    else
    {
      /*
       * We must remove the return entity from the
       * tmp list if it is there, otherwise there
       * will be trouble when bltin() puts it on the list.
       */
      
      if (e_name (eval_ret.u.ent) == 0)
      {
	retd = remove_tmp (eval_ret.u.ent);
	*return_ptr = retd;
      }
      else
      {
	/*
	 * eval returned an existing entitiy,
	 * let bltin() handle it.
	 */

	e_type (eval_ret.u.ent) = LISTNODE;
	*return_ptr = (VPTR) eval_ret.u.ent;
      }
    }
  
  return;
}

void
Sign (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  Datum arg, targ;
  Matrix *m, *msign;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("sign: 1 argument allowed", 0);

  arg = get_bltin_arg ("sign", d_arg, 1, 0);
  targ = convert_all_to_matrix (arg);
  m = (Matrix *) e_data (targ.u.ent);
  msign = matrix_Sign (m);
  TARG_DESTROY (arg, targ);
  *return_ptr = (VPTR) msign;
}
