/* btree.c */

/* Binary tree implementation using ListNode structure for tree nodes.
   The ListNode structure was originally designed for doubley linked
   lists, so for binary tree purposes, left == prev, and right == next. */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "btree.h"
#include "function.h"
#include "util.h"
#include "mem.h"

#include <stdio.h>

/* **************************************************************
 * Create a binary tree structure and initialize it.
 * ************************************************************** */
Btree *
btree_Create ()
{
  Btree *new;

  new = (Btree *) MALLOC (sizeof (Btree));
  new->type = BTREE;
  new->name = 0;
  new->numNodes = 0;
  new->root_node = 0;

  return (new);
}

/* **************************************************************
 * Destroy a binary tree, and all of the nodes in it.
 * ************************************************************** */

static void btree_destroy_nodes _PROTO ((ListNode * node));
static void btree_destroy_nodes_except _PROTO ((ListNode * node, int type));

void
btree_Destroy (root)
     Btree *root;
{
  ASSERT (root);
  {
    btree_destroy_nodes (root->root_node);
    FREE (root->name);
    root->type = 0;
    root->numNodes = 0;
    root->root_node = 0;
    FREE (root);
  }
}

void
btree_DestroyExcept (root, type)
     Btree *root;
     int type;
{
  btree_destroy_nodes_except (root->root_node, type);
  FREE (root->name);
  root->numNodes = 0;
  root->root_node = 0;
  FREE (root);
}

static void
btree_destroy_nodes (node)
     ListNode *node;
{
  if (node != 0)
  {
    btree_destroy_nodes (node->prev);
    btree_destroy_nodes (node->next);
    listNode_Destroy (node);
  }
}

static void
btree_destroy_nodes_except (node, type)
     ListNode *node;
     int type;
{
  if (node != 0)
  {
    btree_destroy_nodes_except (node->prev, type);
    btree_destroy_nodes_except (node->next, type);
    if (listNode_GetType (node) == type)
      listNode_DestroyNodeOnly (node);
    else
      listNode_Destroy (node);
  }
}

/* **************************************************************
 * Add a new node to the given binary tree.
 * ************************************************************** */

static ListNode *btree_add_node _PROTO ((ListNode * rnode, ListNode * anode));

ListNode *
btree_AddNode (root, node)
     Btree *root;
     ListNode *node;
{
  ListNode *add;

  if (root->root_node == 0)
  {
    root->numNodes++;
    return (root->root_node = node);
  }
  else
  {
    if ((add = btree_add_node (root->root_node, node)) != 0)
      root->numNodes++;
    return (add);
  }
}

static ListNode *
btree_add_node (rnode, anode)
     ListNode *rnode, *anode;
{
  int cond;

  if (rnode == 0)
  {
    /* End of tree, add node */
    return (anode);
  }
  else if ((cond = strcmp (listNode_GetKey (anode), listNode_GetKey (rnode)))
	   == 0)
  {
    /* Error, found duplicate */
    return (0);
  }
  else if (cond < 0)
    /* Traverse left branch */
    rnode->prev = btree_add_node (rnode->prev, anode);
  else
    /* Traverse right branch */
    rnode->next = btree_add_node (rnode->next, anode);
  return (rnode);
}

/* **************************************************************
 * Search a tree for the node with the given key.
 * ************************************************************** */

static ListNode *btree_find_node _PROTO ((ListNode * node, char *key));

ListNode *
btree_FindNode (root, key)
     Btree *root;
     char *key;
{
  return (btree_find_node (root->root_node, key));
}

static ListNode *
btree_find_node (node, key)
     ListNode *node;
     char *key;
{
  int cond;

  if (node == 0)
    /* End of tree */
    return (0);
  else if ((cond = strcmp (key, listNode_GetKey (node))) == 0)
    /* Found it */
    return (node);
  else if (cond < 0)
    /* Traverse left branch */
    return (btree_find_node (node->prev, key));
  else
    /* Traverse right branch */
    return (btree_find_node (node->next, key));
}

/*
 * Recusive version of the previous function.
 * This version will look for ListNodes that are themselves
 * btrees, and will walk them also.
 */

static ListNode *btree_find_node_r _PROTO ((ListNode * node, ListNode * fnode));

ListNode *
btree_FindNodeR (root, fnode)
     Btree *root;
     ListNode *fnode;
{
  return (btree_find_node_r (root->root_node, fnode));
}

/*
 * We have to check EVERY node until we find it, since
 * the nodes are not ordered by pointer value.
 */

static ListNode *
btree_find_node_r (node, fnode)
     ListNode *node, *fnode;
{
  ListNode *n;

  if (node == 0)
    /* End of tree */
    return (0);
  else if (node == fnode)
    /* Found it */
    return (node);
  if ((n = btree_find_node_r (node->prev, fnode)) != 0)
    return (n);
  if ((n = btree_find_node_r (node->next, fnode)) != 0)
    return (n);
  return (0);
}

/* **************************************************************
 * Some miscellaneous binary tree functions.
 * ************************************************************** */

void
btree_SetName (root, name)
     Btree *root;
     char *name;
{
  ASSERT (root);
  FREE (root->name);
  root->name = name;
}

char *
btree_GetName (root)
     Btree *root;
{
  return (root->name);
}

int
btree_GetNumNodes (root)
     Btree *root;
{
  return (root->numNodes);
}

/*
 * Return the "real" number of nodes. Count only the ones
 * that are not UNDEF.
 */

static int btree_get_real_num_nodes _PROTO ((ListNode * node, int num));

int
btree_GetRealNumNodes (root)
     Btree *root;
{
  int num = 0;
  return (btree_get_real_num_nodes (root->root_node, num));
}

static int
btree_get_real_num_nodes (node, num)
     ListNode *node;
     int num;
{
  if (node != 0)
  {
    num = btree_get_real_num_nodes (node->prev, num);
    if (listNode_GetType (node) != UNDEF)
    {
      num++;
    }
    num = btree_get_real_num_nodes (node->next, num);
  }
  return (num);
}

/* **************************************************************
 * Print binary tree.
 * ************************************************************** */
#define N_WORD     5
static int count = 0;

static char **
btree_get_node_names (node, names)
     ListNode *node;
     char **names;
{
  if (node != 0)
  {
    names = btree_get_node_names (node->prev, names);
    if (listNode_GetType (node) != UNDEF)
    {
      *(names++) = cpstr (listNode_GetKey (node));
      count++;
    }
    names = btree_get_node_names (node->next, names);
  }
  return (names);
}

void
btree_Print (root, fn)
     Btree *root;
     FILE *fn;
{
  int i;
  char **names;

  count = 0;

  if (root->name != 0 && strncmp (root->name, "-", 1))
    fprintf (fn, " %s =\n", root->name);

  if (root->numNodes != 0)
  {
    names = (char **) MALLOC ((root->numNodes) * sizeof (char *));
    btree_get_node_names (root->root_node, names);

    if (count)
    {
      /* Now print out names array */
      fprintf (fn, "   ");
      for (i = 1; i <= count; i++)
      {
	fprintf (fn, "%-13s", names[i - 1]);
	FREE (names[i - 1]);
	if ((i % N_WORD) == 0)
	  fprintf (fn, "\n   ");
      }
      fprintf (fn, "\n");
      FREE (names);
    }
    else
    {
      fprintf (fn, "\t<<>>\n");
    }
  }
  else
    fprintf (fn, "\t<<>>\n");
}

/* **************************************************************
 * Copy a tree
 * ************************************************************** */

#include "scalar.h"
#include "matrix.h"
#include "r_string.h"

static void btree_copy_nodes _PROTO ((Btree * new_tree, ListNode * node));

Btree *
btree_CopyTree (root)
     Btree *root;
{
  Btree *new_tree;

  if (root->name && (!strncmp (root->name, "$$", 2)))
    error_1 ("Cannot copy global symbol table", 0);
  new_tree = btree_Create ();
  btree_copy_nodes (new_tree, root->root_node);
  new_tree->numNodes = root->numNodes;

  return (new_tree);
}

static void
btree_copy_nodes (new_tree, node)
     Btree *new_tree;
     ListNode *node;
{
  char *name;
  ListNode *lnode;

  if (node != 0)
  {
    btree_copy_nodes (new_tree, node->prev);
    btree_copy_nodes (new_tree, node->next);

    name = cpstr (listNode_GetKey (node));

    /* Since we are going to copy each node, we must know what
       type of data is hanging off of it */
    switch (e_type (node))
    {
    case SCALAR:
      lnode = install (new_tree, name, SCALAR, 
		       scalar_Copy (e_data (node)));
      scalar_SetName (e_data (lnode), cpstr (name));
      break;
    case MATRIX:
      lnode = install (new_tree, name, MATRIX, 
		       matrix_Copy (e_data (node)));
      matrix_SetName (e_data (lnode), cpstr (name));
      break;
    case STRING:
      lnode = install (new_tree, name, STRING, 
		       string_Copy (e_data (node)));
      string_SetName (e_data (lnode), cpstr (name));
      break;
    case BTREE:
      lnode = install (new_tree, name, BTREE, 
		       btree_CopyTree (e_data (node)));
      btree_SetName (e_data (lnode), cpstr (name));
      break;
    case UNDEF:
      break;
    case U_FUNCTION:
      lnode = install (new_tree, name, U_FUNCTION, 
		       function_Copy (e_data (node)));
      function_SetName (e_data (lnode), cpstr (name));
      break;
    default:
      error_1 ("invalid type for list copy", 0);
      break;
    }
  }
}

/* **************************************************************
 * Write out the contents of a binary tree. Another variation
 * of btree_Print.
 * ************************************************************** */

static void btree_write_nodes _PROTO ((ListNode *, FILE *));

void
btree_Write (btree, fn)
     Btree *btree;
     FILE *fn;
{
  /* Write out the header information */
  if (btree_GetName (btree) == 0)
    fprintf (fn, "# list : %s no_of_elements: %d \n",
	     "LIST", btree_GetNumNodes (btree));
  else
    fprintf (fn, "# list : %s no_of_elements: %d \n",
	     btree_GetName (btree), btree_GetNumNodes (btree));

  btree_write_nodes (btree->root_node, fn);
}

static void
btree_write_nodes (node, fn)
     ListNode *node;
     FILE *fn;
{
  if (node != 0)
  {
    btree_write_nodes (node->prev, fn);
    switch (e_type (node))
    {
    case SCALAR:
      scalar_Write (e_data (node), fn);
      break;
    case MATRIX:
      matrix_Write (e_data (node), fn);
      break;
    case BTREE:
      btree_Write (e_data (node), fn);
      break;
    case STRING:
      string_Write (e_data (node), fn);
      break;
    case UNDEF:
    case U_FUNCTION:
    case BLTIN:
      break;
    default:
      fprintf (stderr, "ERROR, Object type: %i\n", e_type (node));
      warning_1 ("Invalid object for btree_write()", 0);
      break;
    }
    btree_write_nodes (node->next, fn);
  }
}

/* **************************************************************
 * Read a list of RLaB objects from file.
 * ************************************************************** */

Btree *
btree_Read (fn)
     FILE *fn;
{
  int i, nel;
  char l_name[80], *name, type[20];
  Btree *btree, *newlist;
  Matrix *m;
  Scalar *s;
  String *str;

  fscanf (fn, "%s no_of_elements: %d \n", l_name, &nel);

  /* Create a list and start reding object(s) from fn */
  newlist = btree_Create ();
  btree_SetName (newlist, cpstr (l_name));

  for (i = 0; i < nel; i++)
  {
    fscanf (fn, "# %s :", type);

    if (!strcmp ("scalar", type))
    {
      s = scalar_Read (fn);
      name = scalar_GetName (s);
      install (newlist, cpstr (name), SCALAR, s);
    }
    else if (!strcmp ("matrix", type))
    {
      m = matrix_Read (fn);
      name = matrix_GetName (m);
      install (newlist, cpstr (name), MATRIX, m);
    }
    else if (!strcmp ("string", type))
    {
      str = string_Read (fn);
      name = string_GetName (str);
      install (newlist, cpstr (name), STRING, str);
    }
    else if (!strcmp ("list", type))
    {
      btree = btree_Read (fn);
      name = string_GetName (btree);
      install (newlist, cpstr (name), BTREE, btree);
    }
    else
      error_1 (type, "do not know how to read this type");
  }
  return (newlist);
}

extern size_t ent_sizeof _PROTO ((ListNode *));
static size_t btree_get_sizeof _PROTO ((ListNode *));

size_t
btree_sizeof (btree)
     Btree *btree;
{
  return btree_get_sizeof (btree->root_node);
}

static size_t
btree_get_sizeof (node)
     ListNode *node;
{
  size_t size = 0;
  if (node != 0)
  {
    size += btree_get_sizeof (node->prev);
    size += ent_sizeof (node);
    size += btree_get_sizeof (node->next);
  }
  return size;
}
