/* function.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "function.h"
#include "util.h"
#include "mem.h"
#include "symbol.h"

/* **************************************************************
 * Create an RLaB Function entity.
 * ************************************************************** */
Function *
function_Create ()
{
  Function *new = (Function *) MALLOC (sizeof (Function));

  new->type = U_FUNCTION;
  new->name = 0;
  new->n_args = 0;
  new->args = 0;
  new->n_local = 0;
  new->local = 0;
  new->ncode = 0;
  new->code = 0;

  return (new);
}

/* **************************************************************
 * Destroy an RLaB Function entity.
 * ************************************************************** */
void
function_Destroy (f)
     Function *f;
{
  f->type = U_FUNCTION;
  FREE (f->name);

  if (f->args)
    list_Destroy (f->args);
  f->args = 0;
  f->n_args = 0;

  if (f->local)
    list_Destroy (f->local);
  f->n_local = 0;
  f->local = 0;

  f->ncode = 0;
  FREE (f->code);
  FREE (f);
}

Function *
function_Copy (f)
     Function *f;
{
  ASSERT (f);
  {
    char *key;
    int i;
    Function *new = function_Create ();

    /* Create/Copy the argument list */
    new->n_args = f->n_args;
    if (new->n_args > 0)
    {
      new->args = list_Create ();
      for (i = 1; i <= f->n_args; i++)
      {
	key = listNode_GetKey (list_GetNodeByPos (f->args, f->n_args - i + 1));
	arg_var_push (new->args, cpstr (key));
      }
    }

    /* Create/Copy the local var list */
    new->n_local = f->n_local;
    if (new->n_local > 0)
    {
      new->local = list_Create ();
      for (i = 1; i <= f->n_local; i++)
      {
	key = listNode_GetKey (list_GetNodeByPos (f->local, f->n_local - i + 1));
	local_var_push (new->local, cpstr (key));
      }
    }
    new->ncode = f->ncode;
    new->code = (Inst *) MALLOC (sizeof (Inst) * (new->ncode + 1));
    /* Copy the compliled code */
    for (i = 0; i < f->ncode; i++)
      new->code[i] = f->code[i];

    return (new);
  }
}

char *
function_GetName (f)
     Function *f;
{
  ASSERT (f);
  {
    return (f->name);
  }
}

void
function_SetName (f, name)
     Function *f;
     char *name;
{
  ASSERT (f);
  {
    FREE (f->name);
    f->name = name;
  }
}

int
function_HasLocalVar (f)
     Function *f;
{
  if (f->n_local > 0)
    return (1);
  else
    return (0);
}

/* **************************************************************
 * Set the function code pointer
 * ************************************************************** */
void
function_SetCodePtr (f, ptr)
     Function *f;
     Inst *ptr;
{
  f->code = ptr;
}

Inst *
function_GetCodePtr (f)
     Function *f;
{
  return (f->code);
}

int
function_SetCodeSize (f, size)
     Function *f;
     int size;
{
  ASSERT (f);
  {
    f->code = (Inst *) MALLOC (size * (sizeof (Inst)));
    f->ncode = size;
    return (1);
  }
}

int
function_GetCodeSize (f)
     Function *f;
{
  ASSERT (f);
  {
    return (f->ncode);
  }
}

void
function_SetArgPtr (f, arg_list)
     Function *f;
     List *arg_list;
{
  f->args = arg_list;
}

void
function_SetLocalPtr (f, local_var_list)
     Function *f;
     List *local_var_list;
{
  f->local = local_var_list;
}

List *
function_GetArgPtr (f)
     Function *f;
{
  return (f->args);
}

List *
function_GetLocalPtr (f)
     Function *f;
{
  return (f->local);
}

int
function_GetNlocal (f)
     Function *f;
{
  return (f->n_local);
}

void
function_SetNargs (f, n_args)
     Function *f;
     int n_args;
{
  f->n_args = n_args;
}

int
function_GetNargs (f)
     Function *f;
{
  return (f->n_args);
}

void
function_SetNlocal (f, n_local)
     Function *f;
     int n_local;
{
  f->n_local = n_local;
}

/* **************************************************************
 * Create a local variable object.
 * ************************************************************** */
LVar *
lvar_Create ()
{
  LVar *new = (LVar *) MALLOC (sizeof (LVar));

  new->type = LOCAL_VAR;
  new->name = 0;
  new->offset = 0;

  return (new);
}

/* **************************************************************
 * Destroy an instance of a LVar.
 * ************************************************************** */
void
lvar_Destroy (l_var)
     LVar *l_var;
{
  ASSERT (l_var);
  {
    l_var->type = 0;
    FREE (l_var->name);
    l_var->offset = 0;

    FREE (l_var);
  }
}

/* **************************************************************
 * Set the offset for a LVar
 * ************************************************************** */
void
lvar_SetOffset (l_var, offset)
     LVar *l_var;
     int offset;
{
  l_var->offset = offset;
}

void
lvar_SetName (l_var, name)
     LVar *l_var;
     char *name;
{
  FREE (l_var->name);
  l_var->name = name;
}

char *
lvar_GetName (l_var)
     LVar *l_var;
{
  return (l_var->name);
}

/* **************************************************************
 * Push a function arg onto a list. At the same time set the arg
 * offset (on the stack).
 * ************************************************************** */
List *
arg_var_push (list, s)
     List *list;
     char *s;
{
  LVar *l_var = lvar_Create ();

  if (list == 0)
    list = list_Create ();
  {
    ListNode *new_node;

    new_node = listNode_Create ();
    listNode_SetKey (new_node, s);
    listNode_AttachData (new_node, ARG_VAR, l_var, lvar_Destroy);
    list_PushNode (list, new_node);
  }
  lvar_SetName (l_var, cpstr (s));
  lvar_SetOffset (l_var, list_GetNumNodes (list));

  return (list);
}

/* **************************************************************
 * Push a local var onto a list. At the same time set the variable
 * offset (on the stack).
 * ************************************************************** */
List *
local_var_push (list, s)
     List *list;
     char *s;
{
  LVar *l_var = lvar_Create ();

  if (list == 0)
    list = list_Create ();
  {
    ListNode *new_node;

    new_node = listNode_Create ();
    listNode_SetKey (new_node, s);
    listNode_AttachData (new_node, LOCAL_VAR, l_var, lvar_Destroy);
    list_PushNode (list, new_node);
  }
  lvar_SetName (l_var, cpstr (s));
  lvar_SetOffset (l_var, list_GetNumNodes (list));

  return (list);
}
