/* op.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "code.h"
#include "listnode.h"
#include "util.h"
#include "scop1.h"
#include "matop1.h"
#include "matop2.h"
#include "r_string.h"

/* bltin.c */
extern double errno_check _PROTO ((double d, char *s));

/*
 * Defines used to flatten out some of the switches.
 */

#define SCALAR_SCALAR 1111
#define SCALAR_MATRIX 1113
#define SCALAR_STRING 1120

#define MATRIX_MATRIX 1133
#define MATRIX_SCALAR 1131
#define MATRIX_STRING 1140

#define STRING_STRING 1210
#define STRING_SCALAR 1201
#define STRING_MATRIX 1203

static Datum ent_ent_add _PROTO ((ListNode * e1, ListNode * e2));

/* **************************************************************
 * General addition operation on two Datums, return a new Datum.
 * ************************************************************** */
Datum
addition_op (d1, d2)
     Datum d1, d2;
{
  Datum new;

  /* Convert all CONSTANT and iCONSTANT into entities in order
     to eliminate switch at this level */
  d1 = convert_const (d1);
  d2 = convert_const (d2);

  /* Remove this function call later */
  new = ent_ent_add (d1.u.ent, d2.u.ent);
  
  remove_tmp_destroy (d1.u.ent);
  remove_tmp_destroy (d2.u.ent);
  return (new);
}

/* **************************************************************
 * Add two entities, return the result in a new Datum.
 * ************************************************************** */

static Datum
ent_ent_add (e1, e2)
     ListNode *e1, *e2;
{
  Datum new;
  Scalar *s;
  String *str;
  Matrix *m;

  switch (STYPE (e_type (e1), e_type (e2)))
  {
  case SCALAR_SCALAR:
    s = scalar_CreateC (SVALr (e_data (e1)) + SVALr (e_data (e2)),
			SVALi (e_data (e1)) + SVALi (e_data (e2)));
    new.u.ent = install_tmp (SCALAR, s, scalar_Destroy);
    new.type = ENTITY;
    break;

  case SCALAR_MATRIX:		/* scalar + matrix */
    m = matrix_scalar_add (e_data (e2), e_data (e1));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  case SCALAR_STRING:
    error_2 (e_name (e1), e_name (e2), "cannot add numeric and string");
    break;

  case MATRIX_SCALAR:		/* matrix + scalar */
    m = matrix_scalar_add (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
    
  case MATRIX_MATRIX:		/* matrix + matrix */
    m = matrix_Add (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
    
  case MATRIX_STRING:
    m = matrix_string_add (e_data (e1), e_data (e2), 1);
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
    
  case STRING_STRING:
    str = string_Add (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (STRING, str, string_Destroy);
    new.type = ENTITY;
    break;
    
  case STRING_MATRIX:
    m = matrix_string_add (e_data (e2), e_data (e1), 2);
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  default:
    error_2 (e_name (e1), e_name (e2), "arg(s) invalid for add-op");
    break;
  }
  return (new);
}

static Datum ent_ent_subtract _PROTO ((ListNode * e1, ListNode * e2));

/* **************************************************************
 * General subtraction operation on two Datums, return a new Datum.
 * d1 - d2
 * ************************************************************** */

Datum
subtraction_op (d1, d2)
     Datum d1, d2;
{
  Datum new;

  /* Convert all CONSTANT and iCONSTANT into entities in order
     to eliminate switch at this level */
  d1 = convert_const (d1);
  d2 = convert_const (d2);

  new = ent_ent_subtract (d1.u.ent, d2.u.ent);

  remove_tmp_destroy (d1.u.ent);
  remove_tmp_destroy (d2.u.ent);
  return (new);
}

/* **************************************************************
 * Subtract two entities, return the result in a new Datum.
 * e1 - e2
 * ************************************************************** */
static Datum
ent_ent_subtract (e1, e2)
     ListNode *e1, *e2;
{
  Datum new;
  Scalar *s;
  Matrix *m;

  switch (STYPE (e_type (e1), e_type (e2)))
  {
  case SCALAR_SCALAR:           /* scalar - scalar */
    s = scalar_Subtract (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (SCALAR, s, scalar_Destroy);
    new.type = ENTITY;
    break;

  case SCALAR_MATRIX:		/* scalar - matrix */
    m = matrix_scalar_sub2 (e_data (e2), e_data (e1));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_SCALAR:		/* matrix - scalar */
    m = matrix_scalar_sub1 (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
    
  case MATRIX_MATRIX:		/* matrix - matrix */
    m = matrix_Sub (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
    
  default:
    error_2 (e_name (e1), e_name (e2), 
	     "arg(s) invalid for subtract-op");
    break;

  }
  return (new);
}

static Datum ent_ent_el_add _PROTO ((ListNode *e1, ListNode *e2));

/* **************************************************************
 * Element-by-element addition operation on two Datums, 
 * return a new Datum.
 * ************************************************************** */

Datum
el_addition_op (d1, d2)
     Datum d1, d2;
{
  Datum new;

  /*
   * Convert all CONSTANT and iCONSTANT into entities in order
   * to eliminate switch at this level
   */

  d1 = convert_const (d1);
  d2 = convert_const (d2);

  /* Remove this function call later */
  new = ent_ent_el_add (d1.u.ent, d2.u.ent);

  remove_tmp_destroy (d1.u.ent);
  remove_tmp_destroy (d2.u.ent);
  return (new);
}

/* **************************************************************
 * Add two entities in an element-by-element sense, 
 * return the result in a new Datum.
 * ************************************************************** */

static Datum
ent_ent_el_add (e1, e2)
     ListNode *e1, *e2;
{
  Datum new;
  Scalar *s;
  String *str;
  Matrix *m;

  switch (STYPE (e_type (e1), e_type (e2)))
  {
  case SCALAR_SCALAR:		/* scalar + scalar */
    s = scalar_Add (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (SCALAR, s, scalar_Destroy);
    new.type = ENTITY;
    break;

  case SCALAR_MATRIX:		/* scalar + matrix */
    m = matrix_scalar_add (e_data (e2), e_data (e1));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
    
  case MATRIX_SCALAR:	        /* matrix + scalar */
    m = matrix_scalar_add (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
    
  case MATRIX_MATRIX:		/* matrix + matrix */
    m = matrix_El_Add (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
    
  case MATRIX_STRING:
    m = matrix_string_add (e_data (e1), e_data (e2), 1);
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
    
  case STRING_STRING:
    str = string_Add (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (STRING, str, string_Destroy);
    new.type = ENTITY;
    break;
    
  case STRING_MATRIX:
    m = matrix_string_add (e_data (e2), e_data (e1), 2);
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  default:
    error_2 (e_name (e1), e_name (e2), 
	     "arg(s) invalid for el-add-op");
    break;

  }
  return (new);
}

static Datum ent_ent_el_subtract _PROTO ((ListNode * e1, ListNode * e2));

/* **************************************************************
 * General subtraction operation on two Datums, return a new Datum.
 * d1 - d2
 * ************************************************************** */

Datum
el_subtraction_op (d1, d2)
     Datum d1, d2;
{
  Datum new;

  /*
   * Convert all CONSTANT and iCONSTANT into entities in order
   * to eliminate switch at this level
   */

  d1 = convert_const (d1);
  d2 = convert_const (d2);

  new = ent_ent_el_subtract (d1.u.ent, d2.u.ent);

  remove_tmp_destroy (d1.u.ent);
  remove_tmp_destroy (d2.u.ent);
  return (new);
}

/* **************************************************************
 * Subtract two entities in an element-by-element fashion, 
 * return the result in a new Datum.
 * e1 - e2
 * ************************************************************** */

static Datum
ent_ent_el_subtract (e1, e2)
     ListNode *e1, *e2;
{
  Datum new;
  Scalar *s;
  Matrix *m;

  switch (STYPE (e_type (e1), e_type (e2)))
  {
  case SCALAR_SCALAR:		/* scalar - scalar */
    s = scalar_Subtract (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (SCALAR, s, scalar_Destroy);
    new.type = ENTITY;
    break;
    
  case SCALAR_MATRIX:		/* scalar - matrix */
    m = matrix_scalar_sub2 (e_data (e2), e_data (e1));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_SCALAR:		/* matrix - scalar */
    m = matrix_scalar_sub1 (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
    
  case MATRIX_MATRIX:		/* matrix - matrix */
    m = matrix_El_Sub (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
    
  default:
    error_2 (e_name (e1), e_name (e2), 
	     "arg(s) invalide for el-subtraction-op");
    break;
    
  }
  return (new);
}

static Datum ent_ent_multiply _PROTO ((ListNode * e1, ListNode * e2));

/* **************************************************************
 * General multiply operation on two Datums, return a new Datum.
 * ************************************************************** */

Datum
multiply_op (d1, d2)
     Datum d1, d2;
{
  Datum new;

  /* Convert all CONSTANT and iCONSTANT into entities in order
     to eliminate switch at this level */
  d1 = convert_const (d1);
  d2 = convert_const (d2);

  new = ent_ent_multiply (d1.u.ent, d2.u.ent);

  remove_tmp_destroy (d1.u.ent);
  remove_tmp_destroy (d2.u.ent);
  return (new);
}

/* **************************************************************
 * Multiply two entities, return the result in a new Datum.
 * ************************************************************** */

static Datum
ent_ent_multiply (e1, e2)
     ListNode *e1, *e2;
{
  Datum new;
  Scalar *s;
  Matrix *m;

  switch (STYPE (e_type (e1), e_type (e2)))
  {
  case SCALAR_SCALAR:		/* scalar * scalar */
    s = scalar_Multiply (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (SCALAR, s, scalar_Destroy);
    new.type = ENTITY;
    break;
    
  case SCALAR_MATRIX:		/* scalar * matrix */
    m = matrix_scalar_mul (e_data (e2), e_data (e1));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
    
  case MATRIX_SCALAR:		/* matrix * scalar */
    m = matrix_scalar_mul (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
    
  case MATRIX_MATRIX:		/* matrix * matrix */
    m = matrix_Multiply (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  default:
    error_2 (e_name (e1), e_name (e2), 
	     "arg(s) invalid for multiply-op");
    break;

  }
  return (new);
}

/* **************************************************************
 * Right Division  A / B
 * ************************************************************** */

Datum
rdivide (d1, d2)
     Datum d1, d2;
{
  Datum new;
  Matrix *m;

  /* Convert anything possible to a matrix */
  d1 = convert_to_matrix_d (d1);
  d2 = convert_to_matrix_d (d2);

  m = matrix_Rdivide (e_data (d1.u.ent), e_data (d2.u.ent));
  new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
  new.type = ENTITY;

  remove_tmp_destroy (d1.u.ent);
  remove_tmp_destroy (d2.u.ent);
  return (new);
}

/*
 * Element - by - element right divide.
 */

static Datum ent_ent_el_rdivide _PROTO ((ListNode * e1, ListNode * e2));

Datum
el_rdivide (d1, d2)
     Datum d1, d2;
{
  Datum new;

  /* Convert all CONSTANT and iCONSTANT into entities in order
     to eliminate switch at this level */
  d1 = convert_const (d1);
  d2 = convert_const (d2);

  new = ent_ent_el_rdivide (d1.u.ent, d2.u.ent);

  remove_tmp_destroy (d1.u.ent);
  remove_tmp_destroy (d2.u.ent);
  return (new);
}

static Datum
ent_ent_el_rdivide (e1, e2)
     ListNode *e1, *e2;
{
  Datum new;
  Scalar *s;
  Matrix *m;

  switch (STYPE (e_type (e1), e_type (e2)))
  {
  case SCALAR_SCALAR:		/* scalar / scalar */
    s = scalar_Divide (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (SCALAR, s, scalar_Destroy);
    new.type = ENTITY;
    break;
    
  case SCALAR_MATRIX:		/* scalar / matrix */
    m = matrix_scalar_div2 (e_data (e2), e_data (e1));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_SCALAR:		/* matrix / scalar */
    m = matrix_scalar_div1 (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
    
  case MATRIX_MATRIX:		/* matrix / matrix */
    m = matrix_El_Div (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  default:
    error_2 (e_name (e1), e_name (e2), 
	     "arg(s) invalid for el-right-divide-op");
    break;

  }
  return (new);
}

/* **************************************************************
 * Left Division  A \ B
 * ************************************************************** */

Datum
ldivide (d1, d2)
     Datum d1, d2;
{
  Datum new;
  Matrix *m;

  /* Convert anything possible to a matrix */
  d1 = convert_to_matrix_d (d1);
  d2 = convert_to_matrix_d (d2);
  
  m = matrix_Ldivide (e_data (d1.u.ent), e_data (d2.u.ent));
  new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
  new.type = ENTITY;

  remove_tmp_destroy (d1.u.ent);
  remove_tmp_destroy (d2.u.ent);
  return (new);
}

/*
 * Element - by element left divide.
 */

static Datum ent_ent_el_ldivide _PROTO ((ListNode * e1, ListNode * e2));

Datum
el_ldivide (d1, d2)
     Datum d1, d2;
{
  Datum new;

  /* Convert all CONSTANT and iCONSTANT into entities in order
     to eliminate switch at this level */
  d1 = convert_const (d1);
  d2 = convert_const (d2);

  new = ent_ent_el_ldivide (d1.u.ent, d2.u.ent);

  remove_tmp_destroy (d1.u.ent);
  remove_tmp_destroy (d2.u.ent);
  return (new);
}

static Datum
ent_ent_el_ldivide (e1, e2)
     ListNode *e1, *e2;
{
  Datum new;
  Scalar *s;
  Matrix *m;

  switch (STYPE (e_type (e1), e_type (e2)))
  {
  case SCALAR_SCALAR:		/* scalar1 \ scalar2 */
    s = scalar_Divide (e_data (e2), e_data (e1));
    new.u.ent = install_tmp (SCALAR, s, scalar_Destroy);
    new.type = ENTITY;
    break;
    
  case SCALAR_MATRIX:		/* scalar \ matrix */
    m = matrix_scalar_div1 (e_data (e2), e_data (e1));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  case MATRIX_SCALAR:		/* matrix \ scalar */
    m = matrix_scalar_div2 (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
    
  case MATRIX_MATRIX:		/* matrix \ matrix */
    m = matrix_El_Div (e_data (e2), e_data (e1));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  default:
    error_2 (e_name (e1), e_name (e2), 
	     "arg(s) invalid for el-left-divide-op");
    break;
  }

  return (new);
}

/* **************************************************************
 * Negate a CONSTANT or an ENTITY.
 * ************************************************************** */

Datum
negate_op (d)
     Datum d;
{
  Datum new;
  Scalar *s;
  Matrix *m;

  switch (d.type)
  {
  case CONSTANT:
    new.u.val = -d.u.val;
    new.type = CONSTANT;
    break;

  case iCONSTANT:
    new.u.val = -d.u.val;
    new.type = iCONSTANT;
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      s = scalar_ChangeSign (e_data (d.u.ent));
      new.u.ent = install_tmp (SCALAR, s, scalar_Destroy);
      new.type = ENTITY;
      remove_tmp_destroy (d.u.ent);
      break;

    case MATRIX:
      m = matrix_Negate (e_data (d.u.ent));
      new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
      new.type = ENTITY;
      remove_tmp_destroy (d.u.ent);
      break;

    default:
      error_1 (e_name (d.u.ent), "Invalid type, cannot change sign");
      break;

    }
    break;

  default:
    error_1 ("invalid entity, cannot change sign", 0);
    break;

  }
  return (new);
}

/* **************************************************************
 * Return a Datum that contains d1 ^ d2
 * ************************************************************** */

static Datum ent_ent_pow _PROTO ((ListNode * d1, ListNode * d2));

Datum
power_op (d1, d2)
     Datum d1, d2;
{
  Datum new;

  /* Convert all CONSTANT and iCONSTANT into entities in order
     to eliminate switch at this level */
  d1 = convert_const (d1);
  d2 = convert_const (d2);

  new = ent_ent_pow (d1.u.ent, d2.u.ent);

  remove_tmp_destroy (d1.u.ent);
  remove_tmp_destroy (d2.u.ent);
  return (new);
}

static Datum
ent_ent_pow (e1, e2)
     ListNode *e1, *e2;
{
  Datum new;
  Scalar *s;
  Matrix *m;

  switch (STYPE (e_type (e1), e_type (e2)))
  {
  case SCALAR_SCALAR:		/* scalar ^ scalar */
    s = scalar_Pow (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (SCALAR, s, scalar_Destroy);
    new.type = ENTITY;
    break;
    
  case MATRIX_SCALAR:		/* matrix ^ scalar */
    m = matrix_scalar_Pow1 (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  case SCALAR_MATRIX:		/* scalar ^ matrix */
    m = matrix_scalar_Pow2 (e_data (e2), e_data (e1));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
    
  case MATRIX_MATRIX:		/* matrix ^ matrix */
    error_1 ("matrix^matrix illegal", 0);
    break;

  default:
    error_2 (e_name (e1), e_name (e2), 
	     "arg(s) invalid for power-op");
    break;

  }
  return (new);
}

/*
 * Element - by - element Power Operation.
 * d1 .^ d2
 */

static Datum ent_ent_el_pow _PROTO ((ListNode * d1, ListNode * d2));

Datum
el_power (d1, d2)
     Datum d1, d2;
{
  Datum new;

  /* Convert all CONSTANT and iCONSTANT into entities in order
     to eliminate switch at this level */
  d1 = convert_const (d1);
  d2 = convert_const (d2);

  new = ent_ent_el_pow (d1.u.ent, d2.u.ent);

  remove_tmp_destroy (d1.u.ent);
  remove_tmp_destroy (d2.u.ent);
  return (new);
}

static Datum
ent_ent_el_pow (e1, e2)
     ListNode *e1, *e2;
{
  Datum new;
  Scalar *s;
  Matrix *m;

  switch (STYPE (e_type (e1), e_type (e2)))
  {
  case SCALAR_SCALAR:		/* scalar .^ scalar */
    s = scalar_Pow (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (SCALAR, s, scalar_Destroy);
    new.type = ENTITY;
    break;
    
  case MATRIX_SCALAR:		/* matrix .^ scalar */
    m = matrix_Pow1 (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  case SCALAR_MATRIX:		/* scalar .^ matrix */
    m = matrix_Pow2 (e_data (e2), e_data (e1));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;
    
  case MATRIX_MATRIX:		/* matrix .^ matrix */
    m = matrix_matrix_el_pow (e_data (e1), e_data (e2));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  default:
    error_2 (e_name (e1), e_name (e2), 
	     "arg(s) invalid for el-power-op");
    break;

  }
  return (new);
}
