/* rlab.y */

/*  This file is a part of RLaB ("Our"-LaB)
    Copyright (C) 1992, 1993, 1994  Ian R. Searle

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    See the file ./COPYING 
 ***********************************************************************/

%{
#include "rlab.h"
#include "mem.h"
#include "list.h"
#include "code.h"
#include "symbol.h"
#include "scalar.h"
#include "function.h"
#include "r_string.h"
#include "util.h"
#include <setjmp.h>
  
/* Trick yyparse() so we can pass yychar into yyerror() */
#define yyerror(s)  ryyerror(s, yychar)
void ryyerror _PROTO((char *s, int yychar));

/* Lookup variables in symbol table */
static Var * name_lookup _PROTO((List *list1, List *list2, char *name, int scope));

extern char *line_contents;   /* scan.l */
extern int lineno;            /* scan.l */
extern char *curr_file_name;  /* scan.l */
extern int line_nos;          /* main.c */
extern int dec_buff _PROTO ((void));
extern jmp_buf jmp[];

extern int do_eval;           /* bltin2.c, main.c */

/* Help and rfile functions */
extern void help _PROTO(( void ));
extern void help_name _PROTO(( char *name ));
extern void rfile _PROTO(( void ));
extern void rfile_load _PROTO(( char *name ));

void fstatic_var_push _PROTO((char *file_name, char *name));
ListNode *static_lookup _PROTO((char *file_name, char *name));

int flush_line = 0;           /* Tells yylex() when to flush rest of line */
int prompt = 0;               /* prompt=0: put out prompt-1, p != 0: prompt-2 */

List *arg_sym_tab;            /* For tracking argument lists */
List *local_sym_tab;          /* For tracking local() statement lists */

int scope = GLOBAL;           /* Determines current scope */
static int looping = 0;       /* for keeping track of BREAK, CONTINUE */
static int psave = 0;         /* Save program pointer (offset) */
static int lsave = 0;         /* Save line number */
int loff = 100;               /* The line # offset, to avoid op-code collisions */
int i;

%}    /* Declarations */

%start program

%union {
  List       *list;   /* ptr to a linked list */
  ListNode   *ent;    /* pointer to an RLaB entity */
  double      d_val;  /* double numbers */
  int         n_int;  /* number of items in a list */
  char       *name;   /* char ptr to name string passed from scanner */
  struct _nn {
    int off;
    int count;
  } nn;               /* For machine offset, and count */
  Var *vinfo;         /* Global, arg, and local variable info */
}

%token <d_val> NUMBER iNUMBER
%token <name>  NAME R_STRING HELP_NAME RFILE_NAME UNDEFINED
%token <n_int> WHILE IF ELSE QUIT FOR IN BREAK CONTINUE
%token <n_int> FUNCTION RETURN SEMIC Q_MARK LOCAL_DEC SELF
%token <n_int> HELP RFILE GST FSTATIC_DEC JNK

%type  <n_int>  program stmt c_stmt stmts line cstmts cline NL texpr_or_empty
%type  <n_int>  expr vec_expr mat_expr mat_list texpr vexpr
%type  <n_int>  cond while if for for_spec end function function_assign
%type  <n_int>  if_stmt while_stmt for_stmt return_stmt
%type  <n_int>  break continue
%type  <n_int>  vid for_vec_expr assign list_expr list_el
%type  <n_int>  separator newlines list_member nl
%type  <list>   opt_arg_name_list local_stmt opt_local_name_list
%type  <n_int>  fstatic_stmt opt_fstatic_name_list
%type  <vinfo>  var
%type  <ent>    function_ent
%type  <nn>     opt_arg_list list_list if_blk texpr_list

/* Operator Precedence and Associativity */
%nonassoc JNK
%nonassoc NUMBER iNUMBER NAME R_STRING WHILE IF QUIT 
%nonassoc FOR BREAK CONTINUE FUNCTION RETURN SELF GST FSTATIC_DEC
%right '='
%left  ','              /* concatenation operator */
%left  ':'              /* vector creation operator */
%left  OR               /* || */
%left  AND              /* && */
%left  EQ NE            /* == != */
%left  GT GE LT LE      /* > >= < <= */
%left  '+' '-' EL_ADD_OP EL_SUB_OP
%left  '*' '/' EL_MUL_OP EL_DIV_OP EL_LDIV_OP LDIV_OP
%nonassoc UNARY_MINUS UNARY_PLUS
%right '^' EL_POW_OP
%nonassoc TRANSPOSE EL_TRANSPOSE
%nonassoc NOT INC DEC
%nonassoc LEFT_LIST RIGHT_LIST
%left '[' ']' '.'

%%      /* Grammar */

 program: /* empty */      { $$ = 0; }
        | program line     { code(STOP);
			     return(1); }
        | program RFILE    { code(OP_RFILE); code(STOP);
			     return (1); }
        | program RFILE_NAME  { $$ = 0; code(OP_RFILE_NAME);
			     codep($2); code (STOP);
			     return (1); }
        | program HELP     { code(OP_HELP); code(STOP);
			     return (1); }
        | program HELP_NAME   { $$ = 0; code(OP_HELP_NAME);
			     codep ($2); code(STOP);
			     return (1); }
        | program error
        ;
 
 /*
  * Line
  */

 line: NL           { $$ = progoff; }
     | stmts NL     { $$ = $1; }
     ;

 /*
  * Statements
  */

 stmts: stmt
      | stmts stmt
      ;

 stmt: texpr separator  { if (do_eval)
			  { 
			    /* Only for eval() */
			    code (OP_SAVE_EVAL);
			  }
			  else
			  { 
			    /* Normal operation */
			    if ($2)
			      code (OP_PRINT);
			    else
			      code (OP_POP_CLEAN);
			  }
			}
     | if_stmt
     | while_stmt
     | for_stmt
     | break
     | continue
     | return_stmt
     | c_stmt
     | function
     | fstatic_stmt
     | QUIT                 { code(OP_QUIT); }
     ;

 separator: /* empty */ { $$ = 1; }
          | SEMIC       { $$ = 0; }
          | Q_MARK      { $$ = 1; }
          ;

 left_paren: '('
           ;

 right_paren: ')'
            ;

 left_brace: '{'
           ;

 right_brace: '}'
            ;

 if_blk: left_brace cstmts right_brace  { $$.off = $2;
					  $$.count = 0; }
       | left_brace cstmts end ELSE cstmts right_brace  { $$.off = $2; 
							  $$.count = $5; }
       ;

 cline: NL    { $$ = progoff; }
      | stmts  %prec JNK
      ; 

 cstmts: cline
       | cstmts cline
       ;

 c_stmt: left_brace cstmts right_brace  { $$ = $2; }
       ;

 if_stmt: if cond newlines if_blk end {
          program->prog[$1+1].op_code = $4.off - $1;   /* if_blk offset */
	  if($4.count != 0) {
	    program->prog[$1+2].op_code = $4.count - $1;
	  } else {
	    program->prog[$1+2].op_code = 0;
	  }
          program->prog[$1+3].op_code = $5 - $1;  /* end, if cond fails */ 
	  prompt--; }
        ;

 while_stmt: while cond newlines c_stmt end {
             program->prog[$1+1].op_code = $4 - $1;    /* body of loop */
	     program->prog[$1+2].op_code = $5 - $1;    /* end, if cond fails */
	     looping--;
	     prompt--;
	   }
           ;

 for_stmt: for for_spec newlines c_stmt end
         {
           program->prog[$1+1].op_code = $4 - $1;   /* body of loop */
           program->prog[$1+2].op_code = $5 - $1;   /* end of loop */  
	   looping--;
	   prompt--;
	 }
         ;

 for: FOR  { $$ = code(OP_FOR); code(STOP); code(STOP); 
	     looping++; 
	     prompt++; }
    ;

 for_spec: left_paren vid IN for_vec_expr right_paren  { $$ = $2; }
         ;

 for_vec_expr: texpr  { code(STOP); }
             ;

 return_stmt: RETURN texpr separator
              { $$ = $2;
		if(scope == GLOBAL)
		  {
		    yyerror("return not allowed outside function");
		    longjmp( jmp[dec_buff()], 1 );
		  }
		code(OP_FUNCTION_RETURN);
	      }

 cond: left_paren texpr right_paren  { code(STOP); $$ = $2; }
     ;

 while: WHILE  { $$ = code(OP_WHILE); code(STOP); code(STOP); 
		 looping ++; 
		 prompt++; }
      ;

 if: IF  { $$ = code(OP_IF); 
	   code(STOP); code(STOP); code(STOP); 
	   prompt++; }
   ;

 break: BREAK separator { if(!looping) 
			  {
			    yyerror("break stmt not allowed outside loop");
			    longjmp( jmp[dec_buff()], 1 );
			  }
			  $$ = code(OP_BREAK); 
			  code(STOP); 
			}
      ;

 continue: CONTINUE separator { if(!looping) 
				{
				  yyerror("continue stmt not allowed outside loop");
				  longjmp( jmp[dec_buff()], 1 );
				}
				$$ = code(OP_CONTINUE);
				code(STOP);
			      }
         ;

 end: /* empty */    { code(STOP); $$ = progoff; }
    ;

 /*
  * Expressions
  */

 /* Top level expression */
 texpr: expr %prec '='
      | vexpr
      | assign
      ;

 expr: '(' texpr ')'     { $$ = $2; }
     ;

 /* Vector expression */
 vexpr: expr ':' expr         { code(OP_VECTOR_CREATE); 
				 code(2); }
      | expr ':' expr ':' expr { code(OP_VECTOR_CREATE);
				 code(3); }
      ;

 /* Assignments */
 assign:   vid '=' texpr       { code(OP_ASSIGN); }
         /* matrix assign */
         | expr '[' vec_expr SEMIC vec_expr ']' '=' texpr
                                  { code(OP_MATRIX_ASSIGN); 
				    code(1); }
         | expr '[' vec_expr SEMIC ']' '=' texpr
                                  { code(OP_MATRIX_ASSIGN); 
				    code(2); }
         | expr '[' SEMIC vec_expr ']' '=' texpr
                                  { code(OP_MATRIX_ASSIGN); 
				    code(3); }
         | expr '[' vec_expr ']' '=' texpr
                                  { code(OP_MATRIX_VEC_ASSIGN); }
         /* list assign */
         | expr '.' '[' expr ']' '=' texpr
                                  { code(OP_LIST_ASSIGN);
				    code(1); }
         | expr '.' NAME '=' texpr
                                  { code(OP_LIST_ASSIGN);
				    code(2);
				    codep($3);
				    string_log($3); }
         | function_assign
         ;

 function_assign: vid '=' function       { code(OP_ASSIGN); }
                | expr '.' '[' expr ']' '=' function
                                         { code(OP_LIST_ASSIGN);
					   code(1); }
                | expr '.' NAME '=' function
                                         { code(OP_LIST_ASSIGN);
					   code(2);
					   codep($3);
					   string_log($3); }
                ;

 /* 
  * Matrix expression 
  */

 mat_expr: '[' mat_list ']'  { $$ = $2; }
         ;

 expr: mat_expr
     ;

 mat_list: /* empty matrix */       { $$ = code(OP_EMPTY_MATRIX_CREATE); }
         | vec_expr                 { code(OP_MATRIX_CREATE); }
         | mat_list SEMIC vec_expr  { code(OP_MATRIX_APPEND); }
         | mat_list SEMIC nl vec_expr  { code(OP_MATRIX_APPEND); }
         ;

 /* Vector concatenation */
 vec_expr: expr
         | vexpr
         | vec_expr ',' vec_expr  { code(OP_VEC_APPEND); }
         ;

 /* sub-matrix */
 expr: expr '[' vec_expr SEMIC vec_expr ']'
                       {  code(OP_MATRIX_SUB);
			  code(1); }
     | expr '[' vec_expr SEMIC ']'
                       {  code(OP_MATRIX_SUB);
			  code(2); }
     | expr '[' SEMIC vec_expr ']'
                       {  code(OP_MATRIX_SUB);
			  code(3); }
     | expr '[' SEMIC ']' { /* no-op */ }
     | expr '[' ']'       { /* no-op */ }
     | expr '[' vec_expr ']'
                       {  code(OP_MATRIX_VEC_SUB); }
     | expr '[' ':' ']'
                       {  code(OP_MATRIX_COL); }
     ;

 /* 
  * List Expression
  */

 expr: list_expr
     | list_member
     ;

 list_expr: LEFT_LIST list_list RIGHT_LIST
                               { $$ = $2.off;
				 code(OP_LIST_CREATE);
				 code($2.count); }
          ;

 list_member: expr '.' '[' expr ']'    { code(OP_LIST_MEMB);
					     code(1); }
            | expr '.' NAME   { code(OP_LIST_MEMB);
				code(2);
				codep($3);
				string_log($3); }
            ;

 list_list: /* empty */        { $$.count = 0; $$.off = progoff; }
          | list_el            { $$.count = 1; $$.off = $1; }
          | list_list SEMIC list_el
                               { $$.count = $1.count + 1;
				 $$.off = $1.off; }
          | list_list SEMIC NL list_el
                               { $$.count = $1.count + 1;
				 $$.off = $1.off; }
          ;

 list_el: NAME '=' texpr  { $$ = $3;
			       code(OP_LIST_EL_CREATE);
			       codep($1); 
			       string_log($1); }
        | texpr
        ;           
              
 /* 
  * General Expressions
  */

 expr: NUMBER      { $$ = code(OP_PUSH_CONSTANT); 
		     coded($1); }
     | iNUMBER     { $$ = code(OP_PUSH_iCONSTANT);
		     coded($1); }
     | R_STRING    { $$ = code(OP_PUSH_STRING); 
		     codep($1);
		     if(scope == GLOBAL)
		       install_tmp (STRING, (VPTR) string_Create($1), 
				    string_Destroy);
		     else
		       string_log($1);
		   }
     | vid
     | vid INC       { code(OP_INC); }
     | vid DEC       { code(OP_DEC); }
     | vid '(' opt_arg_list ')'
                     { 
		       code(OP_FUNCTION_CALL);
		       code($3.count);   /* number of args on stack */
		     }
     | self '(' opt_arg_list ')'
                     { 
		       code(OP_FUNCTION_CALL_SELF);
		       code($3.count);   /* number of args on stack */
		       $$ = $3.off;
		     }
     | list_member '(' opt_arg_list ')'
                     { 
		       code(OP_FUNCTION_CALL);
		       code($3.count);   /* number of args on stack */
		       $$ = $3.off;
		     }
     | expr '+' expr     { code(OP_ADD); }
     | expr EL_ADD_OP expr   { code(OP_EL_ADD); }
     | expr '-' expr     { code(OP_SUB); }
     | expr EL_SUB_OP expr   { code(OP_EL_SUB); }
     | expr '*' expr     { code(OP_MUL); }
     | expr EL_MUL_OP expr   { code(OP_EL_MUL); }
     | expr '/' expr         { code(OP_DIV); }
     | expr LDIV_OP expr     { code(OP_LDIV); }
     | expr EL_DIV_OP expr   { code(OP_EL_DIV); }
     | expr EL_LDIV_OP expr  { code(OP_EL_LDIV); }
     | expr '^' expr         { code(OP_POWER); }
     | expr EL_POW_OP expr   { code(OP_EL_POWER); }
     | expr TRANSPOSE    { code(OP_TRANSPOSE); }
     | expr EL_TRANSPOSE { code(OP_EL_TRANSPOSE); }
     | expr GT expr      { code(OP_GT);  }
     | expr GE expr      { code(OP_GE);  }
     | expr LT expr      { code(OP_LT);  }
     | expr LE expr      { code(OP_LE);  }
     | expr EQ expr      { code(OP_EQ);  }
     | expr NE expr      { code(OP_NE);  }
     | expr AND expr     { code(OP_AND); }
     | expr OR expr      { code(OP_OR);  }
     | NOT expr          { $$ = $2; code(OP_NOT); }
     | '-' expr %prec UNARY_MINUS  { $$ = $2;   code(OP_NEGATE); }
     | '+' expr %prec UNARY_PLUS   { $$ = $2; }
     ;

 /*
  * Variables
  */

 vid: var   { if($1->type == GLOBAL) 
		{
		  $$ = code(OP_PUSH_VAR); 
		  codep($1->ent);
		}
	      else if($1->type == LOCAL_VAR)
		{
		  $$ = code(OP_PUSH_LOCAL_VAR);
		  code($1->offset);
		}
	      else if($1->type == ARG_VAR)
		{
		  $$ = code(OP_PUSH_ARG);
		  code($1->offset);
		}
	      FREE($1);
	    }
    ;

 var: NAME  { $$ = name_lookup (arg_sym_tab, local_sym_tab, $1, scope); }      
    | GST   { $$ = gst (); }
    ;

 nl: NL             { $$ = 0; }
   | nl NL          { $$ = 0; }
   ;

 newlines: /* empty */    { $$ = 0; }
         | NL             { $$ = 0; }
         | newlines NL    { $$ = 0; }
         ;

 NL: '\n'  { if (line_nos)
	     {
	       code (OP_LINE_NO);
	       code (lineno+loff);
	     }
	   }
   ;

 /*
  * Functions
  */

 function: function_ent { $$ = code(OP_PUSH_VAR);
			  codep($1); }
         ;

 function_ent: FUNCTION  { prompt++; }
               left_paren opt_arg_name_list right_paren newlines '{' newlines
                   { if(scope == LOCAL) 
		       {
			 scope = GLOBAL;
			 if(arg_sym_tab != 0) 
			   list_Destroy(arg_sym_tab);
			 if(local_sym_tab != 0)
			   list_Destroy(local_sym_tab);
			 yyerror("function decl not allowed inside function");
			 longjmp( jmp[dec_buff()], 1 );
		       }
		     scope = LOCAL;
		     arg_sym_tab = $4; 
		     psave = progoff;
		     lsave = lineno+loff; 
		   }
               local_stmt  { local_sym_tab = $10; }
               cstmts '}' 
                   {  code(OP_DEF_FUNC_RET);
		      $$ = function_define($4, $10, psave, lsave);
		      arg_sym_tab = 0;
		      local_sym_tab = 0;
		      scope = GLOBAL; 
		      prompt--;
		    }
             ;

 self: SELF { if(scope == GLOBAL)
		{
		  yyerror("$self not allowed outside function");
		  longjmp( jmp[dec_buff()], 1 );
		}
	    }
     ;

 /* For each argument in a User-Function create a special 
    type of variable */
 opt_arg_name_list: /* empty */  { $$ = list_Create(); }
                  | NAME  { $$ = arg_var_push(0, $1); }
                  | opt_arg_name_list ',' NAME
                          { arg_var_push($1, $3); }
                  ;

 /* List of local variables for a function */
 local_stmt: /* empty */  { $$ = list_Create(); }
           | LOCAL_DEC '(' opt_local_name_list ')' separator
                          { $$ = $3; }
           ;

 opt_local_name_list: NAME  { $$ = local_var_push(0, $1); }
                    | opt_local_name_list ',' NAME
                            { local_var_push($1, $3); }
                    ;

 /* As we accumulate args on the stack, keep count */
 opt_arg_list: /* empty */            { $$.count = 0; $$.off = progoff; }
             | texpr                  { $$.count = 1; $$.off = $1; }
             | texpr_list             { $$.count = $1.count; 
					$$.off = $1.off; }
             ;

 texpr_list: texpr_or_empty ',' texpr_or_empty  { $$.count = 2; $$.off = $1; }
           | texpr_list ',' texpr_or_empty      { $$.count = $1.count + 1;
						  $$.off = $1.off; }
           ;

 texpr_or_empty: /* empty */   { code(OP_PUSH_UNDEF); }
               | texpr
               ;

 /* List of file-static variables */
 fstatic_stmt: FSTATIC_DEC '(' opt_fstatic_name_list ')' separator
                            { $$ = progoff; }
             ;

 opt_fstatic_name_list: NAME  { fstatic_var_push(curr_file_name, $1);
				$$ = 0; }
                      | opt_fstatic_name_list ',' NAME
                              { fstatic_var_push(curr_file_name, $3);
				$$ = 0; }
                      ;
%%

/* **************************************************************
 * Lookup a NAME in the symbol table, and return a ptr to the 
 * entity. This function pays attention to the scope of the 
 * variable usage, and looks for the NAME in the appropriate
 * place.
 * ************************************************************** */

extern int get_function_scope _PROTO ((void));       /* code.c */
extern List *get_function_arglist _PROTO ((void));   /* code.c */
extern List *get_function_locallist _PROTO ((void)); /* code.c */

static Var *
name_lookup(arg_list, local_list, name, scope)
     List *arg_list, *local_list;
     char *name;
     int scope;
{
  ListNode *ent;
  Var *retval = (Var *) MALLOC(sizeof(Var));

  if (do_eval && !scope)
  {
    /* Adjust things so eval string will get proper scope, etc... */
    scope = get_function_scope ();
    if (scope == LOCAL)
    {
      arg_list = get_function_arglist ();
      local_list = get_function_locallist ();
    }
  }

  if(scope == LOCAL)
    {
      /* 1st check function's local and arg lists */
      if(local_list && ((ent = lookup(local_list, name)) != 0))
      {
	/* LOCAL_VAR */
	FREE(name);
	retval->type = LOCAL_VAR;
	retval->offset = lvar_GetOffset(e_data(ent));
      }
      else if((ent = lookup(arg_list, name)) != 0)
      {
	/* ARG */
	retval->type = ARG_VAR;
	retval->offset = lvar_GetOffset(e_data(ent));
	retval->name = name;
      }
      else if ((ent = static_lookup(curr_file_name, name)))
      {
	FREE (name);
	retval->type = GLOBAL;
	retval->ent = ent;
      }
      else
	{ /* we did not find it, look on global symbol table */
	  if((ent = lookup(0, name)) != 0)
	    {
	      FREE(name);
	      retval->type = GLOBAL;
	      retval->ent = ent;
	    }
	  else
	    { /* we could not find it, so create it */
	      ent = install(0, name, UNDEF, 0);
	      /* ent = install(0, name, SCALAR, scalar_Create(0.0));*/
	      /* scalar_SetName(e_data(ent), cpstr(name));*/
	      retval->type = GLOBAL;
	      retval->ent = ent;
	    }
	}
    }
  else  /* scope == GLOBAL */
    {
      if ((ent = static_lookup(curr_file_name, name)))
      {
	FREE (name);
	retval->type = GLOBAL;
	retval->ent = ent;
      }
      else if((ent = lookup(0, name)) != 0)
	{
	  FREE(name);
	  retval->type = GLOBAL;
	  retval->ent = ent;
	}
      else
	{ /* we could not find it, so create it */
	  ent = install(0, name, UNDEF, 0);
	  /* ent = install(0, name, SCALAR, scalar_Create(0.0));*/
	  /* scalar_SetName(e_data(ent), cpstr(name));*/
	  retval->type = GLOBAL;
	  retval->ent = ent;
	}
    }
  return(retval);
}

/* **************************************************************
 * File Static Functions.
 * ************************************************************** */

static Btree *static_tree;

void
init_static_tree ()
{
  static_tree = btree_Create ();
}

void
destroy_fstatic_tree ()
{
  btree_Destroy (static_tree);
}

/*
 * Push a variable onto a file-static tree.
 * Don't return anything, this function is only called when
 * a fstatic() is encountered.
 */

void
fstatic_var_push (file_name, name)
     char *file_name, *name;
{
  ListNode *file_tree, *lnode;

  /* Check file-static tree for existence of file_name */

  if ((file_tree = btree_FindNode (static_tree, file_name)))
  {
    /* Look for name */
    if (!(lnode = btree_FindNode (e_data (file_tree), name)))
    { /* Did not find the name, create it */
      lnode = listNode_Create ();
      listNode_SetKey (lnode, cpstr (name));
      listNode_SetScope (lnode, STATIC);
      btree_AddNode (e_data (file_tree), lnode);
      listNode_SetType (lnode, UNDEF);
    }
  }
  else
  {
    /* Create file_tree, and add name */
    file_tree = listNode_Create ();
    listNode_SetKey (file_tree, cpstr (file_name));
    listNode_AttachData (file_tree, BTREE, btree_Create(), btree_Destroy);
    btree_AddNode (static_tree, file_tree);

    lnode = listNode_Create ();
    listNode_SetKey (lnode, cpstr (name));
    listNode_SetScope (lnode, STATIC);
    btree_AddNode (e_data (file_tree), lnode);    
    listNode_SetType (lnode, UNDEF);
  }
}

ListNode *
static_lookup (file_name, name)
     char *file_name, *name;
{
  ListNode *file_tree, *lnode;

  if ((file_tree = btree_FindNode (static_tree, file_name)))
  {
    if ((lnode = btree_FindNode (e_data (file_tree), name)))
      return (lnode);
  }
  return (0);
}

/* **************************************************************
 * Called for YACC syntax error. Write parser error messages to
 * stderr (and diary file if necc.). Also force machine and scanner
 * to reset.
 * ************************************************************** */
void
ryyerror(s, yychar)
     char *s;
     int yychar;
{
  char  *loc, *diag_str;
  int i, j;

  /* Print out error message */  
  if(write_diary)
    {
      if( strcmp("stdin", curr_file_name) )
	fprintf(diary_file_ptr, "%s on line: %d, file: %s\n",
		s, lineno, curr_file_name);
      else
	fprintf(diary_file_ptr, "%s\n", s);
      
      if(strrchr(line_contents, '\n') == 0)
	fprintf(diary_file_ptr, "%s\n", line_contents);
      else
	fprintf(diary_file_ptr, "%s", line_contents);
    }
  if( strcmp("stdin", curr_file_name) )
    fprintf(stderr, "%s on line: %d, file: %s\n",
	    s, lineno, curr_file_name);
  else
    fprintf(stderr, "%s\n", s );
    
  /* Now try and print out offending tokens */
  if(strrchr(line_contents, '\n') == 0)
    fprintf(stderr, "%s\n", line_contents);
  else
    fprintf(stderr, "%s", line_contents);

  /* find next token/char on line */
  if((loc = strrchr(line_contents, yychar)) == 0)
    i = strlen(line_contents);
  else
    i = loc - line_contents;

  /* Create a string of blanks with a '^' where the 
     parse error is. Check value of loc, error may be at
     beginning of string (loc = 0) */
  if(i == 0)
    {
      diag_str = (char *) MALLOC(2*sizeof(char));
      diag_str[0] = '^';
      diag_str[1] = '\0';
    }
  else
    {
      diag_str = (char *) MALLOC((i+1)*sizeof(char));
      for(j = 0; j < i-1; j++)
	diag_str[j] = ' ';
      diag_str[j] = '^';
      diag_str[j+1] = '\0';
    }
    
  if(write_diary)
    {
      fprintf(diary_file_ptr, "%s\n", diag_str);
      fflush(diary_file_ptr);
    }
  fprintf(stderr, "%s\n", diag_str);
  fflush(stderr);

  FREE(diag_str);

  /* Reset the loop detection flag */
  looping = 0;

  /*
   * Flush the input line since the rest is probably
   * Uninteligble.
   */

  flush_line = 1;

  /* Reset the scanner, in case we error'ed during a load() */
  /* new_file(0); */

  /* Most likely there are some bad op-codes in the 
     machine... reset */
  initcode();

  /* Reset the prompt, in case we error'ed in a loop */
  prompt = 0;

  /* 
   * Handle some clean-up if we were in the midst of 
   * parsing a function, then longjmp back to the prompt.
   */

  if (scope == LOCAL)
  {
    scope = GLOBAL;
    if(arg_sym_tab != 0) 
      list_Destroy(arg_sym_tab);
    if(local_sym_tab != 0)
      list_Destroy(local_sym_tab);
  }
  longjmp( jmp[dec_buff()], 1 );
}
