/* 
 * dissassem.c
 * Print human readable version of op-codes for inspection
 */

/*  This file is a part of RLaB ("Our"-LaB)
    Copyright (C) 1992, 1994  Ian R. Searle

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    See the file ./COPYING
 ***********************************************************************/

#include "rlab.h"
#include "code.h"
#include "bltin.h"
#include "r_string.h"
#include "print.h"
#include "util.h"

#include <stdio.h>

static Inst *pc;		/* Program counter (ptr) */
static int lineno;

/* **************************************************************
 * Functions to print various forms of op-codes.
 * ************************************************************** */
void
print_op (p, s)
     Inst *p;
     char *s;
{
  fprintf (stderr, " %3d: %s\n", lineno++, s);
}

void
print_code_ptr (p)
     Inst *p;
{
  /* pointers */
  fprintf (stderr, " %3d: %p\n", lineno++, (VPTR) (*p).ptr);
}
void
print_code_string (p)
     Inst *p;
{
  /* pointers */
  fprintf (stderr, " %3d: %s\n", lineno++, (char *) ((*p).ptr));
}

void
print_code_int (p)
     Inst *p;
{
  /* print ints */
  fprintf (stderr, " %3d: %d\n", lineno++, (*p).op_code);
}

void
print_code_dval (p)
     Inst *p;
{
  /* doubles */
  fprintf (stderr, " %3d: %g\n", lineno++, (*p).d_val);
}

void
print_code_var (p)
     Inst *p;
{
  /* print variables */
  if ( ((ListNode *) ((*p).ptr))->key != 0 )
    fprintf (stderr, " %3d: %s\n", lineno++, ((ListNode *) ((*p).ptr))->key);
}

void
diss_assemble (p, pstop)
     Inst *p;
     int pstop;
{
  int i;

  lineno = 1;
  pc = p;
  for (i = 0; i < pstop; i++)
  {
    switch ((*pc).op_code)
    {
    case OP_PUSH_VAR:
      print_op (pc++, "push var");
      print_code_var (pc++);
      i++;
      break;

    case OP_PUSH_ARG:
      print_op (pc++, "push arg var");
      print_code_int (pc++);
      i++;
      break;

    case OP_PUSH_LOCAL_VAR:
      print_op (pc++, "push local var");
      print_code_int (pc++);
      i++;
      break;

    case OP_ADD:
      print_op (pc++, "add");
      break;

    case OP_SUB:
      print_op (pc++, "sub");
      break;

    case OP_MUL:
      print_op (pc++, "multiply");
      break;

    case OP_DIV:
      print_op (pc++, "right divide");
      break;

    case OP_LDIV:
      print_op (pc++, "left divide");
      break;

    case OP_NEGATE:
      print_op (pc++, "negate");
      break;

    case OP_POWER:
      print_op (pc++, "power");
      break;

    case OP_ASSIGN:
      print_op (pc++, "assign");
      break;

    case OP_FOR:
      print_op (pc++, "for loop");
      print_code_int (pc++);
      i++;
      print_code_int (pc++);
      i++;
      break;

    case OP_EL_MUL:
      print_op (pc++, "matrix-el-multiply");
      break;

    case OP_EL_DIV:
      print_op (pc++, "matrix-el-r-divide");
      break;

    case OP_EL_LDIV:
      print_op (pc++, "matrix-el-l-divide");
      break;

    case OP_EL_POWER:
      print_op (pc++, "matrix-el-power");
      break;

    case OP_PUSH_CONSTANT:
      print_op (pc++, "push constant");
      print_code_dval (pc++);
      i++;
      break;

    case OP_PUSH_iCONSTANT:
      print_op (pc++, "push imaginary constant");
      print_code_dval (pc++);
      i++;
      break;

    case OP_PRINT:
      print_op (pc++, "print");
      break;

    case OP_GT:
      print_op (pc++, "gt");
      break;

    case OP_LT:
      print_op (pc++, "lt");
      break;

    case OP_EQ:
      print_op (pc++, "eq");
      break;

    case OP_GE:
      print_op (pc++, "ge");
      break;

    case OP_LE:
      print_op (pc++, "le");
      break;

    case OP_NE:
      print_op (pc++, "ne");
      break;

    case OP_AND:
      print_op (pc++, "and");
      break;

    case OP_OR:
      print_op (pc++, "or");
      break;

    case OP_NOT:
      print_op (pc++, "not");
      break;

    case OP_IF:
      print_op (pc++, "ifcode");
      print_code_int (pc++);
      i++;
      print_code_int (pc++);
      i++;
      print_code_int (pc++);
      i++;
      break;

    case OP_WHILE:
      print_op (pc++, "whilecode");
      print_code_int (pc++);
      i++;
      print_code_int (pc++);
      i++;
      break;

    case OP_SWAP:
      print_op (pc++, "swap");
      break;

    case OP_INC:
      print_op (pc++, "inc");
      break;

    case OP_DEC:
      print_op (pc++, "dec");
      break;

    case OP_POP:
      print_op (pc++, "pop");
      break;

    case OP_POP_CLEAN:
      print_op (pc++, "pop clean");
      break;

    case OP_VECTOR_CREATE:
      print_op (pc++, "vector_create");
      print_code_int (pc++);
      i++;
      break;

    case OP_VEC_APPEND:
      print_op (pc++, "vector append");
      break;

    case OP_MATRIX_VEC_SUB:
      print_op (pc++, "matrix-vector sub");
      break;

    case OP_MATRIX_VEC_ASSIGN:
      print_op (pc++, "matrix-vector assign");
      break;

    case OP_MATRIX_CREATE:
      print_op (pc++, "matrix create");
      break;

    case OP_MATRIX_APPEND:
      print_op (pc++, "stack matrix");
      break;

    case OP_MATRIX_ASSIGN:
      print_op (pc++, "matrix assign");
      print_code_int (pc++);
      i++;
      break;

    case OP_MATRIX_SUB:
      print_op (pc++, "sub matrix");
      print_code_int (pc++);
      i++;
      break;

    case OP_LIST_CREATE:
      print_op (pc++, "create list");
      print_code_int (pc++);
      i++;
      break;

    case OP_LIST_MEMB:
      print_op (pc++, "list member");
      if (((*pc).op_code) == 1)
      {
	print_code_int (pc++);
	i++;
      }
      else
      {
	print_code_int (pc++);
	i++;
	print_code_string (pc++);
	i++;
      }
      break;

    case OP_LIST_ASSIGN:
      print_op (pc++, "list assign");
      if (((*pc).op_code) == 1)
      {
	print_code_int (pc++);
	i++;
      }
      else
      {
	print_code_int (pc++);
	i++;
	print_code_dval (pc++);
	i++;
      }
      break;

    case OP_LIST_EL_CREATE:
      print_op (pc++, "list-el-create");
      print_code_ptr (pc++);
      i++;
      break;

    case OP_FUNCTION_CALL:
      print_op (pc++, "function call");
      print_code_int (pc++);
      i++;
      break;

    case OP_FUNCTION_CALL_1:
      print_op (pc++, "function call 1");
      print_code_int (pc++);
      i++;
      break;

    case OP_FUNCTION_CALL_SELF:
      print_op (pc++, "function call-self");
      print_code_int (pc++);
      i++;
      break;

    case OP_FUNCTION_RETURN:
      print_op (pc++, "function return");
      break;

    case OP_DEF_FUNC_RET:
      print_op (pc++, "default function return");
      break;

    case OP_TRANSPOSE:
      print_op (pc++, "matrix transpose");
      break;

    case OP_PUSH_STRING:
      print_op (pc++, "push_string");
      print_code_string (pc++);
      i++;
      break;

    case OP_BREAK:
      print_op (pc++, "break");
      break;

    case OP_CONTINUE:
      print_op (pc++, "continue");
      break;

    case OP_QUIT:
      print_op (pc++, "quit");
      break;

    case OP_LINE_NO:
      fprintf (stderr, " %3d: %s", lineno++, "line # ");
      pc++;
      fprintf (stderr, " %d\n", (*pc++).op_code);
      i++;
      break;

    case OP_FILE_NAME:
      fprintf (stderr, " %3d: %s", lineno++, "file: ");
      pc++;
      fprintf (stderr, " %s\n", (char *) (*pc++).ptr);
      i++;
      break;

    case OP_JMP:
      print_op (pc++, "jmp");
      print_code_int (pc++);
      i++;
      break;

    case OP_EMPTY_MATRIX_CREATE:
      print_op (pc++, "create_empty_matrix");
      break;

    case OP_MATRIX_COL:
      print_op (pc++, "matrix_reshape_col");
      break;

    case OP_EL_TRANSPOSE:
      print_op (pc++, "matrix el-transpose");
      break;

    case OP_RFILE:
      print_op (pc++, "rfile command");
      break;

    case OP_RFILE_NAME:
      print_op (pc++, "rfile-name command");
      print_code_string (pc++);
      i++;
      break;

    case OP_HELP:
      print_op (pc++, "help command");
      break;

    case OP_HELP_NAME:
      print_op (pc++, "help-name command");
      print_code_string (pc++);
      i++;
      break;

    case OP_PUSH_UNDEF:
      print_op (pc++, "push-UNDEF");
      break;

    case OP_EL_ADD:
      print_op (pc++, "el-add");
      break;

    case OP_EL_SUB:
      print_op (pc++, "el-sub");
      break;
      
    case STOP:
      print_op (pc++, "stop");
      break;

    case OP_SAVE_EVAL:
      print_op (pc++, "save_eval");
      break;

    default:
      fprintf (stderr, "Invalid op-code: %d\n", (*pc).op_code);
      fflush (stderr);
      error_1 ("dissassemble:", "Invalid op-code");
      break;
    }
  }
}

/* **************************************************************
 * Function Variable Scope analyzer.
 * Go through a function's op-codes, looking for variable references.
 * When one is found, print out the scope of the variable, and
 * it's name.
 * ************************************************************** */

#include "function.h"
extern ListNode *static_lookup _PROTO((char *file_name, char *name));

void
FVScope (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *fname, *fstring, *name;
  int i, lineno, narg, ncode, nlocal, off;
  Function *f;
  List *localv, *argv;
  ListNode *ltmp, *FUNC, *FN;
  FILE *fptr;
  Inst *pc;

  fstring = 0;   /* Initialize */
  FUNC = 0; FN = 0;

  if (n_args > 2)
    error_1 ("fvscope: at most 2 arguments allowed", 0);
  FUNC = bltin_get_ufunc ("fvscope", d_arg, 1);

  fptr = stdout;      /* Default */
  if (n_args == 2)
  {
    /*
     * Extract the string ptr to the output file
     * or process.
     */

    FN = bltin_get_string ("fvscope", d_arg, 2);
    fstring = string_GetString (e_data (FN));

    if ((fptr = get_file_ds (fstring, "w", 0)) == 0)
    {
      warning_1 (fstring, "cannot open for write");
      remove_tmp_destroy (FUNC);
      remove_tmp_destroy (FN);
      *return_ptr = (VPTR) scalar_Create (0.0);
      return;
    }
  }
  
  /* 
   * Set up function searching defaults.
   */

  f = (Function *) e_data (FUNC);	/* Function ptr */
  pc = function_GetCodePtr (f);	        /* Code List ptr */
  ncode = function_GetCodeSize (f);	/* # of instructions */
  argv = f->args;		        /* Arg List ptr */
  narg = f->n_args;		        /* # of arguments */
  localv = f->local;		        /* Local var list ptr */
  nlocal = f->n_local;		        /* # of local vars */
  lineno = 1;

  fname = pc[1].ptr;     		/* File name */

  fprintf (fptr, "\tFunction Variable SCOPE analysis for : %s\n",
	   e_name (FUNC));
  fprintf (fptr, "\tFilename: %s\n\n", fname);
  fprintf (fptr, "\tline\tGLOBAL\t\t\tARG\t\tLOCAL\n\n");

  for (i = 0; i < ncode; i++)
  {
    switch ((*pc++).op_code)
    {
    case OP_PUSH_VAR:
      name = ((ListNode *) ((*pc).ptr))->key;
      i++;
      if (static_lookup (fname, name))
	fprintf (fptr, "\t%4i\tStatic-Var: %s\n", lineno, name);
      else
      {
	if (((ListNode *) ((*pc).ptr))->type == U_FUNCTION ||
	    ((ListNode *) ((*pc).ptr))->type == BLTIN)
	{
	  fprintf (fptr, "\t%4i\tGlobal-Var: %s\n", lineno, name);
	}
	else
	{
	  fprintf (fptr, "\t%4i\tGlobal-Var: %s*\n", lineno, name);
	}
      }
      pc++;
      break;

    case OP_PUSH_ARG:
      off = (*pc++).op_code;
      i++;
      ltmp = list_GetNodeByPos (argv, (narg - off) + 1);
      name = ltmp->key;
      fprintf (fptr, "\t%4i\t\t\t\tArg-Var: %s\n", lineno, name);
      break;

    case OP_PUSH_LOCAL_VAR:
      off = (*pc++).op_code;
      i++;
      ltmp = list_GetNodeByPos (localv, (nlocal - off) + 1);
      name = ltmp->key;
      fprintf (fptr, "\t%4i\t\t\t\t\t\tLocal-Var: %s\n", lineno, name);
      break;

    case OP_ADD:
    case OP_SUB:
    case OP_MUL:
    case OP_DIV:
    case OP_LDIV:
    case OP_NEGATE:
    case OP_POWER:
    case OP_ASSIGN:
      break;

    case OP_FOR:
      pc += 2;
      i += 2;
      break;

    case OP_EL_MUL:
    case OP_EL_DIV:
    case OP_EL_LDIV:
    case OP_EL_POWER:
      break;

    case OP_PUSH_CONSTANT:
    case OP_PUSH_iCONSTANT:
      pc++;
      i++;
      break;

    case OP_PRINT:
    case OP_GT:
    case OP_LT:
    case OP_EQ:
    case OP_GE:
    case OP_LE:
    case OP_NE:
    case OP_AND:
    case OP_OR:
    case OP_NOT:
      break;

    case OP_IF:
      pc += 3;
      i += 3;
      break;

    case OP_WHILE:
      pc += 2;
      i += 2;
      break;

    case OP_SWAP:
    case OP_INC:
    case OP_DEC:
    case OP_POP:
    case OP_POP_CLEAN:
      break;

    case OP_VECTOR_CREATE:
      pc++;
      i++;
      break;

    case OP_VEC_APPEND:
    case OP_MATRIX_VEC_SUB:
    case OP_MATRIX_VEC_ASSIGN:
    case OP_MATRIX_CREATE:
    case OP_MATRIX_APPEND:
      break;

    case OP_MATRIX_ASSIGN:
    case OP_MATRIX_SUB:
    case OP_LIST_CREATE:
      pc++;
      i++;
      break;

    case OP_LIST_MEMB:
    case OP_LIST_ASSIGN:
      if (((*pc).op_code) == 1)
      {
	pc++;
	i++;
      }
      else
      {
	pc += 2;
	i += 2;
      }
      break;

    case OP_LIST_EL_CREATE:
    case OP_FUNCTION_CALL:
    case OP_FUNCTION_CALL_1:
    case OP_FUNCTION_CALL_SELF:
      pc++;
      i++;
      break;

    case OP_FUNCTION_RETURN:
      break;

    case OP_DEF_FUNC_RET:
      i = ncode;
      break;

    case OP_TRANSPOSE:
      break;

    case OP_PUSH_STRING:
      pc++;
      i++;
      break;

    case OP_BREAK:
    case OP_CONTINUE:
    case OP_QUIT:
      break;

    case OP_LINE_NO:
      lineno = (*pc++).op_code - 100;
      i++;
      break;

    case OP_FILE_NAME:
    case OP_JMP:
      pc++;
      i++;
      break;

    case OP_EMPTY_MATRIX_CREATE:
    case OP_MATRIX_COL:
    case OP_EL_TRANSPOSE:
    case OP_RFILE:
      break;

    case OP_RFILE_NAME:
      pc++;
      i++;
      break;

    case OP_HELP:
      break;

    case OP_HELP_NAME:
      pc++;
      i++;
      break;

    case OP_PUSH_UNDEF:      
    case OP_EL_ADD:
    case OP_EL_SUB:
    case STOP:
      break;

    default:
      error_1 ("fvscope: error in function codes", 0);
      break;
    }
  }
  *return_ptr = (VPTR) scalar_Create (1.0);
  remove_tmp_destroy (FUNC);
  if (FN)
    remove_tmp_destroy (FN);
}
