/*
 * fi_1.c
 * Fortran Interfaces
 * Contains most of the computational interfaces to LAPACK.
 */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1993, 1994  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "mem.h"
#include "symbol.h"
#include "scalar.h"
#include "matrix.h"
#include "matop2.h"
#include "matop3.h"
#include "util.h"

#include <math.h>

#include "fi.h"
#include "lp.h"
#include "bl.h"

/*
 * Define our own abs(). We need to do this, cause we need
 * and abs() that does double, as well as int.
 */
#define rabs(x) ((x) >= 0 ? (x) : -(x))

/*
 * Create Infs, and NaNs
 */
#include "mathl.h"
static const unsigned char __nan[8] = r__nan_bytes;
#define	R_NAN	(*(const double *) __nan)

static const unsigned char __inf_val[8] = r__inf_val_bytes;
#define	R_INF   (*(const double *) __inf_val)

double matrix_Rcond _PROTO ((Matrix * m));
double matrix_Norm _PROTO ((Matrix * m, char *type));

/*
 * Pointers for temporary accounting.
 * Note that we CANNOT use these in a function, if that
 * function calls another function(s) that uses one of
 * the tmp? pointers.
 */
static ListNode *tmp1, *tmp2, *tmp3, *tmp4, *tmp5, *tmp6;
static ListNode *tmp7, *tmp8, *tmp9;

/* **************************************************************
 * Compute the inverse of the given matrix.
 * Return the result in a new matrix.
 * ************************************************************** */

static Matrix *matrix_inverse_real _PROTO ((Matrix * m));
static Matrix *matrix_inverse_complex _PROTO ((Matrix * m));

Matrix *
matrix_Inverse (m)
     Matrix *m;
{
  ASSERT (m);
  {
    Matrix *minv;

    matrix_screen_string (m);
    matrix_Detect_Inf (m);
    matrix_Detect_Nan (m);
    
    if (MTYPE (m) == REAL)
      minv = matrix_inverse_real (m);
    else
      minv = matrix_inverse_complex (m);
    return (minv);
  }
}

static Matrix *
matrix_inverse_real (m)
     Matrix *m;
{
  ASSERT (m);
  {
    double anorm, rcond;
    F_INT info, mm, n, lda, *ipiv, *iwork, lwork, norm;
    ListNode *t1, *t2, *t3, *t5;
    Matrix *inv, *work;

    lda = mm = MNR (m);
    n = MNC (m);
    lwork = mm;
    norm = (F_INT) '1';

    if (mm != n)
      error_1 (matrix_GetName (m), "matrix must be square for inv()");

    t1 = install_tmp (D_VOID, ipiv = (F_INT *) MALLOC (sizeof (F_INT) * mm),
		      free);
    t2 = install_tmp (D_VOID, iwork = (F_INT *) MALLOC (sizeof (F_INT) * n),
		      free);
    t3 = install_tmp (MATRIX, work = matrix_Create (1, 4 * lwork),
		      matrix_Destroy);
    t5 = install_tmp (MATRIX, inv = matrix_Copy (m),
		      matrix_Destroy);

    signal (SIGINT, intcatch);
    RGETRF (&mm, &n, MDPTRr (inv), &lda, ipiv, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DGETRF", 0);
    if ((int) info > 0)
      error_1 (matrix_GetName (m), "matrix is singular");

    /*
     * Check input for ill-conditioning.
     */

    anorm = matrix_Norm (m, "1");
    signal (SIGINT, intcatch);
    RGECON (&norm, &n, MDPTRr (inv), &lda, &anorm, &rcond, MDPTRr (work),
	    iwork, &info);
    signal (SIGINT, intcatch_wait);
    if (rcond <= DBL_EPSILON)
      warning_1 ("WARNING, ill-conditioned input", 0);

    signal (SIGINT, intcatch);
    RGETRI (&mm, MDPTRr (inv), &lda, ipiv, MDPTRr (work), &lwork, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DGETRI", 0);
    if ((int) info > 0)
      error_1 (matrix_GetName (m), "matrix is singular");

    /* Now destroy temporary work arrays */
    remove_tmp_destroy (t1);
    remove_tmp_destroy (t2);
    remove_tmp_destroy (t3);

    /* Remove the return value from the tmplist. */
    remove_tmp (t5);

    return (inv);
  }
}

static Matrix *
matrix_inverse_complex (m)
     Matrix *m;
{
  ASSERT (m);
  {
    double anorm, rcond, *rwork;
    F_INT info, mm, n, lda, *ipiv, lwork, norm;
    ListNode *t1, *t2, *t3, *t5;
    Matrix *inv, *work;

    lda = mm = MNR (m);
    n = MNC (m);
    lwork = mm;

    norm = (F_INT) '1';

    if (mm != n)
      error_1 (matrix_GetName (m), "matrix must be square for inv()");

    t1 = install_tmp (D_VOID, ipiv = (F_INT *) MALLOC (sizeof (F_INT) * mm),
		      free);
    t2 = install_tmp (D_VOID, rwork = (double *) MALLOC (sizeof (double) * 2 * n),
		      free);
    t3 = install_tmp (MATRIX, work = matrix_CreateC (1, 2 * lwork),
		      matrix_Destroy);
    t5 = install_tmp (MATRIX, inv = matrix_Copy (m),
		      matrix_Destroy);

    signal (SIGINT, intcatch);
    XGETRF (&mm, &n, MDPTRc (inv), &lda, ipiv, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK ZGETRF", 0);
    if ((int) info > 0)
      error_1 (matrix_GetName (m), "matrix is singular");

    /*
     * Check input for ill-conditioning.
     */

    anorm = matrix_Norm (m, "1");
    signal (SIGINT, intcatch);
    XGECON (&norm, &n, MDPTRc (inv), &lda, &anorm, &rcond, MDPTRc (work),
	    rwork, &info);
    signal (SIGINT, intcatch_wait);
    if (rcond <= DBL_EPSILON)
      warning_1 ("WARNING, ill-conditioned input", 0);

    signal (SIGINT, intcatch);
    XGETRI (&mm, MDPTRc (inv), &lda, ipiv, MDPTRc (work), &lwork, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK ZGETRI", (char *) 0);
    if ((int) info > 0)
      error_1 (matrix_GetName (m), "matrix is singular");

    /* Now destroy temporary work arrays */
    remove_tmp_destroy (t1);
    remove_tmp_destroy (t2);
    remove_tmp_destroy (t3);

    /* Remove the return value from the tmplist. */
    remove_tmp (t5);

    return (inv);
  }
}

/* **************************************************************
 * Solve the system of equations [a]{x} = {b}
 * ************************************************************** */

extern int matrix_is_symm _PROTO ((Matrix *m));
Matrix * matrix_solve_ge _PROTO ((Matrix *a, Matrix *b));
Matrix * matrix_solve_sy _PROTO ((Matrix *a, Matrix *b));

Matrix *
solve_eq (a, b)
     Matrix *a, *b;
{
  ASSERT (a);
  ASSERT (b);
  {
    Matrix *m;
    if (matrix_is_symm (a))
      m = matrix_solve_sy (a, b);
    else
      m = matrix_solve_ge (a, b);
    
    return (m);
  }
}
 
Matrix *
matrix_solve_ge (a, b)
     Matrix *a, *b;
{
  F_INT info, lda, ldb, mm, n, nrhs, *ipiv;
  F_INT trans, lwork, norm, *iwork;
  double anorm, rcond, *rwork;
  Matrix *A, *B, *work;
  
  matrix_screen_string (a);
  matrix_Detect_Inf (a);
  matrix_Detect_Nan (a);
  
  matrix_screen_string (b);
  matrix_Detect_Inf (b);
  matrix_Detect_Nan (b);
  
  lda = mm = MNR (a);
  n = MNC (a);
  lwork = mm;
  trans = (F_INT) 'N';
  norm = (F_INT) '1';
  
  if (mm != n)
    error_1 (matrix_GetName (a), "matrix must be square for solve()");
  
  if (mm != MNR (b))
    error_2 (matrix_GetName (a), matrix_GetName (b),
	     "RHS row dim. must match LHS row dim.");
  
  tmp1 = install_tmp (D_VOID, ipiv = (F_INT *) MALLOC (sizeof (F_INT) * mm),
		      free);
  tmp2 = install_tmp (MATRIX, A = matrix_Copy (a),
		      matrix_Destroy);
  tmp3 = install_tmp (MATRIX, B = matrix_Copy (b),
		      matrix_Destroy);
  tmp4 = install_tmp (MATRIX, work = matrix_Create (1, 4 * lwork),
		      matrix_Destroy);
  
  if (MTYPE (a) == REAL && MTYPE (b) == REAL)
  {
    signal (SIGINT, intcatch);
    RGETRF (&mm, &n, MDPTRr (A), &lda, ipiv, &info);
    signal (SIGINT, intcatch_wait);
    
    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DGETRF", 0);
    if ((int) info > 0)
      error_1 (matrix_GetName (a), "matrix is singular");
    
    /*
     * Check input for ill-conditioning.
     */
    
    tmp5 = install_tmp (D_VOID, iwork = (F_INT *) MALLOC (sizeof (F_INT) * n),
			free);
    
    anorm = matrix_Norm (a, "1");
    signal (SIGINT, intcatch);
    RGECON (&norm, &n, MDPTRr (A), &lda, &anorm, &rcond, MDPTRr (work),
	    iwork, &info);
    signal (SIGINT, intcatch_wait);
    if (rcond <= DBL_EPSILON)
      warning_1 ("WARNING, ill-conditioned input", 0);
  }
  else
  {
    /* Force both A and B to be COMPLEX */
    if (MTYPE (A) != COMPLEX)
    {
      remove_tmp_destroy (tmp2);
      tmp2 = install_tmp (MATRIX, A = matrix_copy_complex (a),
			  matrix_Destroy);
    }
    if (MTYPE (B) != COMPLEX)
    {
      remove_tmp_destroy (tmp3);
      tmp3 = install_tmp (MATRIX, B = matrix_copy_complex (b),
			  matrix_Destroy);
    }
    
    signal (SIGINT, intcatch);
    XGETRF (&mm, &n, MDPTRc (A), &lda, ipiv, &info);
    signal (SIGINT, intcatch_wait);
    
    if ((int) info < 0)
      error_1 ("bad argument to LAPACK ZGETRF", 0);
    if ((int) info > 0)
      error_1 (matrix_GetName (a), "matrix is singular");
    
    /*
     * Check input for ill-conditioning.
     */
    
    tmp5 = install_tmp (D_VOID, 
			rwork = (double *) MALLOC (sizeof(double)*2*n),
			free);
    
    anorm = matrix_Norm (a, "1");
    signal (SIGINT, intcatch);
    XGECON (&norm, &n, MDPTRr (A), &lda, &anorm, &rcond, MDPTRr (work),
	    rwork, &info);
    signal (SIGINT, intcatch_wait);
    if (rcond <= DBL_EPSILON)
      warning_1 ("WARNING, ill-conditioned input", 0);
  }
  
  nrhs = MNC (b);
  ldb = MNR (b);
  
  if (MTYPE (a) == REAL && MTYPE (b) == REAL)
  {
    signal (SIGINT, intcatch);
    RGETRS (&trans, &n, &nrhs, MDPTRr (A), &lda, ipiv, MDPTRr (B), &ldb,
	    &info);
    signal (SIGINT, intcatch_wait);
  }
  else
  {
    signal (SIGINT, intcatch);
    XGETRS (&trans, &n, &nrhs, MDPTRc (A), &lda, ipiv, MDPTRc (B), &ldb,
	    &info);
    signal (SIGINT, intcatch_wait);
  }
  
  if ((int) info < 0)
    error_1 ("Bad argument(s) to LAPACK DGETRS", 0);
  
  remove_tmp_destroy (tmp1);
  remove_tmp_destroy (tmp2);
  remove_tmp (tmp3);
  remove_tmp_destroy (tmp4);
  remove_tmp_destroy (tmp5);
  
  return (B);
}

Matrix *
matrix_solve_sy (a, b)
     Matrix *a, *b;
{
  F_INT info, lda, ldb, mm, n, nrhs, *ipiv;
  F_INT trans, lwork, norm, *iwork;
  F_INT uplo;
  double anorm, rcond;
  Matrix *A, *B, *work;
  
  matrix_screen_string (a);
  matrix_Detect_Inf (a);
  matrix_Detect_Nan (a);
  
  matrix_screen_string (b);
  matrix_Detect_Inf (b);
  matrix_Detect_Nan (b);
  
  n = lda = mm = MNR (a);
  uplo = (F_INT) 'L';
  trans = (F_INT) 'N';
  norm = (F_INT) '1';
  
  /*
   *Try and pick a good NB, without ILAENV.
   */

  if (n < 100)
    lwork = n;
  else
    lwork = 64 * n;
  
  if (mm != n)
    error_1 (matrix_GetName (a), "matrix must be square for solve()");
  
  if (mm != MNR (b))
    error_2 (matrix_GetName (a), matrix_GetName (b),
	     "RHS row dim. must match LHS row dim.");
  
  tmp1 = install_tmp (D_VOID, ipiv = (F_INT *) MALLOC (sizeof (F_INT) * n),
		      free);
  tmp2 = install_tmp (MATRIX, A = matrix_Copy (a),
		      matrix_Destroy);
  tmp3 = install_tmp (MATRIX, B = matrix_Copy (b),
		      matrix_Destroy);
  if (MTYPE (a) == REAL && MTYPE (b) == REAL)
  {
    tmp4 = install_tmp (MATRIX, work = matrix_Create (1, lwork),
			matrix_Destroy);
    signal (SIGINT, intcatch);
    RSYTRF (&uplo, &n, MDPTRr (A), &lda, ipiv, MDPTRr (work), &lwork, &info);
    signal (SIGINT, intcatch_wait);
    
    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DSYTRF", 0);
    if ((int) info > 0)
      error_1 (matrix_GetName (a), "matrix is singular");
    
    /*
     * Check input for ill-conditioning.
     */
    
    remove_tmp_destroy (tmp4);
    tmp4 = install_tmp (MATRIX, work = matrix_Create (1, 2*n),
                        matrix_Destroy);
    tmp5 = install_tmp (D_VOID, iwork = (F_INT *) MALLOC (sizeof (F_INT) * n),
			free);
    
    anorm = matrix_Norm (a, "1");
    signal (SIGINT, intcatch);
    RSYCON (&uplo, &n, MDPTRr (A), &lda, ipiv, &anorm, &rcond, 
            MDPTRr (work), iwork, &info);
    signal (SIGINT, intcatch_wait);

    if (rcond <= DBL_EPSILON)
      warning_1 ("WARNING, ill-conditioned input", 0);

    remove_tmp_destroy (tmp5);
  }
  else
  {
    /* Force both A and B to be COMPLEX */
    if (MTYPE (A) != COMPLEX)
    {
      remove_tmp_destroy (tmp2);
      tmp2 = install_tmp (MATRIX, A = matrix_copy_complex (a),
			  matrix_Destroy);
    }
    if (MTYPE (B) != COMPLEX)
    {
      remove_tmp_destroy (tmp3);
      tmp3 = install_tmp (MATRIX, B = matrix_copy_complex (b),
			  matrix_Destroy);
    }
    
    tmp4 = install_tmp (MATRIX, work = matrix_CreateC (1, lwork),
			matrix_Destroy);
    signal (SIGINT, intcatch);
    XHETRF (&uplo, &n, MDPTRc (A), &lda, ipiv, MDPTRc (work), &lwork, &info);
    signal (SIGINT, intcatch_wait);
    
    if ((int) info < 0)
      error_1 ("bad argument to LAPACK ZHETRF", 0);
    if ((int) info > 0)
      error_1 (matrix_GetName (a), "matrix is singular");
    
    /*
     * Check input for ill-conditioning.
     */
    
    remove_tmp_destroy (tmp4);
    tmp4 = install_tmp (MATRIX, work = matrix_CreateC (1, 2*n),
			matrix_Destroy);
    
    anorm = matrix_Norm (a, "1");
    signal (SIGINT, intcatch);
    XHECON (&uplo, &n, MDPTRr (A), &lda, ipiv, &anorm, &rcond, 
            MDPTRc (work), &info);
    signal (SIGINT, intcatch_wait);

    if (rcond <= DBL_EPSILON)
      warning_1 ("WARNING, ill-conditioned input", 0);
  }
  
  nrhs = MNC (b);
  ldb = MNR (b);
  
  if (MTYPE (a) == REAL && MTYPE (b) == REAL)
  {
    signal (SIGINT, intcatch);
    RSYTRS (&uplo, &n, &nrhs, MDPTRr (A), &lda, ipiv, MDPTRr (B), &ldb,
	    &info);
    signal (SIGINT, intcatch_wait);
  }
  else
  {
    signal (SIGINT, intcatch);
    XHETRS (&uplo, &n, &nrhs, MDPTRc (A), &lda, ipiv, MDPTRc (B), &ldb,
	    &info);
    signal (SIGINT, intcatch_wait);
  }
  
  if ((int) info < 0)
    error_1 ("Bad argument(s) to LAPACK DSYTRS or ZHETRS", 0);
  
  remove_tmp_destroy (tmp1);
  remove_tmp_destroy (tmp2);
  remove_tmp (tmp3);
  remove_tmp_destroy (tmp4);
  
  return (B);
}


static void matrix_Svd_Real _PROTO ((Matrix * M, Matrix ** rsv,
				     Matrix ** lsv, Matrix ** sigma, int flag));
static void matrix_Svd_Complex _PROTO ((Matrix * M, Matrix ** rsv,
					Matrix ** lsv, Matrix ** sigma, int flag));

double
matrix_Norm (m, type)
     Matrix *m;
     char *type;
{
  F_DOUBLE norm, *work;
  F_INT lda, nrow, ncol;
  F_INT l, itype;
  ListNode *t1;
  Matrix *lsv, *rsv, *sigma;

  norm = 0.0;   /* Initialize */
  matrix_Detect_Inf (m);
  matrix_Detect_Nan (m);

  matrix_screen_string (m);

  /* Argument error checking */
  if (!strcmp (type, "1"));
  else if (!strcmp (type, "M"));
  else if (!strcmp (type, "m"));
  else if (!strcmp (type, "O"));
  else if (!strcmp (type, "o"));
  else if (!strcmp (type, "I"));
  else if (!strcmp (type, "i"));
  else if (!strcmp (type, "F"));
  else if (!strcmp (type, "f"));
  else if (!strcmp (type, "E"));
  else if (!strcmp (type, "e"));
  else if (!strcmp (type, "2"));
  else
    error_1 ("incorrect STRING specifier for norm()", (char *) 0);

  /* Need the string length for FORTRAN */
  l = strlen (type);

  nrow = (F_INT) MNR (m);
  ncol = (F_INT) MNC (m);
  lda = nrow;

  if (strcmp (type, "2"))
  {
    t1 = install_tmp (D_VOID,
		      work=(double *) MALLOC ((size_t) (nrow*sizeof(double))),
		      free);
    itype = (F_INT) (type[0] - '0' + '0');
    if (MTYPE (m) == REAL)
    {
      signal (SIGINT, intcatch);
      norm = RLANGE (&itype, &nrow, &ncol, MDPTRr (m), &lda, work);
      signal (SIGINT, intcatch_wait);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      signal (SIGINT, intcatch);
      norm = XLANGE (&itype, &nrow, &ncol, MDPTRc (m), &lda, work);
      signal (SIGINT, intcatch_wait);
    }
    remove_tmp_destroy (t1);
  }
  else
  {
    /* Compute the matrix 2-norm */
    if (MTYPE (m) == REAL)
      matrix_Svd_Real (m, &rsv, &lsv, &sigma, 3);
    else if (MTYPE (m) == COMPLEX)
      matrix_Svd_Complex (m, &rsv, &lsv, &sigma, 3);

    /*
     * Return the largest singular value s[1]
     */
    norm = MAT (sigma, 1, 1);
    matrix_Destroy (rsv);
    matrix_Destroy (lsv);
    matrix_Destroy (sigma);
  }
  return (norm);
}

/* **************************************************************
 * Compute the recipricol of the condition number.
 * ************************************************************** */
double
matrix_Rcond (m)
     Matrix *m;
{
  ASSERT (m);
  {
    F_INT lda, n, info, *ipiv, *iwork, norm;
    double anorm, rcond, *rwork;
    ListNode *t1, *t2, *t3, *t4;
    Matrix *A, *work;

    t1 = t2 = t3 = t4 = 0;   /* Initialize */

    matrix_screen_string (m);
    matrix_Detect_Inf (m);
    matrix_Detect_Nan (m);

    lda = MNR (m);
    n = min (MNC (m), lda);

    norm = (F_INT) '1';

    if (MTYPE (m) == REAL)
    {
      t1 = install_tmp (MATRIX, work = matrix_Create (1, 4 * n),
			matrix_Destroy);
      t2 = install_tmp (D_VOID, iwork = (F_INT *) MALLOC (sizeof (F_INT) * n),
			free);
      t3 = install_tmp (D_VOID, ipiv = (F_INT *) MALLOC (sizeof (F_INT) * lda),
			free);
      t4 = install_tmp (MATRIX, A = matrix_Copy (m),
			matrix_Destroy);

      anorm = matrix_Norm (m, "1");

      signal (SIGINT, intcatch);
      RGETRF (&lda, &n, MDPTRr (A), &lda, ipiv, &info);

      RGECON (&norm, &n, MDPTRr (A), &lda, &anorm, &rcond, MDPTRr (work),
              iwork, &info);
      signal (SIGINT, intcatch_wait);

      if ((int) info < 0)
	error_1 ("illegal argument to LAPACK DGECON", 0);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      t1 = install_tmp (MATRIX, work = matrix_CreateC (1, 4 * n),
			matrix_Destroy);
      t2 = install_tmp (D_VOID, rwork=(double *) MALLOC (sizeof(double)*2*n),
			free);
      t3 = install_tmp (D_VOID, ipiv = (F_INT *) MALLOC (sizeof (F_INT) * lda),
			free);
      t4 = install_tmp (MATRIX, A = matrix_Copy (m),
			matrix_Destroy);

      anorm = matrix_Norm (m, "1");

      signal (SIGINT, intcatch);
      XGETRF (&lda, &n, MDPTRc (A), &lda, ipiv, &info);

      XGECON (&norm, &n, MDPTRc (A), &lda, &anorm, &rcond, MDPTRc (work),
              rwork, &info);
      signal (SIGINT, intcatch_wait);

      if ((int) info < 0)
	error_1 ("illegal argument to LAPACK ZGECON", 0);
    }

    remove_tmp_destroy (t1);
    remove_tmp_destroy (t2);
    remove_tmp_destroy (t3);
    remove_tmp_destroy (t4);

    return (rcond);
  }
}

/* **************************************************************
 * Compute the determinant of a general matrix.
 * ************************************************************** */

static double matrix_Det_Real _PROTO ((Matrix * m));
static Complex matrix_Det_Complex _PROTO ((Matrix * m));

Scalar *
matrix_Det (m)
     Matrix *m;
{
  ASSERT (m);
  {
    Scalar *new;

    matrix_screen_string (m);
    matrix_Detect_Inf (m);
    matrix_Detect_Nan (m);

    new = scalar_CreateC (0.0, 0.0);

    if (MTYPE (m) == REAL)
      SVALr (new) = matrix_Det_Real (m);
    else
      SVAL (new) = matrix_Det_Complex (m);
    return (new);
  }
}

static double
matrix_Det_Real (m)
     Matrix *m;
{
  ASSERT (m);
  {
    int i;
    F_INT info, mm, n, lda, *ipiv, one;
    double det[2], ten;
    ListNode *t1, *t2;
    Matrix *lu;

    lda = mm = MNR (m);
    n = MNC (m);
    one = 1;

    if (mm != n)
      error_1 (matrix_GetName (m), "matrix must be square for det()");

    t1 = install_tmp (D_VOID, ipiv = (F_INT *) MALLOC (sizeof (F_INT) * mm),
		      free);
    t2 = install_tmp (MATRIX, lu = matrix_Copy (m),
		      matrix_Destroy);

    signal (SIGINT, intcatch);
    RGETRF (&mm, &n, MDPTRr (lu), &lda, ipiv, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DGETRF", (char *) 0);
    if ((int) info > 0)
    {
      warning_1 (matrix_GetName (m), "matrix is singular");
      return (0.0);
    }

    /*
     * Now compute the determinant using the algorithm contained
     * in the LINPACK subroutines DGEDI, ZGEDI. I'm doing it this
     * way because the LAPACK does not provide a direct method for
     * computing the determinant?
     */

    det[0] = 1.;
    det[1] = 0.;
    ten = 10.;
    for (i = 1; i <= n; ++i)
    {
      if (ipiv[i - 1] != i)
      {
	det[0] = -det[0];
      }
      det[0] = MAT (lu, i, i) * det[0];
      /*        ...exit */
      if (det[0] == 0.)
      {
	goto L60;
      }
    L10:
      if (rabs (det[0]) >= 1.)
      {
	goto L20;
      }
      det[0] = ten * det[0];
      det[1] += -1.;
      goto L10;
    L20:
    L30:
      if (rabs (det[0]) < ten)
      {
	goto L40;
      }
      det[0] /= ten;
      det[1] += 1.;
      goto L30;
    L40:
      ;
    }
  L60:

    remove_tmp_destroy (t1);
    remove_tmp_destroy (t2);

    return (det[0] * pow (10.0, det[1]));
  }
}

static Complex
matrix_Det_Complex (m)
     Matrix *m;
{
  ASSERT (m);
  {
    int i;
    F_INT info, mm, n, lda, *ipiv, one;
    double d1, d2, ten;
    Complex det[2], z1, z2;
    ListNode *t1, *t2;
    Matrix *lu;

    lda = mm = MNR (m);
    n = MNC (m);
    one = 1;

    if (mm != n)
      error_1 (matrix_GetName (m), "matrix must be square for det()");

    t1 = install_tmp (D_VOID, ipiv = (F_INT *) MALLOC (sizeof (F_INT) * mm),
		      free);
    t2 = install_tmp (MATRIX, lu = matrix_Copy (m),
		      matrix_Destroy);

    signal (SIGINT, intcatch);
    XGETRF (&mm, &n, MDPTRc (lu), &lda, ipiv, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK ZGETRF", (char *) 0);
    if ((int) info > 0)
    {
      warning_1 (matrix_GetName (m), "matrix is singular");
      z1.r = 0.0;
      z1.i = 0.0;
      return (z1);
    }

    /*
     * Now compute the determinant using the algorithm contained
     * in the LINPACK subroutines DGEDI, ZGEDI. I'm doing it this
     * way because the LAPACK does not provide a direct method for
     * computing the determinant?
     */

    det[0].r = 1., det[0].i = 0.;
    det[1].r = 0., det[1].i = 0.;
    ten = 10.;
    for (i = 1; i <= n; ++i)
    {
      if (ipiv[i - 1] != i)
      {
	z1.r = -det[0].r;
	z1.i = -det[0].i;
	det[0].r = z1.r;
	det[0].i = z1.i;
      }
      z1.r = MATr (lu, i, i) * det[0].r - MATi (lu, i, i) * det[0].i;
      z1.i = MATr (lu, i, i) * det[0].i + MATi (lu, i, i) * det[0].r;
      det[0].r = z1.r;
      det[0].i = z1.i;
      /*        ...exit */
      z1.r = det[0].r * 0. - det[0].i * -1.;
      z1.i = det[0].r * -1. + det[0].i * 0.;
      if ((d1 = det[0].r, rabs (d1)) + (d2 = z1.r, rabs (d2)) == 0.)
      {
	goto L60;
      }
    L10:
      z1.r = det[0].r * 0. - det[0].i * -1.;
      z1.i = det[0].r * -1. + det[0].i * 0.;
      if ((d1 = det[0].r, rabs (d1)) + (d2 = z1.r, rabs (d2)) >= 1.)
      {
	goto L20;
      }
      z2.r = ten;
      z2.i = 0.;
      z1.r = z2.r * det[0].r - z2.i * det[0].i;
      z1.i = z2.r * det[0].i + z2.i * det[0].r;
      det[0].r = z1.r;
      det[0].i = z1.i;
      z1.r = det[1].r - 1.;
      z1.i = det[1].i + 0.;
      det[1].r = z1.r;
      det[1].i = z1.i;
      goto L10;
    L20:
    L30:
      z1.r = det[0].r * 0. - det[0].i * -1.;
      z1.i = det[0].r * -1. + det[0].i * 0.;
      if ((d1 = det[0].r, rabs (d1)) + (d2 = z1.r, rabs (d2)) < ten)
      {
	goto L40;
      }
      z2.r = ten;
      z2.i = 0.;
      z1 = complex_Div (det[0], z2);
      det[0].r = z1.r;
      det[0].i = z1.i;
      z1.r = det[1].r + 1.;
      z1.i = det[1].i + 0.;
      det[1].r = z1.r;
      det[1].i = z1.i;
      goto L30;
    L40:
      ;
    }
  L60:

    remove_tmp_destroy (t1);
    remove_tmp_destroy (t2);
    z1 = complex_pow (10.0, 0.0, det[1].r, det[1].i);
    return (complex_Multiply (det[0].r, det[0].i, z1.r, z1.i));
  }
}

/* **************************************************************
 * Singular Value Decompositions.
 * ************************************************************** */

void
matrix_Svd (M, rsv, lsv, sigma, flag)
     Matrix *M, **rsv, **lsv, **sigma;
     int flag;
{
  ASSERT (M);
  {
    matrix_screen_string (M);
    matrix_Detect_Inf (M);
    matrix_Detect_Nan (M);

    if (MTYPE (M) == REAL)
      matrix_Svd_Real (M, rsv, lsv, sigma, flag);
    else
      matrix_Svd_Complex (M, rsv, lsv, sigma, flag);
  }
}


static void
matrix_Svd_Real (M, rsv, lsv, sigma, flag)
     Matrix *M, **rsv, **lsv, **sigma;
     int flag;
{
  ASSERT (M);
  {
    F_INT k, lda, ldu, ldvt, lwork, m, n, info;
    F_INT jobu, jobvt;
    ListNode *t1, *t4, *t5, *t6, *t7;
    Matrix *s, *work;
    Matrix *A, *u, *vt;

    t1 = t4 = t5 = t6 = t7 = 0;   /* Initialize */
    u = vt = 0;                   /* Initialize */

    m = (F_INT) MNR (M);
    n = (F_INT) MNC (M);
    lda = m;
    k = (F_INT) min (m, n);
    lwork = (F_INT) max (3 * min (m, n) + max (m, n), 5 * min (m, n) - 4);

    if (flag == 1)
    {
      jobu = (F_INT) 'A';
      jobvt = (F_INT) 'A';
      ldu = m;
      ldvt = n;
      t5 = install_tmp (MATRIX, u = matrix_Create (ldu, m),
			matrix_Destroy);
      t6 = install_tmp (MATRIX, vt = matrix_Create (ldvt, n),
			matrix_Destroy);
    }
    else if (flag == 2)
    {
      jobu = (F_INT) 'S';
      jobvt = (F_INT) 'S';
      ldu = m;
      ldvt = k;
      t5 = install_tmp (MATRIX, u = matrix_Create (ldu, k),
			matrix_Destroy);
      t6 = install_tmp (MATRIX, vt = matrix_Create (ldvt, n),
			matrix_Destroy);
    }
    else if (flag == 3)
    {
      jobu = (F_INT) 'N';
      jobvt = (F_INT) 'N';
      ldu = 1;
      ldvt = 1;
      t5 = install_tmp (MATRIX, u = matrix_Create (0, 0),
			matrix_Destroy);
      t6 = install_tmp (MATRIX, vt = matrix_Create (0, 0),
			matrix_Destroy);
    }

    t1 = install_tmp (MATRIX, A = matrix_Copy (M),
		      matrix_Destroy);
    t4 = install_tmp (MATRIX, s = matrix_Create (1, k),
		      matrix_Destroy);
    t7 = install_tmp (MATRIX, work = matrix_Create (1, lwork),
		      matrix_Destroy);

    signal (SIGINT, intcatch);
    RGESVD (&jobu, &jobvt, &m, &n, MDPTRr (A), &lda, MDPTRr (s),
	    MDPTRr (u), &ldu, MDPTRr (vt), &ldvt,
	    MDPTRr (work), &lwork, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
    {
      fprintf (stderr, "ERROR: %ith argument to DGESVD is bad\n", (int) -info);
      error_1 ("bad argument to LAPACK DGESVD", 0);
    }
    if ((int) info > 0)
      error_1 ("svd algorithm failed to converge", 0);

    /* Clean Up */
    remove_tmp_destroy (t1);
    remove_tmp (t4);
    remove_tmp (t5);
    remove_tmp (t6);
    remove_tmp_destroy (t7);

    /* Set proper addresses */
    *rsv = vt;
    *lsv = u;
    *sigma = s;
  }
}

static void
matrix_Svd_Complex (M, rsv, lsv, sigma, flag)
     Matrix *M, **rsv, **lsv, **sigma;
     int flag;
{
  ASSERT (M);
  {
    F_INT k, lda, ldu, ldvt, lwork, m, n, info;
    F_INT jobu, jobvt;
    ListNode *t1, *t2, *t4, *t5, *t6, *t7;
    Matrix *s, *rwork, *work;
    Matrix *A, *u, *vt;

    u = vt = 0;                         /* Initialize */
    t1 = t2 = t4 = t5 = t6 = t7 = 0;    /* Initialize */
    m = MNR (M);
    n = MNC (M);
    lda = m;
    k = min (m, n);
    lwork = 2 * min (m, n) + max (m, n);

    if (flag == 1)
    {
      jobu = (F_INT) 'A';
      jobvt = (F_INT) 'A';
      ldu = m;
      ldvt = n;
      t5 = install_tmp (MATRIX, u = matrix_CreateC (ldu, m),
			matrix_Destroy);
      t6 = install_tmp (MATRIX, vt = matrix_CreateC (ldvt, n),
			matrix_Destroy);
    }
    else if (flag == 2)
    {
      jobu = (F_INT) 'S';
      jobvt = (F_INT) 'S';
      ldu = m;
      ldvt = k;
      t5 = install_tmp (MATRIX, u = matrix_CreateC (ldu, k),
			matrix_Destroy);
      t6 = install_tmp (MATRIX, vt = matrix_CreateC (ldvt, n),
			matrix_Destroy);
    }
    else if (flag == 3)
    {
      jobu = (F_INT) 'N';
      jobvt = (F_INT) 'N';
      ldu = 1;
      ldvt = 1;
      t5 = install_tmp (MATRIX, u = matrix_CreateC (0, 0),
			matrix_Destroy);
      t6 = install_tmp (MATRIX, vt = matrix_CreateC (0, 0),
			matrix_Destroy); 
   }

    t1 = install_tmp (MATRIX, A = matrix_Copy (M), matrix_Destroy);
    t2 = install_tmp (MATRIX, rwork = matrix_Create (1, 5 * max (m, n)),
		      matrix_Destroy);
    t4 = install_tmp (MATRIX, s = matrix_Create (1, k), matrix_Destroy);
    t7 = install_tmp (MATRIX, work = matrix_CreateC (1, lwork),
		      matrix_Destroy);
    
    signal (SIGINT, intcatch);
    XGESVD (&jobu, &jobvt, &m, &n, MDPTRc (A), &lda, MDPTRr (s),
	    MDPTRc (u), &ldu, MDPTRc (vt), &ldvt,
	    MDPTRc (work), &lwork, MDPTRr (rwork), &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK ZGESVD", (char *) 0);
    if ((int) info > 0)
      error_1 ("svd algorithm failed to converge", (char *) 0);

    /* Clean Up */
    remove_tmp_destroy (t1);
    remove_tmp_destroy (t2);
    remove_tmp (t4);
    remove_tmp (t5);
    remove_tmp (t6);
    remove_tmp_destroy (t7);

    /* Set proper addresses */
    *rsv = vt;
    *lsv = u;
    *sigma = s;
  }
}

/* **************************************************************
 * Compute eigenvalues, and vectors for symmetric prob
 * [A]x = Lambda*x.
 * ************************************************************** */
void
matrix_Eig_SEP (M, val, vec)
     Matrix *M, **vec, **val;
{
  ASSERT (M);
  {
    F_INT info, lda, lwork, m, n;
    F_INT jobz, uplo;
    Matrix *A;
    Matrix *w, *work, *rwork;

    w = 0;   /* Initialize */

    matrix_screen_string (M);
    matrix_Detect_Inf (M);
    matrix_Detect_Nan (M);

    /* Some rudimentary checks */
    m = MNR (M);
    n = MNC (M);
    lda = m;

    jobz = (F_INT) 'V';
    uplo = (F_INT) 'L';

    if (m != n)
      error_1 (matrix_GetName (M), "must input square matrix to eig(A)");

    /* Copy [a] cause it will get destroyed */
    tmp1 = install_tmp (MATRIX, A = matrix_Copy (M),
			matrix_Destroy);

    if (MTYPE (A) == REAL)
    {
      lwork = 3*n - 1;
      tmp2 = install_tmp (MATRIX, w = matrix_Create (1, n), matrix_Destroy);
      tmp5 = install_tmp (MATRIX, work = matrix_Create (1, lwork),
			  matrix_Destroy);

      signal (SIGINT, intcatch);
      RSYEV (&jobz, &uplo, &n, MDPTRr (A), &lda, MDPTRr (w), MDPTRr (work), 
             &lwork, &info);
      signal (SIGINT, intcatch_wait);
    }
    else if (MTYPE (A) == COMPLEX)
    {
      lwork = 2*n - 1;
      tmp2 = install_tmp (MATRIX, w = matrix_Create (1, n),
			  matrix_Destroy);
      tmp4 = install_tmp (MATRIX, work = matrix_CreateC (1, lwork),
			  matrix_Destroy);
      tmp5 = install_tmp (MATRIX, rwork = matrix_Create (1, 3*n-2),
			  matrix_Destroy);

      signal (SIGINT, intcatch);
      XHEEV (&jobz, &uplo, &n, MDPTRc (A), &lda, MDPTRr (w), MDPTRc (work), 
             &lwork, MDPTRr (rwork), &info);
      signal (SIGINT, intcatch_wait);
    }

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DSYEV or ZHEEV", 0);
    if ((int) info > 0)
      error_1 ("Eigensolver failed to converge", 0);

    *val = w;
    *vec = A;

    if (MTYPE (A) == REAL)
    {
      remove_tmp (tmp1);
      remove_tmp (tmp2);
      remove_tmp (tmp5);
      return;
    }
    else if (MTYPE (A) == COMPLEX)
    {
      remove_tmp (tmp1);
      remove_tmp (tmp2);
      remove_tmp (tmp4);
      remove_tmp_destroy (tmp5);
      return;
    }
  }
}

/* **************************************************************
 * Compute eigenvalues, and vectors for non-symmetric prob
 * [A]x = Lambda*x.
 * lflag =  0 No left-eigenvectors
 * lflag != 0 Return left-eigenvectors
 * ************************************************************** */
void
matrix_Eig_NEP (M, val, vec, lvec, lflag)
     Matrix *M, **vec, **val, **lvec;
     int *lflag;
{
  ASSERT (M);
  {
    int j, k;
    F_INT info, lda, ldvl, ldvr, lwork, m, n;
    F_INT jobvl, jobvr;
    Matrix *A, *etmp, *ptmp, *ltmp, *vl, *vr;
    Matrix *w, *wr, *wi, *work, *rwork;

    rwork = 0;         /* Initialize */
    w = wr = wi = 0;   /* Initialize */
    ltmp = 0;          /* Initialize */

    matrix_screen_string (M);
    matrix_Detect_Inf (M);
    matrix_Detect_Nan (M);

    /* Some rudimentary checks */
    m = MNR (M);
    n = MNC (M);
    lda = m;
    ldvl = n;
    ldvr = n;

    if (m != n)
      error_1 (matrix_GetName (M), "must input square matrix to eig(A)");

    /* Copy [a] cause it will get destroyed */
    tmp1 = install_tmp (MATRIX, A = matrix_Copy (M),
			matrix_Destroy);

    if (MTYPE (A) == REAL)
    {
      lwork = 4 * n;
      tmp2 = install_tmp (MATRIX, wr = matrix_Create (1, n),
			  matrix_Destroy);
      tmp3 = install_tmp (MATRIX, wi = matrix_Create (1, n),
			  matrix_Destroy);
      tmp4 = install_tmp (MATRIX, vr = matrix_Create (ldvr, n),
			  matrix_Destroy);
      if (lflag == 0)
	tmp8 = install_tmp (MATRIX, vl = matrix_Create (1, 1),
			    matrix_Destroy);
      else
	tmp8 = install_tmp (MATRIX, vl = matrix_Create (ldvr, n),
			    matrix_Destroy);
      tmp5 = install_tmp (MATRIX, work = matrix_Create (1, lwork),
			  matrix_Destroy);
    }
    else
    {
      lwork = 2 * n;
      tmp2 = install_tmp (MATRIX, w = matrix_CreateC (1, n),
			  matrix_Destroy);
      tmp3 = install_tmp (MATRIX, vr = matrix_CreateC (ldvr, n),
			  matrix_Destroy);
      if (lflag == 0)
	tmp8 = install_tmp (MATRIX, vl = matrix_CreateC (1, 1),
			    matrix_Destroy);
      else
	tmp8 = install_tmp (MATRIX, vl = matrix_CreateC (ldvr, n),
			    matrix_Destroy);
      tmp4 = install_tmp (MATRIX, work = matrix_CreateC (1, lwork),
			  matrix_Destroy);
      tmp5 = install_tmp (MATRIX, rwork = matrix_Create (1, 2 * n),
			  matrix_Destroy);
    }
    if (MTYPE (A) == REAL)
    {
      signal (SIGINT, intcatch);
      if (lflag == 0)
      {
        jobvl = (F_INT) 'N'; jobvr = (F_INT) 'V';
	RGEEV (&jobvl, &jobvr, &n, MDPTRr (A), &lda, MDPTRr (wr), MDPTRr (wi),
               MDPTRr (vl), &ldvl, MDPTRr (vr), &ldvr, MDPTRr (work),
               &lwork, &info);
      }
      else
      {
        jobvl = (F_INT) 'V'; jobvr = (F_INT) 'V';
	RGEEV (&jobvl, &jobvr, &n, MDPTRr (A), &lda, MDPTRr (wr), MDPTRr (wi),
               MDPTRr (vl), &ldvl, MDPTRr (vr), &ldvr, MDPTRr (work),
               &lwork, &info);
      }
      signal (SIGINT, intcatch_wait);
    }
    else
    {
      signal (SIGINT, intcatch);
      if (lflag == 0)
      {
        jobvl = (F_INT) 'N'; jobvr = (F_INT) 'V';
	XGEEV (&jobvl, &jobvr, &n, MDPTRc (A), &lda, MDPTRc (w), MDPTRc (vl),
               &ldvl, MDPTRc (vr), &ldvr, MDPTRc (work), &lwork,
               MDPTRr (rwork), &info);
      }
      else
      {
        jobvl = (F_INT) 'V'; jobvr = (F_INT) 'V';
	XGEEV (&jobvl, &jobvr, &n, MDPTRc (A), &lda, MDPTRc (w), MDPTRc (vl),
               &ldvl, MDPTRc (vr), &ldvr, MDPTRc (work), &lwork,
               MDPTRr (rwork), &info);
      }
      signal (SIGINT, intcatch_wait);
    }
    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DGEEV or ZGEEV", 0);
    if ((int) info > 0)
      error_1 ("failed to compute all of the eigenvalues", 0);

    if (MTYPE (A) == COMPLEX)
    {
      *val = w;
      *vec = vr;
      if (lflag != 0)
      {
	*lvec = vl;
	remove_tmp (tmp8);
      }
      else
	remove_tmp_destroy (tmp8);

      remove_tmp_destroy (tmp1);
      remove_tmp (tmp2);
      remove_tmp (tmp3);
      remove_tmp_destroy (tmp4);
      remove_tmp_destroy (tmp5);
      return;
    }

    /* First unpack eigenvectors */
    /* Loop accros rows, get all vectors */

    tmp6 = install_tmp (MATRIX, ptmp = matrix_CreateC (m, n),
			matrix_Destroy);

    for (k = 0; k < n; k++)
    {
      /* Right Eigenvectors */
      if (MATrv (wi, k) == 0.0)
      {
	/* Real Eigenvector */
	for (j = 0; j < n; j++)
	{
	  MATr0 (ptmp, j, k) = MAT0 (vr, j, k);
	  MATi0 (ptmp, j, k) = 0.0;
	}
      }
      else
      {
	/* Imaginary Eigenvector */
	if (MATrv (wi, k) >= 0.0)
	{
	  /* Construct positive part of complex conjugate pair */
	  for (j = 0; j < n; j++)
	  {
	    MATr0 (ptmp, j, k) = MAT0 (vr, j, k);
	    MATi0 (ptmp, j, k) = MAT0 (vr, j, (k + 1));
	  }
	}
	else
	{
	  /* Construct negative part of CC pair */
	  for (j = 0; j < n; j++)
	  {
	    MATr0 (ptmp, j, k) = MATr0 (ptmp, j, k - 1);
	    MATi0 (ptmp, j, k) = -MATi0 (ptmp, j, k - 1);
	  }
	}
      }
    }

    if (lflag != 0)
    {
      tmp9 = install_tmp (MATRIX, ltmp = matrix_CreateC (m, n),
			  matrix_Destroy);
      for (k = 0; k < n; k++)
      {
	/* Left Eigenvectors */
	if (MATrv (wi, k) == 0.0)
	{
	  /* Real Eigenvector */
	  for (j = 0; j < n; j++)
	  {
	    MATr0 (ltmp, j, k) = MAT0 (vl, j, k);
	    MATi0 (ltmp, j, k) = 0.0;
	  }
	}
	else
	{
	  /* Imaginary Eigenvector */
	  if (MATrv (wi, k) >= 0.0)
	  {
	    /* Construct positive part of complex conjugate pair */
	    for (j = 0; j < n; j++)
	    {
	      MATr0 (ltmp, j, k) = MAT0 (vl, j, k);
	      MATi0 (ltmp, j, k) = MAT0 (vl, j, (k + 1));
	    }
	  }
	  else
	  {
	    /* Construct negative part of CC pair */
	    for (j = 0; j < n; j++)
	    {
	      MATr0 (ltmp, j, k) =  MATr0 (ltmp, j, k - 1);
	      MATi0 (ltmp, j, k) = -MATi0 (ltmp, j, k - 1);
	    }
	  }
	}
      }
    }

    /* Now load up eigenvalues */
    tmp7 = install_tmp (MATRIX, etmp = matrix_CreateCLoad (wr, wi),
			matrix_Destroy);

    /* Set argument pointers */
    *val = etmp;
    *vec = ptmp;
    if (lflag != 0)
    {
      *lvec = ltmp;
      remove_tmp (tmp9);
    }
    
    /* Clean Up */
    remove_tmp_destroy (tmp1);
    remove_tmp_destroy (tmp2);
    remove_tmp_destroy (tmp3);
    remove_tmp_destroy (tmp4);
    remove_tmp_destroy (tmp5);
    remove_tmp (tmp6);
    remove_tmp (tmp7);
    remove_tmp_destroy (tmp8);
  }
}

void
matrix_Eig_GSEP (MA, MB, val, vec)
     Matrix *MA, *MB, **vec, **val;
{
  ASSERT (MA);
  ASSERT (MB);
  {
    int sol_type = 0;
    F_INT info, itype, lda, lwork, n, jobz, uplo;
    Matrix *A, *B;
    Matrix *w, *work, *rwork;

    A = B = rwork = work = 0;    /* Initialize */

    matrix_screen_string (MA);
    matrix_screen_string (MB);
    matrix_Detect_Inf (MA);
    matrix_Detect_Nan (MA);
    matrix_Detect_Inf (MB);
    matrix_Detect_Nan (MB);

    /* Some rudimentary checks, MA, MB must be square */
    if (MNR (MA) != MNC (MA))
      error_1 (matrix_GetName (MA), "must be symmetric for eig(a,b)");
    if (MNR (MB) != MNC (MB))
      error_1 (matrix_GetName (MB), "must be symmetric for eig(a,b)");
    if (MNR (MA) != MNR (MB))
      error_2 (matrix_GetName (MA), matrix_GetName (MB),
	       "must be same size for eig(a,b)");

    itype = 1;
    n = MNR (MA);
    lda = n;

    jobz = (F_INT) 'V';
    uplo = (F_INT) 'L';

    tmp3 = install_tmp (MATRIX, w = matrix_Create (1, n),
			matrix_Destroy);

    if (MTYPE (MA) == REAL && MTYPE (MB) == REAL)
    {
      sol_type = REAL;
      lwork = max (1, 3 * n - 1);
      tmp1 = install_tmp (MATRIX, A = matrix_Copy (MA), matrix_Destroy);
      tmp2 = install_tmp (MATRIX, B = matrix_Copy (MB), matrix_Destroy);
      tmp4 = install_tmp (MATRIX, work = matrix_Create (1, lwork), 
			  matrix_Destroy);
    }
    else if (MTYPE (MA) == COMPLEX || MTYPE (MB) == COMPLEX)
    {
      sol_type = COMPLEX;
      lwork = max (1, 2 * n - 1);
      tmp1 = install_tmp (MATRIX, A = matrix_copy_complex (MA), 
			  matrix_Destroy);
      tmp2 = install_tmp (MATRIX, B = matrix_copy_complex (MB),
			  matrix_Destroy);
      tmp4 = install_tmp (MATRIX, work = matrix_CreateC (1, lwork),
			  matrix_Destroy);
      tmp5 = install_tmp (MATRIX, rwork = matrix_Create (1, 3 * n - 2),
			  matrix_Destroy);
    }

    if (sol_type == REAL)
    {
      signal (SIGINT, intcatch);
      RSYGV (&itype, &jobz, &uplo, &n, MDPTRr (A), &lda, MDPTRr (B), &lda,
             MDPTRr (w), MDPTRr (work), &lwork, &info);
      signal (SIGINT, intcatch_wait);
    }
    else
    {
      signal (SIGINT, intcatch);
      XHEGV (&itype, &jobz, &uplo, &n, MDPTRc (A), &lda, MDPTRc (B), &lda,
             MDPTRr (w), MDPTRc (work), &lwork, MDPTRr (rwork), &info);
      signal (SIGINT, intcatch_wait);
    }

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DSYGV or ZHEGV", (char *) 0);
    if ((int) info > 0)
      error_1 ("failure in eigensolver DSYGV or ZHEGV", (char *) 0);

    *val = w;
    *vec = A;

    /* Clean Up */
    remove_tmp (tmp1);
    remove_tmp_destroy (tmp2);
    remove_tmp (tmp3);
    remove_tmp_destroy (tmp4);
    if (sol_type == COMPLEX)
      remove_tmp_destroy (tmp5);
  }
}

void
matrix_Eig_GNEP (MA, MB, val, vecr)
     Matrix *MA, *MB, **vecr, **val;
{
  ASSERT (MA);
  ASSERT (MB);
  {
    int i, j, sol_type;
    F_INT info, k, lda, ldvl, ldvr, lwork, n;
    F_INT jobvl, jobvr;
    ListNode *tmp1, *tmp2, *tmp4, *tmp5, *tmp6;
    ListNode *tmp7, *tmp8, *tmp9, *tmp11, *tmp12;
    Matrix *A, *B;
    Matrix *alpha, *alphar, *alphai, *beta;
    Matrix *eval, *rvec, *vl, *vr;
    Matrix *work, *rwork;

    sol_type = 0;                              /* Initialize */
    tmp1 = tmp2 = tmp4 = tmp5 = tmp6 = 0;      /* Initialize */
    tmp7 = tmp8 = tmp9 = tmp11 = 0;            /* Initialize */
    A = B = alpha = alphai = alphar = 0;       /* Initialize */
    rvec = vl = vr = work = rwork = beta = 0;  /* Initialize */

    matrix_screen_string (MA);
    matrix_screen_string (MB);
    matrix_Detect_Inf (MA);
    matrix_Detect_Nan (MA);
    matrix_Detect_Inf (MB);
    matrix_Detect_Nan (MB);

    /* Some rudimentary checks, MA, MB must be square */
    if (MNR (MA) != MNC (MA))
      error_1 (matrix_GetName (MA), "must be symmetric for eig(a,b)");
    if (MNR (MB) != MNC (MB))
      error_1 (matrix_GetName (MB), "must be symmetric for eig(a,b)");
    if (MNR (MA) != MNR (MB))
      error_2 (matrix_GetName (MA), matrix_GetName (MB),
	       "must be same size for eig(a,b)");

    n = MNR (MA);
    lda = ldvl = ldvr = n;
   
    jobvl = (F_INT) 'N';
    jobvr = (F_INT) 'V';

    if (MTYPE (MA) == REAL && MTYPE (MB) == REAL)
    {
      sol_type = REAL;
      lwork = max (1, 8*n);
      tmp1 = install_tmp (MATRIX, A = matrix_Copy (MA),
			  matrix_Destroy);
      tmp2 = install_tmp (MATRIX, B = matrix_Copy (MB),
			  matrix_Destroy);
      tmp4 = install_tmp (MATRIX, work = matrix_Create (1, lwork),
			  matrix_Destroy);
      tmp5 = install_tmp (MATRIX, alphar = matrix_Create (1, n),
			  matrix_Destroy);
      tmp6 = install_tmp (MATRIX, alphai = matrix_Create (1, n),
			  matrix_Destroy);
      tmp7 = install_tmp (MATRIX, beta = matrix_Create (1, n),
			  matrix_Destroy);
      tmp8 = install_tmp (MATRIX, vl = matrix_Create (1, 1),
			  matrix_Destroy);
      tmp9 = install_tmp (MATRIX, vr = matrix_Create (ldvr, n),
			  matrix_Destroy);
    }
    else if (MTYPE (MA) == COMPLEX || MTYPE (MB) == COMPLEX)
    {
      sol_type = COMPLEX;
      lwork = max (1, 2*n);
      tmp1 = install_tmp (MATRIX, A = matrix_copy_complex (MA), 
			  matrix_Destroy);
      tmp2 = install_tmp (MATRIX, B = matrix_copy_complex (MB),
			  matrix_Destroy);
      tmp4 = install_tmp (MATRIX, work = matrix_CreateC (1, lwork),
			  matrix_Destroy);
      tmp5 = install_tmp (MATRIX, rwork = matrix_Create (1, 8*n),
			  matrix_Destroy);
      tmp6 = install_tmp (MATRIX, alpha = matrix_CreateC (1, n),
			  matrix_Destroy);
      tmp7 = install_tmp (MATRIX, beta = matrix_CreateC (1, n),
			  matrix_Destroy);
      tmp8 = install_tmp (MATRIX, vl = matrix_CreateC (1, 1),
			  matrix_Destroy);
      tmp9 = install_tmp (MATRIX, vr = matrix_CreateC (ldvr, n),
			  matrix_Destroy);
    }

    if (sol_type == REAL)
    {
      signal (SIGINT, intcatch);
      RGEGV (&jobvl, &jobvr, &n, MDPTRr (A), &lda, MDPTRr (B), &lda, 
	     MDPTRr (alphar), MDPTRr (alphai), MDPTRr (beta), 
	     MDPTRr (vl), &ldvl, MDPTRr (vr), &ldvr, MDPTRr (work), 
	     &lwork, &info);
      signal (SIGINT, intcatch_wait);
    }
    else
    {
      signal (SIGINT, intcatch);
      XGEGV (&jobvl, &jobvr, &n, MDPTRc (A), &lda, MDPTRc (B), &lda, 
             MDPTRc (alpha), MDPTRc (beta), MDPTRc (vl), &ldvl, 
             MDPTRc (vr), &ldvr, MDPTRc (work), &lwork, MDPTRr (rwork),
             &info);
      signal (SIGINT, intcatch_wait);
    }

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DGEGV or ZGEGV", 0);
    if ((int) info > 0)
      error_1 ("failure in eigensolver DGEGV or ZGEGV", 0);

    /*
     * Now sort out results
     * First unpack eigenvectors
     */

    if (sol_type == REAL)
    {
      tmp11 = install_tmp (MATRIX, rvec = matrix_CreateC (n, n),
			   matrix_Destroy);
      
      for (k = 0; k < n; k++)
      {
	if (MATrv (alphai, k) == 0.0)
	{
	  /* Real Eigenvector */
	  for (j = 0; j < n; j++)
	  {
	    MATr0 (rvec, j, k) = MAT0 (vr, j, k);
	    MATi0 (rvec, j, k) = 0.0;
	  }
	}
	else
	{
	  /* Imaginary Eigenvector */
	  if (MATrv (alphai, k) >= 0.0)
	  {
	    /* Construct positive part of complex conjugate pair */
	    for (j = 0; j < n; j++)
	    {
	      MATr0 (rvec, j, k) = MAT0 (vr, j, k);
	      MATi0 (rvec, j, k) = MAT0 (vr, j, (k + 1));
	    }
	  }
	  else
	  {
	    /* Construct negative part of CC pair */
	    for (j = 0; j < n; j++)
	    {
	      MATr0 (rvec, j, k) =  MATr0 (rvec, j, k - 1);
	      MATi0 (rvec, j, k) = -MATi0 (rvec, j, k - 1);
	    }
	  }
	}
      }
      remove_tmp_destroy (tmp8);
      remove_tmp_destroy (tmp9);
    }

    /*
     * Now compute eigenvalues
     */
    tmp12 = install_tmp (MATRIX, eval = matrix_CreateC (1, n),
			 matrix_Destroy);

    if (sol_type == REAL)
    {
      for (i = 1; i <= n; i++)
      {
	if (MAT (beta, 1, i) == 0.0)
	{
	  if (MAT (alphar, 1, i) == 0.0)
	    MATr (eval, 1, i) = R_NAN;
	  else
	    MATr (eval, 1, i) = R_INF;
	  
	  if (MAT (alphai, 1, i) == 0.0)
	    MATi (eval, 1, i) = R_NAN;
	  else
	    MATi (eval, 1, i) = R_INF;
	}
	else
	{
	  MATr (eval, 1, i) = MAT (alphar, 1, i) / MAT (beta, 1, i);
	  MATi (eval, 1, i) = MAT (alphai, 1, i) / MAT (beta, 1, i);
	}
      }
    }
    else if (sol_type == COMPLEX)
    {
      for (i = 1; i <= n; i++)
      {
	if (MATr (beta, 1, i) == 0.0 && MATi (beta, 1, i) == 0.0)
	{
	  if (MATr (alpha, 1, i) == 0.0 && MATi (alpha, 1, i) == 0.0)
	  {
	    MATr (eval, 1, i) = R_NAN;
	    MATi (eval, 1, i) = R_NAN;
	  }
	  else
	  {
	    MATr (eval, 1, i) = R_INF;
	    MATi (eval, 1, i) = R_INF;
	  }
	}
	else
	{
	  MATc (eval, 1, i) = complex_div (MATr(alpha,1,i), MATi(alpha,1,i),
					   MATr(beta, 1,i), MATi(beta,1,i));
	}
      }
    }

    if (sol_type == REAL)
    {
      *vecr = rvec;
      *val = eval;

      /* Clean Up */
      remove_tmp_destroy (tmp1);
      remove_tmp_destroy (tmp2);
      remove_tmp_destroy (tmp4);
      remove_tmp_destroy (tmp5);
      remove_tmp_destroy (tmp6);
      remove_tmp_destroy (tmp7);
      
      remove_tmp (tmp11);
      remove_tmp (tmp12);
    }
    else if (sol_type == COMPLEX)
    {
      *vecr = vr;
      *val = eval;

      /* Clean Up */
      remove_tmp_destroy (tmp1);
      remove_tmp_destroy (tmp2);
      remove_tmp_destroy (tmp4);
      remove_tmp_destroy (tmp5);
      remove_tmp_destroy (tmp6);
      remove_tmp_destroy (tmp7);

      remove_tmp (tmp8);
      remove_tmp (tmp9);
      remove_tmp (tmp12);
    }
  }
}

void
matrix_Eig_GNEPa (MA, MB, valpha, vbeta, vecl, vecr)
     Matrix *MA, *MB, **valpha, **vbeta, **vecl, **vecr;
{
  ASSERT (MA);
  ASSERT (MB);
  {
    int j, sol_type;
    F_INT info, k, lda, ldvl, ldvr, lwork, n;
    F_INT jobvl, jobvr;
    ListNode *tmp1, *tmp2, *tmp4, *tmp5, *tmp6;
    ListNode *tmp7, *tmp8, *tmp9, *tmp10, *tmp11, *tmp12;
    Matrix *A, *B;
    Matrix *alpha, *alphar, *alphai, *beta;
    Matrix *eval, *rvec, *lvec, *vl, *vr;
    Matrix *work, *rwork;

    sol_type = 0;                                  /* Initialize */
    tmp1 = tmp2 = tmp4 = tmp5 = tmp6 = 0;          /* Initialize */
    tmp7 = tmp8 = tmp9 = 0;                        /* Initialize */
    tmp10 = tmp11 = tmp12 = 0;                     /* Initialize */
    A = B = alpha = alphar = alphai = beta = 0;    /* Initialize */
    vl = vr = work = rwork = 0;                    /* Initialize */
    eval = rvec = lvec = 0;                        /* Initialize */

    matrix_screen_string (MA);
    matrix_screen_string (MB);
    matrix_Detect_Inf (MA);
    matrix_Detect_Nan (MA);
    matrix_Detect_Inf (MB);
    matrix_Detect_Nan (MB);

    /* Some rudimentary checks, MA, MB must be square */
    if (MNR (MA) != MNC (MA))
      error_1 (matrix_GetName (MA), "must be symmetric for eig(a,b)");
    if (MNR (MB) != MNC (MB))
      error_1 (matrix_GetName (MB), "must be symmetric for eig(a,b)");
    if (MNR (MA) != MNR (MB))
      error_2 (matrix_GetName (MA), matrix_GetName (MB),
	       "must be same size for eig(a,b)");

    n = MNR (MA);
    lda = ldvl = ldvr = n;

    jobvl = (F_INT) 'V';
    jobvr = (F_INT) 'V';

    if (MTYPE (MA) == REAL && MTYPE (MB) == REAL)
    {
      sol_type = REAL;
      lwork = max (1, 8*n);
      tmp1 = install_tmp (MATRIX, A = matrix_Copy (MA),
			  matrix_Destroy);
      tmp2 = install_tmp (MATRIX, B = matrix_Copy (MB),
			  matrix_Destroy);
      tmp4 = install_tmp (MATRIX, work = matrix_Create (1, lwork),
			  matrix_Destroy);
      tmp5 = install_tmp (MATRIX, alphar = matrix_Create (1, n),
			  matrix_Destroy);
      tmp6 = install_tmp (MATRIX, alphai = matrix_Create (1, n),
			  matrix_Destroy);
      tmp7 = install_tmp (MATRIX, beta = matrix_Create (1, n),
			  matrix_Destroy);
      tmp8 = install_tmp (MATRIX, vl = matrix_Create (ldvl, n),
			  matrix_Destroy);
      tmp9 = install_tmp (MATRIX, vr = matrix_Create (ldvr, n),
			  matrix_Destroy);
    }
    else if (MTYPE (MA) == COMPLEX || MTYPE (MB) == COMPLEX)
    {
      sol_type = COMPLEX;
      lwork = max (1, 2*n);
      tmp1 = install_tmp (MATRIX, A = matrix_copy_complex (MA),
			  matrix_Destroy);
      tmp2 = install_tmp (MATRIX, B = matrix_copy_complex (MB),
			  matrix_Destroy);
      tmp4 = install_tmp (MATRIX, work = matrix_CreateC (1, lwork),
			  matrix_Destroy);
      tmp5 = install_tmp (MATRIX, rwork = matrix_Create (1, 8*n),
			  matrix_Destroy);
      tmp6 = install_tmp (MATRIX, alpha = matrix_CreateC (1, n),
			  matrix_Destroy);
      tmp7 = install_tmp (MATRIX, beta = matrix_CreateC (1, n),
			  matrix_Destroy);
      tmp8 = install_tmp (MATRIX, vl = matrix_CreateC (ldvl, n),
			  matrix_Destroy);
      tmp9 = install_tmp (MATRIX, vr = matrix_CreateC (ldvr, n),
			  matrix_Destroy);
    }

    if (sol_type == REAL)
    {
      signal (SIGINT, intcatch);
      RGEGV (&jobvl, &jobvr, &n, MDPTRr (A), &lda, MDPTRr (B), &lda, 
	     MDPTRr (alphar), MDPTRr (alphai), MDPTRr (beta), 
	     MDPTRr (vl), &ldvl, MDPTRr (vr), &ldvr, MDPTRr (work), 
	     &lwork, &info);
      signal (SIGINT, intcatch_wait);
    }
    else
    {
      signal (SIGINT, intcatch);
      XGEGV (&jobvl, &jobvr, &n, MDPTRc (A), &lda, MDPTRc (B), &lda, 
             MDPTRc (alpha), MDPTRc (beta), MDPTRc (vl), &ldvl, 
             MDPTRc (vr), &ldvr, MDPTRc (work), &lwork, MDPTRr (rwork),
             &info);
      signal (SIGINT, intcatch_wait);
    }

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DGEGV or ZGEGV", 0);
    if ((int) info > 0)
      error_1 ("failure in eigensolver DGEGV or ZGEGV", 0);

    /*
     * Now sort out results
     * First unpack eigenvectors
     */

    if (sol_type == REAL)
    {
      tmp10 = install_tmp (MATRIX, lvec = matrix_CreateC (n, n),
			   matrix_Destroy);
      tmp11 = install_tmp (MATRIX, rvec = matrix_CreateC (n, n),
			   matrix_Destroy);
      tmp12 = install_tmp (MATRIX, eval = matrix_CreateC (1, n),
			   matrix_Destroy);
      
      for (k = 0; k < n; k++)
      {
	MATr0 (eval, 1, k) = MAT0 (alphar, 1, k);
	MATi0 (eval, 1, k) = MAT0 (alphai, 1, k);

	if (MATrv (alphai, k) == 0.0)
	{
	  /* Real Eigenvector */
	  for (j = 0; j < n; j++)
	  {
	    MATr0 (lvec, j, k) = MAT0 (vl, j, k);
	    MATi0 (lvec, j, k) = 0.0;
	    MATr0 (rvec, j, k) = MAT0 (vr, j, k);
	    MATi0 (rvec, j, k) = 0.0;
	  }
	}
	else
	{
	  /* Imaginary Eigenvector */
	  if (MATrv (alphai, k) >= 0.0)
	  {
	    /* Construct positive part of complex conjugate pair */
	    for (j = 0; j < n; j++)
	    {
	      MATr0 (lvec, j, k) = MAT0 (vl, j, k);
	      MATi0 (lvec, j, k) = MAT0 (vl, j, (k + 1));
	      MATr0 (rvec, j, k) = MAT0 (vr, j, k);
	      MATi0 (rvec, j, k) = MAT0 (vr, j, (k + 1));
	    }
	  }
	  else
	  {
	    /* Construct negative part of CC pair */
	    for (j = 0; j < n; j++)
	    {
	      MATr0 (lvec, j, k) =  MATr0 (lvec, j, k - 1);
	      MATi0 (lvec, j, k) = -MATi0 (lvec, j, k - 1);
	      MATr0 (rvec, j, k) =  MATr0 (rvec, j, k - 1);
	      MATi0 (rvec, j, k) = -MATi0 (rvec, j, k - 1);
	    }
	  }
	}
      }
      remove_tmp_destroy (tmp8);
      remove_tmp_destroy (tmp9);
    }
    
    if (sol_type == REAL)
    {
      *vecl = lvec;
      *vecr = rvec;
      *valpha = eval;
      *vbeta = beta;

      /* Clean Up */
      remove_tmp_destroy (tmp1);
      remove_tmp_destroy (tmp2);
      remove_tmp_destroy (tmp4);
      remove_tmp_destroy (tmp5);
      remove_tmp_destroy (tmp6);
      remove_tmp (tmp7);
      
      remove_tmp (tmp10);
      remove_tmp (tmp11);
      remove_tmp (tmp12);
    }
    else if (sol_type == COMPLEX)
    {
      *vecl = vl;
      *vecr = vr;
      *valpha = alpha;
      *vbeta = beta;

      /* Clean Up */
      remove_tmp_destroy (tmp1);
      remove_tmp_destroy (tmp2);
      remove_tmp_destroy (tmp4);
      remove_tmp_destroy (tmp5);
      remove_tmp (tmp6);
      remove_tmp (tmp7);

      remove_tmp (tmp8);
      remove_tmp (tmp9);
    }
  }
}

Matrix *
matrix_Chol (m)
     Matrix *m;
{
  ASSERT (m);
  {
    F_INT i, j, info, lda, n, uplo;
    Matrix *A;

    matrix_screen_string (m);
    matrix_Detect_Inf (m);
    matrix_Detect_Nan (m);

    lda = n = MNC (m);
    uplo = (F_INT) 'U';

    /* [m] gets overwritten, so copy it */
    tmp1 = install_tmp (MATRIX, A = matrix_Copy (m), matrix_Destroy);

    /* Call LAPACK routine */
    if (MTYPE (m) == REAL)
    {
      signal (SIGINT, intcatch);
      RPOTRF (&uplo, &n, MDPTRr (A), &lda, &info);
      signal (SIGINT, intcatch_wait);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      signal (SIGINT, intcatch);
      XPOTRF (&uplo, &n, MDPTRc (A), &lda, &info);
      signal (SIGINT, intcatch_wait);
    }

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK [DZ]OPTRF", (char *) 0);
    if ((int) info > 0)
      error_1 (matrix_GetName (m), "not positive definite");

    /* We must make sure and zero out the lower left triangle of [A] */
    if (MTYPE (A) == REAL)
    {
      for (i = 1; i < n; i++)
	for (j = 0; j < i; j++)
	  MAT0 (A, i, j) = 0.0;
    }
    else if (MTYPE (A) == COMPLEX)
    {
      for (i = 1; i < n; i++)
      {
	for (j = 0; j < i; j++)
	{
	  MATr0 (A, i, j) = 0.0;
	  MATi0 (A, i, j) = 0.0;
	}
      }
    }
    
    remove_tmp (tmp1);
    return (A);
  }
}

/* **************************************************************
 * Compute the QR decomposition of [m].
 * ************************************************************** */

static void matrix_QrR _PROTO ((Matrix * m, Matrix ** q, Matrix ** r));
static void matrix_QrZ _PROTO ((Matrix * m, Matrix ** q, Matrix ** r));

void
matrix_Qr (m, q, r)
     Matrix *m, **q, **r;
{
  matrix_screen_string (m);
  matrix_Detect_Inf (m);
  matrix_Detect_Nan (m);

  if (MTYPE (m) == REAL)
    matrix_QrR (m, q, r);
  else
    matrix_QrZ (m, q, r);
}

static void
matrix_QrR (M, q, r)
     Matrix *M, **q, **r;
{
  ASSERT (M);
  {
    int i, j;
    double *tau, *work;
    F_INT info, k, lwork, m, n;
    Matrix *A, *Atmp, *R;

    /* Set dimensional paramaters */
    m = MNR (M);
    n = MNC (M);
    k = min (m, n);
    lwork = max (m, n);

    /* 
     * over/under determined q: M x M
     *                       r: M x N
     *                       p: N x N
     */

    /* Create work arrays */
    tmp1 = install_tmp (MATRIX, A = matrix_Copy (M), matrix_Destroy);
    tmp2 = install_tmp (D_VOID, 
			tau = (double *) MALLOC (sizeof (double) * k),
			free);
    tmp3 = install_tmp (D_VOID, 
			work = (double *) MALLOC (sizeof (double) * lwork),
			free);

    signal (SIGINT, intcatch);
    RGEQRF (&m, &n, MDPTRr (A), &m, tau, work, &lwork, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DGEQRF", (char *) 0);

    /* Extract [R] */
    tmp4 = install_tmp (MATRIX, R = matrix_Create (m, n), 
			matrix_Destroy);
    matrix_Zero (R);
    for (j = 1; j <= MNC (R); j++)
      for (i = 1; (i <= j) && (i <= MNR (R)); i++)
	MAT (R, i, j) = MAT (A, i, j);

    signal (SIGINT, intcatch);
    /* Grow [A] if necessary to hold computed [Q] */
    if ((m-n) > 0)
    {
      matrix_AppendColR (A, m - n);
    }
    RORGQR (&m, &m, &k, MDPTRr (A), &m, tau, work, &lwork, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DORGQR", (char *) 0);

    /*
     * Shrink Q if necessary
     */

    if ((m-n) < 0)
    {
      Atmp = matrix_ExtractRowMatrix (A, matrix_CreateFill (1.0, (double) m,
							    1.0, 0));
      *q = Atmp;
      remove_tmp_destroy (tmp1);
    }
    else
    {
      *q = A;
      remove_tmp (tmp1);
    }

    /* Clean - Up */
    *r = R;

    remove_tmp_destroy (tmp2);
    remove_tmp_destroy (tmp3);
    remove_tmp (tmp4);
  }
}

static void
matrix_QrZ (M, q, r)
     Matrix *M, **q, **r;
{
  ASSERT (M);
  {
    int i, j;
    Complex *tau, *work;
    F_INT info, k, lwork, m, n;
    Matrix *A, *Atmp, *R;

    /* Set dimensional paramaters */
    m = MNR (M);
    n = MNC (M);
    k = min (m, n);
    lwork = max (m, n);

    /* 
     *  over / over determined q: M x M
     *                         r: M x N
     *                         p: N x N
     */

    /* Create work arrays */
    tmp1 = install_tmp (MATRIX, A = matrix_Copy (M),
			       matrix_Destroy);
    tmp2 = install_tmp (D_VOID, 
			tau = (Complex *) MALLOC (sizeof (Complex) * k),
			free);
    tmp3 = install_tmp (D_VOID, 
			work = (Complex *) MALLOC (sizeof (Complex) * lwork),
			free);

    signal (SIGINT, intcatch);
    XGEQRF (&m, &n, MDPTRc (A), &m, tau, work, &lwork, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK ZGEQRF", (char *) 0);

    /* Extract [R] */
    tmp4 = install_tmp (MATRIX, R = matrix_CreateC (m, n), matrix_Destroy);
    matrix_Zero (R);
    for (j = 1; j <= MNC (R); j++)
      for (i = 1; (i <= j) && (i <= MNR (R)); i++)
      {
	MATr (R, i, j) = MATr (A, i, j);
	MATi (R, i, j) = MATi (A, i, j);
      }

    /* Now re-assemble [Q] */
    signal (SIGINT, intcatch);
    if ((m-n) > 0)
    {
      matrix_AppendColC (A, m - n);
    }
    XUNGQR (&m, &m, &k, MDPTRc (A), &m, tau, work, &lwork, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK ZUNGQR", (char *) 0);

    if ((m-n) < 0)
    {
      Atmp = matrix_ExtractRowMatrix (A, matrix_CreateFill (1.0, (double) m,
							    1.0, 0));
      *q = Atmp;
      remove_tmp_destroy (tmp1);
    }
    else
    {
      *q = A;
      remove_tmp (tmp1);
    }
    
    /* Clean - Up */
    *r = R;

    remove_tmp_destroy (tmp2);
    remove_tmp_destroy (tmp3);
    remove_tmp (tmp4);
  }
}

static void matrix_QrPR _PROTO ((Matrix * m, Matrix ** q, 
                                 Matrix ** r, Matrix **p));
static void matrix_QrPZ _PROTO ((Matrix * m, Matrix ** q, 
                                 Matrix ** r, Matrix **p));

void
matrix_QrP (m, q, r, p)
     Matrix *m, **q, **r, **p;
{
  matrix_screen_string (m);
  matrix_Detect_Inf (m);
  matrix_Detect_Nan (m);

  if (MTYPE (m) == REAL)
    matrix_QrPR (m, q, r, p);
  else
    matrix_QrPZ (m, q, r, p);
}

static void
matrix_QrPR (M, q, r, p)
     Matrix *M, **q, **r, **p;
{
  ASSERT (M);
  {
    int i, j, *jpvt;
    double *tau, *work;
    F_INT info, k, lwork, m, n;
    Matrix *A, *Atmp, *R, *P;

    /* Set dimensional paramaters */
    m = MNR (M);
    n = MNC (M);
    k = min (m, n);
    lwork = 3 * max (m, n);

    /* 
     *  over / under determined q: M x M
     *                          r: M x N
     *                          p: N x N
     */

    /* Create work arrays */
    tmp1 = install_tmp (MATRIX, A = matrix_Copy (M), matrix_Destroy);
    tmp2 = install_tmp (D_VOID, 
			tau = (double *) MALLOC (sizeof (double) * k),
			free);
    tmp3 = install_tmp (D_VOID, 
			work = (double *) MALLOC (sizeof (double) * lwork),
			free);
    tmp5 = install_tmp (D_VOID, jpvt = (int *) MALLOC (sizeof (int) * n),
			free);
    for (i = 0; i < n; i++)
      jpvt[i] = 0;          /* It is important to zero-out jpvt */

    signal (SIGINT, intcatch);
    RGEQPF (&m, &n, MDPTRr (A), &m, jpvt, tau, work, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DGEQPF", (char *) 0);

    /* Extract [R] */
    tmp4 = install_tmp (MATRIX, R = matrix_Create (m, n), matrix_Destroy);
    matrix_Zero (R);
    for (j = 1; j <= MNC (R); j++)
      for (i = 1; (i <= j) && (i <= MNR (R)); i++)
	MAT (R, i, j) = MAT (A, i, j);

    /* Form [P] */
    tmp6 = install_tmp (MATRIX, P = matrix_Create (n, n), matrix_Destroy);
    matrix_Zero (P);
    for (i = 1; i <= n; i++)
    {
      if (jpvt[i-1] != 0)
        MAT (P, jpvt[i-1], i) = 1.0;
      else
        MAT (P, i, i) = 1.0;
    }

    signal (SIGINT, intcatch);
    if ((m-n) > 0)
    {
      matrix_AppendColR (A, m - n);
    }
    RORGQR (&m, &m, &k, MDPTRr (A), &m, tau, work, &lwork, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DORGQR", (char *) 0);

    if ((m-n) < 0)
    {
      Atmp = matrix_ExtractRowMatrix (A, matrix_CreateFill (1.0, (double) m,
							    1.0, 0));
      *q = Atmp;
      remove_tmp_destroy (tmp1);
    }
    else
    {
      *q = A;
      remove_tmp (tmp1);
    }

    /* Clean - Up */
    *r = R;
    *p = P;

    remove_tmp_destroy (tmp2);
    remove_tmp_destroy (tmp3);
    remove_tmp (tmp4);
    remove_tmp_destroy (tmp5);
    remove_tmp (tmp6);
  }
}

static void
matrix_QrPZ (M, q, r, p)
     Matrix *M, **q, **r, **p;
{
  ASSERT (M);
  {
    int i, j, *jpvt;
    Complex *tau, *work;
    F_INT info, k, lwork, m, n;
    Matrix *A, *Atmp, *R, *P, *rwork;

    /* Set dimensional paramaters */
    m = MNR (M);
    n = MNC (M);
    k = min (m, n);
    lwork = max (m, n);

    /* 
     *  over / under determined q: M x M
     *                          r: M x N
     *                          p: N x N
     */

    /* Create work arrays */
    tmp1 = install_tmp (MATRIX, A = matrix_Copy (M), matrix_Destroy);
    tmp2 = install_tmp (D_VOID, 
			tau = (Complex *) MALLOC (sizeof (Complex) * k),
			free);
    tmp3 = install_tmp (D_VOID, 
			work = (Complex *) MALLOC (sizeof (Complex) * lwork),
			free);

    tmp5 = install_tmp (D_VOID, jpvt = (int *) MALLOC (sizeof (int) * n),
			free);
    for (i = 0; i < n; i++)
      jpvt[i] = 0;          /* It is important to zero-out jpvt */

    tmp7 = install_tmp (MATRIX, rwork = matrix_Create (2*n, 1), matrix_Destroy);

    signal (SIGINT, intcatch);
    XGEQPF (&m, &n, MDPTRc (A), &m, jpvt, tau, work, MDPTRr (rwork), &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK ZGEQPF", (char *) 0);

    /* Extract [R] */
    tmp4 = install_tmp (MATRIX, R = matrix_CreateC (m, n), matrix_Destroy);
    matrix_Zero (R);
    for (j = 1; j <= MNC (R); j++)
      for (i = 1; (i <= j) && (i <= MNR (R)); i++)
      {
	MATr (R, i, j) = MATr (A, i, j);
	MATi (R, i, j) = MATi (A, i, j);
      }

    /* Form [P] */
    tmp6 = install_tmp (MATRIX, P = matrix_Create (n, n), matrix_Destroy);
    matrix_Zero (P);
    for (i = 1; i <= n; i++)
    {
      if (jpvt[i-1] != 0)
        MAT (P, jpvt[i-1], i) = 1.0;
      else
        MAT (P, i, i) = 1.0;
    }

    /* Now re-assemble [Q] */
    signal (SIGINT, intcatch);
    if ((m-n) > 0)
    {
      matrix_AppendColC (A, m - n);
    }
    XUNGQR (&m, &m, &k, MDPTRc (A), &m, tau, work, &lwork, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK ZUNGQR", (char *) 0);

    if ((m-n) < 0)
    {
      Atmp = matrix_ExtractRowMatrix (A, matrix_CreateFill (1.0, (double) m,
							    1.0, 0));
      *q = Atmp;
      remove_tmp_destroy (tmp1);
    }
    else
    {
      *q = A;
      remove_tmp (tmp1);
    }

    /* Clean - Up */
    *r = R;
    *p = P;

    remove_tmp_destroy (tmp2);
    remove_tmp_destroy (tmp3);
    remove_tmp (tmp4);
    remove_tmp_destroy (tmp5);
    remove_tmp (tmp6);
    remove_tmp_destroy (tmp7);
  }
}

/* **************************************************************
 * Compute the Hessenberg form of a matrix. Also compute an
 * orthogonal matrix [p], such that, [m] = [p][h][p]'.
 * ************************************************************** */

static void matrix_Hess_Real _PROTO ((Matrix * m, Matrix ** p, Matrix ** h));
static void matrix_Hess_Complex _PROTO ((Matrix * m, Matrix ** p, Matrix ** h));

void
matrix_Hess (m, p, h)
     Matrix *m, **p, **h;
{
  ASSERT (m);
  {
    matrix_screen_string (m);
    matrix_Detect_Inf (m);
    matrix_Detect_Nan (m);

    if (MTYPE (m) == REAL)
      matrix_Hess_Real (m, p, h);
    else
      matrix_Hess_Complex (m, p, h);
  }
}

static void
matrix_Hess_Real (M, q, h)
     Matrix *M, **q, **h;
{
  ASSERT (M);
  {
    int i, j;
    F_INT ilo, ihi, lda, m, n, info, lwork;
    Matrix *A, *tau, *work;

    /* Get matrix dimensions */
    m = MNR (M);
    n = MNC (M);
    lda = m;
    lwork = n;
    ilo = 1;
    ihi = n;

    /* The input to orthes() will be destroyed, so copy it */
    tmp1 = install_tmp (MATRIX, A = matrix_Copy (M), matrix_Destroy);
    tmp2 = install_tmp (MATRIX, tau = matrix_Create (1, n), matrix_Destroy);
    tmp3 = install_tmp (MATRIX, work = matrix_Create (1, lwork), 
			matrix_Destroy);

    signal (SIGINT, intcatch);
    RGEHRD (&n, &ilo, &ihi, MDPTRr (A), &lda, MDPTRr (tau),
	    MDPTRr (work), &lwork, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DGEHRD", (char *) 0);

    /* Save [h] */
    tmp4 = install_tmp (MATRIX, *h = matrix_Copy (A), matrix_Destroy);
    for (i = 2; i < m; i++)
      for (j = 0; (j < i - 1) && (j < n); j++)
	MAT0 (*h, i, j) = (double) 0.0;

    signal (SIGINT, intcatch);
    RORGHR (&n, &ilo, &ihi, MDPTRr (A), &lda, MDPTRr (tau),
	    MDPTRr (work), &lwork, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK DORGHR", (char *) 0);

    /* Assign pointer */
    *q = A;

    /* clean up */
    remove_tmp (tmp1);
    remove_tmp_destroy (tmp2);
    remove_tmp_destroy (tmp3);
    remove_tmp (tmp4);
  }
}

static void
matrix_Hess_Complex (M, q, h)
     Matrix *M, **q, **h;
{
  ASSERT (M);
  {
    int i, j;
    F_INT ilo, ihi, lda, m, n, info, lwork;
    Matrix *A, *tau, *work;

    m = MNR (M);
    n = MNC (M);
    lda = m;
    lwork = n;
    ilo = 1;
    ihi = n;

    /* Set up work arrays */
    tmp1 = install_tmp (MATRIX, A = matrix_Copy (M), matrix_Destroy);
    tmp2 = install_tmp (MATRIX, tau = matrix_CreateC (1, n), 
			matrix_Destroy);
    tmp3 = install_tmp (MATRIX, work = matrix_CreateC (1, lwork), 
			matrix_Destroy);

    signal (SIGINT, intcatch);
    XGEHRD (&n, &ilo, &ihi, MDPTRc (A), &lda, MDPTRc (tau),
	    MDPTRc (work), &lwork, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK ZGEHRD", (char *) 0);

    /* Save [h] */
    tmp4 = install_tmp (MATRIX, *h = matrix_Copy (A), matrix_Destroy);
    for (i = 2; i < m; i++)
      for (j = 0; (j < i - 1) && (j < n); j++)
      {
	MATr0 (*h, i, j) = 0.0;
	MATi0 (*h, i, j) = 0.0;
      }

    signal (SIGINT, intcatch);
    XUNGHR (&n, &ilo, &ihi, MDPTRc (A), &lda, MDPTRc (tau),
	    MDPTRc (work), &lwork, &info);
    signal (SIGINT, intcatch_wait);

    if ((int) info < 0)
      error_1 ("bad argument to LAPACK ZORGHR", (char *) 0);

    /* Assign pointer */
    *q = A;

    /* clean up */
    remove_tmp (tmp1);
    remove_tmp_destroy (tmp2);
    remove_tmp_destroy (tmp3);
    remove_tmp (tmp4);
  }
}

/* **************************************************************
 * Balance a matrix.
 * ************************************************************** */

void
matrix_Balance (m, Ab, t)
     Matrix *m, **Ab, **t;
{
  double *scale;
  int i;
  F_INT info, nm, n, low, igh, job;
  Matrix *A, *T;

  matrix_screen_string (m);
  matrix_Detect_Inf (m);
  matrix_Detect_Nan (m);

  if (MNR (m) != MNC (m))
    error_1 (matrix_GetName (m), "input to balance must be square matrix");

  nm = (F_INT) MNR (m);
  n = nm;
  job = (F_INT) 'S';

  tmp1 = install_tmp (MATRIX, A = matrix_Copy (m), matrix_Destroy);
  tmp2 = install_tmp (D_VOID, scale = (double *) MALLOC (n * sizeof (double)),
		      free);

  /* LAPACK balance */
  if (MTYPE (m) == REAL)
  {
    signal (SIGINT, intcatch);
    RGEBAL (&job, &n, MDPTRr (A), &n, &low, &igh, scale, &info);
    signal (SIGINT, intcatch_wait);
  }
  else if (MTYPE (m) == COMPLEX)
  {
    signal (SIGINT, intcatch);
    XGEBAL (&job, &n, MDPTRc (A), &n, &low, &igh, scale, &info);
    signal (SIGINT, intcatch_wait);
  }

  if (info)
    error_1 ("error in argument to balance()", (char *) 0);

  tmp3 = install_tmp (MATRIX, T = matrix_Create ((int) nm, (int) nm),
		      matrix_Destroy);
  matrix_Zero (T);
  for (i = 0; i < n; i++)
    MAT0 (T, i, i) = scale[i];

  *Ab = A;
  *t = T;

  remove_tmp (tmp1);
  remove_tmp_destroy (tmp2);
  remove_tmp (tmp3);
}
