/* matop3.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1994  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "mem.h"
#include "matrix.h"
#include "complex.h"
#include "matop1.h"
#include "matop2.h"
#include "matop3.h"
#include "r_string.h"
#include "fi_1.h"
#include "fi_2.h"
#include "bltin.h"
#include "mathl.h"
#include "util.h"
#include <math.h>

/*
 * Define our own abs(). We need to do this, cause we need
 * and abs() that does double, as well as int.
 */

#define rabs(x) ((x) >= 0 ? (x) : -(x))

/*
 * Compute the minimum values shared by two matrices.
 */

Matrix *
matrix_2_min (m1, m2)
     Matrix *m1, *m2;
{
  ASSERT (m1);
  ASSERT (m2);
  {
    int i, size;
    Matrix *m = 0;

    /* Check sizes */

    if (MNR (m1) == MNR (m2) && MNC (m1) == MNC (m2))
    {
      if (MTYPE (m1) == REAL && MTYPE (m2) == REAL)
      {
	m = matrix_Copy (m1);
	size = MNR (m1) * MNC (m1);
	for (i = 0; i < size; i++)
	{
	  if (MATrv (m1, i) > MATrv (m2, i))
	    MATrv (m, i) = MATrv (m2, i);
	}
      }
      else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == COMPLEX)
      {
	m = matrix_Copy (m1);
	size = MNR (m1) * MNC (m1);
	for (i = 0; i < size; i++)
	{
	  if (complex_Abs (MATcv (m1, i)) > complex_Abs (MATcv (m2, i)))
	    MATcv (m, i) = MATcv (m2, i);
	}
      }
      else if (MTYPE (m1) == REAL && MTYPE (m2) == COMPLEX)
      {
	m = matrix_Copy (m2);
	size = MNR (m1) * MNC (m1);
	for (i = 0; i < size; i++)
	{
	  if (MATrv (m1, i) < complex_Abs (MATcv (m2, i)))
	  {
	    MATcvr (m, i) = MATrv (m1, i);
	    MATcvi (m, i) = 0.0;
	  }
	}
      }
      else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == REAL)
      {
	m = matrix_Copy (m1);
	size = MNR (m1) * MNC (m1);
	for (i = 0; i < size; i++)
	{
	  if (complex_Abs (MATcv (m1, i)) > MATrv (m2, i))
	  {
	    MATcvr (m, i) = MATrv (m2, i);
	    MATcvi (m, i) = 0.0;
	  }
	}
      }
    }
    else
    {
      error_2 (matrix_GetName (m1), matrix_GetName (m2),
	       "matrix sizes must match");
    }
    return (m);
  }
}

Matrix *
matrix_2_max (m1, m2)
     Matrix *m1, *m2;
{
  ASSERT (m1);
  ASSERT (m2);
  {
    int i, size;
    Matrix *m = 0;

    /* Check sizes */

    if (MNR (m1) == MNR (m2) && MNC (m1) == MNC (m2))
    {
      if (MTYPE (m1) == REAL && MTYPE (m2) == REAL)
      {
	m = matrix_Copy (m1);
	size = MNR (m1) * MNC (m1);
	for (i = 0; i < size; i++)
	{
	  if (MATrv (m1, i) < MATrv (m2, i))
	    MATrv (m, i) = MATrv (m2, i);
	}
      }
      else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == COMPLEX)
      {
	m = matrix_Copy (m1);
	size = MNR (m1) * MNC (m1);
	for (i = 0; i < size; i++)
	{
	  if (complex_Abs (MATcv (m1, i)) < complex_Abs (MATcv (m2, i)))
	    MATcv (m, i) = MATcv (m2, i);
	}
      }
      else if (MTYPE (m1) == REAL && MTYPE (m2) == COMPLEX)
      {
	m = matrix_Copy (m2);
	size = MNR (m1) * MNC (m1);
	for (i = 0; i < size; i++)
	{
	  if (MATrv (m1, i) > complex_Abs (MATcv (m2, i)))
	  {
	    MATcvr (m, i) = MATrv (m1, i);
	    MATcvi (m, i) = 0.0;
	  }
	}
      }
      else if (MTYPE (m1) == COMPLEX && MTYPE (m2) == REAL)
      {
	m = matrix_Copy (m1);
	size = MNR (m1) * MNC (m1);
	for (i = 0; i < size; i++)
	{
	  if (complex_Abs (MATcv (m1, i)) < MATrv (m2, i))
	  {
	    MATcvr (m, i) = MATrv (m2, i);
	    MATcvi (m, i) = 0.0;
	  }
	}
      }
    }
    else
    {
      error_2 (matrix_GetName (m1), matrix_GetName (m2),
	       "matrix sizes must match");
    }
    return (m);
  }
}

/*
 * Compute the P-Norm of a vector
 */

double
matrix_PNorm (m, p)
     Matrix *m;
     double p;
{
  ASSERT (m);
  {
    double pnorm = 0.0;
    double sum = 0.0;
    int i, size;

    matrix_screen_string (m);

    if (MNR (m) != 1 && MNC (m) != 1)
      error_1 (matrix_GetName (m),
	       "cannot compute P-norm of a matrix");

    if (MTYPE (m) == REAL)
    {
      size = MNR (m) * MNC (m);
      for (i = 0; i < size; i++)
	sum += errcheck (pow ((rabs (MATrv (m, i))), p), "pow");
      pnorm = pow (sum, 1.0 / p);
    }
    if (MTYPE (m) == COMPLEX)
    {
      size = MNR (m) * MNC (m);
      for (i = 0; i < size; i++)
	sum += errcheck (pow ((complex_Abs (MATcv (m, i))), p), "pow");
      pnorm = pow (sum, 1.0 / p);
    }
    return pnorm;
  }
}

/* **************************************************************
 * Look for an Inf or a NaN in a matrix, error() if you find one.
 * ************************************************************** */

void
matrix_Detect_Inf (m)
     Matrix *m;
{
  ASSERT (m);
  {
    int n = MNR (m) * MNC (m);

    if (MTYPE (m) == REAL)
    {
      if (detect_inf_r (MDPTRr (m), n))
	error_1 (matrix_GetName (m),
		 "error, contains Inf value");
    }
    else if (MTYPE (m) == COMPLEX)
    {
      if (detect_inf_c (MDPTRc (m), n))
	error_1 (matrix_GetName (m),
		 "error, contains Inf value");
    }
    else
      error_1 (matrix_GetName (m),
	       "Bad Error, string matrix in numeric context ?");
  }
}

void
matrix_Detect_Nan (m)
     Matrix *m;
{
  ASSERT (m);
  {
    int n = MNR (m) * MNC (m);

    if (MTYPE (m) == REAL)
    {
      if (detect_nan_r (MDPTRr (m), n))
	error_1 (matrix_GetName (m),
		 "error, contains NaN value");
    }
    else if (MTYPE (m) == COMPLEX)
    {
      if (detect_nan_c (MDPTRc (m), n))
	error_1 (matrix_GetName (m),
		 "error, contains NaN value");
    }
    else
      error_1 (matrix_GetName (m),
	       "Bad Error, string matrix in numeric context ?");
  }
}
