/* scop1.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1993  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "scalar.h"
#include "bltin.h"
#include "mathl.h"

#include <math.h>

/* 
 * Define our own abs(). We need to do this, cause we need
 * and abs() that does double, as well as int.
 */
#define rabs(x) ((x) >= 0 ? (x) : -(x))

Scalar *
scalar_Add (s1, s2)
     Scalar *s1, *s2;
{
  ASSERT (s1);
  ASSERT (s2);
  {
    Scalar *new = scalar_CreateC (s1->val.r + s2->val.r,
				  s1->val.i + s2->val.i);
    return (new);
  }
}

Scalar *
scalar_Subtract (s1, s2)
     Scalar *s1, *s2;
{
  ASSERT (s1);
  ASSERT (s2);
  {
    Scalar *new = scalar_CreateC (s1->val.r - s2->val.r,
				  s1->val.i - s2->val.i);
    return (new);
  }
}

Scalar *
scalar_Multiply (s1, s2)
     Scalar *s1, *s2;
{
  ASSERT (s1);
  ASSERT (s2);
  {
    Scalar *new = scalar_CreateC (0.0, 0.0);

    if ((s1->val.i == 0.0) && (s2->val.i == 0.0))
      new->val.r = s1->val.r * s2->val.r;
    else
      new->val = complex_Multiply (SVALr (s1), SVALi (s1),
				   SVALr (s2), SVALi (s2));
    return (new);
  }
}

Scalar *
scalar_Divide (s1, s2)
     Scalar *s1, *s2;
{
  ASSERT (s1);
  ASSERT (s2);
  {
    Scalar *new = scalar_CreateC (0.0, 0.0);

    if ((s1->val.i == 0.0) && (s2->val.i == 0.0))
      new->val.r = s1->val.r / s2->val.r;
    else
      new->val = complex_div (SVALr (s1), SVALi (s1),
			      SVALr (s2), SVALi (s2));
    return (new);
  }
}

Scalar *
scalar_Exp (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Scalar *new = scalar_CreateC (0.0, 0.0);

    if (SVALi (s) == 0.0)	/* REAL */
      SVALr (new) = errcheck (exp (s->val.r), "exp");
    else
      SVAL (new) = complex_exp (SVALr (s), SVALi (s));
    return (new);
  }
}

/* 
 * Calculate the principal root. s1^s2
 */

Scalar *
scalar_Pow (s1, s2)

     Scalar *s1, *s2;
{
  ASSERT (s1);
  ASSERT (s2);
  {
    Scalar *new = scalar_CreateC (1.0, 0.0);

    /* s1 ^ 0 */
    if (SVALr (s2) == 0.0 && SVALi (s2) == 0.0)
      return (new);

    /* s1r ^ int (s2r) */
    else if (SVALi (s1) == 0.0 && 
	     floor (SVALr (s2)) == SVALr (s2) && SVALi (s2) == 0.0)
      SVALr (new) = errcheck (pow (s1->val.r, s2->val.r), "pow");

    /* +s1r ^ s2r */
    else if (SVALr (s1) > 0.0 && SVALi (s1) == 0.0 && SVALi (s2) == 0.0)
      SVALr (new) = errcheck (pow (s1->val.r, s2->val.r), "pow");

    /* Must do complex */
    else
      SVAL (new) = complex_Pow (SVAL (s1), SVAL (s2));
    return (new);
  }
}

/* **************************************************************
 * Change the sign of a scalar (Unary Negation)
 * ************************************************************** */
Scalar *
scalar_ChangeSign (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Scalar *new = scalar_CreateC (-SVALr (s), -SVALi (s));
    return (new);
  }
}

Scalar *
scalar_Abs (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Scalar *new = scalar_CreateC (0.0, 0.0);
    if (SVALi (s) == 0.0)
      SVALr (new) = rabs (SVALr (s));
    else
      SVALr (new) = complex_abs (SVALr (s), SVALi (s));
    return (new);
  }
}

Scalar *
scalar_Log (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Scalar *new = scalar_CreateC (0.0, 0.0);
    if (SVALi (s) == 0.0 && SVALr (s) >= 0.0)
      SVALr (new) = errcheck (log (SVALr (s)), "log");
    else
      SVAL (new) = complex_log (SVALr (s), SVALi (s));
    return (new);
  }
}

#define log10e 0.43429448190325182765

Scalar *
scalar_Log10 (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Complex ctmp;
    Scalar *new = scalar_CreateC (0.0, 0.0);

    if (SVALi (s) == 0.0 && SVALr (s) >= 0.0)
      SVALr (new) = errcheck (log10 (SVALr (s)), "log");
    else
    {
      ctmp = complex_log (SVALr (s), SVALi (s));
      SVAL (new) = complex_Multiply (log10e, 0.0, ctmp.r, ctmp.i);
    }

    return (new);
  }
}

Scalar *
scalar_Sin (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Scalar *new = scalar_CreateC (0.0, 0.0);
    if (SVALi (s) == 0.0)
      SVALr (new) = errcheck (sin (SVALr (s)), "sin");
    else
      SVAL (new) = complex_sin (SVALr (s), SVALi (s));
    return (new);
  }
}

Scalar *
scalar_Cos (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Scalar *new = scalar_CreateC (0.0, 0.0);
    if (SVALi (s) == 0.0)
      SVALr (new) = errcheck (cos (SVALr (s)), "cos");
    else
      SVAL (new) = complex_cos (SVALr (s), SVALi (s));
    return (new);
  }
}

Scalar *
scalar_Tan (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Scalar *new = scalar_CreateC (0.0, 0.0);
    if (SVALi (s) == 0.0)
      SVALr (new) = errcheck (tan (SVALr (s)), "tan");
    else
      SVAL (new) = complex_tan (SVALr (s), SVALi (s));
    return (new);
  }
}

Scalar *
scalar_Asin (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Scalar *new = scalar_CreateC (0.0, 0.0);
    if (SVALi (s) == 0.0)
    {
      if (SVALr (s) >= 1. || SVALr (s) <= -1.)
      {
	SVAL (new) = complex_Asin (SVAL (s));
      }
      else
      {
	SVALr (new) = errcheck (asin (SVALr (s)), "sin");
      }
    }
    else
    {
      SVAL (new) = complex_Asin (SVAL (s));
    }
    return (new);
  }
}

Scalar *
scalar_Acos (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Scalar *new = scalar_CreateC (0.0, 0.0);
    if (SVALi (s) == 0.0)
    {
      if (SVALr (s) >= 1. || SVALr (s) <= -1.)
      {
	SVAL (new) = complex_Acos (SVAL (s));
      }
      else
      {
	SVALr (new) = errcheck (acos (SVALr (s)), "cos");
      }
    }
    else
    {
      SVAL (new) = complex_Acos (SVAL (s));
    }
    return (new);
  }
}

Scalar *
scalar_Atan (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Scalar *new = scalar_CreateC (0.0, 0.0);
    if (SVALi (s) == 0.0)
      SVALr (new) = errcheck (atan (SVALr (s)), "atan");
    else
      SVAL (new) = complex_Atan (SVAL (s));
    return (new);
  }
}

Scalar *
scalar_Sqrt (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Scalar *new = scalar_CreateC (0.0, 0.0);
    if (s->val.i == 0.0)
    {
      if (s->val.r < 0.0)
	new->val.i = errcheck (sqrt (rabs (s->val.r)), "sqrt");
      else
	new->val.r = errcheck (sqrt (s->val.r), "sqrt");
    }
    else
      SVAL (new) = complex_sqrt (SVALr (s), SVALi (s));

    return (new);
  }
}

Scalar *
scalar_Int (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Scalar *new = scalar_CreateC (0.0, 0.0);
    new->val.r = (double) ((int) s->val.r);
    new->val.i = (double) ((int) s->val.i);
    return (new);
  }
}

Scalar *
scalar_Ceil (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Scalar *new = scalar_CreateC (0.0, 0.0);
    new->val.r = errcheck (ceil (s->val.r), "ceil");
    new->val.i = errcheck (ceil (s->val.i), "ceil");
    return (new);
  }
}

Scalar *
scalar_Floor (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Scalar *new = scalar_CreateC (0.0, 0.0);
    new->val.r = errcheck (floor (s->val.r), "floor");
    new->val.i = errcheck (floor (s->val.i), "floor");
    return (new);
  }
}

Scalar *
scalar_Round (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Scalar *new = scalar_CreateC (0.0, 0.0);
    new->val.r = errcheck (rint (s->val.r), "rint");
    new->val.i = errcheck (rint (s->val.i), "rint");
    return (new);
  }
}

Scalar *
scalar_Inv (s)
     Scalar *s;
{
  ASSERT (s);
  {
    Scalar *new;
    Scalar *tmp = scalar_CreateC (1.0, 0.0);
    new = scalar_Divide (tmp, s);
    return (new);
  }
}

Scalar *
scalar_Mod (s1, s2)
     Scalar *s1, *s2;
{
  ASSERT (s1);
  ASSERT (s2);
  {
    Scalar *new = scalar_Create (0.0);

    if (SVALi (s1) == 0.0 && SVALi (s2) == 0.0)
    {
      SVALr (new) = errcheck (fmod (SVALr (s1), SVALr (s2)), "fmod");
      SVALi (new) = 0.0;
    }
    else
    {
      SVAL (new) = complex_Mod (SVAL (s1), SVAL (s2));
    }
    return new;
  }
}

/* **************************************************************
 * SCALAR relational functions.
 * ************************************************************** */
double
scalar_eq (s1, s2)
     Scalar *s1, *s2;
{
  ASSERT (s1);
  ASSERT (s2);
  {
    if ((SVALr (s1) == SVALr (s2)) && (SVALi (s1) == SVALi (s2)))
      return (1.0);
    else
      return (0.0);
  }
}

double
scalar_ne (s1, s2)
     Scalar *s1, *s2;
{
  ASSERT (s1);
  ASSERT (s2);
  {
    if ((SVALr (s1) != SVALr (s2)) || (SVALi (s1) != SVALi (s2)))
      return (1.0);
    else
      return (0.0);
  }
}

double
scalar_lt (s1, s2)
     Scalar *s1, *s2;
{
  ASSERT (s1);
  ASSERT (s2);
  {
    if (SVALi (s1) == 0.0 && SVALi (s2) == 0.0)
      return ((double) (SVALr (s1) < SVALr (s2)));
    else
      return (complex_lt (SVALr (s1), SVALi (s1), SVALr (s2), SVALi (s2)));
  }
}

double
scalar_le (s1, s2)
     Scalar *s1, *s2;
{
  ASSERT (s1);
  ASSERT (s2);
  {
    if (SVALi (s1) == 0.0 && SVALi (s2) == 0.0)
      return ((double) (SVALr (s1) <= SVALr (s2)));
    else
      return (complex_le (SVALr (s1), SVALi (s1), SVALr (s2), SVALi (s2)));
  }
}

double
scalar_gt (s1, s2)
     Scalar *s1, *s2;
{
  ASSERT (s1);
  ASSERT (s2);
  {
    if (SVALi (s1) == 0.0 && SVALi (s2) == 0.0)
      return ((double) (SVALr (s1) > SVALr (s2)));
    else
      return (complex_gt (SVALr (s1), SVALi (s1), SVALr (s2), SVALi (s2)));
  }
}

double
scalar_ge (s1, s2)
     Scalar *s1, *s2;
{
  ASSERT (s1);
  ASSERT (s2);
  {
    if (SVALi (s1) == 0.0 && SVALi (s2) == 0.0)
      return ((double) (SVALr (s1) >= SVALr (s2)));
    else
      return (complex_ge (SVALr (s1), SVALi (s1), SVALr (s2), SVALi (s2)));
  }
}

double
scalar_and (s1, s2)
     Scalar *s1, *s2;
{
  ASSERT (s1);
  ASSERT (s2);
  {
    if (SVALi (s1) == 0.0 && SVALi (s2) == 0.0)
      return ((double) ((SVALr (s1) != 0.0) && (SVALr (s2) != 0.0)));
    else
      return (complex_and (SVALr (s1), SVALi (s1), SVALr (s2), SVALi (s2)));
  }
}

double
scalar_or (s1, s2)
     Scalar *s1, *s2;
{
  ASSERT (s1);
  ASSERT (s2);
  {
    if (SVALi (s1) == 0.0 && SVALi (s2) == 0.0)
      return ((double) ((SVALr (s1) != 0.0) || (SVALr (s2) != 0.0)));
    else
      return (complex_or (SVALr (s1), SVALi (s1), SVALr (s2), SVALi (s2)));
  }
}

double
scalar_not (s)
     Scalar *s;
{
  ASSERT (s);
  {
    double r1;
    if (SVALi (s) == 0.0)
      return ((double) (SVALr (s) == 0.0));
    else
    {
      r1 = sqrt (SVALr (s) * SVALr (s) + SVALi (s) * SVALi (s));
      return ((double) (r1 == 0.0));
    }
  }
}
