/* util.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1994  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "code.h"
#include "mem.h"
#include "list.h"
#include "bltin.h"
#include "btree.h"
#include "scalar.h"
#include "matrix.h"
#include "r_string.h"
#include "function.h"

#include <string.h>
#include <setjmp.h>

/* **************************************************************
 * Set the program name...
 * ************************************************************** */

static char *progname;

char *cpstr _PROTO ((char *string));

void
set_progname (value)
     char *value;
{
  progname = cpstr (value);
}

/* **************************************************************
 * Run-time error handling and signal catching...
 * ************************************************************** */
/*
 * These hold the env for longjmp()s back to the prompt, 
 * and error exit
 */

#define NJMP 40			/* Depth of load() recursion */
static jmp_buf jmp[NJMP];
static int ijmp;		/* Counter for tracking jmp[] */

static char *null_string = "NULL";
static char no_file[] = "no file info available";

/* **************************************************************
 * Jump buffer handling routines. These routines merely inc, or 
 * an integer, whilst checking to see that the bounds of the array 
 * have not been exceeded .
 * ************************************************************** */

int
get_ijmp ()
{
  return (ijmp);
}

int
inc_buff ()
{
  if (ijmp >= NJMP)
    fprintf (stderr, "NJMP too small\n");

  return (ijmp++);
}

int
dec_buff ()
{
  if (ijmp <= 0)
    fprintf (stderr, "error while decrementing ijmp\n");

  return (--ijmp);
}

jmp_buf *
jmp_inc_buff ()
{
  return (&jmp[inc_buff ()]);
}

jmp_buf *
jmp_dec_buff ()
{
  return (&jmp[dec_buff ()]);
}

/* **************************************************************
 * Recover from a run-time error. 1 name, 1 message
 * ************************************************************** */

void warning_1 _PROTO ((char *s, char *t));
void warning_2 _PROTO ((char *s1, char *s2, char *t));
void warning_3 _PROTO ((char *s1, char *s2, char *s3, char *t));

static int line_nos;		/* if TRUE include line numbers in code */

void
set_util_line_nos (val)
     int val;
{
  line_nos = val;
}

void
error_1 (s, t)
     char *s, *t;
{
  warning_1 (s, t);
  longjmp (jmp[dec_buff ()], 1);
}

/* **************************************************************
 * Recover from a run-time error. 2 names, 1 message.
 * ************************************************************** */
void
error_2 (s1, s2, t)
     char *s1, *s2, *t;
{
  warning_2 (s1, s2, t);
  longjmp (jmp[dec_buff ()], 1);
}

/* **************************************************************
 * Recover from a run-time error. 3 names, 1 message.
 * ************************************************************** */
void
error_3 (s1, s2, s3, t)
     char *s1, *s2, *s3, *t;
{
  warning_3 (s1, s2, s3, t);
  longjmp (jmp[dec_buff ()], 1);
}

/* **************************************************************
 * Print warning messages. We are not worried about speed, since
 * an error has occurred. We must check the string arguments,
 * because under some circumstances they may be NULL. If the input
 * is not coming from the command line (stdin) then print out
 * file and line number information.
 * ************************************************************** */
void
warning_1 (s, t)
     char *s, *t;
{
  char *fn;
  int lineno, write_diary;

  write_diary = get_write_diary ();

  /* 
   * Check s, if errors cascade, it is possible that s may be NULL.
   * Or s may be something obscure, like an argument tag.
   */

  if (s == 0)
  {
    s = null_string;
  }
  else if (!strncmp (s, "-", 1))
  {
    s = null_string;
  }

  /* Print file and line info if available */
  if (line_nos)
  {
    fn = find_file_name ();
    lineno = find_lineno ();
  }
  else
  {
    fn = no_file;
    lineno = 0;
  }
  if (write_diary)
  {
    FILE *diary_file_ptr = get_diary_file_ptr ();
    fprintf (diary_file_ptr, "%s: %s", progname, s);
    if (t)
      fprintf (diary_file_ptr, ", %s", t);
    fprintf (diary_file_ptr, "\n");

    if (strcmp ("stdin", fn))
      fprintf (diary_file_ptr, "near line %d, file: %s\n",
	       lineno, fn);
    fflush (diary_file_ptr);
  }

  fprintf (stderr, "%s: %s", progname, s);

  if (t)
    fprintf (stderr, ", %s", t);
  fprintf (stderr, "\n");

  if (strcmp ("stdin", fn))
    fprintf (stderr, "near line %d, file: %s\n", lineno, fn);
  fflush (stderr);
}

void
warning_2 (s1, s2, t)
     char *s1, *s2, *t;
{
  char *fn;
  int lineno, write_diary;

  write_diary = get_write_diary ();

  /* 
   * Check s, if errors cascade, it is possible that s may be NULL.
   * Or s may be something obscure, like an argument tag.
   */
  if (s1 == 0)
    s1 = null_string;
  else if (!strncmp (s1, "-", 1))
    s1 = null_string;
  if (s2 == 0)
    s2 = null_string;
  else if (!strncmp (s2, "-", 1))
    s2 = null_string;

  /* Print file and line info if available */
  if (line_nos)
  {
    fn = find_file_name ();
    lineno = find_lineno ();
  }
  else
  {
    fn = no_file;
    lineno = 0;
  }
  if (write_diary)
  {
    FILE *diary_file_ptr = get_diary_file_ptr ();
    fprintf (diary_file_ptr, "%s: %s, %s", progname, s1, s2);
    if (t)
      fprintf (diary_file_ptr, ", %s", t);
    fprintf (diary_file_ptr, "\n");

    if (strcmp ("stdin", fn))
      fprintf (diary_file_ptr, "near line %d, file: %s\n", lineno, fn);

    fflush (diary_file_ptr);
  }

  fprintf (stderr, "%s: %s, %s", progname, s1, s2);

  if (t)
    fprintf (stderr, ", %s", t);
  fprintf (stderr, "\n");

  if (strcmp ("stdin", fn))
    fprintf (stderr, "near line %d, file: %s\n", lineno, fn);
  fflush (stderr);
}

void
warning_3 (s1, s2, s3, t)
     char *s1, *s2, *s3, *t;
{
  char *fn;
  int lineno, write_diary;

  write_diary = get_write_diary ();

  /* 
   * Check s, if errors cascade, it is possible that s may be NULL.
   * Or s may be something obscure, like an argument tag.
   */
  if (s1 == 0)
    s1 = null_string;
  else if (!strncmp (s1, "-", 1))
    s1 = null_string;
  if (s2 == 0)
    s2 = null_string;
  else if (!strncmp (s2, "-", 1))
    s2 = null_string;
  if (s3 == 0)
    s3 = null_string;
  else if (!strncmp (s3, "-", 1))
    s3 = null_string;

  /* Print file and line info if available */
  if (line_nos)
  {
    fn = find_file_name ();
    lineno = find_lineno ();
  }
  else
  {
    fn = no_file;
    lineno = 0;
  }
  if (write_diary)
  {
    FILE *diary_file_ptr = get_diary_file_ptr ();
    fprintf (diary_file_ptr, "%s: %s, %s, %s", progname, s1, s2, s3);
    if (t)
      fprintf (diary_file_ptr, ", %s", t);
    fprintf (diary_file_ptr, "\n");

    if (strcmp ("stdin", fn))
      fprintf (diary_file_ptr, "near line %d, file: %s\n", lineno, fn);

    fflush (diary_file_ptr);
  }

  fprintf (stderr, "%s: %s, %s, %s", progname, s1, s2, s3);

  if (t)
    fprintf (stderr, ", %s", t);
  fprintf (stderr, "\n");

  if (strcmp ("stdin", fn))
    fprintf (stderr, "near line %d, file: %s\n", lineno, fn);
  fflush (stderr);
}

/* **************************************************************
 * Signal catching functions.
 * ************************************************************** */
void
fpecatch (tmp)
     int tmp;
{
#ifdef __EMX__
  /* to re-enable signals under EMX */
  signal (SIGFPE, SIG_ACK);
#endif

  signal (SIGFPE, SIG_IGN);
  error_1 ("floating point exception", 0);
}

void
pipecatch (tmp)
     int tmp;
{
#ifdef __EMX__
  /* to re-enable signals under EMX */
  signal (SIGPIPE, SIG_ACK);
#endif

  signal (SIGPIPE, SIG_IGN);
  /* Don't print anything, just jump */
  longjmp (jmp[dec_buff ()], 1);
}

/* **************************************************************
 * Copy a string to a new char ptr, Return the ptr to the newly
 * created string.
 * ************************************************************** */
char *
cpstr (string)
     char *string;
{
  char *new_string;
  if (string != 0)
  {
    new_string = (char *) MALLOC ((size_t) (strlen (string) + 1));
    strcpy (new_string, string);
    return (new_string);
  }
  return (0);
}

/* **************************************************************
 * Copy a string to a new char *, strip the surrounding "".
 * This function is primarily used by the RLaB scanner.
 * ************************************************************** */
char *
cpstr_strip (string)
     char *string;
{
  int len;
  char *new_string;
  if (string != 0)
  {
    len = strlen (string);
    string[len - 1] = '\0';	/* get rid of trailing " */
    string++;			/* get rid of 1st " */

    new_string = (char *) MALLOC ((size_t) (len - 1));

    strcpy (new_string, string);
    return (new_string);
  }
  return (0);
}

/* **************************************************************
 * Get a double value from a Datum.
 * ************************************************************** */
double
get_datum_value (datum, s)
     Datum datum;
     char *s;
{
  double d = 0.0;
  Matrix *m;

  switch (datum.type)
  {
  case CONSTANT:
    d = datum.u.val;
    break;

  case iCONSTANT:
    d = datum.u.val;
    break;

  case ENTITY:
    switch (e_type (datum.u.ent))
    {
    case SCALAR:
      d = SVALr (e_data (datum.u.ent));
      break;

    case MATRIX:
      m = (Matrix *) e_data (datum.u.ent);
      if (MNR (m) == 1 && MNC (m) == 1)
      {
	if (MTYPE (m) == REAL)
	  d = MAT (m, 1, 1);
	else if (MTYPE (m) == COMPLEX)
	  d = MATr (m, 1, 1);
	else
	  error_1 (s, 0);
      }
      else
	error_1 (s, 0);
      break;

    case UNDEF:
      error_1 (s, "UNDEFINED");
      break;

    default:
      error_1 (s, 0);
      break;
    }
    break;
  }
  return (d);
}

Datum
convert_const (d)
     Datum d;
{
  if (d.type == CONSTANT)
  {
    d.u.ent = install_tmp (SCALAR, scalar_CreateC (d.u.val, 0.0),
			   scalar_Destroy);
    d.type = ENTITY;
  }
  else if (d.type == iCONSTANT)
  {
    d.u.ent = install_tmp (SCALAR, scalar_CreateC (0.0, d.u.val),
			   scalar_Destroy);
    d.type = ENTITY;
  }
  return (d);
}

/* **************************************************************
 * Convert a Datum to a SCALAR entity if appropriate.
 * Note: this function is not really named properly, since it
 * will return a matrix, without an error. It could almost be
 * called convert_to_entity, but it balks on certain types of
 * entities.
 * ************************************************************** */

Datum
convert_to_scalar (d)
     Datum d;
{
  switch (d.type)
  {

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
    case MATRIX:
    case STRING:
      return (d);

    case BTREE:
      error_1 (e_name (d.u.ent), "is a LIST, cannot convert to SCALAR");

    case LIST:
      error_1 (e_name (d.u.ent), "is a LIST, cannot convert to SCALAR");

    case U_FUNCTION:
      error_1 (e_name (d.u.ent),
	       "is a USER-FUNCTION, cannot convert to SCALAR");

    case BLTIN:
      error_1 (e_name (d.u.ent),
	       "is a BUILT-IN function, cannot convert to SCALAR");

    case UNDEF:
      error_1 (e_name (d.u.ent), "UNDEFINED");

    default:
      return (d);
      break;
    }
    break;

  case CONSTANT:
    d.u.ent = install_tmp (SCALAR, scalar_CreateC (d.u.val, 0.0),
			   scalar_Destroy);
    d.type = ENTITY;
    return (d);

  case iCONSTANT:
    d.u.ent = install_tmp (SCALAR, scalar_CreateC (0.0, d.u.val),
			   scalar_Destroy);
    d.type = ENTITY;
    return (d);

  default:
    error_1 (e_name (d.u.ent), "cannot convert to SCALAR");
    break;
  }

  return (d);
}

/* **************************************************************
 * Get the numeric scalar value of an ENTITY if appropriate.
 * CONSTANT, iCONSTANT, SCALAR, MATRIX (1x1) all yield a
 * simple numeric value. The imaginary part of a complex
 * number is ignored. This function used primarily for dealing
 * with  builtin function arguments.
 * ************************************************************** */

double
get_num_scalar_val (d)
     Datum d;
{
  double retd = 0.0;
  Matrix *m;

  switch (d.type)
  {
  case CONSTANT:
    retd = d.u.val;
    break;

  case iCONSTANT:
    retd = 0.0;
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      retd = SVALr (e_data (d.u.ent));
      break;

    case MATRIX:
      m = (Matrix *) e_data (d.u.ent);
      if (MNR (m) == 1 && MNC (m) == 1)
      {
	if (MTYPE (m) == REAL)
	  retd = MAT (m, 1, 1);
	else if (MTYPE (m) == COMPLEX)
	  retd = MATr (m, 1, 1);
	else
	  error_1 (e_name (d.u.ent),
		   "String matrix not allowed in this context");
      }
      else
	error_1 (e_name (d.u.ent), "Matrix dim must be 1x1 in this context");
      break;

    case UNDEF:
      error_1 (e_name (d.u.ent), "UNDEFINED");

    default:
      error_1 (e_name (d.u.ent), "Invalid type in this context");

    }
  }
  return (retd);
}

/*
 * Convert anything to an entity.
 */

Datum
convert_to_matrix_entity (d)
     Datum d;
{
  Datum new;
  Matrix *m;

  switch (d.type)
  {
  case CONSTANT:
    m = matrix_Create (1, 1);
    MAT (m, 1, 1) = d.u.val;
    matrix_SetName (m, cpstr ("CONSTANT"));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  case iCONSTANT:
    m = matrix_CreateC (1, 1);
    matrix_Zero (m);
    MATi (m, 1, 1) = d.u.val;
    matrix_SetName (m, cpstr ("iCONSTANT"));
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      if (SVALi (e_data (d.u.ent)) == 0.0)
      {
	m = matrix_Create (1, 1);
	MAT (m, 1, 1) = SVALr (e_data (d.u.ent));
      }
      else
      {
	m = matrix_CreateC (1, 1);
	MATr (m, 1, 1) = SVALr (e_data (d.u.ent));
	MATi (m, 1, 1) = SVALi (e_data (d.u.ent));
      }
      matrix_SetName (m, cpstr (scalar_GetName (e_data (d.u.ent))));
      new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
      new.type = ENTITY;
      break;

    case MATRIX:
      new = d;
      break;

    case STRING:
      m = matrix_CreateS (1, 1);
      MATs (m, 1, 1) = cpstr (string_GetString (e_data (d.u.ent)));
      matrix_SetName (m, cpstr (string_GetName (e_data (d.u.ent))));
      new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
      new.type = ENTITY;
      break;

    case BTREE:
    case LIST:
    case U_FUNCTION:
    case BLTIN:
    case UNDEF:
      new = d;
      break;
    }
    break;
  }
  return (new);
}

/* **************************************************************
 * Convert a Datum to a MATRIX entity if appropriate. This function
 * should probably convert a string to a matrix (according to the
 * function name). But, it does'nt, and it should'nt.
 *
 * Note that the original datum is not destroyed.
 * ************************************************************** */

Datum
convert_to_matrix (d)
     Datum d;
{
  Datum new;
  Matrix *m;

  switch (d.type)
  {
  case CONSTANT:
    m = matrix_Create (1, 1);
    MAT (m, 1, 1) = d.u.val;
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  case iCONSTANT:
    m = matrix_CreateC (1, 1);
    matrix_Zero (m);
    MATi (m, 1, 1) = d.u.val;
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      if (SVALi (e_data (d.u.ent)) == 0.0)
      {
	m = matrix_Create (1, 1);
	MAT (m, 1, 1) = SVALr (e_data (d.u.ent));
      }
      else
      {
	m = matrix_CreateC (1, 1);
	MATr (m, 1, 1) = SVALr (e_data (d.u.ent));
	MATi (m, 1, 1) = SVALi (e_data (d.u.ent));
      }
      new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
      new.type = ENTITY;
      break;

    case MATRIX:
      return (d);
      break;

    case STRING:
      error_1 (e_name (d.u.ent), "is a STRING, cannot convert to numeric MATRIX");
    case BTREE:
      error_1 (e_name (d.u.ent), "cannot convert a LIST to a MATRIX");
    case LIST:
      error_1 (e_name (d.u.ent), "cannot convert a LIST to a MATRIX");
    case U_FUNCTION:
      error_1 (e_name (d.u.ent),
	       "cannot convert a USER-FUNCTION to a MATRIX");
    case BLTIN:
      error_1 (e_name (d.u.ent),
	       "cannot convert a BLTIN-FUNCTION to a MATRIX");
    case UNDEF:
      error_1 (e_name (d.u.ent), "UNDEFINED");

    default:
      error_1 (e_name (d.u.ent), "cannot convert to MATRIX");
    }
  }
  return (new);
}

/*
 * Same as the previous function, but this function DOES
 * destroy the input datum if necessary.
 */

Datum
convert_to_matrix_d (d)
     Datum d;
{
  Datum new;
  Matrix *m;

  switch (d.type)
  {
  case CONSTANT:
    m = matrix_Create (1, 1);
    MAT (m, 1, 1) = d.u.val;
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  case iCONSTANT:
    m = matrix_CreateC (1, 1);
    matrix_Zero (m);
    MATi (m, 1, 1) = d.u.val;
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      if (SVALi (e_data (d.u.ent)) == 0.0)
      {
	m = matrix_Create (1, 1);
	MAT (m, 1, 1) = SVALr (e_data (d.u.ent));
      }
      else
      {
	m = matrix_CreateC (1, 1);
	MATr (m, 1, 1) = SVALr (e_data (d.u.ent));
	MATi (m, 1, 1) = SVALi (e_data (d.u.ent));
      }
      remove_tmp_destroy (d.u.ent);
      new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
      new.type = ENTITY;
      break;

    case MATRIX:
      return (d);
      break;

    case STRING:
      error_1 (e_name (d.u.ent), "is a STRING, cannot convert to numeric MATRIX");
    case BTREE:
      error_1 (e_name (d.u.ent), "cannot convert a LIST to a MATRIX");
    case LIST:
      error_1 (e_name (d.u.ent), "cannot convert a LIST to a MATRIX");
    case U_FUNCTION:
      error_1 (e_name (d.u.ent),
	       "cannot convert a USER-FUNCTION to a MATRIX");
    case BLTIN:
      error_1 (e_name (d.u.ent),
	       "cannot convert a BLTIN-FUNCTION to a MATRIX");
    case UNDEF:
      error_1 (e_name (d.u.ent), "UNDEFINED");

    default:
      error_1 (e_name (d.u.ent), "cannot convert to MATRIX");
    }
  }
  return (new);
}

/* **************************************************************
 * This version does convert a string to a string matrix!
 * ************************************************************** */

Datum
convert_all_to_matrix (d)
     Datum d;
{
  Datum new;
  Matrix *m;

  switch (d.type)
  {
  case CONSTANT:
    m = matrix_Create (1, 1);
    MAT (m, 1, 1) = d.u.val;
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  case iCONSTANT:
    m = matrix_CreateC (1, 1);
    matrix_Zero (m);
    MATi (m, 1, 1) = d.u.val;
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      if (SVALi (e_data (d.u.ent)) == 0.0)
      {
	m = matrix_Create (1, 1);
	MAT (m, 1, 1) = SVALr (e_data (d.u.ent));
      }
      else
      {
	m = matrix_CreateC (1, 1);
	MATr (m, 1, 1) = SVALr (e_data (d.u.ent));
	MATi (m, 1, 1) = SVALi (e_data (d.u.ent));
      }
      new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
      new.type = ENTITY;
      break;

    case MATRIX:
      return (d);
      break;

    case STRING:
      m = matrix_CreateS (1, 1);
      MATs (m, 1, 1) = cpstr (string_GetString (e_data (d.u.ent)));
      new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
      new.type = ENTITY;
      break;

    case BTREE:
      error_1 (e_name (d.u.ent), "cannot convert a LIST to a MATRIX");
    case LIST:
      error_1 (e_name (d.u.ent), "cannot convert a LIST to a MATRIX");
    case U_FUNCTION:
      error_1 (e_name (d.u.ent),
	       "cannot convert a USER-FUNCTION to a MATRIX");
    case BLTIN:
      error_1 (e_name (d.u.ent),
	       "cannot convert a BLTIN-FUNCTION to a MATRIX");
    case UNDEF:
      error_1 (e_name (d.u.ent), "UNDEFINED");

    default:
      error_1 (e_name (d.u.ent), "cannot convert to MATRIX");
    }
  }
  return (new);
}

/*
 * Same as above, but destroys the datum if necessary.
 */

Datum
convert_all_to_matrix_d (d)
     Datum d;
{
  Datum new;
  Matrix *m;

  switch (d.type)
  {
  case CONSTANT:
    m = matrix_Create (1, 1);
    MAT (m, 1, 1) = d.u.val;
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  case iCONSTANT:
    m = matrix_CreateC (1, 1);
    matrix_Zero (m);
    MATi (m, 1, 1) = d.u.val;
    new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    new.type = ENTITY;
    break;

  case ENTITY:
    switch (e_type (d.u.ent))
    {
    case SCALAR:
      if (SVALi (e_data (d.u.ent)) == 0.0)
      {
	m = matrix_Create (1, 1);
	MAT (m, 1, 1) = SVALr (e_data (d.u.ent));
      }
      else
      {
	m = matrix_CreateC (1, 1);
	MATr (m, 1, 1) = SVALr (e_data (d.u.ent));
	MATi (m, 1, 1) = SVALi (e_data (d.u.ent));
      }
      remove_tmp_destroy (d.u.ent);
      new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
      new.type = ENTITY;
      break;

    case MATRIX:
      return (d);
      break;

    case STRING:
      m = matrix_CreateS (1, 1);
      MATs (m, 1, 1) = cpstr (string_GetString (e_data (d.u.ent)));
      remove_tmp_destroy (d.u.ent);
      new.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
      new.type = ENTITY;
      break;

    case BTREE:
      error_1 (e_name (d.u.ent), "cannot convert a LIST to a MATRIX");
    case LIST:
      error_1 (e_name (d.u.ent), "cannot convert a LIST to a MATRIX");
    case U_FUNCTION:
      error_1 (e_name (d.u.ent),
	       "cannot convert a USER-FUNCTION to a MATRIX");
    case BLTIN:
      error_1 (e_name (d.u.ent),
	       "cannot convert a BLTIN-FUNCTION to a MATRIX");
    case UNDEF:
      error_1 (e_name (d.u.ent), "UNDEFINED");

    default:
      error_1 (e_name (d.u.ent), "cannot convert to MATRIX");
    }
  }
  return (new);
}

Datum
convert_to_rhs_matrix (d)
     Datum d;
{
  Matrix *m;

  if (d.type == CONSTANT)
  {
    m = matrix_Create (1, 1);
    MAT (m, 1, 1) = d.u.val;
    d.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    d.type = ENTITY;
    return (d);
  }
  else if (d.type == iCONSTANT)
  {
    m = matrix_CreateC (1, 1);
    matrix_Zero (m);
    MATi (m, 1, 1) = d.u.val;
    d.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
    d.type = ENTITY;
    return (d);
  }
  else if (d.type == ENTITY)
  {
    if (e_type (d.u.ent) == SCALAR)
    {
      if (SVALi (e_data (d.u.ent)) == 0.0)
      {
	m = matrix_Create (1, 1);
	MAT (m, 1, 1) = SVALr (e_data (d.u.ent));
      }
      else
      {
	m = matrix_CreateC (1, 1);
	MATr (m, 1, 1) = SVALr (e_data (d.u.ent));
	MATi (m, 1, 1) = SVALi (e_data (d.u.ent));
      }
      remove_tmp_destroy (d.u.ent);
      d.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
      d.type = ENTITY;
      return (d);
    }
    else if (e_type (d.u.ent) == MATRIX)
      return (d);
    else if (e_type (d.u.ent) == STRING)
    {
      m = matrix_CreateS (1, 1);
      MATs (m, 1, 1) = cpstr (string_GetString (e_data (d.u.ent)));
      remove_tmp_destroy (d.u.ent);
      d.u.ent = install_tmp (MATRIX, m, matrix_Destroy);
      d.type = ENTITY;
      return (d);
    }
    else if (e_type (d.u.ent) == BTREE)
      error_1 (e_name (d.u.ent), "is a LIST, cannot convert to RHS MATRIX");
    else if (e_type (d.u.ent) == LIST)
      error_1 (e_name (d.u.ent), "is a LIST, cannot convert to RHS MATRIX");
    else if (e_type (d.u.ent) == U_FUNCTION)
      error_1 (e_name (d.u.ent),
	       "is a USER-FUNCTION, cannot convert to RHS MATRIX");
    else if (e_type (d.u.ent) == BLTIN)
      error_1 (e_name (d.u.ent),
	       "is a BUILT-IN function, cannot convert to RHS MATRIX");
    else if (e_type (d.u.ent) == UNDEF)
      error_1 (e_name (d.u.ent), "UNDEFINED");
    else
      return (d);
  }
  else
    error_1 (e_name (d.u.ent), "cannot convert to RHS MATRIX");

  return (d);
}

/* **************************************************************
 * Special function for creating string matrices for who(), what().
 * Each variant gets certain types of nodes from the TREE.
 * ************************************************************** */

#define N_WORD     5		/* Number of columns to print for what(), and who(). */
static int count = 0;

/*
 * Get a BTREEs element or member names.
 * The strings are copied in this function, so
 * the return'ed object is the caller's.
 */

static char **
btree_get_node_names (node, names)
     ListNode *node;
     char **names;
{
  if (node != 0)
  {
    names = btree_get_node_names (node->prev, names);
    if (listNode_GetType (node) != UNDEF)
    {
      *(names++) = cpstr (listNode_GetKey (node));
      count++;
    }
    names = btree_get_node_names (node->next, names);
  }
  return (names);
}

/* Special print functions for RLaB */

/*
 * Print out the ListNodes for the what() function. Only print
 * nodes with BLTIN, and U_FUNCTION types. Do not print names
 * that begin with a "_".
 */

static char **
btree_get_node_names_what (node, names)
     ListNode *node;
     char **names;
{
  if (node != 0)
  {
    names = btree_get_node_names_what (node->prev, names);
    if (((listNode_GetType (node) == BLTIN)
	 || listNode_GetType (node) == U_FUNCTION)
	&& strncmp (listNode_GetKey (node), "_", 1))
    {
      *(names++) = cpstr (listNode_GetKey (node));
      count++;
    }
    names = btree_get_node_names_what (node->next, names);
  }
  return (names);
}

/*
 * Print out the ListNodes for the who() functio.
 * Only print nodes with data types, not UNDEF, or
 * variable names beginning with a "_".
 */

static char **
btree_get_node_names_who (node, names)
     ListNode *node;
     char **names;
{
  if (node != 0)
  {
    names = btree_get_node_names_who (node->prev, names);
    if ((listNode_GetType (node) != BLTIN)
	&& listNode_GetType (node) != U_FUNCTION
	&& listNode_GetType (node) != UNDEF
	&& strncmp (listNode_GetKey (node), "_", 1))
    {
      *(names++) = cpstr (listNode_GetKey (node));
      count++;
    }
    names = btree_get_node_names_who (node->next, names);
  }
  return (names);
}

Matrix *
btree_Print_What (root)
     Btree *root;
{
  int i, nrow, ncol;
  char **names;
  Matrix *m;

  count = 0;
  names = (char **) MALLOC ((root->numNodes) * sizeof (char *));

  btree_get_node_names_what (root->root_node, names);

  /* Divide # of elements by number of columns we want */
  nrow = count / N_WORD;
  ncol = N_WORD;

  if (count <= ncol)
    nrow = 1;
  else if (count % ncol != 0)
    nrow = nrow + 1;

  m = matrix_CreateS (nrow, ncol);

  /* Now load names into the matrix */
  for (i = 0; i < count; i++)
    MATsv (m, i) = names[i];

  for (i = count; i < nrow * ncol; i++)
    MATsv (m, i) = cpstr ("");

  FREE (names);
  return (m);
}

Matrix *
btree_Print_Who (root)
     Btree *root;
{
  int i, nrow, ncol;
  char **names;
  Matrix *m;

  count = 0;
  names = (char **) MALLOC ((root->numNodes) * sizeof (char *));

  btree_get_node_names_who (root->root_node, names);

  nrow = count / N_WORD;
  ncol = N_WORD;

  if (count <= ncol)
    nrow = 1;
  else if (count % ncol != 0)
    nrow = nrow + 1;

  m = matrix_CreateS (nrow, ncol);

  /* Now load names into the matrix */
  for (i = 0; i < count; i++)
    MATsv (m, i) = names[i];

  for (i = count; i < nrow * ncol; i++)
    MATsv (m, i) = cpstr ("");

  FREE (names);
  return (m);
}

/*
 * Return a row matrix of the members in a BTREE.
 */

Matrix *
btree_members (root)
     Btree *root;
{
  int i;
  char **names;
  Matrix *m;

  count = 0;
  names = (char **) MALLOC ((root->numNodes) * sizeof (char *));

  btree_get_node_names (root->root_node, names);

  m = matrix_CreateS (1, count);

  /* Now load names into the matrix */
  for (i = 0; i < count; i++)
    MATsv (m, i) = names[i];
  
  FREE (names);
  return (m);
}

/*
 * Create an integer MxN matrix for Sort()
 */

Matrix *
matrix_CreateFillSind (nrow, ncol)
     int nrow, ncol;
{
  int i, j;
  Matrix *new = matrix_Create (nrow, ncol);

  for (i = 1; i <= nrow; i++)
    for (j = 1; j <= ncol; j++)
      MAT (new, i, j) = (double) i;

  return (new);
}

/* **************************************************************
 * Call a RLaB User-Function from C source code. The arguments are:
 * fname: A null-terminated character string containing the 
 *        User-Function name.
 * args: A pointer to an array of the function arguments.
 * nargs: The number of arguments.
 *
 * call_rlab_script returns a void pointer that points to the
 * entity returned by the User-Function.
 * ************************************************************** */

extern void extern_push _PROTO ((Datum d));
extern Datum extern_pop _PROTO ((void));
extern void userf _PROTO ((ListNode * sp, int nargs, int self));
extern void bltin _PROTO ((ListNode * sp, int nargs, int popf));

VPTR
call_rlab_script (fname, args, nargs)
     char *fname;
     Datum *args;
     int nargs;
{
  int i;
  VPTR ret;
  Datum ret_datum;
  Function *func;
  ListNode *efunc;

  /* Check to see if fname exists */
  if (!(efunc = btree_FindNode (get_symtab_ptr (), fname)))
    error_1 (fname, "does not exist on symbol table");

  /* Error-check the function */
  if (e_type (efunc) != U_FUNCTION && e_type (efunc) != BLTIN)
    error_1 (fname, "must be class \"function\"");

  func = (Function *) e_data (efunc);

  /*
   * Set up the stack and code arrays.
   */

  for (i = 0; i < nargs; i++)
  {
    extern_push (args[i]);
  }

  /*
   * Execute the User/Bltin Function code.
   * Do not increment the program counter (pc)
   * cause we are NOT calling this function
   * from an environment that is executing
   * other code.
   */

  if (e_type (efunc) == BLTIN)
  {
    bltin (efunc, nargs, 0);
  }
  else
  {
    userf (efunc, nargs, 0);
  }

  /* Pop the return entity */
  ret_datum = extern_pop ();

  /*
   * Now check the returned object.
   * If it is on the tmp-list, get it off.
   * Else, convert it to a matrix and return it.
   */

  if (ret_datum.type == ENTITY && listNode_GetKey (ret_datum.u.ent) == 0)
  {
    ret = remove_tmp (ret_datum.u.ent);
    return (ret);
  }
  else
  {
    ret_datum = convert_all_to_matrix (ret_datum);
    return ((VPTR) listNode_GetData (ret_datum.u.ent));
  }
}
