/* function.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1994  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "code.h"
#include "function.h"
#include "util.h"
#include "mem.h"
#include "symbol.h"

/* **************************************************************
 * Create an RLaB Function entity.
 * ************************************************************** */
Function *
function_Create ()
{
  Function *new = (Function *) MALLOC (sizeof (Function));

  new->type = U_FUNCTION;
  new->name = 0;

  new->n_args = 0;
  new->args = 0;

  new->n_local = 0;
  new->local = 0;

  new->n_global = 0;
  new->global = 0;

  new->ncode = 0;
  new->code = 0;

  return (new);
}

/* **************************************************************
 * Destroy an RLaB Function entity.
 * ************************************************************** */

void
function_Destroy (f)
     Function *f;
{
  f->type = U_FUNCTION;
  FREE (f->name);

  if (f->args)
    list_Destroy (f->args);
  f->args = 0;
  f->n_args = 0;

  if (f->local)
    list_Destroy (f->local);
  f->n_local = 0;
  f->local = 0;

  if (f->global)
    list_Destroy (f->global);
  f->n_global = 0;
  f->global = 0;

  f->ncode = 0;
  FREE (f->code);
  FREE (f);
}

Function *
function_Copy (f)
     Function *f;
{
  ASSERT (f);
  {
    char *key;
    int i;
    Function *new = function_Create ();

    /* Create/Copy the argument list */
    new->n_args = f->n_args;
    new->args = list_Create ();
    if (new->n_args > 0)
    {
      for (i = 1; i <= f->n_args; i++)
      {
	key = listNode_GetKey (list_GetNodeByPos (f->args, f->n_args - i + 1));
	arg_var_push (new->args, cpstr (key));
      }
    }

    /* Create/Copy the local var list */
    new->n_local = f->n_local;
    new->local = list_Create ();
    if (new->n_local > 0)
    {
      for (i = 1; i <= f->n_local; i++)
      {
	key = listNode_GetKey (list_GetNodeByPos (f->local, f->n_local - i + 1));
	local_var_push (new->local, cpstr (key));
      }
    }

    /* Create/Copy the global var list */
    new->n_global = f->n_global;
    new->global = list_Create ();
    if (new->n_global > 0)
    {
      for (i = 1; i <= f->n_global; i++)
      {
	key = listNode_GetKey (list_GetNodeByPos (f->global, f->n_global - i + 1));
	local_var_push (new->global, cpstr (key));
      }
    }

    new->ncode = f->ncode;
    new->code = (Inst *) MALLOC (sizeof (Inst) * (new->ncode + 1));
    /* Copy the compliled code */
    for (i = 0; i < f->ncode; i++)
      new->code[i] = f->code[i];

    return (new);
  }
}

/*--------------------------------------------------------------------*/

char *
function_GetName (f)
     Function *f;
{
  ASSERT (f);
  {
    return (f->name);
  }
}

void
function_SetName (f, name)
     Function *f;
     char *name;
{
  ASSERT (f);
  {
    FREE (f->name);
    f->name = name;
  }
}

/*--------------------------------------------------------------------*/

int
function_HasLocalVar (f)
     Function *f;
{
  if (f->n_local > 0)
    return (1);
  else
    return (0);
}

/* **************************************************************
 * Set the function code pointer
 * ************************************************************** */
void
function_SetCodePtr (f, ptr)
     Function *f;
     Inst *ptr;
{
  f->code = ptr;
}

Inst *
function_GetCodePtr (f)
     Function *f;
{
  return (f->code);
}

int
function_SetCodeSize (f, size)
     Function *f;
     int size;
{
  ASSERT (f);
  {
    f->code = (Inst *) MALLOC (size * (sizeof (Inst)));
    f->ncode = size;
    return (1);
  }
}

int
function_GetCodeSize (f)
     Function *f;
{
  ASSERT (f);
  {
    return (f->ncode);
  }
}

/*--------------------------------------------------------------------*/

void
function_SetArgPtr (f, arg_list)
     Function *f;
     List *arg_list;
{
  f->args = arg_list;
}

void
function_SetLocalPtr (f, local_var_list)
     Function *f;
     List *local_var_list;
{
  f->local = local_var_list;
}

void
function_SetGlobalPtr (f, global_var_list)
     Function *f;
     List *global_var_list;
{
  f->global = global_var_list;
}

/*--------------------------------------------------------------------*/

List *
function_GetArgPtr (f)
     Function *f;
{
  return (f->args);
}

List *
function_GetLocalPtr (f)
     Function *f;
{
  return (f->local);
}

List *
function_GetGlobalPtr (f)
     Function *f;
{
  return (f->global);
}

/*--------------------------------------------------------------------*/

int
function_GetNlocal (f)
     Function *f;
{
  return (f->n_local);
}

int
function_GetNglobal (f)
     Function *f;
{
  return (f->n_global);
}

/*--------------------------------------------------------------------*/

void
function_SetNargs (f, n_args)
     Function *f;
     int n_args;
{
  f->n_args = n_args;
}

int
function_GetNargs (f)
     Function *f;
{
  return (f->n_args);
}

void
function_SetNlocal (f, n_local)
     Function *f;
     int n_local;
{
  f->n_local = n_local;
}

void
function_SetNglobal (f, n_global)
     Function *f;
     int n_global;
{
  f->n_global = n_global;
}

/* **************************************************************
 * Create a local variable object.
 * ************************************************************** */
LVar *
lvar_Create ()
{
  LVar *new = (LVar *) MALLOC (sizeof (LVar));

  new->type = LOCAL_VAR;
  new->name = 0;
  new->offset = 0;

  return (new);
}

/* **************************************************************
 * Destroy an instance of a LVar.
 * ************************************************************** */
void
lvar_Destroy (l_var)
     LVar *l_var;
{
  ASSERT (l_var);
  {
    l_var->type = 0;
    FREE (l_var->name);
    l_var->offset = 0;

    FREE (l_var);
  }
}

/* **************************************************************
 * Set the offset for a LVar
 * ************************************************************** */
void
lvar_SetOffset (l_var, offset)
     LVar *l_var;
     int offset;
{
  l_var->offset = offset;
}

void
lvar_SetName (l_var, name)
     LVar *l_var;
     char *name;
{
  FREE (l_var->name);
  l_var->name = name;
}

char *
lvar_GetName (l_var)
     LVar *l_var;
{
  return (l_var->name);
}

/* **************************************************************
 * Push a function arg onto a list. At the same time set the arg
 * offset (on the stack).
 * ************************************************************** */
List *
arg_var_push (list, s)
     List *list;
     char *s;
{
  LVar *l_var = lvar_Create ();

  if (list == 0)
    list = list_Create ();
  {
    ListNode *new_node;

    new_node = listNode_Create ();
    listNode_SetKey (new_node, s);
    listNode_AttachData (new_node, ARG_VAR, l_var, lvar_Destroy);
    list_PushNode (list, new_node);
  }
  lvar_SetName (l_var, cpstr (s));
  lvar_SetOffset (l_var, list_GetNumNodes (list));

  return (list);
}

/* **************************************************************
 * Push a local var onto a list. At the same time set the variable
 * offset (on the stack).
 * ************************************************************** */
List *
local_var_push (list, s)
     List *list;
     char *s;
{
  LVar *l_var = lvar_Create ();

  if (list == 0)
    error_1 ("Something terrible has happened, no function symbol table", 0);

  {
    ListNode *new_node;

    new_node = listNode_Create ();
    listNode_SetKey (new_node, s);
    listNode_AttachData (new_node, LOCAL_VAR, l_var, lvar_Destroy);
    list_PushNode (list, new_node);
  }
  lvar_SetName (l_var, cpstr (s));
  lvar_SetOffset (l_var, list_GetNumNodes (list));

  return (list);
}

List *
global_var_push (list, s)
     List *list;
     char *s;
{
  LVar *l_var = lvar_Create ();

  if (list == 0)
    error_1 ("Something terrible has happened, no function symbol table", 0);

  {
    ListNode *new_node;

    new_node = listNode_Create ();
    listNode_SetKey (new_node, s);
    listNode_AttachData (new_node, GLOBAL_VAR, l_var, lvar_Destroy);
    list_PushNode (list, new_node);
  }
  lvar_SetName (l_var, cpstr (s));
  lvar_SetOffset (l_var, list_GetNumNodes (list));

  return (list);
}

/* **************************************************************
 * Functions that help create RLaB user-functions. Re-direct program
 * code generation, and help handle local functions someday.
 * ************************************************************** */

extern Program *program_Get _PROTO ((void));
extern Inst *get_program_counter _PROTO ((void));
extern void set_program_counter _PROTO ((Inst * prgm));

#define NFUNC 10          /* Max number of recursive function definition */
static Inst *oldpc[NFUNC];
static Program *np[NFUNC];
static Program *op[NFUNC];
static int nf = 0;

int
function_setup1 (lsave, curr_file_name)
     int lsave;
     char *curr_file_name;
{
  Program *npp;
  /*
   * Create new program, save ptr to existing program
   * and point code generation at new program
   * array.
   */
  
  np[nf] = program_Create (50);
  op[nf] = program_Get ();
  oldpc[nf] = get_program_counter ();
  program_Set (np[nf]);		       

  npp = np[nf];

  /*
   * Setup new program array.
   */
  
  npp->prog[0].op_code = OP_FILE_NAME;
  npp->prog[1].ptr = curr_file_name;
  npp->prog[2].op_code = OP_LINE_NO;
  if (lsave == 0 || lsave == 1)
    npp->prog[3].op_code = 1;
  else
    npp->prog[3].op_code = lsave;

  npp->progp = &(npp->prog[4]);
  nf++;

  return (1);
}

ListNode *
function_setup2 (arg, local, global, poff)
     List *arg, *local, *global;
     int poff;
{
  Function *f;
  Program *npp = np[--nf];
  
  f = function_Create ();
  function_SetArgPtr (f, arg);
  function_SetNargs (f, list_GetNumNodes (arg));
  function_SetLocalPtr (f, local);
  function_SetNlocal (f, list_GetNumNodes (local));
  function_SetGlobalPtr (f, global);
  function_SetNglobal (f, list_GetNumNodes (global));
  
  /* Hook the program array into the function. */
  f->code = npp->prog;
  f->ncode = npp->ncode;
  
  /*
   * Clean up the back of the program array
   * (remove the extra STOPs).
   */

  f->code = (Inst *) REALLOC (f->code, sizeof (Inst) * (poff + 1));

  /* Free up the original program. */
  npp->prog = 0;
  npp->ncode = 0;
  npp->off = 0;
  npp->progp = (Inst *) 0;
  FREE (npp);

  /* Reset program stuff. */
  program_Set (op[nf]);
  set_program_counter (oldpc[nf]);

  return (install_tmp (U_FUNCTION, f, function_Destroy));  
}
