/* main.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1993  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "mem.h"
#include "code.h"
#include "util.h"
#include "rfile.h"
#include "bltin.h"
#include "version.h"
#include "r_string.h"
#include "getopt.h"

#include <stdio.h>
#include <sys/types.h>

#ifdef __STDC__
#include <stdlib.h>
#else
extern char *getenv ();
#endif

#ifdef unix
static char PATH_DELIM[] = "/";
#endif

#ifdef THINK_C

static char PATH_DELIM[] = ":";

/* ColorQuickDraw Windows */
#include <myconsole16.h>
/* Preference */
char *PreferenceName = "RLaB-Preference";
char *getpref (char *);
#define getenv(env_name) getpref(env_name)
int mac_rows = 25;		/* no. of rows */
char *txFont, *txSize, *nrows;	/* console font and size */
int txFontNum = 4, txSizeNum = 9;

static void think_c_setup _PROTO ((void));

#endif /* THINK_C */

#ifdef HAVE_DIRENT_H
#include <dirent.h>
#else
#include <sys/dir.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

/* Scanner declarations */
extern void set_rlab_input _PROTO ((int type));
extern int yyparse ();
extern FILE *yyin;
extern int lineno;
extern int flush_line;
extern char *curr_file_name;

/* print.c */
extern close_file_ds _PROTO ((char *fn));

/* rfile.c */
extern int rfile_test _PROTO ((char *name));
int rfile_dir _PROTO ((char *dirname));

/* code.c */
extern void init_frame _PROTO ((void));
extern void init_stack _PROTO ((void));
extern void init_flstack _PROTO ((void));

void init_misc _PROTO ((void));
void init_var _PROTO ((void));
void init_file_list _PROTO ((void));
void init_static_tree _PROTO ((void));
int run_program _PROTO ((char *input));
void run _PROTO ((Program * p));
void run_no_clean _PROTO ((Program * p));
void warning_1 _PROTO ((char *s, char *t));
void warning_2 _PROTO ((char *s1, char *s2, char *t));
void print_greeting _PROTO ((void));

#ifdef unix
void init_environment _PROTO ((void));
#endif

#ifdef THINK_C
void init_tc_environment _PROTO ((void));
#endif

extern void diss_assemble _PROTO ((Inst * p, int progoff));

static char *progname;

#ifdef THINK_C
char *DEFAULT_RC0 = ".rlab";
char *DEFAULT_HELP = "help";
char *DEFAULT_LIB = "rlib";
char *DEFAULT_PAGER = " ";
char *DEFAULT_SEARCH_PATH = "toolbox;examples";
char rlab_dir[100];		/* absolute rlab directory path */
#endif

static char *rlab_rc0;		/* RLAB_RC0, DEFAULT_RC0 */
static char *help_dir;		/* RLAB_HELP_DIR, DEFAULT_HELP */
static char *lib_dir;		/* RLAB_LIB_DIR, DEFAULT_LIB */
static char *pager;		/* RLAB_PAGER, DEFAULT_PAGER */
static char *search_path;	/* RLAB_SEARCH_PATH, DEFAULT_SEARCH_PATH */

static int yydebug = 0;		/* set to `1' for parser debugging */

static int print_machine = 0;	/* If TRUE, print contents of machine queue */
int use_readline = 1;		/* If FALSE DO NOT use GNU readline */
static int use_rc0 = 1;		/* If TRUE run rlab_rc0 on start-up */
static int use_pager = 1;	/* if TRUE use default pager (more) */
static int load_lib = 1;	/* if TRUE load libraries */
int line_nos = 1;		/* If TRUE use line #s in op-codes */
static int message = 1;		/* If TRUE print the start-up message */

/* If TRUE we will go interactive after start-up */
static int go_interactive = 1;
/* If TRUE we are running interactively */
static int interactive = 0;

static Program *mprog;		/* Main program array */
static char usage_string[] = "rlab -Vdhlmnpqr [file(s)] [-]";

/* **************************************************************
 * main, RLaB
 * ************************************************************** */
int
main (argc, argv)
     int argc;
     char *argv[];
{
  int c, r;

#ifdef THINK_C
  think_c_setup ();
  argc = ccommand (&argv);
#endif

  /* Process command line args */
  progname = argv[0];
  set_progname (progname);

  while ((c = getopt (argc, argv, "Vdhlmnpqr")) != EOF)
  {
    switch (c)
    {
    case 'V':
      fprintf (stderr, "%s version: %s\n", progname, version_string);
      return (0);
      break;
    case 'd':
      print_machine = 1;
      break;
    case 'h':
      fprintf (stderr, "usage: %s\n", usage_string);
      return (1);
      break;
    case 'l':
      load_lib = 0;
      break;
    case 'm':
      message = 0;
      break;
    case 'n':
      line_nos = 0;
      break;
    case 'p':
      use_pager = 0;
      break;
    case 'q':
      use_rc0 = 0;
      break;
    case 'r':
      use_readline = 0;
      break;
    default:
      fprintf (stderr, "usage: %s\n", usage_string);
      return (1);
      break;
    }
  }

  /* Perform initialization */
  init_misc ();

#ifdef unix
  init_environment ();
#endif
#ifdef THINK_C
  init_tc_environment ();
#endif

  init_var ();
  init_symbol_table ();
  init_file_list ();
  init_static_tree ();

  /* Process .rlab file */
  if (use_rc0)
  {
    if ((r = run_program (rlab_rc0)) == 0)
    {
      fprintf (stderr, "\nCould not open RLaB init script\n");
      fprintf (stderr, "try setting environment variable RLAB_RC0 ");
      fprintf (stderr, "and re-run RLaB\n\n\n");
    }
    else if (r < 0)
    {
      fprintf (stderr, "ERROR in rlab_rc0 file\n");
      return (0);
    }
  }

  /* Process rfiles in library */
  if (load_lib)
    rfile_dir (lib_dir);

  if (optind < argc)
    go_interactive = 0;

  /* Process files on the command line */
  while ((c = optind++) < argc)
  {
    if (!strcmp (argv[c], "-"))
      go_interactive = 1;
    else
      run_program (argv[c]);
  }

  /* Finally, go interactive */
  if (go_interactive)
  {
    if (message)
      print_greeting ();
    interactive = 1;
    run_program (0);
  }
  return (0);
}

/* **************************************************************
 * Misc initialization.
 * ************************************************************** */

#include "fpe.h"

void
init_misc ()
{
  /* Set the input function pointer */
  set_rlab_input (0);

  /* Initialize the interpreter stacks */
  init_frame ();
  init_stack ();
  init_flstack ();

  /* Try and setup for fpe exceptions. */
  setup_fpe_handling ();
}

/* **************************************************************
 * Get all RLaB environment/default variables.
 * ************************************************************** */

#ifdef unix
void
init_environment ()
{

  if ((rlab_rc0 = getenv ("RLAB_RC0")) == 0)
    rlab_rc0 = DEFAULT_RC0;

  if ((help_dir = getenv ("RLAB_HELP_DIR")) == 0)
    help_dir = DEFAULT_HELP;

  if ((lib_dir = getenv ("RLAB_LIB_DIR")) == 0)
    lib_dir = DEFAULT_LIB;

  if (use_pager)
  {
    if ((pager = getenv ("RLAB_PAGER")) == 0)
    {
      if ((pager = getenv ("PAGER")) == 0)
	pager = DEFAULT_PAGER;
    }
  }
  else
    pager = cpstr ("cat");

  if ((search_path = getenv ("RLAB_SEARCH_PATH")) == 0)
    search_path = DEFAULT_SEARCH_PATH;
}
#endif /* unix */

#ifdef THINK_C
void
init_tc_environment ()
{
  char *tmp;
  int counter = 1;
  getcwd (rlab_dir, 99);

  if ((rlab_rc0 = getenv ("RLAB_RC0")) == 0)
    rlab_rc0 = DEFAULT_RC0;

  if ((help_dir = getenv ("RLAB_HELP_DIR")) == 0)
  {
    help_dir = (char *) malloc (strlen (rlab_dir) + strlen (DEFAULT_HELP) + 8);
    sprintf (help_dir, "%s%s", rlab_dir, DEFAULT_HELP);
  }

  if ((lib_dir = getenv ("RLAB_LIB_DIR")) == 0)
  {
    lib_dir = (char *) malloc (strlen (rlab_dir) + strlen (DEFAULT_LIB) + 8);
    sprintf (lib_dir, "%s%s", rlab_dir, DEFAULT_LIB);
  }

  if (use_pager)
  {
    if ((pager = getenv ("RLAB_PAGER")) == 0)
    {
      if ((pager = getenv ("PAGER")) == 0)
	pager = DEFAULT_PAGER;
    }
  }
  else
    pager = cpstr ("cat");

  if ((search_path = getenv ("RLAB_SEARCH_PATH")) == 0)
  {
    tmp = strchr (DEFAULT_SEARCH_PATH, ';');
    while (tmp)
    {
      counter++;
      tmp++;
      tmp = strchr (tmp, ';');
    }
    search_path = (char *) malloc (strlen (rlab_dir) * counter +
				   strlen (DEFAULT_SEARCH_PATH) + 8);
    search_path[0] = '\0';
    tmp = strtok (DEFAULT_SEARCH_PATH, ";");
    while (tmp)
    {
      strcat (search_path, rlab_dir);
      strcat (search_path, tmp);
      tmp = strtok ('\0', ";");
      if (tmp)
	strcat (search_path, ";");
    }
  }
}
#endif /* THINK_C */

void
init_var ()
{
  /* Initialize miscellaneous variables in code.c */
  set_print_machine (print_machine);
  set_line_nos (line_nos);
  set_util_line_nos (line_nos);
  set_use_pager (use_pager);
  set_code_pager (pager);

  /* Initialize miscellaneous variables in rfile.c */
  set_search_path (search_path);
  set_help_dir (help_dir);
  set_lib_dir (lib_dir);
  set_pager (pager);
}

/* **************************************************************
 * Run the parser/machine.
 * ************************************************************** */

extern Program *program_Get _PROTO ((void));
extern Inst *get_program_counter _PROTO ((void));
extern void set_program_counter _PROTO ((Inst * prgm));
extern int reset_frame_ptr _PROTO ((void));
extern int reset_stack_ptr _PROTO ((void));
extern int clean_up_bc_lists _PROTO ((void));
extern Datum *get_stackp _PROTO ((void));
extern void set_stackp _PROTO ((Datum * new_stackp));

int
run_program (input)
     char *input;
{
  int retval;
  Inst *oldpc;
  Program *old_program, *program;

  if (input == 0)
  {
    if (!new_file ("stdin"))
      return (0);
  }
  else
  {
    close_file_ds (input);	/* In case it was open by accident */
    if (!new_file (input))
      return (0);
  }

  program = program_Create (500);	/* Create a new program array */
  old_program = program_Get ();	        /* Save the old program array */
  oldpc = get_program_counter ();	/* Save the old counter */

  /* Point the parser at the new program array */
  program_Set (program);

  /* Run the parser/machine */
  while (1)
  {
    if (!setjmp (*jmp_inc_buff ()))
    {
      /* Normal operation */
      signal (SIGFPE, fpecatch);
      signal (SIGINT, intcatch_wait);
#ifdef HAVE_PIPE
      signal (SIGPIPE, pipecatch);
#endif
      if (input == 0)
	run (program);
      else
	run_no_clean (program);

      /* Decrement the jmp buffer counter */
      dec_buff ();
      retval = 1;
      break;
    }
    else
    {
      /* An error (longjmp) has occurred */
      reset_frame_ptr ();
      reset_stack_ptr ();
      clean_up_bc_lists ();
      retval = -1;
      if (input == 0)
	continue;
      else
      {
	/*
	 * We error'ed, keep longjmp'ing until we hit
	 * input == 0, or we run out of jmps.
	 */

	new_file (0);
	flush_line = 0;
	program_Destroy (program);
	program_Set (old_program);
	set_program_counter (oldpc);

	if (get_ijmp () > 0)
	  longjmp (*jmp_dec_buff (), 1);
	else
	  return (retval);
      }
    }
  }

  /* Reset the old program array, etc, ... */
  program_Destroy (program);
  program_Set (old_program);
  set_program_counter (oldpc);

  return (retval);
}

/*
 * Run the parser / machine on a character string.
 */

char *eval_string;
extern void set_rlab_input _PROTO ((int type));		/* scan.l */
int do_eval = 0;

int
run_program_eval ()
{
  int retval;
  Datum *old_stackp;
  Inst *oldpc;
  Program *old_program, *program;

  /*
   * Set up new program space, and save current state
   * so we can get back.
   */

  program = program_Create (50);	/* Create a new program array */
  old_program = program_Get ();	/* Save the old program array */
  oldpc = get_program_counter ();	/* Save the old counter */
  old_stackp = get_stackp ();	/* Save current stackp */

  /* Point the parser at the new program array */
  program_Set (program);

  /* Run the parser/machine */
  while (1)
  {
    if (!setjmp (*jmp_inc_buff ()))
    {
      /* Normal operation */
      signal (SIGFPE, fpecatch);
      signal (SIGINT, intcatch_wait);
#ifdef HAVE_PIPE
      signal (SIGPIPE, pipecatch);
#endif

      initcode ();
      yyparse ();
      if (print_machine)
	diss_assemble (program->prog, get_progoff ());
      execute (program->prog);

      /* Decrement the jmp buffer counter */
      dec_buff ();
      retval = 1;		/* Successfull return */
      break;
    }
    else
    {
      /*
       * An error (longjmp) has occurred.
       * Do not reset the frame and stack pointers
       * since we will no longjmp back to the prompt.
       * Instead we will return to the caller.
       */

      retval = 0;
      program_Destroy (program);
      program_Set (old_program);
      set_program_counter (oldpc);
      set_stackp (old_stackp);

      return (retval);
    }
  }

  /* Reset the old program array, etc, ... */
  program_Destroy (program);
  program_Set (old_program);
  set_program_counter (oldpc);

  return (retval);
}

void
Eval (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  int retv;
  void *retd;
  Datum eval_ret;
  ListNode *EVAL;
  Scalar *s = 0;
  String *str;

  if (n_args != 1)
    error_1 ("eval: 1 argument allowed", 0);

  EVAL = bltin_get_string ("eval", d_arg, 1);
  str = (String *) e_data (EVAL);
  eval_string = string_GetString (str);

  do_eval++;
  set_rlab_input (1);
  retv = run_program_eval ();
  if (--do_eval == 0)
    set_rlab_input (0);
  
  /* Now return eval_ret */
  eval_ret = get_eval_ret ();

  if (retv == 0)
    *return_ptr = (VPTR) scalar_Create (0.0);
  else if (eval_ret.type != ENTITY)
  {
    /* Make an ENTITY out of it. */
    if (eval_ret.type == CONSTANT)
      s = scalar_Create (eval_ret.u.val);
    else if (eval_ret.type == iCONSTANT)
      s = scalar_CreateC (0.0, eval_ret.u.val);

    *return_ptr = (VPTR) s;
  }
  else
  {
    /*
     * We must remove the return entity from the
     * tmp list if it is there, otherwise there
     * will be trouble when bltin() puts it on the list.
     */

    if (e_name (eval_ret.u.ent) == 0)
    {
      retd = remove_tmp (eval_ret.u.ent);
      *return_ptr = retd;
    }
    else
    {
      /*
       * eval returned an existing entitiy,
       * let bltin() handle it.
       */

      e_type (eval_ret.u.ent) = LISTNODE;
      *return_ptr = (VPTR) eval_ret.u.ent;
    }
  }

  remove_tmp_destroy (EVAL);
  return;
}

void
run (program)
     Program *program;
{
  for (initcode (); yyparse (); initcode ())
  {
    if (print_machine)
      diss_assemble (program->prog, get_progoff ());
    execute (program->prog);
    clean_list ();
  }
}

void
run_no_clean (program)
     Program *program;
{
  for (initcode (); yyparse (); initcode ())
  {
    if (print_machine)
      diss_assemble (program->prog, get_progoff ());
    execute (program->prog);
  }
}

/* **************************************************************
 * Run a whole directory of rfiles through the machine.
 * ************************************************************** */

int
rfile_dir (dirname)
     char *dirname;
{
  /* FIX */
  char tmp[100];
  DIR *dirp;
#ifdef HAVE_DIRENT_H
  struct dirent *direntp;
#else
  struct direct *direntp;
#endif

  mprog = program_Create (1000);
  program_Set (mprog);
  if ((dirp = opendir (dirname)) != 0)
  {
    while ((direntp = readdir (dirp)) != NULL)
    {
      if (rfile_test (direntp->d_name))
      {
	strcpy (tmp, dirname);
	strcat (tmp, PATH_DELIM);
	strcat (tmp, direntp->d_name);
	run_program (tmp);
      }
    }
    closedir (dirp);
    return (1);
  }
  else
  {
    fprintf (stderr, "ERROR opening directory: %s\n", dirname);
  }
  return (0);
}

/* **************************************************************
 * Print a greeting to RLaB users.
 * ************************************************************** */

void
print_greeting ()
{
  printf ("Welcome to RLaB. New users type `help INTRO'\n");
  printf ("RLaB version %s ", version_string);
  printf ("Copyright (C) 1992-95 Ian Searle\n");
  printf ("RLaB comes with ABSOLUTELY NO WARRANTY; for details type");
  printf (" `help WARRANTY'\n");
  printf ("This is free software, and you are welcome to redistribute it");
  printf (" under\ncertain conditions; type `help CONDITIONS' for details\n");
}

/* **************************************************************
 * Cover for pclose() so we don't get messed up by pclose() that
 * cause a SIGPIPE, when closing an already broken pipe.
 * ************************************************************** */

#ifdef HAVE_PIPE

void
rpclose (fp)
     FILE *fp;
{
  signal (SIGPIPE, SIG_IGN);
  pclose (fp);
  signal (SIGPIPE, pipecatch);
}

#else

void
rpclose (fp)
     FILE *fp;
{
  fclose (fp);
}

#endif /* HAVE_PIPE */

/* **************************************************************
 * Platform specific setup, in this case THINK-C.
 * ************************************************************** */

#ifdef THINK_C

static void
think_c_setup ()
{
  if (txFont = getenv ("RLAB_FONT"))
    txFontNum = atoi (txFont);
  if (txSize = getenv ("RLAB_SIZE"))
    txSizeNum = atoi (txSize);
  if (nrows = getenv ("RLAB_ROWS"))
    mac_rows = atoi (nrows);
  console_options.pause_atexit = 1;
  console_options.procID = 0;
  console_options.title = "\pMacRLaB";
  console_options.nrows = mac_rows;
  console_options.txFont = txFontNum;
  console_options.txSize = txSizeNum;
  cgotoxy (1, 1, stdout);
}

#endif /* THINK_C */
