/* 
 * mathl.c
 * Math Library Functions.
 * Some pieces of code borrowed from the GNU C-library
 * (some of which, in turn, were borrowed from UCB)
 */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1994  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "mathl.h"
#include "util.h"

#include <stdio.h>
#include <math.h>
#include <errno.h>

/* **************************************************************
 * Check errno when calling math library functions.
 * ************************************************************** */
double
errno_check (d, s)
     double d;
     char *s;
{
  if (errno == EDOM)
  {
    errno = 0;
    error_1 (s, "argument out of domain");
  }
  else if (errno == ERANGE)
  {
    errno = 0;
    error_1 (s, "result out of range");
  }
  return (d);
}

#ifdef USE_MATHERR

/* **************************************************************
 * RLaB's matherr(). For now, use error_1 when matherr() is called.
 * This will suffice until I figure out what to do with the
 * various "standard", but incompatible math libraries.
 * ************************************************************** */

int
matherr (x)
     struct exception *x;
{
  switch (x->type)
  {
  case DOMAIN:
    error_1 ("argument out of DOMAIN", 0);
    break;
  case SING:
    error_1 ("argument SINGULARITY", 0);
    break;
  case OVERFLOW:
    error_1 ("OVERFLOW", 0);
    break;
  case UNDERFLOW:
    error_1 ("UNDERFLOW", 0);
    break;
  case TLOSS:
    error_1 ("Total LOSS of precision", 0);
    break;
  case PLOSS:
    error_1 ("Partial LOSS of precision", 0);
    break;
  }
  return (1);
}
#endif /* USE_MATHERR */

#ifndef HAVE_RINT
/* **************************************************************
 * A replacement rint() for deficient systems. This is not an IEEE
 * compatible rint(). This is a SIMPLE-MINDED function (it is better
 * than nothing :-).
 * ************************************************************** */

double
Rrint (x)
     double x;
{
  if (x != x)			/* NaN */
    return (x);
  return (x >= 0.0 ? floor (x + 0.5) : -floor (0.5 - x));
}

#endif /* ! HAVE_RINT */

/* **************************************************************
 * A function to check a double array for the presence of an
 * Inf or a NaN. This is to "protect" the Fortran subroutines
 * from getting "hung" on this type of input.
 * ************************************************************** */

/*
 * Create Infs, and NaNs
 */

#include "mathl.h"

static const unsigned char __inf_val[8] = r__inf_val_bytes;
#define	R_INF   (*(const double *) __inf_val)

int
detect_inf_r (a, n)
     double *a;
     int n;
{
  int i;

  for (i = 0; i < n; i++)
  {
    if (R_INF == a[i])
      return (1);
  }
  return (0);
}

int
detect_inf_c (a, n)
     Complex *a;
     int n;
{
  int j;

  for (j = 0; j < n; j++)
  {
    if (R_INF == a[j].r || R_INF == a[j].i)
      return (1);
  }
  return (0);
}

int
detect_nan_r (a, n)
     double *a;
     int n;
{
  int i;

  for (i = 0; i < n; i++)
  {
    if (a[i] != a[i])
      return (1);
  }
  return (0);
}

int
detect_nan_c (a, n)
     Complex *a;
     int n;
{
  int j;

  for (j = 0; j < n; j++)
  {
    if ((a[j].r != a[j].r) || (a[j].i != a[j].i))
      return (1);
  }
  return (0);
}

Matrix *
matrix_isinf (m)
     Matrix *m;
{
  int i, size;
  Matrix *mi;

  size = MNR (m) * MNC (m);
  mi = matrix_Create (MNR (m), MNC (m));
  
  for (i = 0; i < size; i++)
  {
    if (MTYPE (m) == REAL)
    {
      if (R_INF == MATrv (m, i))
	MATrv (mi, i) = 1.0;
      else
	MATrv (mi, i) = 0.0;
    }
    else if (MTYPE (m) == COMPLEX)
    {
      if (R_INF == MATcvr (m, i) || R_INF == MATcvi (m, i))
	MATrv (mi, i) = 1.0;
      else
	MATrv (mi, i) = 0.0;
    }
    else
      error_1 (matrix_GetName (m), "String matrix not allowed");
  }
  return (mi);
}

Matrix *
matrix_isnan (m)
     Matrix *m;
{
  int i, size;
  Matrix *mi;

  size = MNR (m) * MNC (m);
  mi = matrix_Create (MNR (m), MNC (m));
  
  for (i = 0; i < size; i++)
  {
    if (MTYPE (m) == REAL)
    {
      if (MATrv (m, i) != MATrv (m, i))
	MATrv (mi, i) = 1.0;
      else
	MATrv (mi, i) = 0.0;
    }
    else if (MTYPE (m) == COMPLEX)
    {
      if (MATcvr (m, i) != MATcvr (m, i) || MATcvi (m, i) != MATcvi (m, i))
	MATrv (mi, i) = 1.0;
      else
	MATrv (mi, i) = 0.0;
    } 
    else
      error_1 (matrix_GetName (m), "String matrix not allowed");
  }
  return (mi);
}

Matrix *
matrix_finite (m)
     Matrix *m;
{
  int i, size;
  Matrix *mi;

  size = MNR (m) * MNC (m);
  mi = matrix_Create (MNR (m), MNC (m));
  
  for (i = 0; i < size; i++)
  {
    if (MTYPE (m) == REAL)
    {
      if (R_INF != MATrv (m, i) && MATrv (m, i) == MATrv (m, i))
	MATrv (mi, i) = 1.0;
      else
	MATrv (mi, i) = 0.0;
    }
    else if (MTYPE (m) == COMPLEX)
    {
      if ((R_INF != MATcvr (m, i) && R_INF != MATcvi (m, i)) &&
	  (MATcvr (m, i) == MATcvr (m, i) && MATcvi (m, i) == MATcvi (m, i)))
	MATrv (mi, i) = 1.0;
      else
	MATrv (mi, i) = 0.0;
    } 
    else
      error_1 (matrix_GetName (m), "String matrix not allowed");
  }
  return (mi);
}
