//-------------------------------------------------------------------//

//  Syntax:	plot ( A, KeyTitles, I ) or plot ( X, Y, I )
//		replot ( A, KeyTitles, I ) or replot ( X, Y, I ) or replot ( I )
//		splot ( X, Y, Z, KeyTitles, I )
//		cont ( X, Y, Z, KeyTitles, I )
//		pstring ( "string", I )
//		postplot ( "file_name", "mode", I )
//		latexplot ( "file_name", "mode", I )
//		printplot ( "printer-name", I )
//		psave ( "file_name", A )
//		pssave ( "file_name", X, Y, Z )
//		multiplot ( rows, columns, I )	!!!!!! Special GNUPLOT-contrib needed !!!!!!
//		nomultiplot ( I )		!!!!!! Special GNUPLOT-contrib needed !!!!!!
//		plotyy ( I )			!!!!!! Special GNUPLOT-contrib needed !!!!!!
//		noplotyy ( I )			!!!!!! Special GNUPLOT-contrib needed !!!!!!
//		pclose ( I )
//		setterm ( "term", I )
//		showplot ( I )
//		showpwin ( )
//		phelp ( "string", I)
//		autoscale ( "axis", I )
//		range ( x-start, x-end, y-start, y-end, z-start, z-end, I )
//		xrange ( start, end, I )
//		yrange ( start, end, I )
//		zrange ( start, end, I )
//		xtics ( start, end, incr, I ) or xtics (vector, I ) or xtics ( I )
//		noxtics ( I )
//		ytics ( start, end, incr, I ) or ytics (vector, I ) or ytics ( I )
//		noytics ( I )
//		ztics ( start, end, incr, I ) or ytics (vector, I ) or ytics ( I )
//		noztics ( I )
//		loglog ( I )
//		semilogx ( I )
//		semilogy ( I )
//		nolog ( I )
//		xlabel ( "label_name", I )
//		ylabel ( "label_name", I )
//		zlabel ( "label_name", I )
//		title ( "title_name", I )
//		notitle ( "title_name", I )
//		linestyle ( "line_style", I )
//		pformat ( "format-string", "axis", I )
//		key ( x, y, z, I )	(*)
//		nokey ( I )
//		grid ( I )
//		nogrid ( I )
//		axis ( I )
//		noaxis ( I )
//		label ( "label_text", x, y, z, labelnr, I )	(*)
//		nolabel ( labelnr, I )
//		time ( I )
//		notime ( I )

//		(*) Be alert how you send these commands. You will get into trouble
//		    if you try to send "I" without fully specified coordinates.
//		    Do not leave out "z" or "labelnr" for example.

// Description:

//  The plot function plots numeric data via the GNUPLOT program. The
//  argument A can be a matrix, or a list of matrices, or a string.
//  plot can also plot a vector Y versus a vector X. X and Y can
//  also be matrixes. 
//  When A is a matrix, columns 2 through N are plotted versus the 1st
//  column. When A is a list of matrices, each matrix is plotted
//  versus it's own 1st column. When A is a string, the string is sent
//  to GNUPLOT as a command.

//  plot ( M )		  		plots the contents of matrix M
//  plot( X, Y )			plots Y verus X
//  plot ( << M1 ; M2>> )		plots M1, and M2 on the same graph
//  plot ( "set noborder" )		sends the string directly to
//  					GNUPLOT as a command

//  Using a list of matrices as an argument to plot is useful when the
//  the independent variable (1st column) of each matrix is different.

//  The second and third, and optional argument to plot denotes the plot process
//  number and sends key-titles. The plot() function will open distinct plot
//  processes when given distinct values of N. When this feature is used in a
//  X-windows environment, the effect is to create separate plot windows.
//  The KeyTitles is a string-vector containing the titles to be plotted as keys.
//  They are chosen from the beginning of KeyTitles[1] and so on.
//  For example:

//  plot ( M );				plot the contents if matrix M
//  plot ( 2.*M, 1 );			plot 2 times M, to another window
//  plot (<<M1;M2>>, KeyTitles, 2);	plot M1, and M2 on the same graph in another
//					window with the keytitles from KeyTitles

//  The replot() command can be used to replot the last plot. It can also be
//  used to superinpose a new plot in an old plot.

//  replot( I );			replot the I-th plot-window
//  replot( A, KeyTitles, I );		superinpose new plot on the I-th plot-window
//  replot( X, Y, I );			superinpose new plot on the I-th plot-window

//  For 3D-plotting there are several commands. splot() makes a 3D-surface plot
//  and cont() makes a 2D-contour plot. This is in GNUPLOT a 3D-plot with no 
//  surface.
//  
//  The splot() function splots numeric data via the GNUPLOT program. The
//  argument Z is a matrix as functions of vectors X and Y.

//  splot ( X, Y, Z )			splots the contents of matrix Z
//					versus X and Y
//  splot ( Z )				splots the contents of matrix Z
//					versus rows and columns
//  splot ( X, Y, [Z1,Z2] )		splots Z1, and Z2 on the same graph
//  cont ( X, Y, Z )			splots the contents of matrix Z
//					as a 2D-contour-plot
//  cont ( X, Y, [Z1,Z2] )		splots Z1, and Z2 on the same graph
//					as a 2D-contour-plot

//  The psave() and pssave() saves the usual temporary file to a specific, 
//  non-temporary file named "file_name". psave() is for 2D-plots and pssave() 
//  is for 3D-plots. This is useful if you wish to save session-output in a 
//  GNUPLOT-way.

//  multiplot() allows you to make multiple plots in one window. After giving 
//  the command you enter multimode. After making rows*columns plot commands 
//  you automaticly leaves this mode. plotyy() allows you to make one plot with 
//  two y-axes. You will enter multimode, and after making two plot commands you 
//  leave. The first plot will be on the left axis and the second on the right. 
//  If you want to leave multiplot() or plotyy() before ending the plot commands
//  just type nomultiplot() or noplotyy().
//  If you want a PostScript output from plots made in multimode you do not 
//  use postplot(). This will not work. plotyy() is for 2D-plots only. 
//  GNUPLOT-3.5 DO NOT SUPPORT THESE FEATURES. If you want to use them 
//  you must add GNUPLOT-contributions and recompile GNUPLOT.

//  The pclose function closes (destroys)  the I-th GNUPLOT
//  subprocess. If I = "all", all GNUPLOT subprocess are closed.

//  The postplot() function generates a PostScript file named "file_name" 
//  of the current I-th plot. "mode" is further postscript definitions to 
//  GNUPLOT if needed, for example "eps color 'Times-Roman' 14" for making
//  a smaller plot in color with Times-Roman-font size 14.
//  The PostScript plot is saved in a file for printing, or whatever. 
//  Immediately after calling postplot(), the GNUPLOT variables
//  term, and output are reset to there former values.
//  printplot() generates a PostScript file and sends it to the printer.
//  latexplot() generates a LaTeX file named "file_name" of the current 
//  I-th plot. 

//  The setterm function allows the user to set/reset the I-th plot's
//  term type. The argument "term" is a string that corresponds to one
//  of GNUPLOT's available term types, for example "vttek".

//  The showplot function prints the contents of the I-th plot list to
//  stdout.
//  The showpwin function prints the currently existing plotwindows 
//  to stdout.

//  A bunch of often used commands to GNUPLOT has special functions.
//  These could just as well be sent as strings, but this makes life easier.
//  These are range(), xrange(), yrange(), zrange(), xtics(), ytics(), 
//  noxtics(), noytics(), loglog(), semilogx(), semilogy(), nolog(), 
//  autoscale(), xlabel(), ylabel(), zlabel(), 
//  title(), notitle(), linestyle(), pformat(), key(), nokey(), 
//  grid(), nogrid(), axis(), noaxis(), label(), nolabel(), 
//  time() and notime().

//  The xtics(), ytics() and ztics() functions sets the tics on the axes. 
//  If it is used as xtics(start,end,incr,I), the tics will be set 
//  for plotwindow I, starting from "start", ending at "end", 
//  with an increment of "incr". 
//  If, however, it is used as xtics([a,b,c,d],I), the tics will 
//  set at "a", "b", "c" and "d" in plotwindow I. 
//  The command xtics(I) will reset the tics in plotwindow I.
//  The noxtics() function results in a tics-less x-axis.

//  The plot function is an interface to the GNUPLOT program. The plot
//  function uses temporary files and Rlab's piping capability to
//  transparently plot numerical data via GNUPLOT. Since the plot
//  function is an rfile, it could easily be modified to to use
//  another plotting program.

//  To be able to use all commands in this r-file you must add some
//  contributions to GNUPLOT and recompile it. These are "multiplot"
//  and "borders". See the GNUPLOT FAQ for help.

//-------------------------------------------------------------------//



_RePlotParameter = zeros(1,100);

//
// List to contain plot-object lists.
//

static (p);
p = <<>>;

//
// Internal plot-related functions
//

static (plotl, plotm, splotm, plots);
static (pobj_Create, pobj_Reset, pobj_Destroy);
static (pobj_TmpFileName, pobj_WriteData, pobj_PlotCmd, pobj_RePlotCmd, pobj_SPlotCmd);
static (pobj_PlotKeyTitle, pobj_Plot, pobj_SetRm, pobj_Rm);



//
// User interface to plot functionality
//

plot = function ( data, titles, N )
{
    local( WaitAMoment, sizemem, flag1, flag2, flag3 )
    global( _MultiPlotParameter, _MultiOrginParameter, _PlotYyParameter );

    flag1 = 0;
    flag2 = 0;
    flag3 = 0;
    // Check the existence titles
    if (exist (titles)) {
	if (class (titles) == "num") {
	    if (max (size (titles)) == 1) {
		N = titles;
		else
		flag1=1;
		sizemem = size (data);
		if (data.nr < data.nc) {
		    flag2 = 1;
		    data = data';
		}
		if (titles.nr < titles.nc) {
		    flag3 = 1;
		    titles = titles';
		}
		data = [data,titles];
	    }
	}
    }
    
    // Set the default plot-object
    if (!exist (N)) { N = 0; }

    // Check the existence of p.[N]
    if (max (size (N)) != 1) { error ("plot(): N must be a 1-by-1"); }
    if (exist (p.[N])) {
	pobj_Reset (N);
	else
	pobj_Create (N);
    }

    // Check for multiplot and/or plotyy mode
    if (exist (_MultiPlotParameter) || exist (_PlotYyParameter)) {
	// Delay (RLaB is a bit to fast)
	for(i in 1:1000) {
	    WaitAMoment=1;
	    clear(WaitAMoment);
	}
	// Check for plotyy mode
	if (exist (_PlotYyParameter)) {
	    if (_PlotYyParameter==1) {
		printf (" Plot on left y-axis done!\n");
		if (!exist (_MultiPlotParameter)) {
		    fprintf (p.[N].prog, "set multiplot\n");
		}
		fprintf (p.[N].prog, "set border 7\n");
		else if (_PlotYyParameter==2) {
		    printf (" Plot on right y-axis done!\n");
		    fprintf (p.[N].prog, "set border 8\n");
	    }   }
	    _PlotYyParameter=_PlotYyParameter+1;
	}
	// Check for multiplot mode
	if (exist (_MultiPlotParameter)) {
	    printf (" Plot number %i/%i done!\n",_MultiPlotParameter[3],_MultiPlotParameter[1]*_MultiPlotParameter[2]);
	    if (_MultiPlotParameter[3]==1) {
		fprintf (p.[N].prog, "set multiplot\n");
		fprintf (p.[N].prog, "set size %s,%s\n",1./_MultiPlotParameter[2],1./_MultiPlotParameter[1]);
	    }
	    N=_MultiPlotParameter[4];
	    fprintf (p.[N].prog, "set orgin %s,%s\n",_MultiOrginParameter[_MultiPlotParameter[3];1],_MultiOrginParameter[_MultiPlotParameter[3];2]);
	    if (exist (_PlotYyParameter)) {
		if (_PlotYyParameter==3) {_MultiPlotParameter[3]=_MultiPlotParameter[3]+1; }
		else
		_MultiPlotParameter[3]=_MultiPlotParameter[3]+1;
	    }
	}
    }
    
    // Time to plot
    if (class (data) == "num") {
	fprintf (p.[N].prog, "set noparametric\n");	
	pobj_Rm (N);
	plotm (data, N, 1, titles);
	pobj_Plot (N);
	pobj_SetRm (N);
	else if (class (data) == "string") {
	    plots (data, N);
	    pobj_Plot (N);
	    else if (class (data) == "list") {
		pobj_Rm (N);
		plotl (data, N, titles);
		pobj_Plot (N);
		pobj_SetRm (N);	
		else
		error ("Invalid data for plot()");
    }   }   }
    
    // Check for multiplot and/or plotyy mode
    // Check for plotyy mode
    if (exist (_PlotYyParameter)) {
	if (_PlotYyParameter==3) {
	    if (!exist (_MultiPlotParameter)) { fprintf (p.[N].prog, "set nomultiplot\n"); }
	    fprintf (p.[N].prog, "set border\n");
	    clear(_PlotYyParameter);
	}
    }
    // Check for multiplot mode
    if (exist (_MultiPlotParameter)) {
	if (_MultiPlotParameter[3]==_MultiPlotParameter[1]*_MultiPlotParameter[2]+1) {
	    fprintf (p.[N].prog, "set nomultiplot\n");
	    fprintf (p.[N].prog, "set size %s,%s\n",1,1);
	    fprintf (p.[N].prog, "set orgin %s,%s\n",0,0);
	    fprintf (p.[N].prog, "set border\n");
	    clear(_MultiPlotParameter);
	    clear(_MultiOrginParameter);
	}
    }
    
    if (flag1 == 1) { 
	if (flag2 == 0) {
	    data = data[;1:sizemem[2]];
	    else
	    data = data[;1:sizemem[1]];
	}
	if (flag2 == 1) { data = data'; }
	if (flag3 == 1) { titles = titles'; }
    }

};

//
// User interface to replot functionality
//

replot = function ( data, titles, N )
{
    local( sizemem, flag1, flag2, flag3, flag4 );
    global( _MultiPlotParameter, _MultiOrginParameter, _PlotYyParameter );
    global( _RePlotParameter );
    
    flag1 = 0;
    flag2 = 0;
    flag3 = 0;
    flag4 = 0;
    // Check the existence titles
    if (exist (titles)) {
	if (class (titles) == "num") {
	    if (max (size (titles)) == 1) {
		N = titles;
		else
		flag1=1;
		sizemem = size (data);
		if (data.nr < data.nc) {
		    flag2 = 1;
		    data = data';
		}
		if (titles.nr < titles.nc) {
		    flag3 = 1;
		    titles = titles';
		}
		data = [data,titles];
	    }
	}
    }
    
    if (exist (data) && !exist (titles)) {
	if (max(size(data)) == 1) {
	    N = data;
	    clear(data);
	    flag4=1;
	}
    }
    // Set the default plot-object
    if (!exist (N)) { N = 0; }

    // Check the existence of p.[N]
    if (max (size (N)) != 1) { error ("plot(): N must be a 1-by-1"); }
    if (exist (p.[N])) {
	pobj_Reset (N);
	else
	pobj_Create (N);
    }

    // Time to plot
    if (!exist(data)) {
	fprintf (p.[N].prog, "replot\n");
	if (flag4 == 1) { data = N; }
	return 0;
    }
    if (class (data) == "num") {
	fprintf (p.[N].prog, "set noparametric\n");
	_RePlotParameter[N+1] = _RePlotParameter[N+1]+1;
	pobj_Rm (N);
	replotm (data, N, 1, titles);
	pobj_Plot (N);
	pobj_SetRm (N);
	else if (class (data) == "string") {
	    plots (data, N);
	    pobj_Plot (N);
	    else if (class (data) == "list") {
		pobj_Rm (N);
		plotl (data, N, titles);
		pobj_Plot (N);
		pobj_SetRm (N);	
		else
		error ("Invalid data for replot()");
    }   }   }

    if (flag1 == 1) { 
	if (flag2 == 0) {
	    data = data[;1:sizemem[2]];
	    else
	    data = data[;1:sizemem[1]];
	}
	if (flag2 == 1) { data = data'; }
	if (flag3 == 1) { titles = titles'; }
    }

};


//
// User interface to splot functionality
//

splot = function ( datax, datay, datazz, titles, N )
{
    local (flag1, flag2, flag3, flag4, flagx, flagy, WaitAMoment)
    global( _MultiPlotParameter, _MultiOrginParameter, _PlotYyParameter );

    // Reset flags
    flag1=0;
    flag2=0;
    flag3=0;
    flag4=0;
    flagx=0;
    flagy=0;

    // Check what has been sent here
    // One input!
    if (!exist (datay)) {
	datazz=datax;
	datax=1:datazz.nc;
	datax=datax';
	datay=1:datazz.nr;
	datay=datay';
	flag1=flag1+1;
	// Two inputs!
	else if (!exist (datazz)) {
	    if (exist (datay)) {
		if (class (datay) == "num") {
		    N = datay;
		    flag2=flag2+0;
		    else
		    titles = datay;
		    flag2=flag2+1;
		}
	    }
	    datazz=datax;
	    datax=1:datazz.nc;
	    datax=datax';
	    datay=1:datazz.nr;
	    datay=datay';
	    flag2=flag2+1;
	    // Three inputs!
	    else if (!exist (titles)) {
		if (class (datay) != "num") {
		    titles = datay;
		    N = datazz;
		    datazz=datax;
		    datax=1:datazz.nc;
		    datax=datax';
		    datay=1:datazz.nr;
		    datay=datay';
		    flag3=flag3+1;
		}
		// Four inputs!
		else if (!exist (N)) {
		    if (class (titles) == "num") {
			N=titles;
			flag4=flag4+1;
		    }
    }   }   }   }
    
    // Transpose x- and y-vectors if nessecary
    if (datax.nc != 1 ) {
	if (datax.nr != 1 ) {
	    error ("Invalid data for x-axis (not a vector)");
	    else
	    datax = datax';
	    flagx=flagx+1;
	}
    }
    if (datay.nc != 1 ) {
	if (datay.nr != 1 ) {
	    error ("Invalid data for y-axis (not a vector)");
	    else
	    datay = datay';
	    flagy=flagy+1;
	}
    }
    
    // Check how many splots to do
    plotantal=(datazz.nc/datazz.nr)/(datax.nr/datay.nr);
    if (plotantal == int(plotantal)) {
	
	// Set the default splot-object
	if (!exist (N)) { N = 0; }

	// Check the existence of p.[N]
	if (max (size (N)) != 1) { error ("splot(): N must be a 1-by-1"); }
	if (exist (p.[N])) {
	    pobj_Reset (N);
	    else
	    pobj_Create (N);
	}

	// Check for multiplot and/or plotyy mode
	if (exist (_MultiPlotParameter) || exist (_PlotYyParameter)) {
	    // Delay (RLaB is a bit to fast)
	    for(i in 1:1000) {
		WaitAMoment=1;
		clear(WaitAMoment);
	    }
	    // Check for plotyy mode
	    if (exist (_PlotYyParameter)) {
		if (_PlotYyParameter==1) {
		    if (!exist (_MultiPlotParameter)) {
			fprintf (p.[N].prog, "set multiplot\n");
		    }
		    fprintf (p.[N].prog, "set border 7\n");
		    else if (_PlotYyParameter==2) {
			fprintf (p.[N].prog, "set border 8\n");
		}   }
		_PlotYyParameter=_PlotYyParameter+1;
	    }
	    // Check for multiplot mode
	    if (exist (_MultiPlotParameter)) {
		printf (" Plot number %i/%i done!\n",_MultiPlotParameter[3],_MultiPlotParameter[1]*_MultiPlotParameter[2]);			
		if (_MultiPlotParameter[3]==1) {
		    fprintf (p.[N].prog, "set multiplot\n");
		    fprintf (p.[N].prog, "set size %s,%s\n",1./_MultiPlotParameter[2],1./_MultiPlotParameter[1]);
		}
		N=_MultiPlotParameter[4];
		fprintf (p.[N].prog, "set orgin %s,%s\n",_MultiOrginParameter[_MultiPlotParameter[3];1],_MultiOrginParameter[_MultiPlotParameter[3];2]);
		if (exist (_PlotYyParameter)) {
		    if (_PlotYyParameter==3) {_MultiPlotParameter[3]=_MultiPlotParameter[3]+1; }
		    else
		    _MultiPlotParameter[3]=_MultiPlotParameter[3]+1;
		}
	    }
	}

	if (class (datax) != "num") {
	    error ("Invalid data class for splot() in x-vector (must be num)");
	    else if (class (datay) != "num") {
		error ("Invalid data class for splot() in y-vector (must be num)");
		else if (class (datazz) != "num") {
		    error ("Invalid data class for splot() in x-matrix (must be num)");
		    else
		    fprintf (p.[N].prog, "set parametric\n");
		    fprintf (p.[N].prog, "set view 60, 30, 1, 1\n");
		    fprintf (p.[N].prog, "set surface\n");
		    fprintf (p.[N].prog, "set nocontour\n");
		    pobj_Rm (N);
		    splotm (datax, datay, datazz, N, 1, titles);
		    pobj_Plot (N);
		    pobj_SetRm (N);
	}   }   }
	else
	error ("Invalid data for splot(), bad x/y-range in z-matrix");
    }
    
    // Restore the output/input
    if (flag1 == 1) { datax=datazz; }
    if (flag2 == 1) {
	datax=datazz;
	datay=N;
	else if (flag2 == 2) {
	    datax=datazz;
	    datay=titles;
    }   }
    if (flag3 == 1) {
	datax=datazz;
	datay=titles;
	datazz=N;
    }
    if (flag4 == 1) { titles=N; }
    if (flagx == 1) { datax=datax'; }
    if (flagy == 1) { datay=datay'; }

    // Check for multiplot and/or plotyy mode
    // Check for plotyy mode
    if (exist (_PlotYyParameter)) {
	if (_PlotYyParameter==3) {
	    if (!exist (_MultiPlotParameter)) { fprintf (p.[N].prog, "set nomultiplot\n"); }
	    fprintf (p.[N].prog, "set border\n");
	    clear(_PlotYyParameter);
	}
    }
    // Check for multiplot mode
    if (exist (_MultiPlotParameter)) {
	if (_MultiPlotParameter[3]==_MultiPlotParameter[1]*_MultiPlotParameter[2]+1) {
	    fprintf (p.[N].prog, "set nomultiplot\n");
	    fprintf (p.[N].prog, "set size %s,%s\n",1,1);
	    fprintf (p.[N].prog, "set orgin %s,%s\n",0,0);
	    fprintf (p.[N].prog, "set border\n");
	    clear(_MultiPlotParameter);
	    clear(_MultiOrginParameter);
	}
    }
};



//
// User interface to splot functionality as 2D-contour
//

cont = function ( datax, datay, datazz, titles, N )
{
    local (flag1, flag2, flag3, flag4, flagx, flagy, WaitAMoment)
    global( _MultiPlotParameter, _MultiOrginParameter, _PlotYyParameter );

    // Reset flags
    flag1=0;
    flag2=0;
    flag3=0;
    flag4=0;
    flagx=0;
    flagy=0;

    // Check what has been sent here
    // One input!
    if (!exist (datay)) {
	datazz=datax;
	datax=1:datazz.nc;
	datax=datax';
	datay=1:datazz.nr;
	datay=datay';
	flag1=flag1+1;
	// Two inputs!
	else if (!exist (datazz)) {
	    if (exist (datay)) {
		if (class (datay) == "num") {
		    N = datay;
		    flag2=flag2+0;
		    else
		    titles = datay;
		    flag2=flag2+1;
		}
	    }
	    datazz=datax;
	    datax=1:datazz.nc;
	    datax=datax';
	    datay=1:datazz.nr;
	    datay=datay';
	    flag2=flag2+1;
	    // Three inputs!
	    else if (!exist (titles)) {
		if (class (datay) != "num") {
		    titles = datay;
		    N = datazz;
		    datazz=datax;
		    datax=1:datazz.nc;
		    datax=datax';
		    datay=1:datazz.nr;
		    datay=datay';
		    flag3=flag3+1;
		}
		// Four inputs!
		else if (!exist (N)) {
		    if (class (titles) == "num") {
			N=titles;
			flag4=flag4+1;
		    }
    }   }   }   }
    
    // Transpose x- and y-vectors if nessecary
    if (datax.nc != 1 ) {
	if (datax.nr != 1 ) {
	    error ("Invalid data for x-axis (not a vector)");
	    else
	    datax = datax';
	    flagx=flagx+1;
	}
    }
    if (datay.nc != 1 ) {
	if (datay.nr != 1 ) {
	    error ("Invalid data for y-axis (not a vector)");
	    else
	    datay = datay';
	    flagy=flagy+1;
	}
    }
    
    // Check how many splots to do
    plotantal=(datazz.nc/datazz.nr)/(datax.nr/datay.nr);
    if (plotantal == int(plotantal)) {

	// Set the default splot-object
	if (!exist (N)) { N = 0; }

	// Check the existence of p.[N]
	if (max (size (N)) != 1) { error ("splot(): N must be a 1-by-1"); }
	if (exist (p.[N])) {
	    pobj_Reset (N);
	    else
	    pobj_Create (N);
	}

	// Check for multiplot and/or plotyy mode
	if (exist (_MultiPlotParameter) || exist (_PlotYyParameter)) {
	    // Delay (RLaB is a bit to fast)
	    for(i in 1:1000) {
		WaitAMoment=1;
		clear(WaitAMoment);
	    }
	    // Check for plotyy mode
	    if (exist (_PlotYyParameter)) {
		if (_PlotYyParameter==1) {
		    if (!exist (_MultiPlotParameter)) {
			fprintf (p.[N].prog, "set multiplot\n");
		    }
		    fprintf (p.[N].prog, "set border 7\n");
		    else if (_PlotYyParameter==2) {
			fprintf (p.[N].prog, "set border 8\n");
		}   }
		_PlotYyParameter=_PlotYyParameter+1;
	    }
	    // Check for multiplot mode
	    if (exist (_MultiPlotParameter)) {
		if (_MultiPlotParameter[3]==1) {
		    fprintf (p.[N].prog, "set multiplot\n");
		    fprintf (p.[N].prog, "set size %s,%s\n",1/_MultiPlotParameter[2],1/_MultiPlotParameter[1]);
		}
		N=_MultiPlotParameter[4];
		fprintf (p.[N].prog, "set orgin %s,%s\n",_MultiOrginParameter[_MultiPlotParameter[3];1],_MultiOrginParameter[_MultiPlotParameter[3];2]);
		if (exist (_PlotYyParameter)) {
		    if (_PlotYyParameter==3) {_MultiPlotParameter[3]=_MultiPlotParameter[3]+1; }
		    else
		    _MultiPlotParameter[3]=_MultiPlotParameter[3]+1;
		}
	    }
	}

	if (class (datax) != "num") {
	    error ("Invalid data class for splot() in x-vector (must be num)");
	    else if (class (datay) != "num") {
		error ("Invalid data class for splot() in y-vector (must be num)");
		else if (class (datazz) != "num") {
		    error ("Invalid data class for splot() in x-matrix (must be num)");
		    else
		    fprintf (p.[N].prog, "set parametric\n");
		    fprintf (p.[N].prog, "set view 0, 0, 1\n");
		    fprintf (p.[N].prog, "set nosurface\n");
		    fprintf (p.[N].prog, "set contour\n");
		    pobj_Rm (N);
		    splotm (datax, datay, datazz, N, 1, titles);
		    pobj_Plot (N);
		    pobj_SetRm (N);
	}   }   }
	else
	error ("Invalid data for splot(), bad x/y-range in z-matrix");
    }
    
    // Restore the output/input
    if (flag1 == 1) { datax=datazz; }
    if (flag2 == 1) {
	datax=datazz;
	datay=N;
	else if (flag2 == 2) {
	    datax=datazz;
	    datay=titles;
    }   }
    if (flag3 == 1) {
	datax=datazz;
	datay=titles;
	datazz=N;
    }
    if (flag4 == 1) { titles=N; }
    if (flagx == 1) { datax=datax'; }
    if (flagy == 1) { datay=datay'; }

    // Check for multiplot and/or plotyy mode
    // Check for plotyy mode
    if (exist (_PlotYyParameter)) {
	if (_PlotYyParameter==3) {
	    if (!exist (_MultiPlotParameter)) { fprintf (p.[N].prog, "set nomultiplot\n"); }
	    fprintf (p.[N].prog, "set border\n");
	    clear(_PlotYyParameter);
	}
    }
    // Check for multiplot mode
    if (exist (_MultiPlotParameter)) {
	if (_MultiPlotParameter[3]==_MultiPlotParameter[1]*_MultiPlotParameter[2]+1) {
	    fprintf (p.[N].prog, "set nomultiplot\n");
	    fprintf (p.[N].prog, "set size %s,%s\n",1,1);
	    fprintf (p.[N].prog, "set orgin %s,%s\n",0,0);
	    fprintf (p.[N].prog, "set border\n");
	    clear(_MultiPlotParameter);
	    clear(_MultiOrginParameter);
	}
    }
};



//
// Send string to GNUPLOT
//

pstring = function ( string, N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "%s\n",string);
};



//
// User interface to save a plot ot splot as postscript hardcopy
//

postplot = function ( file, mode, N )
{
    local ( flag1 );
    flag1 = 0;
    if (!exist (N) && exist (mode)) {
	if (class (mode) == "num") {
	    N = mode;
	    flag1 = 1;
	}
    }

    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { error ("No existing plot to make hardcopy from"); }

    // Make hardcopy file
    if (!exist (mode)) { mode="default"; }
    if (class (mode) == "num") { mode="default"; }
    fprintf (p.[N].prog, "set term post %s\n", mode);
    fprintf (p.[N].prog, "set output \"%s\"\n", file);
    fprintf (p.[N].prog, "replot\n");

    // Reset to original term type, and replot
    fprintf (p.[N].prog, "set term %s\n", p.[N].term);
    fprintf (p.[N].prog, "replot\n");

    if (flag1 == 1) {
	mode = N;
    }
};



//
// User interface to save a plot ot splot as LaTeX hardcopy
//

latexplot = function ( file, mode, N )
{
    local ( flag1 );
    flag1 = 0;
    if (!exist (N) && exist (mode)) {
	if (class (mode) == "num") {
	    N = mode;
	    flag1 = 1;
	}
    }

    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { error ("No existing plot to make hardcopy from"); }

    // Make hardcopy file
    if (!exist (mode)) { mode="default"; }
    if (class (mode) == "num") { mode="default"; }
    fprintf (p.[N].prog, "set term latex %s\n", mode);
    fprintf (p.[N].prog, "set output \"%s\"\n", file);
    fprintf (p.[N].prog, "replot\n");

    // Reset to original term type, and replot
    fprintf (p.[N].prog, "set term %s\n", p.[N].term);
    fprintf (p.[N].prog, "replot\n");

    if (flag1 == 1) {
	mode = N;
    }
};


//
// User interface to print a plot ot splot
//

printplot = function ( printername, N )
{
    local(fn, flag1);
    
    flag1 = 0;
    if (exist (printername) && !exist(N)) {
	if (class (printername) == "num") {
	    N = printername;
	    clear(printername);
	    flag1 = 1;
	}
    }
    
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    
    // Create tmp-file-name
    fn = "/usr/tmp/rlab-tmpf.ps";

    postplot (fn, "default", N);
        // Delay (RLaB is a bit to fast)
    for(i in 1:1000) {
	WaitAMoment=1;
	clear(WaitAMoment);
    }
    if (exist (printername)) {
	system ("lp -d" + printername + " " + fn);
	else
	system ("lp " + fn);
    }
    // Delay (RLaB is a bit to fast)
    for(i in 1:1000) {
	WaitAMoment=1;
	clear(WaitAMoment);
    }
    system ("rm -f " + fn);
    
    if (flag1 == 1) {
	printername = N;
    }
};


//
// User interface to save plots as GNUPLOT-datafiles
//

psave = function ( file, data )
{
    local (ans, i, nplot, flag1);

    flag1 = 0;
    // Check file-name
    if (class (file) != "string") {
	error ("Invalid string for file_name");
    }

    // Determine how many lines to save
    nplot = max ([1, data.nc - 1]);

    if (nplot > data.nr) {
	printf (" Save %i columns, are you sure [y(es)/n(o)/t(ranspose) (def. no)] ? ", data.nc);
	ans = getline ("stdin");
	if (ans.[1] != "y" && ans.[1] != "t") {
	    return 0;
	}
	if (ans.[1] == "t") {
	    data = data';
	    nplot = max ([1, data.nc - 1]);
	    flag1 = 1;
	}
    }

    // Generate two column matrices for saving
    for (i in 1:nplot) {

	// Write data to file
	if (nplot == 1) {
	    pobj_WriteData (real (data), file);
	    else
	    pobj_WriteData (real (data[;1,i+1]), file);
	}
    }

    // Check flag1
    if (flag1 == 1) {
	data = data';
    }
};



//
// User interface to save splots as GNUPLOT-datafiles
//

pssave = function ( file, datax, datay, datazz)
{
    local (ans, i, nsplot, m, n, flag2, flagx, flagy);

    // Reset flags
    flag2=0;
    flagx=0;
    flagy=0;

    // Determine how many plots to draw
    if (!exist (datazz)) { 
	datazz=datax;
	datax=1:datazz.nr;
	datax=datax';
	datay=1:datazz.nc;
	datay=datay';
	flag2=flag2+1;
    }

    if (datax.nc != 1 ) {
	if (datax.nr != 1 ) {
	    error ("Invalid data for x-axis (not a vector)");
	    else
	    datax = datax';
	    flagx=flagx+1;
	}
    }
    if (datay.nc != 1 ) {
	if (datay.nr != 1 ) {
	    error ("Invalid data for y-axis (not a vector)");
	    else
	    datay = datay';
	    flagy=flagy+1;
	}
    }

    // Create file-name
    if (!exist (file)) { error ("Invalid string for file_name"); }
    if (class (file) != "string") { error ("Invalid string for file_name"); }

    nsplot=(datazz.nr/datazz.nc)/(datax.nr/datay.nr);

    // Generate three column matrices for splot program
    // Write data to new data-file
    for (i in 1:nsplot) {
	newdata=[];
	for (m in 1:datax.nr){
	    rownumber=0;
	    for (n in 1:datay.nr){
		rownumber=rownumber+1;
		// x-value
		newdata[rownumber;1]=datax[m];
		// y-value
		newdata[rownumber;2]=datay[n];
		// z-value
		newdata[rownumber;3]=datazz[m;n+(i-1)*datax.nr];
	    }
	    write (file,real (newdata[;1,2,3]));
	    write (file,"");
	}
	close (file);
    }
    
    if (flag2 == 1) {
	datax=datazz;
    }
    if (flagx == 1) { datax=datax'; }
    if (flagy == 1) { datay=datay'; }

};



//
// Make multiplots
//

multiplot = function ( row, col, N )
{
    local (i, j, k)
    global( _MultiPlotParameter, _MultiOrginParameter );

    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    _MultiPlotParameter=[row,col,1,N];
    _MultiOrginParameter=[];
    printf (" Make %i plots!\n",_MultiPlotParameter[1]*_MultiPlotParameter[2]);	
    k=1;
    for (i in 1:row) {
	for (j in 1:col) {
	    _MultiOrginParameter[k;1]=(j-1)/col;
	    _MultiOrginParameter[k;2]=(row-i)/row;
	    k=k+1;
	}
    }
};



//
// Make plot with two y-axes
//

plotyy = function ( N )
{
    global( _PlotYyParameter );

    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    printf (" Make 2 plots!\n The first on left y-axis, the second on the right!\n");
    _PlotYyParameter=1;
};



//
// User interface to close a GNUPLOT-window
//

pclose = function ( N )
{
    local (i)
    if (!exist (N)) { N = 0; }
    if (N == "all") {
	for (i in 0:99) {
	    if (exist (p.[i])) {
		close (p.[i].rmf);
		close (p.[i].prog);
		pobj_Destroy (i);
	    }
	}
	else
	if (!exist (p.[N])) { return -1; }
	close (p.[N].rmf);
	close (p.[N].prog);
	pobj_Destroy (N);
    }
    return N;
};



//
// Set the terminal type for the I-th GNUPLOT process
//

setterm = function ( TERM, N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    p.[N].term = TERM;

    // Now send the "set term" command to GNUPLOT
    fprintf (p.[N].prog, "set term %s\n", TERM);
};



//
// Print out the plot list
//

showplot = function ( N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) {
	pobj_Create (N);
    }

    printf ("\tPlot List %i\n", N);
    printf ("\t\tTerm:\t\t\t%s\n", p.[N].term);
    printf ("\t\tOutput:\t\t\t%s\n", p.[N].output);
    printf ("\t\tTmp Files:\t\t%s\n", p.[N].files);
    //	printf ("\t\tHardCopy:\t\t%s\n", p.[N].hrd);
    //	printf ("\t\tHardCopy Files:\t\t%s\n", p.[N].hrdf);
};

showpwin = function ()
{
    local ( i )
    for ( i in 0:99) {
	if (exist (p.[i])) {
	    printf ("\tPlot window number %i exists!\n", i);
	}
    }
};

//
// End multiplot mode
//

nomultiplot = function(N)
{
    global( _MultiPlotParameter, _MultiOrginParameter, _PlotYyParameter );

    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set size %s,%s\n",1,1);
    fprintf (p.[N].prog, "set orgin %s,%s\n",0,0);
    fprintf (p.[N].prog, "set nomultiplot\n");
    if (exist (_MultiPlotParameter)) {clear(_MultiPlotParameter);}
    if (exist (_MultiOrginParameter)) {clear(_MultiOrginParameter);}
    if (exist (_PlotYyParameter)) {clear(_PlotYyParameter);}
};



//
// End plotyy mode
//

noplotyy = function(N)
{
    global( _PlotYyParameter );

    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }

    fprintf (p.[N].prog, "set nomultiplot\n");
    if (exist (_PlotYyParameter)) {clear(_PlotYyParameter);}
};



//
// Here comes a bunch of functions that really is not nessecary.
// They could be done with string("set key") etc, but this is more convinient.
//


xrange = function ( start, end, N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set xrange [%s:%s]\n",start,end);
};

yrange = function ( start, end, N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set yrange [%s:%s]\n",start,end);
};

zrange = function ( start, end, N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set zrange [%s:%s]\n",start,end);
};

range = function ( startx, endx, starty, endy, startz, endz, N )
{
    if (exist (startz) && !exist (endz)) { N = startz; }
    if (exist (starty) && !exist (endy)) { N = starty; }
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set xrange [%s:%s]\n",startx,endx);
    if (exist(endy)) {
	fprintf (p.[N].prog, "set yrange [%s:%s]\n",starty,endy);
    }
    if (exist(endz)) {
	fprintf (p.[N].prog, "set zrange [%s:%s]\n",startz,endz);
    }
};

xtics = function ( start, end, incr, N )
{
    local ( string, i )
    if (!exist (start)) { start = 0; }
    if (!exist (incr)) {
	if (length(start) > 1) {
	    if (!exist (end)) { end = 0; }
	    if (!exist (p.[end])) { pobj_Create (end); }
	    string = "set xtics (" + num2str(start[1])[1];
	    for (i in 2:length(start)) {
		string = string + "," + num2str(start[i])[1];
	    }
	    string = string + ")\n";
	    fprintf (p.[end].prog, string);
	    else
	    if (!exist (p.[start])) { pobj_Create (start); }
	    fprintf (p.[start].prog, "set xtics\n");
	}
	else
	if (!exist (N)) { N = 0; }
	if (!exist (p.[N])) { pobj_Create (N); }
	fprintf (p.[N].prog, "set xtics %s,%s,%s\n",start,incr,end);
    }
};

noxtics = function ( N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set noxtics\n");
};

ytics = function ( start, end, incr, N )
{
    local ( string, i )
    if (!exist (start)) { start = 0; }
    if (!exist (incr)) {
	if (length(start) > 1) {
	    if (!exist (end)) { end = 0; }
	    if (!exist (p.[end])) { pobj_Create (end); }
	    string = "set ytics (" + num2str(start[1])[1];
	    for (i in 2:length(start)) {
		string = string + "," + num2str(start[i])[1];
	    }
	    string = string + ")\n";
	    fprintf (p.[end].prog, string);
	    else
	    if (!exist (p.[start])) { pobj_Create (start); }
	    fprintf (p.[start].prog, "set ytics\n");
	}
	else
	if (!exist (N)) { N = 0; }
	if (!exist (p.[N])) { pobj_Create (N); }
	fprintf (p.[N].prog, "set ytics %s,%s,%s\n",start,incr,end);
    }
};

noytics = function ( N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set noytics\n");
};

ztics = function ( start, end, incr, N )
{
    local ( string, i )
    if (!exist (start)) { start = 0; }
    if (!exist (incr)) {
	if (length(start) > 1) {
	    if (!exist (end)) { end = 0; }
	    if (!exist (p.[end])) { pobj_Create (end); }
	    string = "set ztics (" + num2str(start[1])[1];
	    for (i in 2:length(start)) {
		string = string + "," + num2str(start[i])[1];
	    }
	    string = string + ")\n";
	    fprintf (p.[end].prog, string);
	    else
	    if (!exist (p.[start])) { pobj_Create (start); }
	    fprintf (p.[start].prog, "set ztics\n");
	}
	else
	if (!exist (N)) { N = 0; }
	if (!exist (p.[N])) { pobj_Create (N); }
	fprintf (p.[N].prog, "set ztics %s,%s,%s\n",start,incr,end);
    }
};

noztics = function ( N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set noztics\n");
};

loglog = function ( N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set nologscale\n");
    fprintf (p.[N].prog, "set logscale xy\n");
};

semilogx = function ( N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set logscale x\n");
};

semilogy = function ( N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set logscale y\n");
};

nolog = function ( N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set nologscale\n");
};

autoscale = function ( axis, N )
{
    local( flag )
    flag = 0;
    if (!exist (axis) && !exist (N)) {
	N = 0;
	axis = "xy";
    }
    if (exist (axis) && !exist (N)) {
	if (class (axis) == "num") {
	    N = axis;
	    axis = "xy";
	    flag = 1;
	}
    }
    if (!exist (p.[N])) { pobj_Create (N); }
    if (class(axis) != "string") {
	error ("Must be of class string");
    }
    fprintf (p.[N].prog, "set autoscale %s\n",axis);
    if (flag == 1) { axis = N; }
};

xlabel = function ( string, N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set xlabel \"%s\"\n",string);
};

ylabel = function ( string, N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set ylabel \"%s\"\n",string);
};

zlabel = function ( string, N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set zlabel \"%s\"\n",string);
};

title = function ( string, N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set title \"%s\"\n",string);
};

notitle = function ( N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set title\n");
};

linestyle = function ( string, N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set data style %s\n",string);
};

pformat = function ( string1, string2, N )
{
    local( flag1);
    flag1 = 0;
    if (!exist(string2)) {
	string2 = "xy";
    }
    if (exist (string2) && !exist (N)) {
	if (class (string2) == "num") {
	    N = string2;
	    string2 = "xy";
	    flag1 = 1;
	}
    }
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set format %s \"%s\"\n",string2,string1);
    if (flag1 == 1) {
	string1 = string2;
	string2 = N;
    }
};


key = function ( x, y, z, N )
{
    local ( flag1 )
    flag1 = 0;
    if (exist(x) && !exist(y)) {
	N = x;
	flag1 = 1;
    }
    if (!exist (z)) { z = 0; }
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    if (exist(y)) {
	fprintf (p.[N].prog, "set key %s,%s,%s\n",x,y,z);
	else
	fprintf (p.[N].prog, "set key\n");
    }
    if (flag1 == 1) {
	x = N;
    }
};

nokey = function ( N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set nokey\n");
};

grid = function ( N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set grid\n");
};

nogrid = function ( N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set nogrid\n");
};

axis = function ( N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set xzeroaxis\n");
    fprintf (p.[N].prog, "set yzeroaxis\n");
};

noaxis = function ( N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set noxzeroaxis\n");
    fprintf (p.[N].prog, "set noyzeroaxis\n");
};

label = function ( label, x, y, z, labelnr, N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    if (class(label) != "string") {
	error ("The label must be a string!")
    }
    if (exist (labelnr)) {
	fprintf (p.[N].prog, "set label %s \"%s\" at %s,%s\n", label, labelnr , x, y, z);
	else
	fprintf (p.[N].prog, "set label \"%s\" at %s,%s\n", label, x, y, z);
    }
};

nolabel = function ( labelnr, N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    if (exist (labelnr)) {
	fprintf (p.[N].prog, "set nolabel %i\n", labelnr);
	else
	fprintf (p.[N].prog, "set nolabel\n");
    }
};

time = function ( N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set time\n");
};

notime = function ( N )
{
    if (!exist (N)) { N = 0; }
    if (!exist (p.[N])) { pobj_Create (N); }
    fprintf (p.[N].prog, "set notime\n");
};


//====================
// Static Functions	=
//====================

plotl = function ( data, N, titles )
{
    local (I, i);
    I = 1;
    for (i in members (data)) {
	if (class (data.[i]) == "num") {
	    // Check for plotting titles
	    pobj_PlotKeyTitle (titles, I, title );

	    plotm (data.[i], N, I, title);
	    I++;
	}
    }
};



//
// Set-Up to plot a matrix. Columns 2...N against 1st column.
//

plotm = function ( data, N, I, titles )
{
    local (ans, fn, i, nplot, title);
    global( _RePlotParameter );
    
    flag1 = 0;
    // Determine how many lines to draw
    nplot = max ([1, data.nc - 1]);

    if (nplot > data.nr) {
	printf (" Plot %i columns, are you sure [y(es)/n(o)/t(ranspose) (def. no)] ? ", data.nc);
	ans = getline ("stdin");
	if (ans.[1] != "y" && ans.[1] != "t") {
	    return 0;
	}
	if (ans.[1] == "t") {
	    data = data';
	    nplot = max ([1, data.nc - 1]);
	    flag1 = 1;
	}
    }

    // Generate two column matrices for plot program
    _RePlotParameter[N+1] = 0;
    for (i in 1:nplot) {
	// Create tmp-file-name
	fn = pobj_TmpFileName (N, i, I);

	// Check for plotting titles
	pobj_PlotKeyTitle (titles, i, title );

	// Write data to tmp-file
	// Add to plot command
	if (nplot == 1) {
	    pobj_WriteData (real (data), fn);
	    pobj_PlotCmd (N, fn, title);
	    else
	    pobj_WriteData (real (data[;1,i+1]), fn);
	    pobj_PlotCmd (N, fn, title);
	}
    }
    
    // Check flag1
    if (flag1 == 1) {
	data = data';
    }

};

replotm = function ( data, N, I, titles )
{
    local ( ans, fn, i, nplot, title );
    global( _RePlotParameter );
    
    flag1 = 0;
    // Determine how many lines to draw
    nplot = max ([1, data.nc - 1]);

    if (nplot > data.nr) {
	printf (" Plot %i columns, are you sure [y(es)/n(o)/t(ranspose) (def. no)] ? ", data.nc);
	ans = getline ("stdin");
	if (ans.[1] != "y" && ans.[1] != "t") {
	    return 0;
	}
	if (ans.[1] == "t") {
	    data = data';
	    nplot = max ([1, data.nc - 1]);
	    flag1 = 1;
	}
    }

    // Generate two column matrices for plot program
    for (i in 1+_RePlotParameter[N+1]:nplot+_RePlotParameter[N+1]) {
	// Create tmp-file-name
	fn = pobj_TmpFileName (N, i, I);

	// Check for plotting titles
	pobj_PlotKeyTitle (titles, i, title );

	// Write data to tmp-file
	// Add to plot command
	if (nplot == 1) {
	    pobj_WriteData (real (data), fn);
	    pobj_RePlotCmd (N, fn, title);
	    else
	    pobj_WriteData (real (data[;1,i+1]), fn);
	    pobj_RePlotCmd (N, fn, title);
	}
    }
    
    // Check flag1
    if (flag1 == 1) {
	data = data';
    }

};


//
// Set-Up to splot a matrix. 
//

splotm = function ( datax, datay, datazz, N, I, titles )
{
    local ( ans, fn, i, nsplot, m, n, title );
    global( _RePlotParameter );
    // Determine how many plots to draw
    nsplot=(datazz.nc/datazz.nr)/(datax.nr/datay.nr);

    // Generate three column matrices for splot program
    _RePlotParameter[N+1] = 0;
    // Write data to new data-file
    // Create tmp-file-name
    for (i in 1:nsplot) {
	fn = pobj_TmpFileName (N, i, I);

	// Check for plotting titles
	pobj_PlotKeyTitle (titles, I, title );
	newdata=[];
	for (m in 1:datax.nr){
	    rownumber=0;
	    for (n in 1:datay.nr){
		rownumber=rownumber+1;
		// x-value
		newdata[rownumber;1]=datax[m];
		// y-value
		newdata[rownumber;2]=datay[n];
		// z-value
		newdata[rownumber;3]=datazz[m;n+(i-1)*datax.nr];
	    }
	    write (fn,real (newdata[;1,2,3]));
	    write (fn,"");
	}
	close (fn);
	// Add to splot command
	pobj_SPlotCmd (N, fn, title);
    }
};



//
// Set-Up to splot a matrix. 
//




//
// Form a plain string to send to GNUPLOT as 
// command
//

plots = function ( data, N )
{
    // Send the string to GNUPLOT

    p.[N].pcmd = data + "\n";
};



//
// Create a plot-object.
//

pobj_Create = function ( N )
{
    local (plist);

    // The tmp files to plot
    plist.files = "";
    // Where the plot command will go
    plist.pcmd = "";

    // Init string for plotting program
    plist.init = "set nogrid\nset data style lines\nset nokey\nset noxzeroaxis\nset noyzeroaxis\n";

    // The program that draws the plot(s)
    sprintf (plist.prog, "|gnuplot #%i", N);

    // To remove tmp-files

    sprintf (plist.rmf, "|rm -f `cat` #%i", N);

    // Keep track of terminal type, and output
    plist.term = "X11";
    plist.output = "stdout";

    // Copy the local list into the static plot-object collection
    p.[N] = plist;
    fprintf (p.[N].prog, "%s", p.[N].init);
    fprintf (p.[N].prog, "set term %s\n", p.[N].term);
};



//
// Reset a plot object to plot new data
//

pobj_Reset = function ( N )
{

    // Close any existing tmp-files that belong
    // to plot-object N. Reset the file-name list.
    p.[N].files = "";
    p.[N].pcmd = "";
};



//
// Destroy a plot-object
//

pobj_Destroy = function ( N )
{
    if (exist (p.[N])) {
	clear (p.[N]);	
    }
};



//
// Create a tmp-file name
//

pobj_TmpFileName = function ( N, i, j )
{
    local (tmp);
    sprintf (tmp, "/usr/tmp/rlab-tmpf-%i-%i-%i", N, i, j);
    p.[N].files = p.[N].files + " " + tmp;
    return tmp;
};



//
// Add data-file (tmp-file) to plot object list of stuff
// to be plotted.
//

pobj_WriteData = function ( m, file )
{
    write (file, m);
    close (file);
};



//
// Create the command(s) to plot all the data in the plot object.
//

pobj_PlotCmd = function ( N, fn, title )
{
    local (tmp);

    if(p.[N].pcmd == "") {
	sprintf(tmp, "plot '%s' title '%s'", fn, title);
	else
	sprintf(tmp,		", '%s' title '%s'", fn, title);
    }

    p.[N].pcmd = p.[N].pcmd + tmp;
};



//
// Create the command(s) to replot all the data in the plot object.
//

pobj_RePlotCmd = function ( N, fn, title )
{
    local (tmp);

    if(p.[N].pcmd == "") {
	sprintf(tmp, "replot '%s' title '%s'", fn, title);
	else
	sprintf(tmp,		", '%s' title '%s'", fn, title);
    }

    p.[N].pcmd = p.[N].pcmd + tmp;
};


//
// Create the command(s) to splot all the data in the splot object.
//

pobj_SPlotCmd = function ( N, fn, title )
{
    local (tmp);

    if(p.[N].pcmd == "") {
	sprintf(tmp, "splot '%s' title '%s'", fn, title);
	else
	sprintf(tmp,		", '%s' title '%s'", fn, title);
    }

    p.[N].pcmd = p.[N].pcmd + tmp;
};



//
// Check for plotting titles
//

pobj_PlotKeyTitle = function (titles, i, title )
{
    if (!exist (titles)) {
	sprintf (title, "%.4g", i+1);
	title = "C-" + title;
	else
	if (i > length(titles)) {
	    sprintf (title, "%.4g", i+1);
	    title = "C-" + title;
	    else
	    title = titles[i];
	}
    }
};



//
// Force the plot program to create the plot
//

pobj_Plot = function ( N )
{
    //	fprintf ("stdout", "%s\n", p.[N].pcmd);
    fprintf (p.[N].prog, "%s\n", p.[N].pcmd);
};



//
// Setup so the tmp-files can be removed
//

pobj_SetRm = function ( N )
{
    fprintf(p.[N].rmf, "%s", p.[N].files);
};



//
// Remove the tmp-files
//

pobj_Rm = function ( N )
{
    if (length (p.[N].files) != 0) {
	close (p.[N].rmf);
    }
};



//
// Provide access to the plot-list
//

_pobj_list = function ( N )
{
    if (!exist (N)) { N = 0; }
    return p.[N];
};

