/*
 *  @(#)PathParserImpl.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.mbtf.v1.engine;


import net.sourceforge.groboutils.mbtf.v1.IPath;
import net.sourceforge.groboutils.mbtf.v1.IState;
import net.sourceforge.groboutils.mbtf.v1.IAction;
import net.sourceforge.groboutils.mbtf.v1.ISystem;
import net.sourceforge.groboutils.mbtf.v1.IErrors;
import net.sourceforge.groboutils.mbtf.v1.IValidate;
import net.sourceforge.groboutils.mbtf.v1.ITransition;
import net.sourceforge.groboutils.mbtf.v1.IPathParser;
import net.sourceforge.groboutils.mbtf.v1.IPathHistory;
import net.sourceforge.groboutils.mbtf.v1.IPathIterator;
import net.sourceforge.groboutils.mbtf.v1.TestHaltRuntimeException;
import net.sourceforge.groboutils.mbtf.v1.TestFailRuntimeException;

import org.apache.log4j.Logger;


/**
 * Knows how to parse an IPath instance, and correctly follow the transitions
 * and record errors.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version    $Date: 2003/02/10 22:52:26 $
 * @since      June 12, 2002
 */
public class PathParserImpl implements IPathParser
{
    private static final Logger LOG = Logger.getLogger( PathParserImpl.class );
    
    /**
     * @exception TestHaltRuntimeException if the path validation reports
     *      a <tt>halt()</tt> to <tt>errors</tt>.
     */
    public IErrors parsePath( IPath path, ISystem sys )
    {
        ErrorsImpl ei = createErrors();
        PathHistoryImpl phi = createPathHistory();
        
        IState currentState = path.getStartState();
        IPathIterator pi = path.iterator();
        while (currentState != null
                && !ei.isHaltPath())
        {
            ErrorsImpl testErrs = createErrors();
            try
            {
                testErrs.setCurrentPathHistory( phi );
                LOG.debug( "Entering state "+currentState );
                phi.addState( currentState );
                LOG.debug( "Validating state "+currentState );
                validate( currentState, testErrs, phi, sys );
                currentState = null;
                if (!testErrs.isHaltPath() && pi.hasNext())
                {
                    ITransition trans = pi.nextTransition();
                    phi.addTransition( trans );
                    LOG.debug( "Vaidatling transition "+trans );
                    validate( trans, testErrs, phi, sys );
                    if (!testErrs.isHaltPath())
                    {
                        IAction act = trans.getAction();
                        LOG.debug( "Start Performing action "+act );
                        act.performAction( sys, testErrs );
                        LOG.debug( "End Performing action "+act );
                        currentState = trans.getDestinationState();
                    }
                }
            }
            catch (TestHaltRuntimeException thre)
            {
                // should have already been caught
                throw thre;
            }
            catch (ThreadDeath td)
            {
                // never catch these.  never.
                throw td;
            }
            catch (TestFailRuntimeException tfre)
            {
                // The error was already added to the error set.
                // This exception ends this path, but the exception is
                // not propigated.
            }
            catch (Throwable t)
            {
                testErrs.addError( "Uncaught exception", t );
                if (t instanceof RuntimeException)
                {
                    throw (RuntimeException)t;
                }
                t.printStackTrace();
                throw new IllegalStateException( t.toString() );
            }
            finally
            {
                ei.addErrors( testErrs );
            }
        }
        
        return ei;
    }
    
    
    protected void validate( IState state, ErrorsImpl ei, PathHistoryImpl phi,
            ISystem sys )
            throws CloneNotSupportedException
    {
        IValidate[] v = state.getValidates();
        if (v.length <= 0)
        {
            ei.setCurrentPathHistory( phi );
            ei.addError( "No validations for state '"+state.getName()+"'" );
            return;
        }
        for (int i = 0; i < v.length && !ei.isHaltPath(); ++i)
        {
            ei.setCurrentPathHistory( phi );
            v[i].validate( sys, ei );
            phi.addValidate( v[i], state, ei );
            ei.setCurrentPathHistory( phi );
        }
    }
    
    
    protected void validate( ITransition trans, ErrorsImpl ei,
            PathHistoryImpl phi, ISystem sys )
            throws CloneNotSupportedException
    {
        IValidate[] v = trans.getValidates();
        if (v.length <= 0)
        {
            ei.setCurrentPathHistory( phi );
            ei.addError( "No validations for transition '"+
                trans.getName()+"'" );
            return;
        }
        for (int i = 0; i < v.length && !ei.isHaltPath(); ++i)
        {
            ei.setCurrentPathHistory( (IPathHistory)phi.clone() );
            v[i].validate( sys, ei );
            phi.addValidate( v[i], trans, ei );
            ei.setCurrentPathHistory( (IPathHistory)phi.clone() );
        }
    }
    
    
    protected ErrorsImpl createErrors()
    {
        return new ErrorsImpl();
    }
    
    
    protected PathHistoryImpl createPathHistory()
    {
        return new PathHistoryImpl();
    }
}

