/* Driver for SanDisk SDDR-09 SmartMedia reader
 *
 *   (c) 2000, 2001 Robert Baruch (autophile@starband.net)
 *   (c) 2002 Andries Brouwer (aeb@cwi.nl)
 *
 * The SanDisk SDDR-09 SmartMedia reader uses the Shuttle EUSB-01 chip.
 * This chip is a programmable USB controller. In the SDDR-09, it has
 * been programmed to obey a certain limited set of SCSI commands.
 * This driver translates the "real" SCSI commands to the SDDR-09 SCSI
 * commands.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Known vendor commands:
 *
 * E7: read scatter gather
 * E8: read
 * E9: write
 * EA: erase
 * EB: reset
 * EC: read status
 * ED: read ID
 * EE: write CIS (?)
 * EF: compute checksum (?)
 */

#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/slab.h>

#include "transport.h"
#include "protocol.h"
#include "raw_bulk.h"
#include "smartmedia.h"
#include "usb.h"
#include "debug.h"
#include "sddr09.h"

#define short_pack(lsb,msb) ( ((u16)(lsb)) | ( ((u16)(msb))<<8 ) )
#define LSB_of(s) ((s)&0xFF)
#define MSB_of(s) ((s)>>8)

/* #define US_DEBUGP printk */

/*
 * On my 16MB card, control blocks have size 64 (16 real control bytes,
 * and 48 junk bytes). In reality of course the card uses 16 control bytes,
 * so the reader makes up the remaining 48. Don't know whether these numbers
 * depend on the card. For now a constant.
 */
#define CONTROL_SHIFT 6

/*
 * On my Combo CF/SM reader, the SM reader has LUN 1.
 * (and things fail with LUN 0).
 * It seems LUN is irrelevant for others.
 */
#define LUN	1
#define	LUNBITS	(LUN << 5)

static int erase_bad_lba_entries = 0;

/* send vendor interface command (0x41) */
/* called for requests 0, 1, 8 */
static int
sddr09_send_command(struct us_data *us,
		    unsigned char request,
		    unsigned char direction,
		    unsigned char *xfer_data,
		    unsigned int xfer_len) {
	int pipe;
	unsigned char requesttype = (0x41 | direction);

	// Get the receive or send control pipe number

	if (direction == USB_DIR_IN)
		pipe = usb_rcvctrlpipe(us->pusb_dev,0);
	else
		pipe = usb_sndctrlpipe(us->pusb_dev,0);

	return usb_storage_send_control(us, pipe, request, requesttype,
					0, 0, xfer_data, xfer_len);
}

static int
sddr09_send_scsi_command(struct us_data *us,
			 unsigned char *command,
			 unsigned int command_len) {
	return sddr09_send_command(us, 0, USB_DIR_OUT, command, command_len);
}

#if 0
/*
 * Test Unit Ready Command: 12 bytes.
 * byte 0: opcode: 00
 */
static int
sddr09_test_unit_ready(struct us_data *us) {
	unsigned char command[6] = {
		0, LUNBITS, 0, 0, 0, 0
	};
	int result;

	result = sddr09_send_scsi_command(us, command, sizeof(command));

	US_DEBUGP("sddr09_test_unit_ready returns %d\n", result);

	return result;
}
#endif

/*
 * Request Sense Command: 12 bytes.
 * byte 0: opcode: 03
 * byte 4: data length
 */
static int
sddr09_request_sense(struct us_data *us, unsigned char *sensebuf, int buflen) {
	unsigned char command[12] = {
		0x03, LUNBITS, 0, 0, buflen, 0, 0, 0, 0, 0, 0, 0
	};
	int result;
	unsigned int act_len;

	result = sddr09_send_scsi_command(us, command, sizeof(command));
	if (result != USB_STOR_TRANSPORT_GOOD) {
		US_DEBUGP("request sense failed\n");
		return result;
	}

	result = usb_storage_raw_bulk(us, SCSI_DATA_READ, sensebuf,
				      buflen, &act_len);
	if (result != USB_STOR_TRANSPORT_GOOD)
		US_DEBUGP("request sense bulk in failed\n");
	else
		US_DEBUGP("request sense worked\n");

	return result;
}

/*
 * Read Command: 12 bytes.
 * byte 0: opcode: E8
 * byte 1: last two bits: 00: read data, 01: read blockwise control,
 *			10: read both, 11: read pagewise control.
 *	 It turns out we need values 20, 21, 22, 23 here (LUN 1).
 * bytes 2-5: address (interpretation depends on byte 1, see below)
 * bytes 10-11: count (idem)
 *
 * A page has 512 data bytes and 64 control bytes (16 control and 48 junk).
 * A read data command gets data in 512-byte pages.
 * A read control command gets control in 64-byte chunks.
 * A read both command gets data+control in 576-byte chunks.
 *
 * Blocks are groups of 32 pages, and read blockwise control jumps to the
 * next block, while read pagewise control jumps to the next page after
 * reading a group of 64 control bytes.
 * [Here 512 = 1<<pageshift, 32 = 1<<blockshift, 64 is constant?]
 *
 * (1 MB and 2 MB cards are a bit different, but I have only a 16 MB card.)
 */

static int
sddr09_readX(struct us_data *us, int x, unsigned long fromaddress,
	     int nr_of_pages, int bulklen, unsigned char *buf,
	     int use_sg) {

	unsigned char command[12] = {
		0xe8, LUNBITS | x, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
	};
	int result;

	command[2] = MSB_of(fromaddress>>16);
	command[3] = LSB_of(fromaddress>>16); 
	command[4] = MSB_of(fromaddress & 0xFFFF);
	command[5] = LSB_of(fromaddress & 0xFFFF); 

	command[10] = MSB_of(nr_of_pages);
	command[11] = LSB_of(nr_of_pages);

	result = sddr09_send_scsi_command(us, command, sizeof(command));

	if (result != USB_STOR_TRANSPORT_GOOD) {
		US_DEBUGP("Result for send_control in sddr09_read2%d %d\n",
			  x, result);
		return result;
	}

	result = usb_storage_bulk_transport(us, SCSI_DATA_READ,
				       buf, bulklen, use_sg);

	if (result != USB_STOR_TRANSPORT_GOOD)
		US_DEBUGP("Result for bulk_transport in sddr09_read2%d %d\n",
			  x, result);

	return result;
}

/*
 * Read Data
 *
 * fromaddress counts data shorts:
 * increasing it by 256 shifts the bytestream by 512 bytes;
 * the last 8 bits are ignored.
 *
 * nr_of_pages counts pages of size (1 << pageshift).
 */
static int
sddr09_read20(struct us_data *us, unsigned long fromaddress,
	      int nr_of_pages, int pageshift, unsigned char *buf, int use_sg) {
	int bulklen = nr_of_pages << pageshift;

	/* The last 8 bits of fromaddress are ignored. */
	return sddr09_readX(us, 0, fromaddress, nr_of_pages, bulklen,
			    buf, use_sg);
}

/*
 * Read Blockwise Control
 *
 * fromaddress gives the starting position (as in read data;
 * the last 8 bits are ignored); increasing it by 32*256 shifts
 * the output stream by 64 bytes.
 *
 * count counts control groups of size (1 << controlshift).
 * For me, controlshift = 6. Is this constant?
 *
 * After getting one control group, jump to the next block
 * (fromaddress += 8192).
 */
static int
sddr09_read21(struct us_data *us, unsigned long fromaddress,
	      int count, int controlshift, unsigned char *buf, int use_sg) {

	int bulklen = (count << controlshift);
	return sddr09_readX(us, 1, fromaddress, count, bulklen,
			    buf, use_sg);
}

/*
 * Read both Data and Control
 *
 * fromaddress counts data shorts, ignoring control:
 * increasing it by 256 shifts the bytestream by 576 = 512+64 bytes;
 * the last 8 bits are ignored.
 *
 * nr_of_pages counts pages of size (1 << pageshift) + (1 << controlshift).
 */
static int
sddr09_read22(struct us_data *us, unsigned long fromaddress,
	      int nr_of_pages, int pageshift, unsigned char *buf, int use_sg) {

	int bulklen = (nr_of_pages << pageshift) + (nr_of_pages << CONTROL_SHIFT);
	US_DEBUGP("sddr09_read22: reading %d pages, %d bytes\n",
		  nr_of_pages, bulklen);
	return sddr09_readX(us, 2, fromaddress, nr_of_pages, bulklen,
			    buf, use_sg);
}

#if 0
/*
 * Read Pagewise Control
 *
 * fromaddress gives the starting position (as in read data;
 * the last 8 bits are ignored); increasing it by 256 shifts
 * the output stream by 64 bytes.
 *
 * count counts control groups of size (1 << controlshift).
 * For me, controlshift = 6. Is this constant?
 *
 * After getting one control group, jump to the next page
 * (fromaddress += 256).
 */
static int
sddr09_read23(struct us_data *us, unsigned long fromaddress,
	      int count, int controlshift, unsigned char *buf, int use_sg) {

	int bulklen = (count << controlshift);
	return sddr09_readX(us, 3, fromaddress, count, bulklen,
			    buf, use_sg);
}
#endif

/*
 * Erase Command: 12 bytes.
 * byte 0: opcode: EA
 * bytes 6-9: erase address (big-endian, counting shorts, sector aligned).
 * 
 * Always precisely one block is erased; bytes 2-5 and 10-11 are ignored.
 * The byte address being erased is 2*Eaddress.
 */
static int
sddr09_erase(struct us_data *us, unsigned long Eaddress) {
	unsigned char command[12] = {
		0xea, LUNBITS, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
	};
	int result;

	command[6] = MSB_of(Eaddress>>16);
	command[7] = LSB_of(Eaddress>>16);
	command[8] = MSB_of(Eaddress & 0xFFFF);
	command[9] = LSB_of(Eaddress & 0xFFFF);

	result = sddr09_send_scsi_command(us, command, sizeof(command));

	if (result != USB_STOR_TRANSPORT_GOOD)
		US_DEBUGP("Result for send_control in sddr09_erase %d\n",
			  result);

	return result;
}

/*
 * Write CIS Command: 12 bytes.
 * byte 0: opcode: EE
 * bytes 2-5: write address in shorts
 * bytes 10-11: sector count
 */
static int
sddr09_writeCIS(struct us_data *us, int nr_of_pages, unsigned char *buf) {
	struct sm_card_info *info = us->extra;

	unsigned char command[12] = {
		0xee, LUNBITS, 0, 0, 0x10, 0, 0, 0, 0, 0, 0, 0x01
	};
	int bulklen, result;

	command[11] = nr_of_pages;
	bulklen = (nr_of_pages << info->pageshift)
		+ (nr_of_pages << CONTROL_SHIFT);
	result = sddr09_send_scsi_command(us, command, sizeof(command));

	if (result != USB_STOR_TRANSPORT_GOOD)
		return result;

	result = usb_storage_bulk_transport(us, SCSI_DATA_WRITE,
					    buf, bulklen, 0);

	if (result != USB_STOR_TRANSPORT_GOOD)
		US_DEBUGP("Result for bulk_transport in sddr09_writeCIS %d\n",
			  result);

	return result;
}

/*
 * Write Command: 12 bytes.
 * byte 0: opcode: E9
 * bytes 2-5: write address (big-endian, counting shorts, sector aligned).
 * bytes 6-9: erase address (big-endian, counting shorts, sector aligned).
 * bytes 10-11: sector count (big-endian, in 512-byte sectors).
 *
 * If write address equals erase address, the erase is done first,
 * otherwise the write is done first. When erase address equals zero
 * no erase is done?
 */
static int
sddr09_writeX(struct us_data *us,
	      unsigned long Waddress, unsigned long Eaddress,
	      int nr_of_pages, int bulklen, unsigned char *buf, int use_sg) {

	unsigned char command[12] = {
		0xe9, LUNBITS, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
	};
	int result;

	command[2] = MSB_of(Waddress>>16);
	command[3] = LSB_of(Waddress>>16);
	command[4] = MSB_of(Waddress & 0xFFFF);
	command[5] = LSB_of(Waddress & 0xFFFF);

	command[6] = MSB_of(Eaddress>>16);
	command[7] = LSB_of(Eaddress>>16);
	command[8] = MSB_of(Eaddress & 0xFFFF);
	command[9] = LSB_of(Eaddress & 0xFFFF);

	command[10] = MSB_of(nr_of_pages);
	command[11] = LSB_of(nr_of_pages);

	result = sddr09_send_scsi_command(us, command, sizeof(command));

	if (result != USB_STOR_TRANSPORT_GOOD)
		return result;

	result = usb_storage_bulk_transport(us, SCSI_DATA_WRITE,
				       buf, bulklen, use_sg);

	if (result != USB_STOR_TRANSPORT_GOOD)
		US_DEBUGP("Result for bulk_transport in sddr09_writeX %d\n",
			  result);

	return result;
}

/* erase address, write same address */
static int
sddr09_write_inplace(struct us_data *us, unsigned long address,
		     int nr_of_pages, int pageshift, unsigned char *buf,
		     int use_sg) {
	int bulklen = (nr_of_pages << pageshift) + (nr_of_pages << CONTROL_SHIFT);
	return sddr09_writeX(us, address, address, nr_of_pages, bulklen,
			     buf, use_sg);
}

#if 0
/*
 * Read Scatter Gather Command: 3+4n bytes.
 * byte 0: opcode E7
 * byte 2: n
 * bytes 4i-1,4i,4i+1: page address
 * byte 4i+2: page count
 * (i=1..n)
 *
 * This reads several pages from the card to a single memory buffer.
 * The last two bits of byte 1 have the same meaning as for E8.
 */
static int
sddr09_read_sg_test_only(struct us_data *us) {
	unsigned char command[15] = {
		0xe7, LUNBITS, 0
	};
	int result, bulklen, nsg, ct;
	unsigned char *buf;
	unsigned long address;

	nsg = bulklen = 0;

	address = 040000; ct = 1;
	nsg++;
	bulklen += (ct << 9);
	command[4*nsg+2] = ct;
	command[4*nsg+1] = ((address >> 9) & 0xFF);
	command[4*nsg+0] = ((address >> 17) & 0xFF);
	command[4*nsg-1] = ((address >> 25) & 0xFF);

	address = 0340000; ct = 1;
	nsg++;
	bulklen += (ct << 9);
	command[4*nsg+2] = ct;
	command[4*nsg+1] = ((address >> 9) & 0xFF);
	command[4*nsg+0] = ((address >> 17) & 0xFF);
	command[4*nsg-1] = ((address >> 25) & 0xFF);

	address = 01000000; ct = 2;
	nsg++;
	bulklen += (ct << 9);
	command[4*nsg+2] = ct;
	command[4*nsg+1] = ((address >> 9) & 0xFF);
	command[4*nsg+0] = ((address >> 17) & 0xFF);
	command[4*nsg-1] = ((address >> 25) & 0xFF);

	command[2] = nsg;

	result = sddr09_send_scsi_command(us, command, 4*nsg+3);

	if (result != USB_STOR_TRANSPORT_GOOD) {
		US_DEBUGP("Result for send_control in sddr09_read_sg %d\n",
			  result);
		return result;
	}

	buf = (unsigned char *) kmalloc(bulklen, GFP_NOIO);
	if (!buf)
		return USB_STOR_TRANSPORT_ERROR;

	result = usb_storage_bulk_transport(us, SCSI_DATA_READ,
				       buf, bulklen, 0);
	if (result != USB_STOR_TRANSPORT_GOOD)
		US_DEBUGP("Result for bulk_transport in sddr09_read_sg %d\n",
			  result);

	kfree(buf);

	return result;
}
#endif

/*
 * Read Status Command: 12 bytes.
 * byte 0: opcode: EC
 *
 * Returns 64 bytes, all zero except for the first.
 * bit 0: 1: Error
 * bit 5: 1: Suspended
 * bit 6: 1: Ready
 * bit 7: 1: Not write-protected
 */

static int
sddr09_read_status(struct us_data *us, unsigned char *status) {

	unsigned char command[12] = {
		0xec, LUNBITS, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
	};
	unsigned char data[64];
	int result;

	US_DEBUGP("Reading status...\n");

	result = sddr09_send_scsi_command(us, command, sizeof(command));
	if (result != USB_STOR_TRANSPORT_GOOD)
		return result;

	result = usb_storage_bulk_transport(us, SCSI_DATA_READ,
				       data, sizeof(data), 0);
	*status = data[0];
	return result;
}

/*
 * Check Checksum Command: 12 bytes.
 * byte 0: opcode: EF
 *
 * Returns 64 bytes, all zero except for the first four.
 * These should be FF FF FF FF.
 */

static int
sddr09_check_checksum(struct us_data *us) {

	unsigned char command[12] = {
		0xef, LUNBITS, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
	};
	unsigned char data[64];
	int result, i;

	US_DEBUGP("Reading checksum...\n");

	result = sddr09_send_scsi_command(us, command, sizeof(command));
	if (result != USB_STOR_TRANSPORT_GOOD)
		return result;

	result = usb_storage_bulk_transport(us, SCSI_DATA_READ,
				       data, sizeof(data), 0);

	printk("checksum bytes:\n");
	for (i=0; i<64; i++) {
		printk(" %02X", data[i]);
		if (i%16 == 15)
			printk("\n");
	}
	return result;
}

/* we never free blocks, so lastpba can only increase */
static unsigned int
sddr09_find_unused_pba(struct sm_card_info *info) {
	static unsigned int lastpba = 1;
	int numblocks = info->capacity >> (info->blockshift + info->pageshift);
	int i;

	for (i = lastpba+1; i < numblocks; i++) {
		if (info->pba_to_lba[i] == UNDEF) {
			lastpba = i;
			return i;
		}
	}
	return 0;
}

static int
sddr09_write_lba(struct us_data *us, unsigned int lba,
		 unsigned int page, unsigned int pages,
		 unsigned char *ptr) {
	struct sm_card_info *info = us->extra;
	unsigned long address;
	unsigned int pba, lbap;
	unsigned int pagelen, blocklen;
	unsigned char *blockbuffer, *bptr, *cptr, *xptr;
	unsigned char ecc[3];
	int i, result;

	lbap = ((lba & 0x3ff) << 1) | 0x1000;
	if (nand_parity[MSB_of(lbap) ^ LSB_of(lbap)])
		lbap ^= 1;
	pba = info->lba_to_pba[lba];

	if (pba == UNDEF) {
		pba = sddr09_find_unused_pba(info);
		if (!pba) {
			printk("sddr09_write_lba: Out of unused blocks\n");
			return USB_STOR_TRANSPORT_ERROR;
		}
		info->pba_to_lba[pba] = lba;
		info->lba_to_pba[lba] = pba;
	}
#if 0
	if (pba == 1) {
		/* Maybe it is impossible to write to PBA 1.
		   Fake success, but don't do anything. */
		printk("sddr09: avoid writing to pba 1\n");
		return USB_STOR_TRANSPORT_GOOD;
	}
#endif

	pagelen = (1 << info->pageshift) + (1 << CONTROL_SHIFT);
	blocklen = (pagelen << info->blockshift);
	blockbuffer = kmalloc(blocklen, GFP_NOIO);
	if (!blockbuffer) {
		printk("sddr09_write_lba: Out of memory\n");
		return USB_STOR_TRANSPORT_ERROR;
	}

	/* read old contents */
	address = (pba << (info->pageshift + info->blockshift));
	result = sddr09_read22(us, address>>1, info->blocksize,
			       info->pageshift, blockbuffer, 0);
	if (result != USB_STOR_TRANSPORT_GOOD)
		goto err;

	/* check old contents */
	for (i = 0; i < info->blockshift; i++) {
		bptr = blockbuffer + i*pagelen;
		cptr = bptr + info->pagesize;
		nand_compute_ecc(bptr, ecc);
		if (!nand_compare_ecc(cptr+13, ecc)) {
			US_DEBUGP("Warning: bad ecc in page %d- of pba %d\n",
				  i, pba);
			nand_store_ecc(cptr+13, ecc);
		}
		nand_compute_ecc(bptr+(info->pagesize / 2), ecc);
		if (!nand_compare_ecc(cptr+8, ecc)) {
			US_DEBUGP("Warning: bad ecc in page %d+ of pba %d\n",
				  i, pba);
			nand_store_ecc(cptr+8, ecc);
		}
	}

	/* copy in new stuff and compute ECC */
	xptr = ptr;
	for (i = page; i < page+pages; i++) {
		bptr = blockbuffer + i*pagelen;
		cptr = bptr + info->pagesize;
		memcpy(bptr, xptr, info->pagesize);
		xptr += info->pagesize;
		nand_compute_ecc(bptr, ecc);
		nand_store_ecc(cptr+13, ecc);
		nand_compute_ecc(bptr+(info->pagesize / 2), ecc);
		nand_store_ecc(cptr+8, ecc);
		cptr[6] = cptr[11] = MSB_of(lbap);
		cptr[7] = cptr[12] = LSB_of(lbap);
	}

	US_DEBUGP("Rewrite PBA %d (LBA %d)\n", pba, lba);

	result = sddr09_write_inplace(us, address>>1, info->blocksize,
				      info->pageshift, blockbuffer, 0);

	US_DEBUGP("sddr09_write_inplace returns %d\n", result);

#if 0
	{
		unsigned char status = 0;
		int result2 = sddr09_read_status(us, &status);
		if (result2 != USB_STOR_TRANSPORT_GOOD)
			US_DEBUGP("sddr09_write_inplace: cannot read status\n");
		else if (status != 0xc0)
			US_DEBUGP("sddr09_write_inplace: status after write: 0x%x\n",
				  status);
	}
#endif

#if 0
	{
		int result2 = sddr09_test_unit_ready(us);
	}
#endif
 err:
	kfree(blockbuffer);

	/* TODO: instead of doing kmalloc/kfree for each block,
	   add a bufferpointer to the info structure */

	return result;
}

static int
sddr09_read_control(struct us_data *us,
		unsigned long address,
		unsigned int blocks,
		unsigned char *content,
		int use_sg) {

	US_DEBUGP("Read control address %08lX blocks %04X\n",
		address, blocks);

	return sddr09_read21(us, address, blocks, CONTROL_SHIFT, content, use_sg);
}

/*
 * Read Device ID Command: 12 bytes.
 * byte 0: opcode: ED
 *
 * Returns 2 bytes: Manufacturer ID and Device ID.
 * On more recent cards 3 bytes: the third byte is an option code A5
 * signifying that the secret command to read an 128-bit ID is available.
 * On still more recent cards 4 bytes: the fourth byte C0 means that
 * a second read ID cmd is available.
 */
static int
sddr09_read_deviceID(struct us_data *us, unsigned char *deviceID) {
	unsigned char command[12] = {
		0xed, LUNBITS, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
	};
	unsigned char content[64];
	int result, i;

	result = sddr09_send_scsi_command(us, command, sizeof(command));
	if (result != USB_STOR_TRANSPORT_GOOD)
		return result;

	result = usb_storage_bulk_transport(us, SCSI_DATA_READ, content, 64, 0);

	for (i = 0; i < 4; i++)
		deviceID[i] = content[i];

	return result;
}

static int
sddr09_get_wp(struct us_data *us) {
	struct sm_card_info *info = us->extra;
	int result;
	unsigned char status;

	result = sddr09_read_status(us, &status);
	if (result != USB_STOR_TRANSPORT_GOOD) {
		US_DEBUGP("sddr09_get_wp: read_status fails\n");
		return result;
	}
	US_DEBUGP("sddr09_get_wp: status %02X", status);
	if ((status & 0x80) == 0) {
		info->flags |= SMARTMEDIA_WP;	/* write protected */
		US_DEBUGP(" WP");
	}
	if (status & 0x40)
		US_DEBUGP(" Ready");
	if (status & LUNBITS)
		US_DEBUGP(" Suspended");
	if (status & 0x1)
		US_DEBUGP(" Error");
	US_DEBUGP("\n");
	return USB_STOR_TRANSPORT_GOOD;
}

#if 0
/*
 * Reset Command: 12 bytes.
 * byte 0: opcode: EB
 */
static int
sddr09_reset(struct us_data *us) {

	unsigned char command[12] = {
		0xeb, LUNBITS, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
	};

	return sddr09_send_scsi_command(us, command, sizeof(command));
}
#endif

static int
sddr09_read_map(struct us_data *us) {
	struct sm_card_info *info = us->extra;
	struct scatterlist *sg;
	int numblocks, alloc_len, alloc_blocks;
	int i, j, result;
	unsigned char *ptr;
	unsigned int lba, lbact;

	if (!info->capacity)
		return -1;

	// read 64 (1<<6) bytes for every block 
	// ( 1 << ( blockshift + pageshift ) bytes)
	//	 of capacity:
	// (1<<6)*capacity/(1<<(b+p)) =
	// ((1<<6)*capacity)>>(b+p) =
	// capacity>>(b+p-6)

	alloc_len = info->capacity >> 
		(info->blockshift + info->pageshift - CONTROL_SHIFT);

	// Allocate a number of scatterlist structures according to
	// the number of 128k blocks in the alloc_len. Adding 128k-1
	// and then dividing by 128k gives the correct number of blocks.
	// 128k = 1<<17

	alloc_blocks = (alloc_len + (1<<17) - 1) >> 17;
	sg = kmalloc(alloc_blocks*sizeof(struct scatterlist), GFP_NOIO);
	if (sg == NULL)
		return 0;

	for (i=0; i<alloc_blocks; i++) {
		if (i<alloc_blocks-1) {
			char *vaddr = kmalloc(1 << 17, GFP_NOIO);
			sg[i].page = virt_to_page(vaddr);
			sg[i].offset = ((unsigned long)vaddr & ~PAGE_MASK);
			sg[i].length = (1<<17);
		} else {
			char *vaddr = kmalloc(alloc_len, GFP_NOIO);
			sg[i].page = virt_to_page(vaddr);
			sg[i].offset = ((unsigned long)vaddr & ~PAGE_MASK);
			sg[i].length = alloc_len;
		}
		alloc_len -= sg[i].length;
	}

	for (i=0; i<alloc_blocks; i++)
		if (sg[i].page == NULL) {
			for (i=0; i<alloc_blocks; i++)
				if (sg[i].page != NULL)
					kfree(page_address(sg[i].page) + sg[i].offset);
			kfree(sg);
			return 0;
		}

	numblocks = info->capacity >> (info->blockshift + info->pageshift);

	result = sddr09_read_control(us, 0, numblocks,
				     (unsigned char *)sg, alloc_blocks);
	if (result != USB_STOR_TRANSPORT_GOOD) {
		for (i=0; i<alloc_blocks; i++)
			kfree(page_address(sg[i].page) + sg[i].offset);
		kfree(sg);
		return -1;
	}

	if (!smartmedia_allocate_map(info)) {
		for (i=0; i<alloc_blocks; i++)
			kfree(page_address(sg[i].page) + sg[i].offset);
		kfree(sg);
		return 0;
	}

	ptr = page_address(sg[0].page)+sg[0].offset;

	/*
	 * Define lba-pba translation table
	 *
	 * Each block is 64 bytes of control data, so block i is located in
	 * scatterlist block i*64/128k = i*(2^6)*(2^-17) = i*(2^-11)
	 */

	for (i=0; i<numblocks; i++) {
		ptr = page_address(sg[i>>11].page) +
			sg[i>>11].offset + ((i&0x7ff)<<6);

		lba = short_pack(ptr[7], ptr[6]);
		lba = (lba & 0x07FF) >> 1;

		if (i < 10 || lba < 10)
			printk("pba %d = lba %d\n", i, lba);

		if (i == 0 || i == 1) {
			info->pba_to_lba[i] = UNUSABLE;
			continue;
		}

		/* special PBAs have control field 0^16 */
		for (j = 0; j < 16; j++)
			if (ptr[j] != 0)
				goto nonz;
		info->pba_to_lba[i] = UNUSABLE;
		printk("sddr09: PBA %04X has no logical mapping\n", i);
		continue;

	nonz:
		/* unwritten PBAs have control field FF^16 */
		for (j = 0; j < 16; j++)
			if (ptr[j] != 0xff)
				goto nonff;
		continue;

	nonff:
		/* normal PBAs start with six FFs */
		if (j < 6) {
			printk("sddr09: PBA %04X has no logical mapping: "
			       "reserved area = %02X%02X%02X%02X "
			       "data status %02X block status %02X\n",
			       i, ptr[0], ptr[1], ptr[2], ptr[3],
			       ptr[4], ptr[5]);
			info->pba_to_lba[i] = UNUSABLE;
			continue;
		}

		if ((ptr[6] >> 4) != 0x01) {
			printk("sddr09: PBA %04X has invalid address field "
			       "%02X%02X/%02X%02X\n",
			       i, ptr[6], ptr[7], ptr[11], ptr[12]);
			info->pba_to_lba[i] = UNUSABLE;
			continue;
		}

		/* check even parity */
		if (nand_parity[ptr[6] ^ ptr[7]]) {
			printk("sddr09: Bad parity in LBA for block %04X"
			       " (%02X %02X)\n", i, ptr[6], ptr[7]);
			info->pba_to_lba[i] = UNUSABLE;
			continue;
		}

		/*
		 * Every 1024 physical blocks ("zone"), the LBA numbers
		 * go back to zero, but are within a higher block of LBA's.
		 * Also, there is a maximum of 1000 LBA's per zone.
		 * In other words, in PBA 1024-2047 you will find LBA 0-999
		 * which are really LBA 1000-1999. This allows for 24 bad
		 * or special physical blocks per zone.
		 */

		if (lba >= 1000) {
			unsigned long address;

			printk("sddr09: Bad LBA %04X for block %04X\n",
			       lba, i);
			info->pba_to_lba[i] = UNDEF /* UNUSABLE */;
			if (erase_bad_lba_entries) {
				/* some cameras cannot erase a card if it has
				   bad entries, so we supply this function */
				address = (i << (info->pageshift + info->blockshift));
				sddr09_erase(us, address>>1);
			}
			continue;
		}

		lba += 1000*(i/0x400);

		if (lba<0x10 || (lba >= 0x3E0 && lba < 0x3EF))
			US_DEBUGP("LBA %04X <-> PBA %04X\n", lba, i);

		if (info->lba_to_pba[lba] != UNDEF) {
			static int ct = 0;
			if (ct++ < 8)
				printk("sddr09.c: Error: duplicate "
				       "LBA %d at %d and %d\n",
				       lba, info->lba_to_pba[lba], i);
		} else {
			info->pba_to_lba[i] = lba;
			info->lba_to_pba[lba] = i;
		}
	}

	/*
	 * Approximate capacity. This is not entirely correct yet,
	 * since a zone with less than 1000 usable pages leads to
	 * missing LBAs. Especially if it is the last zone, some
	 * LBAs can be past capacity.
	 */
	lbact = 0;
	for (i = 0; i < numblocks; i += 1024) {
		int ct = 0;

		for (j = 0; j < 1024 && i+j < numblocks; j++) {
			if (info->pba_to_lba[i+j] != UNUSABLE) {
				if (ct >= 1000)
					info->pba_to_lba[i+j] = SPARE;
				else
					ct++;
			}
		}
		lbact += ct;
	}
	info->lbact = lbact;
	US_DEBUGP("Found %d LBA's\n", lbact);

	for (i=0; i<alloc_blocks; i++)
		kfree(page_address(sg[i].page)+sg[i].offset);
	kfree(sg);

	{ unsigned char *bufr = kmalloc(8192, GFP_NOIO);
	if(bufr){
		memset(bufr, 0x11, 8192);
		sddr09_writeCIS(us, 10, bufr);
		kfree(bufr);
	}}

	return 0;
}

/*
 * This is needed at a very early stage. If this is not listed in the
 * unusual devices list but called from here then LUN 0 of the combo reader
 * is not recognized. But I do not know what precisely these calls do.
 */
int
sddr09_init(struct us_data *us) {
	int result;
	unsigned char data[18];

	result = sddr09_send_command(us, 0x01, USB_DIR_IN, data, 2);
	if (result != USB_STOR_TRANSPORT_GOOD) {
		US_DEBUGP("sddr09_init: send_command fails\n");
		return result;
	}

	US_DEBUGP("SDDR09init: %02X %02X\n", data[0], data[1]);
	// get 07 02

	result = sddr09_send_command(us, 0x08, USB_DIR_IN, data, 2);
	if (result != USB_STOR_TRANSPORT_GOOD) {
		US_DEBUGP("sddr09_init: 2nd send_command fails\n");
		return result;
	}

	US_DEBUGP("SDDR09init: %02X %02X\n", data[0], data[1]);
	// get 07 00

	result = sddr09_request_sense(us, data, sizeof(data));
	if (result == USB_STOR_TRANSPORT_GOOD && data[2] != 0) {
		int j;
		for (j=0; j<sizeof(data); j++)
			printk(" %02X", data[j]);
		printk("\n");
		// get 70 00 00 00 00 00 00 ** 00 00 00 00 00 00
		// 70: current command
		// sense key 0, sense code 0, extd sense code 0
		// additional transfer length ** = sizeof(data) - 7
		// Or: 70 00 06 00 00 00 00 0b 00 00 00 00 28 00 00 00 00 00
		// sense key 06, sense code 28: unit attention,
		// not ready to ready transition
	}

	// test unit ready

	return USB_STOR_TRANSPORT_GOOD;		/* not result */
}

static int
sddr09_do_scsi_command(struct us_data *us, Scsi_Cmnd *srb, int cmdlen) {
	int result;

	srb->cmnd[1] = LUNBITS;

	result = sddr09_send_scsi_command(us, srb->cmnd, cmdlen);
	if (result != USB_STOR_TRANSPORT_GOOD)
		return result;

	if (srb->request_bufflen == 0)
		return USB_STOR_TRANSPORT_GOOD;

	if (srb->sc_data_direction == SCSI_DATA_WRITE ||
	    srb->sc_data_direction == SCSI_DATA_READ) {

		result = usb_storage_bulk_transport(us,
					       srb->sc_data_direction,
					       srb->request_buffer, 
					       srb->request_bufflen,
					       srb->use_sg);

		return result;
	} 

	return USB_STOR_TRANSPORT_GOOD;
}

static int
sddr09_test_unit_ready(struct us_data *us, Scsi_Cmnd *srb, int cmdlen) {
	return sddr09_do_scsi_command(us, srb, cmdlen);
}

static int
sddr09_scsi_request_sense(struct us_data *us, Scsi_Cmnd *srb, int cmdlen) {
	return sddr09_do_scsi_command(us, srb, cmdlen);
}

/* read from a single erase block; address and count in pages */
static int
sddr09_read_data(struct us_data *us,
		 unsigned long address, unsigned int count,
		 unsigned char *buf, int use_sg) {
	struct sm_card_info *info = us->extra;
	return sddr09_read20(us, address << (info->pageshift - 1),
			     count, info->pageshift, buf, use_sg);
}

static struct sm_ops sddr09_ops = {
	sddr09_get_wp,
	sddr09_read_deviceID,
	sddr09_read_map,
	sddr09_read_data,
	sddr09_write_lba,
	sddr09_test_unit_ready,
	sddr09_scsi_request_sense
};

int
sddr09_transport(Scsi_Cmnd *srb, struct us_data *us) {
	return smartmedia_transport(srb, us, &sddr09_ops);
}
