package resources

import (
	"testing"

	"github.com/stretchr/testify/assert"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/pkg/agentcfg"
	"k8s.io/apimachinery/pkg/api/resource"
)

func TestGetResources(t *testing.T) {
	q, err := resource.ParseQuantity("1000")
	assert.NoError(t, err)
	tcs := []struct {
		name         string
		expectedErr  error
		requirements *agentcfg.ResourceRequirements
		expQuantity  objectQuantity
	}{
		{
			name: "fails if Requests requirements are not valid",
			requirements: &agentcfg.ResourceRequirements{
				Requests: &agentcfg.Resource{
					Cpu:    "",
					Memory: "",
				},
				Limits: &agentcfg.Resource{
					Cpu:    "1000",
					Memory: "1000",
				},
			},
			expectedErr: resource.ErrFormatWrong,
		},
		{
			name: "fails if Limits CPU requirements are not valid",
			requirements: &agentcfg.ResourceRequirements{
				Requests: &agentcfg.Resource{
					Cpu:    "1000",
					Memory: "1000",
				},
				Limits: &agentcfg.Resource{
					Cpu:    "",
					Memory: "1000",
				},
			},
			expectedErr: resource.ErrFormatWrong,
		},
		{
			name: "fails if Limits Memory requirements are not valid",
			requirements: &agentcfg.ResourceRequirements{
				Requests: &agentcfg.Resource{
					Cpu:    "1000",
					Memory: "1000",
				},
				Limits: &agentcfg.Resource{
					Cpu:    "1000",
					Memory: "",
				},
			},
			expectedErr: resource.ErrFormatWrong,
		},
		{
			name: "succeeds",
			requirements: &agentcfg.ResourceRequirements{
				Requests: &agentcfg.Resource{
					Cpu:    "1000",
					Memory: "1000",
				},
				Limits: &agentcfg.Resource{
					Cpu:    "1000",
					Memory: "1000",
				},
			},
			expectedErr: nil,
			expQuantity: objectQuantity{
				RequestsCPU:    q,
				RequestsMemory: q,
				LimitsCPU:      q,
				LimitsMemory:   q,
			},
		},
	}

	for _, tc := range tcs {
		t.Run(tc.name, func(t *testing.T) {
			m := Manager{
				Requirements: tc.requirements,
			}
			quantity, err := m.GetResources()
			if tc.expectedErr != nil {
				assert.Equal(t, tc.expectedErr, err)
				assert.Empty(t, quantity)
			} else {
				assert.NoError(t, err)
				assert.Equal(t, tc.expQuantity, quantity)
			}
		})
	}
}
