#ifndef __LINUX_USB_H
#define __LINUX_USB_H

#include <linux/types.h>

/*
 * Descriptor codes..
 */
#define USB_DEVICE_DESC			0x0100
#define USB_CONFIG_DESC			0x0200
#define USB_STRING_DESC			0x0300
#define USB_INTERFACE_DESC		0x0400
#define USB_ENDPOINT_DESC		0x0500
#define USB_HUB_DESC			0x2900

/*
 * Class codes..
 */
#define USB_HUB_CLASS		(9)

/*
 * Request numbers ("bRequest") combined with
 * their direction (bit seven of bmRequestType)
 */
#define USBREQ_GET_STATUS		0x0080
#define USBREQ_CLEAR_FEATURE		0x0100
#define USBREQ_SET_FEATURE		0x0300
#define USBREQ_SET_ADDRESS		0x0500
#define USBREQ_GET_DESCRIPTOR		0x0680
#define USBREQ_SET_DESCRIPTOR		0x0700
#define USBREQ_GET_CONFIGURATION	0x0880
#define USBREQ_SET_CONFIGURATION	0x0900
#define USBREQ_GET_INTERFACE		0x0A80
#define USBREQ_SET_INTERFACE		0x0B00
#define USBREQ_SYNC_FRAME		0x0C80

/*
 * Request target types.
 */
#define USBREQ_DIRECT			0x0000
#define USBREQ_INTERFACE		0x0001
#define USBREQ_ENDPOINT			0x0002
#define USBREQ_HUB			0x0020
#define USBREQ_PORT			0x0023

/*
 * Command codes (this is the combination of the request
 * type and the target type). Not all types are defined,
 * even fewer used.
 */
#define SET_ADDRESS			(USBREQ_SET_ADDRESS)
#define GET_DESCRIPTOR			(USBREQ_GET_DESCRIPTOR)

#define GET_HUB_DESCRIPTOR		(USBREQ_GET_DESCRIPTOR | USBREQ_HUB)

#define SET_PORT_FEATURE		(USBREQ_SET_FEATURE | USBREQ_PORT)
#define CLEAR_PORT_FEATURE		(USBREQ_CLEAR_FEATURE | USBREQ_PORT)
#define GET_PORT_STATUS			(USBREQ_GET_STATUS | USBREQ_PORT)

/*
 * Port feature numbers.
 */
#define PORT_ENABLE		(1)
#define PORT_SUSPEND		(2)
#define PORT_RESET		(4)
#define PORT_POWER		(8)
#define PORT_LOWSPEED		(9)

/*
 * USB device number allocation bitmap. There's one bitmap
 * per USB tree.
 */
struct usb_devmap {
	unsigned long devicemap[128 / (8*sizeof(unsigned long))];
};

/*
 * This is a USB device descriptor.
 *
 * USB device information
 *
 * Make this MUCH dynamic, right now
 * it contains enough information for
 * a USB floppy controller, and nothing
 * else.
 *
 * I'm not proud. I just want this dang
 * thing to start working.
 */
#define USB_MAXCONFIG		1
#define USB_MAXINTERFACES	3
#define USB_MAXENDPOINTS	3

struct usb_device_descriptor {
	__u8  bLength;
	__u8  bDescriptorType;
	__u16 bcdUSB;
	__u8  bDeviceClass;
	__u8  bDeviceSubClass;
	__u8  bDeviceProtocol;
	__u8  bMaxPacketSize0;
	__u16 idVendor;
	__u16 idProduct;
	__u16 bcdDevice;
	__u8  iManufacturer;
	__u8  iProduct;
	__u8  iSerialNumber;
	__u8  bNumConfigurations;
};

/* Endpoint descriptor */
struct usb_endpoint_descriptor {
	__u8  bLength;
	__u8  bDescriptorType;
	__u8  bEndpointAddress;
	__u8  bmAttributes;
	__u16 wMaxPacketSize;
	__u8  bInterval;
};

/* Interface descriptor */
struct usb_interface_descriptor {
	__u8  bLength;
	__u8  bDescriptorType;
	__u8  bInterfaceNumber;
	__u8  bAlternateSetting;
	__u8  bNumEndpoints;
	__u8  bInterfaceClass;
	__u8  bInterfaceSubClass;
	__u8  bInterfaceProtocol;
	__u8  iInterface;

	struct usb_endpoint_descriptor endpoint[USB_MAXENDPOINTS];
};

/* Configuration descriptor information.. */
struct usb_config_descriptor {
	__u8  bLength;
	__u8  bDescriptorType;
	__u16 wTotalLength;
	__u8  bNumInterfaces;
	__u8  bConfigurationValue;
	__u8  iConfiguration;
	__u8  bmAttributes;
	__u8  MaxPower;

	struct usb_interface_descriptor interface[USB_MAXINTERFACES];
};

/* String descriptor */
struct usb_string_descriptor {
	__u8  bLength;
	__u8  bDescriptorType;
};

/* Hub descriptor */
struct usb_hub_descriptor {
	__u8  bLength;
	__u8  bDescriptorType;
};

struct usb_device {
	int devnum;
	int slow;

	struct usb_device_descriptor	descriptor;

	struct usb_config_descriptor	config[USB_MAXCONFIG];
};

extern void usb_init_root_hub(struct usb_device *dev);
extern void usb_connect(struct usb_device *dev, struct usb_devmap *devmap);
extern void usb_disconnect(struct usb_device *dev, struct usb_devmap *devmap);
extern void usb_device_descriptor(struct usb_device *dev);

extern int  usb_parse_configuration(struct usb_device *dev, void *buf, int len);

/*
 * Calling this entity a "pipe" is glorifying it. A USB pipe
 * is something embarrassingly simple: it basically consists
 * of the following information:
 *  - device number (7 bits)
 *  - endpoint number (4 bits)
 *  - current Data0/1 state (1 bit)
 *  - direction (1 bit)
 *  - speed (1 bit)
 *  - max packet size (2 bits: 8, 16, 32 or 64)
 *  - pipe type (2 bits: control, interrupt, bulk, isochronous)
 *
 * That's 18 bits. Really. Nothing more. And the USB people have
 * documented these eighteen bits as some kind of glorious
 * virtual data structure.
 *
 * Let's not fall in that trap. We'll just encode it as a simple
 * unsigned int. The encoding is:
 *
 *  - device:		bits 8-14
 *  - endpoint:		bits 15-18
 *  - Data0/1:		bit 19
 *  - direction:	bit 7		(0 = Host-to-Device, 1 = Device-to-Host)
 *  - speed:		bit 26		(0 = High, 1 = Low Speed)
 *  - max size:		bits 0-1	(00 = 8, 01 = 16, 10 = 32, 11 = 64)
 *  - pipe type:	bits 30-31	(00 = isochronous, 01 = interrupt, 10 = control, 11 = bulk)
 *
 * Why? Because it's arbitrary, and whatever encoding we select is really
 * up to us. This one happens to share a lot of bit positions with the UCHI
 * specification, so that much of the uhci driver can just mask the bits
 * appropriately.
 */

#define usb_maxpacket(pipe)	(8 << ((pipe) & 3))
#define usb_packetid(pipe)	(((pipe) & 0x80) ? 0x69 : 0xE1)

#define usb_pipedevice(pipe)	(((pipe) >> 8) & 0x7f)
#define usb_pipeendpoint(pipe)	(((pipe) >> 15) & 0xf)
#define usb_pipedata(pipe)	(((pipe) >> 19) & 1)
#define usb_pipeout(pipe)	(((pipe) & 0x80) == 0)
#define usb_pipeslow(pipe)	(((pipe) >> 26) & 1)
#define usb_pipetype(pipe)	(((pipe) >> 30) & 3)

static inline unsigned int __create_pipe(struct usb_device *dev, unsigned int endpoint)
{
	return (dev->devnum << 8) | (endpoint << 15) | (dev->slow << 26);
}

static inline unsigned int __default_pipe(struct usb_device *dev)
{
	return (dev->slow << 26);
}

/* Create control pipes.. */
#define usb_sndctrlpipe(dev,endpoint)	((2 << 30) | __create_pipe(dev,endpoint))
#define usb_rcvctrlpipe(dev,endpoint)	((2 << 30) | __create_pipe(dev,endpoint) | 0x80)
#define usb_snddefctrl(dev)		((2 << 30) | __default_pipe(dev))
#define usb_rcvdefctrl(dev)		((2 << 30) | __default_pipe(dev) | 0x80)

/* Create .. */

/*
 * Debugging helpers..
 */
void usb_show_device_descriptor(struct usb_device_descriptor *);
void usb_show_config_descriptor(struct usb_config_descriptor *);
void usb_show_interface_descriptor(struct usb_interface_descriptor *);
void usb_show_endpoint_descriptor(struct usb_endpoint_descriptor *);
void usb_show_hub_descriptor(struct usb_hub_descriptor *);
void usb_show_device(struct usb_device *);

#endif
