# ------------------------------------
# Copyright (c) Microsoft Corporation.
# Licensed under the MIT License.
# ------------------------------------
"""Keys for use in test cases"""

from azure.keyvault.keys import KeyCurveName, KeyVaultKey

EC_KEYS = {
    KeyCurveName.p_256: KeyVaultKey(
        key_id="https://localhost/keys/key-name/version",
        jwk={
            "kty": "EC",
            "key_ops": ["encrypt", "decrypt", "wrapKey", "unwrapKey", "sign", "verify"],
            "d": b"^T\xf9F\r\xb5\xa8\x1a \n\xa6[\xfa\x12-0\n\xb7\x92\t\xee\xd1\x03\x9d\xd5\xca\x9c\n\xc4\x10\xefI",
            "crv": "P-256",
            "x": b"\xba\xb3Z\\\xea\x01\x10\xb4\xbe\x81\xb0\xc4h\xc2\xeb1\xa3\xbfb\x11A\xd939\xcc\x8aY\x05\xf3\x83\x12E",
            "y": b"\x1fs\xbe\xfa\xc7\xef\xd3{H\xea\x1a\xcb:r\xc2\x8c\xf2\xde#z2B]\xc3\xa6>\xc0\xa6.pUN",
        },
    ),
    KeyCurveName.p_384: KeyVaultKey(
        key_id="https://localhost/keys/key-name/version",
        jwk={
            "kty": "EC",
            "key_ops": ["encrypt", "decrypt", "wrapKey", "unwrapKey", "sign", "verify"],
            "d": b"\xe1\x13g}\xdeV\xad(`\x85\xed\x8c\xc4}\x9f\x9eE\x14\x1c\x12\xff\xa4s\xa5L\x05j*W\xb3X\x88A\xe2?\x1f\x80\xd6\xf1\x0fR\x91\x0fh\xf3EH\xcf",
            "crv": "P-384",
            "x": b"\xa5\xd1\tFX\x12\xd5\xa8\xb5\x00\xe4p\xf8S\x8e\x8d\x80q\xdc.\xbe\x86\xef\xf1\x99\x15\xaa\xde0\xb1'(\xc0\xcb;43\xca\x14\x17,\xe1T\x10\xc2iK\x08",
            "y": b"H6\x19\x8c\xe6\xcc\x0brhJ\x98\xd3\x9b\xf2\x08\xf5\xf0z&/S\xd2\xb3x \x18[\xcc\xb9E\xb6M\x87\xfb\x06z\xd9\x95\xcbj\x84\x8aPq$\xf3t\x9c",
        },
    ),
    KeyCurveName.p_256_k: KeyVaultKey(
        key_id="https://localhost/keys/key-name/version",
        jwk={
            "kty": "EC",
            "key_ops": ["encrypt", "decrypt", "wrapKey", "unwrapKey", "sign", "verify"],
            "d": b"\x84\x9f\xb9\x07\x91\x1c\xb9V#\x8d\xb8\xec+\x8e\xba\x17%\xc9[\xb1\xd0J\x94\xfd\xcb\x1c\x89\xd1[\x10\x86R",
            "crv": "P-256K",
            "x": b"~\xa5i\xa0\n\xe5wo\xf7\x08of\xb8\xf1\xee\xc8\x0b\x18\xde\x99t\x95\x06\x9c\xccb&L\xe4|;\xf6",
            "y": b"t\xe9\x9d\x07^\x81\xb3\xdb(8\x90\x95\xdd \xa7\xf6\x0ep\xec$\x1cq\xc7\xcf\x08\xf8\xb5r\x9e04\xf3",
        },
    ),
    KeyCurveName.p_521: KeyVaultKey(
        key_id="https://localhost/keys/key-name/version",
        jwk={
            "kty": "EC",
            "key_ops": ["encrypt", "decrypt", "wrapKey", "unwrapKey", "sign", "verify"],
            "d": b"\xb55`\xbe\xd7\x0f'0D\xdf\xe0 \xd4\xed?\xf4\xee\xe0\xf0l\xd7\x08\x87l0\x04R\x01l1\x85\xb7\xf5\x14H\xde\x19\xeb\x13\xab_\xf0\t\xbe\x88\x18\x1b\xbd.\x89\xe9\xda3\xdc\x97\xc5\x18\x0f8\xa4\xbeb\x83z\xf0",
            "crv": "P-521",
            "x": b"\x83\x95Q\x1e\xa3\xf3\xad\xde\xc6\xb4\xdd\xe8\x1d^}\x15\x85\\\xcc\x95\xe4\xfc\xea\x89\x7f\x8d\xd2\xe5\x81/u\xbb^\xd0W\t\x93P\xdb3\xd6T\x8f\x9c\x90\xa1\x95\xffQ\xa0\x96\xd7\x11a.C\x03?\x81\x02D\xb2x\xf3\xaa",
            "y": b'\x01\x8c\xda\xc9\x1ej\xe5B7\xcc\xaaFD\x0e\xb2\xe9\xaad\xf9\n\xc5\x93:\xcaBE\xbc\x04\x94\x03\xe0\xb8h\x00K3L"\xfa\r\x91\x8fJ\xbfJF2\x17\xd0\xfb\xbb@9\xbe\x87+\x05I\xab\x9cu\xafV\xaa\x1c\x99',
        },
    ),
}


RSA_KEYS = {
    2048: KeyVaultKey(
        key_id="https://localhost/keys/key-name/version",
        jwk={
            "kty": "RSA",
            "key_ops": ["encrypt", "decrypt", "wrapKey", "unwrapKey", "sign", "verify"],
            "n": b"\xeb<P\x14a\xed\x0c\xf4\xfd/\xa8\x85\xa3U\xceL\xd3\x1f\xfb\x13#\xedd4\x9e]ZGv\x9c]b\xad\tT\x88\xb9\xa43\xdd\xc9\xb7\x86\x16\x19\x98{\xf0\xafJ\xb9\xd4\xdaw\xe8\x9d\xb7\x7fc\xb40\xaf\xff`\x9b\xcc\x8b7\xa7'\x01\x93/5h#\xff\xdfq:\x90\xfd.q(\x99\xf1K)\xb5n\xb5\xd6\x89\x84\x8a\x03I\xc5a@\xd9\x9d\xc3aG\xf6\xfc\xdb9c\xd8\xa4pQX\x0fz-\x86\xe7!Ge\xdf#\xcefB/\xb5\xea\x12\xc6\x05\x9dH+T\x9c/\xd3}\xb4\x06\xa5\xc1.\xd47\x13\xd8n\xa80,\x85o\x94\\\xe4\xb6.\xbc\\\xfb\x06\xefcT?\x05ezD\xb3\xc2\x01\xfe(\x15p\xb3x\x84\x80s\x99a\x91-\x05\xb3\xcaf_\xa3M\x82C4l\xf4\x05\x97<\x15I;~\x9f\xc5\xce\x9f%\xa1\xf8\xd9\xddW\xc7\\\xaf\n\x92r\x8e\xe6\xb4\xdf\xeb\xae\xb4\x1b\xcc\xff\xb3\xffI\x0f\xa9\xc0\x0bf2\x00\x82\x1a\xd7\xa6bT\x87G-U",
            "e": b"\x01\x00\x01",
            "d": b'\xc8\xbd\xec|\xa8`\xb9\xd8\xeb\xa3[\x05\xc806`\x1b\x1fD\x8c\xbd\xf1\x1c\xbdF\x93\xd5\xb1\xfc\x87M\xae"\x8a>\xc8z\xdb\xa4\xa3U;\xb7Bn\xcfP\x9dA\x89\xdcq\xc7\x89\xe2I\x8dI9\xd5\x90b\xf0\xe5\t\xaf\x90\xf2\x8aC`5\xb1h\xbb\x19U\x86\xe1\xc2\x9f\xee_N\x93\xdec*\xc8J\x1bS3\xe4\xa0\xe5\x82\x0e\xd7P\xba\x1d\xbbb\xfbF\x85\x80\x9c\xb8)\xfb\xbb\xc0Y\xbasb\xc9-AZ\x85\xeajt\xcc\xf1\x96\x14\xaeD\x88,\xa3\x13\xcaE\xb3\xb2\xd7\x152P\xee\xf6lS\x1c\xdbp\xe4^\xd3\xf1\x18\xe6?\x94\xc9\x1dS\xdc\xc9\x08s\xd1bA\xa5\x1a?\x0cT\x7fAs\xb8\xfc\xdb\x0c\x82\xad+\x80\xb8=\xa2:\x85~\xe1\xef\xdc@w\x04\\j\x9c\x18h\x14j?*\xa8g\xd5\x91+\xe1\t!:\x0b\x10\xe0\x99\xc9\xe3g\x1e\xfao\xb5\x8fD\xd6r\x7f\x94\x7fX\x00\x9e#]\xcf@z\xae\x82\xa1\xa2)\x00\x8e\xb3\xdc\xc1\xdczNm\xb5',
            "dp": b"\x91|\xbd\xa3\xf6\x01\x08\xb7\x92\x1c\x1a}\xe3\xc4\xb1#\x97\x0bxT\xb6\x13\xecx\x8f\xf6\xf3=B=\x80\xb2@\xf2\xce6rSDtB\x04u\x8d\x0c\xb3J\x84\xa3\xedz>\x0fW\xdd\xeeu_[\xf2X-\xdc\xb5}\xd2\xd0b\x11\x9e\x16l\xbc\xcdGZ}h\xa8&\x12 \x08Y\x9d}\x08\x06\xd3c\x93\x106\x8a\x1a\x19\x9f\xf5\x95P\xcce#\x05N\x01\xb8\xbb\x07\x04@\x19\x8e\xb3\x81WzIow\xff\xe1a\xe9\xd0`e\xff",
            "dq": b"\x0eV\x80:\xd9k\xa8\x02U(Ew\x14\x16\x01C\xd8\xe2\x06\x10\x9a\x9aZ\xf1\xf3\x0f\xbe&E\xaa<\t\xb2\xbf&\x0c1\xf3B\xf0j\xc4\x8d\xf7\xa2^\x8f9\x9fY\x0f\xf8\x0f\xa7\x9c\xbd\xb9\xb6<\xaf\xc6\x9b\xb8wJ\x9d\x0c\xa0\xd0|\x97?b\xb0\x0e\x19\xf9\x96\xf3\x1e\x96\xfcp\xe7\x7f\x0f\xad(h\x878'\xccJ\x022\xc7\x06\x89|\x99DH\x9e\x8d\x97\xb2\xbe#(\xb9\xef\xb3\xc9\x00!\\8\x98WVO\x1e\xb0u\xb1l\x11",
            "qi": b"ny\xfb\x87\x05-\xd6\x8dvt\xfd\xe4\x7f$7\xe0\xe3aU(\xa0b[\xc9\xb1PN\xfe\xfd\xf1\x80\xe4?:\x1eA\x8f\x86\xdd\xfd)\xcf\x93C\xac\x14\xbf\xe7\xd9#\x0e\x9e\x14\x15\xd7\xc9\x0b\xf4\xdd<\xe6\xe1,/u\xeb,\x0b\xaaX\xadMo\x86Q\xb00\xd4A\x97\x03\x8d\x86E\x0f;\xcc\xf0\x88\x0f\x90\xd2\xfdc]\xdd\xed\xd8\xfd\xd3\x8d\x99\xc7Uz\xa4\x8d)\x03\xd1\xf7\xb1\xdf\x85\x05Uvsc2W\xa8\x19\x8eS55u",
            "p": b'\xf8\xa8\x80\x8f\xe9\x11.\xc8\xcc\xbe\x91\xfe\xe8&S\xdea6\x85E\xa7\xe40\xc444qq\xa1\xf3KrI\xb9\xda\x9b\xae\r\xd6\x9e\x18\xa3\xdb\xc5\x81*\xd2\xa2r\xf0Q\x94\x97\x0eMd\x9c\xe0v\xb0"\x13\xac\x91\xcc\x8e\x15\xba\xf7\xc1\x82\xc2k\x83\x9c\n\xbd\x8f$^\xcaz4r3\xd6\xab\xc0p:\xe6\x95\t\x98\x89<\x17\xd7\x9a\x9e\t\xaa\xaf\xefW\xceR\x07v[\xa8\xa22\x99|\x9eM\xe4\x17{\xb3\x12\x86p\xc6\xdc\r\x0f',
            "q": b"\xf2.Z\xe0pT\x8cE|\xbd\xaar\xa4}\xfaP\xe6\x86\x00,a\xd8@&\xed\xa3.Q\x89\xccbo\xd2\"\\\x97}\x0f\xa1;~{\x08\xb8\xc3\xb5'\x12aL}\xb2\xe9H\x13Z\xaf\x80\xaf\xa3Q\x84l\xee\xc8\xdd\x1c\xe3\xd5\x9a\x1c<\xd4'\x1c\xa3\x14\xa5\x85\xe8b\xe7\xb6\\\xe5\x8d\xbc\xf8u'\xa1\xfb\x03\xd0e!\xdc\xb8\xc6\xacA\x82\x7f\xecZ$\xf8w\xae\x1cO\xbb;)\xf5\x12;\xc6=9q\xe9\x1a\xe4PZ\xe7[",
        },
    ),
    3072: KeyVaultKey(
        key_id="https://localhost/keys/key-name/version",
        jwk={
            "kty": "RSA",
            "key_ops": ["encrypt", "decrypt", "wrapKey", "unwrapKey", "sign", "verify"],
            "n": b"\xd4\x17\x03\t\xed\x10q\x96~\xaf\xf1\xa2\xd6Cb\xbf5\x9a\t\xda\xe0\xfe\x0c\xfa8C\x94\xad\x84\xdb\x90B\xb17\xf4\xb4\x16\x1b\x8d\xdc\xd2\t3\xae\x86\xffP\xfa\x07\x8cm\xb2?ZH\x0e\xb3\xdf\x18y`\x95\xf1\xe0\xf5\xa80O\xfdZ\x07\x00}\xd7g@\xd7\xce\x1a3|\xeb\x80(\xf5\xb9XR\x82`\xd9m\xd3\x9d\xf4\x02BPN\xdf\xfb\xcd\xad\r\xe7mM\xf9'\xdd\x1b9vK\x82!\xb6\x8b\x04\xc3\xc9b\xe1\xe6\x9b\n\xe1\xb6\x82\xf3\x93k\xba\xd0g\x91\x08\x80\x8eR\xd5*\xe0\x85\x1a\xe6\xe7\x94\xb4O]\x07\x14\x92\xc70\x807J\x00i\x19A \x878\xaf%C\x17\xb9\x0e\x01\x9f\x9b\x1a7um\xd1\x1c\xb0\xd9\tyR\xaf\xac\xc80C\xe4Ks-Q#\x07 5\xd0s\xa0c2\x9b-[b\x93\x00\x96\x0bWRN\xde\x0eF\xb6%\xdf\xfc`\xfa[r\xa7\x99\xcd\xfaX\xa1\x17\xd8JA\xbbE>\x1c\xe6\xe8\xf6\x8f\xcdE\x88\xae\xc8zc\xf1\xc2N\x9f+x\xab\x08\xde4Jc\xdc6\xac\xba\x1bh\xaav\xdbR\x96\x1a\xe3\xb2x\xb3x\xc3\xf0\x19\xfav\x0e\x1c\xcc\x8f\xfb\xbe!\xa8\x13\xc8\xa8\xa0\xa3\xcc\xda3?\x05,%\x10_t\x92\x80D\xec&UY1\x91\x9bV\x95\xc9\x80\xeaF\x14\xdb\x87ORp\x0b\xff1Eh\xb7?\xf2\xdbX\xbax\xc3G\xd0\x8fms#\x82]0@N\x17\xd7\xe4i\x11:C\x1b\x0bJ\x11\xda\x19s/\x7f\xbfi`\x8b\xf5\x91\x19d< \xdc\x1f\xef",
            "e": b"\x01\x00\x01",
            "d": b"r\xa5\xf9\xe3\x8c\xbd\xfd\xd9\x0f\xa4\xc6\xa7r\xd1\xbcx\x86^=\x83\x8fje\xd3\x81D\xaf\x03Y\xb5b\xf7Z[X1]^\t\xc2\xdb\xdbS\x8f\xbd\xb4\x18\xaf\x00\x13\x9e\\\xe3\xb8G\t\x83\x9cN\xdb\x96%\x98:61s\xe4Y\x9d'\xf9\xd1\x0c\xf6\xe8\xeb\x7f\xa3T?\xec\x02<\xa2\x10N9S\xf7L\xb1\xfc\xd9\xfe\xc4\x95\xdaY\xb8\xe4\xa2[1\x0f\xf0[]\x1bz\x7f\xb4\x01Ir\x10\r`z\xf3\xe7RG\xb1d\x10n\xd1=Z\xb4Oa\x0e5^\x06M\x90\" \xcfk\xcd(,\xb7\xf4\xff\xbf\x83\x17G\xbc\xccf`y\x02\xcc\xd4\x03m\xdcJ\x9c\x8e\xb9B\xday\x13\xa47\x06\x1e\x16\xea\xa8\xb1\x88\xa3\xe4\xbf%QX[\xbb_\xb3\xe2\xa1\xbb~\n\xb8X\xbbv\xe2\x8d`\xb6^\xe2\xdb\x85L\xff\x12|\x8b\xb0\x07\x8d\x97\xcb\xbe\x1e9<\xc1Aq\xc8)%\xd44\xd4\xd7\xbc\x90\x1b\xae\x8f\xf0\xc4\x86O\xc6Y@\xfa\x89\xfb\xcfc\xba,\x83\xe9\x1bR\\\x8b\x9c\xec4\xc7\xcd\xfb$\x9d\xdau\x0cy\xcci\xbd\x04S\x13\xea\xd1\xf0-<\x8c\xd7\xd5\x16\x88f\t\xfa\x9c\x95\xda\xf9\xffePZ\x94{[1\xaa\x9c\xf9\x9c\x12|\xf5\xe2\xb2\xd4\x114Q\x9f\x0c\xed6\x06\x1e\xbb\x9d&Z\xa2\xf5\x1c\x1c\xc1M\xbb\x7f\xaf\xb5~\xec\x13\xef ;\x8a\x85\x0f\x93\xfb+\xafJ\x8e\xb5\xb8\x02\xa8|*7\x8c\x7f\t0\xb8\xc4\xcb\x8f\xbf<~\x958Y\xd1\xdf\xc6\x893\xe5\x17\xcb\xca\xeaSp\x894q",
            "dp": b"\x824'\xf8\xaf\x99\xbf\xa0\x05\x1a\xca\x1a\xf8\xf3\x8ed\xd5\x8a\xa9/+\\\xf8\xfa\x11\xd7\x00\xd1\t\x95\xbb\x8f\x1e\xdd\x03\xf4[\xa3E\xb3Q\xc1\x9cG\x83t\x00\xe0\xcf}'\xafj\x94sB\xde\xab\xf1J\xcb\x13F\xd3[jf1\xec\xf4\xcdG.#\xe7\xb5)1\xa7{\x8cQ\xc4GI\x81\x8c']\xc9\xb2\x9by\x93\x82\x8cD\xc1\x8e\x12\x04\xef\xb7w5\xd7F\xea\xcf\x93tu\x86kM\xf1\xd5\xdd\x8d\x88j\x9f\x85F\xa7\xf6\xe2\n\xbc\xdey\xcak\xa0(\xb3\x18Ci\xc0\xfd\xaa\xda\x01\xee\xc1>\xb33\xb0\xe9-\xe0B\xe4\xc8\x81\xeb\xd8\x19\xa8G\x05\x93T\xccx\xaa\xe1h\xd4\x8f\xf5\xf3\xdfTb\x1cUkE1\xaap\x89\xe0\xddR<\xfa\xfb1",
            "dq": b'xs\x81\x0bF\xa1\xf6\x05j\x03V~GD\x1dnHq{\x88b\xfd\xc5p\x07U\x8c\xd6o\x93N\xc8I\xc2\x1d\xc3\x16v\xf5\x0b\x94\xd3\x04b\xb2\xb3rkpO\xdfv,\xd5\x1e\xe6\x0b\x98\xeap\xc9\x84\xbc\xcf\x1f\x8b\x91\xc5tY\xa3w+/^\x15%\x18\x12@\xb5\xb6\xa8\'}\xd7w\xb6\x84\xaaQp\x06o\xce@]p\xa4!\xf5\x1bQ5\xe1=PMkA\x8f\x95\xe7\xdd-\x16\x00p.e\xd0l\xe6\r\xbb\x9cI\xcd\x18i\x9e\xdb\xe6\x86\xec\xad[\xee"\x8a\x12\xb2\xc7\xcc\xdcV\xc9\xce\xbdG\xbf\xa7K <V/K]\x9b>\xa8\xec\xf6\xecF\x98\x9cI\xde\x18\x12tjQ2+\xaa_\xef\xf1\xaaWZf\xdd\x0c\x1e\xaf"xq',
            "qi": b'8\xa1B\xe2@\xae\x10(_z\xfa\x13\xfd\xd6\x85b\xc3\x8d\x0e!\xfe\x18\x8f4\x80d[ p`-\xfc)\xfc\xa2\x93\x18\x1f\xe3=8\x08qxa9\x99i\xc6\x99\x1e\xb9f\x93\xdc"\x10\x1f\x89\x9a\x85\x89\xb0.\x9c\x04=\xd1`L+\xc8ZR\x07\xd7\x96u\\rzH\xc4\x1b\x13\x97\xda\x89\x8eB&\xe3\x07\x8bNv\xb2\xbe\xe2\xb2\x98\xfb4\x8b*\\\x17\xe5\xf2\xb7-9c\xee&r1\xd1\xd4\xcfR\xfa\xa7\x0b\x95\xb0\x15\x15\x1c\x11"\x1fX\xed\x86\xe7\x15\xce\xe6\x87n\x9b\xe2>z*i\xe0\xcc\xa1\xa0z\x18h\xa8R @\xd1M.\x9bG|V\x12T\xdc\xf2o\x8c\xc9\x1b\xd2\xebu\xf8\xf7\xcd\xbf\xea\x99\x0b\xc0tt\x17\xc5\xd0\xa0\x03\x8d',
            "p": b"\xf6\xd9\xfa\xff\xff0G\x85\xc0\x91\x9e\xa4\x9bT\xb7^+\x99\xd9\xeb-\xe3\x00w\x84\xe5L\xc8S\xb4p\xd6\xe6\xca\xc2U\xe7,b\x83\xd1;\x86\xf8<\xa2]h\x18\xde9S\x05\x10\x83u\xf8\xe1\xb9\xaa\x8a\x80uZ\xf5`\xe0\xden]r_\x96\xa0\x01\x9b\x86\xb716[7\xdf{d^>\to66l\xfb\xc9:yE=\xee\xba\r\xa3\xab60\x91\xf4\xfe\xec#\xd0n\xde\xa6\xa1{o\xca\xb4D\xe7Zw\x9an\xc5\\\x9c\xdf\xc7\x87q#\xcd\xac^\x90\"#\xefwYW\x02\x05\x1e_\xc6\x13X1\x851\xd4\xc9\xc1\x9a\xd7A\x08y\x93\xae\xe9\x07\xc2{\xff\x02\x19\xe0\x12\x8f\xfe\xbcW\xeeY\xa2\x91\x1c\x97\xc8\xcc\xaa\x84\xe1\xbf\x9d\xfc\xbe'",
            "q": b"\xdb\xf3:~\xd2\x1d\x9e\xd8i\xa1\"\xceK\xd3Z\\\xccW\x81\x9eD\xc6\xcf\xb9\x1a\xbb+.\xb96\x11\xea;\xf6K\xbbD\xd8\xdd\x89\xa2\xb8Y\xf8\xf28b\xf0\x91\xd1\xfa\x14|\xe0\xbd\xdf<&\xc9\xfc\xcd\xdd\xd5\xf1\x82*\x8c\xcf\x95+\xcf\x1a@\xf4\xc0\xd9Q\xcbi[\xcc\xdd\xe7]/+l\x05G\xf2\xce\x0b'\xac\tMrg\x9c\x0b1w\xfaM]\xe0\xa7\x9d?\xb7\xd5\xd3\x92\x84\x81\xf4\xa7\xc0\xc3\xffz`\x91\x96M]\x180\xbfp#k)\x10\x19\xb5\x82\x1dg\xbak\x91\xff\xf8\niZ\xa4@\x03\xa1SB\xc6`e\xa7\xd1\xde\xec\x03\xf7\x94\x9b\x1bTr\x17\n\xee*\xd4s0\x0bc9\x91\x02W\x84\x96\xa7|f@\xdf\xa8Q&\xf4\xf9",
        },
    ),
    4096: KeyVaultKey(
        key_id="https://localhost/keys/key-name/version",
        jwk={
            "kty": "RSA",
            "key_ops": ["encrypt", "decrypt", "wrapKey", "unwrapKey", "sign", "verify"],
            "n": b"\xea\x1e\x81=\n\x98\x81\xdf'T\x9e\xbaH\x19\n\x1d\x8b\x9a\xf1DJ\xe9\x7f6Q\x84o\x9f\x13}5p/\xce\xf7\xc5;\xef\x1a\x88\xb6\nvm\x94\xc1_N\xc4\xbd)\x80Z)\x12\x1e\xa5&\x8a]\xff\xbe\xfc1\x8e\xc8\xfa\x0b\x14\x0c\xeen\xa4S\xc5P\x08\x0e|\x1b\x01/\x98\xe1\xd8A\xa5\x88\x9b\xbd\xb9i\xe83A(p\x9f\xbf3Gp\xc7\xb7\xcc\xf7\x86\x08\xff\x10\x9f)=W\xe8\x8a\xa4H\xb6\x0c\xd44\xe3K\xa7L\x11.\xdd\x1d\xe5\xc9\xcf\xed.4mo\xe2\x03\x8e\xc9\xc2F\xc1\xf3\xdc\x18\\3\xe7\xe2Z\xf8\xccr\xea\xc4\x0c\x82I\x01E\x8e\x875\x87\x8f\xb4r\xd6\x1fZts\x1f\x05\xfa\xa9\xec\xe6z5\xd7\x19V*\xfeY\t$\xeegG\xfdz\x92\x10\xcb\xbb\x19$\xaa\xeb\x0c\t\xd1Nk6\x84c[`\x16\xd2!\x97\x19\xcc\x08\x9f\xc4\x1c\xed\x12k3{\xa1`\x97\x97\xa7\x05:L\x11t\xfa\xc4I\xab\xber\xd0M\xf4Fi\r\xaa-\x9b\xdb\xdaX?\x98w\xf1\xc4O\x01\xd2\x9b\xb0\x1a\x8d`Wmh3h@\x08\rP)\x9bt0t\xad4$\xfc\xd8\xe2\xd8M3\x91\xa2Y\x91\xcb\xeb\x1e\x12\xe2\xc2\x84\xa8Zs\xf3o\xa5\xa7\x00\x14>~<\x81\x10\x8d\x9d\xe4\\%\xc1\xf2\xd8~[\xb7!\xb9\x96]tb\xf7\xddt\x9f\xc1o<m\xf2\xfc\x16Kr\xd7j{M\xfd\xe7\x96\x93G!\xee\x05\x04\xa2\x0cO\xd5\x9e\xe9\xd3\nX\x13~\xec\x14\xe0s#\xddgD\x8c\x7fB\xb6W\xb4\tH8\xa1\xef\x9d(f\xfc\x02\xc9\\\xe4\xf8\xc9x\x1b\x8c\xda\xaf\xc8\xb8\x8d\xaf\xf9\xaa\xa4!\xf1\xfaB#\xec\xb2\xb0b%\xeb~?X\xae\x01U\x19\x10(\x0bU\x01\xef\xdf\xfb\xf6\xc6\x9f\x16Z8\xc3.\xbdm%y\xd0\xc6\x90\x8b\xca\xaa\xc0\x0f\xf3\x02\xb0S\xea\x05<\x82\xe7\x83|\xa3\r\x0bH?\xfa\xb6\xa9\xa06q%A\xb0\x1e(\xd7T*4\x1dC\xd1\x1emp'a\x07\xbf\x88\x90\x08\xf9\x99\xcb`\x8d\xeb\xe8\x05\xe3",
            "e": b"\x01\x00\x01",
            "d": b"[G\xa9{\xf35\xa2\xed.\x98\x7f\x8a\x81\xb1\xc1\x8f\xe3nu\xc2\xfa\xb2/7.qDS\xf1uO\xf0}\xb3dG2&\xf7av\x18~4\x1b[\x89\x9d\\w\xa4F\xb6\xe0l\x80&C\xf0\x8e\xd5\xb7\xd0\xfcA\x7fz}\xf7CSs/\xfe\xb9_\x8a\xc4\x9a\xfe\xacg\xb2c\xdb\xbd<\x9d\x16\x9b\xd2>\xc8\x99\xb3=\xe1+g\x90\xc6\xb9\xe2$wu\xca|\x97v\x13\x980\xa2\xa5\xb7\xe8u\xb1\x05/w\xafg?\xc3\x18\xaf\xfc\xe6;A\x84 %s\xabw\x83\xd7\x7fD\x8a\xca\xab\xb48\x8c\x06\xe6\xf5\xf1\xfb&&\x10h\x18(\x08d\x0ej\xe077W\xb3_\xd7\xa6Ds0\x13\xf2t\x92V\xceV\xce/O\xbeA*b \xb4\xc0\x01s\xa3M\xe1\x0c2 <\xc2\x80\xd9b\xeb>\xa8\xac\xba\x80v\xfd\x86\x12\xe0\xd2\xec\xe5\xcc\xd9\x10C\xbab\xcc\xcb\xf1\xa0\x11\x03\xcauP[~^LjB\x01\xc3\xa65\x9f\xfb\x83\xdb\xe0\xf1\xd2\xe3\xd1\x06\x17N\x90\x04\x02\x0b\xa4f\xc4\x01\xd5\x02\x85\xa7l\xd3\x17\x1c}\"hf\x16\xf2\xfc\xcfl\x9e\x99t\x82{9\x8cH\xa98\xbe\xd1|y\xc6a\xa2\x81#*\x84\xb8N\xb2gM\xd8i\x03\xcf\xaf\xd8v\x80\x04\\\x98\xbf\xe4\xca\x85\xe7\xb9C)\xff\x8bN\x07\xd2\xa9d\xafB\xaesi\x12\xf9\xddx\xeb{|1\xd3tH\xc6\xfc0~\xd8BB\xd0\x0b\xa1\x7fS\x8e\xf5\x88\xc4\xe8\xfa2\xf2\xff~\x07\xfe\x08\xab\x80\xd6,\xdfx>\x898W\x98+\xaf\x84\xf1\xd7\x01q\xbf\xdd\xf9\xee\"M\xa8\xf7\xc5\xf5`\x05\xde\xf9TS\x07\xd6\xc7\x90\xf0+\xa0`r\xc7\x8f\xd9\x0b\x12W\x17\xb9\xa3\x02\xf6\x02K\xfb6[-82\x14^Iz\t\x80.\x13gL<L\xe7\xed\x0e\xd3\xc2\x0f\xc1\xa5\xce\x8e\x9fj\xf8T\xc6H\xebq\xf8\xee}\xc9\r\xb0\xc73v\xc2\xc8\xda\xf4{\xdf\xf12\xaf'\xe5\x87\xfd\xe2\xc5\xd6z\x8eTs9\xf7\x0c<\xfc\xde\xc8'\xcb\x8b\xc8\xf7[\x03@\xe5E\x01",
            "dp": b'\xbd\x0e1o\xd3H\xf1$W\x1d\xb5\x045\x10s\xf3r\x1b\xc0~\xc6\xc1\xee\r\xc5\xb1\x14F\xb7\x90d\xe72\x11\r\xbck\xe7\x12\x1e\xbfno\x94G\x8d\x19\xfa\n\xe0\x1e\x88\x9e\xb7\xcb\xfc \xebSa8\xcf\xd4\x0c\x85A\xe4\x9b\n\xe9/\x1cn\xea\xc9\xcd\x0c\xa8\x11\xc2\xc9\x1f\x8c\xb4\xdf\x1bU_\x81\x92\x92i\xd10\xd5T\x11\xc6\x88Wg9B\xaa8\xb0+\xc0\xf5\xb7\xa6\xfb\xd1\xfem\xd2\xa2Zv\xed.i\xd1\x9f9P\x9eI\'"\x04\xd7\xa3\xe0H^\x01\xa0K\xca\xb8p\xc8\xf5s\xb9\x0f\x9a\xc2\x98\xea\xde9\x16DZ\xc2;\xd9\xa8\xa7T\x18Rb|9\x85\xb4\xe0B\xa3\xa3\xb8\xa8\x12\x15,\x1f\xb3\x06\xae\x1a\xf9\x98\xc7\xef\x08\xc9\x89T\x93\x97\x0b\xf0M}\xb0-Og\xe0\x80\xc8\xc2\xc2\xad\x7f\xd4D\xbf:\xf4\x19\xf85\xeb>\x17\xa3=\x9e\xad\xf7\xder\x85P\xcb\xb9\xcb\xa8\xf4\xee\x87e\x93\xe2\x0f\xf7]m\xb94N\xd3\xc6\xb0\xcc"\x90s\xeb\xf5\xb0\x19',
            "dq": b'\xed\xdbX\xa5\xbbx\xd0\xb8UI\x80\xc6\xd0k\xe0*\xa4\xa3\x9ahk;\xb3\xc6L\xce\xd3N\x0e\xc3\xce\x00\x9e\xe4-\xc7H=\xd1\x05y\xef\xb1\xe8\xd9\xb9\xdd\x0e\x85#\x95\xa7\xfcyM\x97?\xbas\x03\xc3\x1a1\xee\xd7-\x841\xc5\x1e\xc5.R\xe2\x86\xa0\xbcM\xcd\xcalt \x06#\r\xfc\xe2\xb9\xd6\x91\xcbzGv\x8eD\xdd\xb7\xf8\xe7\\7\xaf"\xf9q\x9fK\x17u\xd2S1\xef\xe5\xb4\xd9u\x0f;\x81\xb4\x88<\x9bRj\xb3\xec\x1f\xef\x7f\x01#J\xf2iO"\x8e\xe9\xa7\xff\x93\xaa,Z\x07\x8f}\xabS\\\x97a\xa2\x1f\xde\xcb\xc2\x8f^tW\x00L\xa51\nD\xc79y\x11\x11\x03\xa5Vy8R\xf3IZ\xa4\xda\xa8\xa2\x16\x03y#\x1b\xe2|o<\x04\xdeZ\xa2j#/\xdd\xe2\xc13P\xe0\x94;\x97}\x83\x0c\xe7[p\x143\x08\xf0\xd4FH\xa8=(\xd5\x1b\xd1\xbb\xdeb\xd1]B\xcf\xbe\xb8=iS o\x8fF\xa6V\x89^\xf5\x00\x99',
            "qi": b'.\x988y\xe9:1Q\x06\x89\xb3\x85\xdc\xfd\xe9]\xb1\xd0\xadb\xef\xbb\x8d\x18.\xa1\xc9\x92m\xe2b\x97\xea\xcf\xd0\xcc\xc8f\x9e\x9a\x9f\xae\x05\x16\x8dD?\xd3\xed}\xfcMA\x97x\xc4\xf5\xa1i\x14\xda\xfcr\x7f\xddN\xefT\x1a\xe5\xf3t\xa4z\x1f\xef\xf8\x8d\x1a7sJ\xfc\xfc\x19\x99Z\x1aR\x9a\xdch\x91,qo\x13\xc3\xf3\xd2\xe13O\x12\xa7t\xbfk\xa2\x84\xd2x\xec\x18M\n\xc6\xf7;\x9a!:"\xa2\nmtsV\xe2\xb8\xdf\xdfzI\xcf7g\xa9\xe7h\x10V%\x9f\x88x*`Gf\xe99\x86\xa8\x8e\x1e\xf2\xf7\xeb\xfd\xf0\xca\x84\xa1?\x8f\x18\xbb\x07{DR+\x86\xdeA\x13V\x1ad`\x05W\x88\x98\x86\xb2Ti25\xfa\xb9\x81?g\x13\xd4\xbb\xbe\xde\xef\x04\'\xd5\xef\xf7!\xb6\x06\n\x9b\xc7d\x90Wk\xb2\x82e3\xa8\t\xc2a\xb5\x1e\xcf\xd6rWa" \xd1\xde\xa8"\x1aBB\x1a0\xaf\x90bjv\xbe\xa9\x83\xfdk$\xb5',
            "p": b"\xfa\xf4[T\xb5<\x06t\xfaN\x96!2\x12,\x87\x82$y\xadHsBX\xe9\xc7\xed\x8a6\xc0\x14\x87\xed\x8a\xd0\xad\x86\x05\xc1or^\x87\xb2\xcc\xa2\x08_\xed\xe2N\x91\xa0#\xba\xaa\x03\xba\xffv\xad\x9az\xcej\x8et\x96S\x1b\xbd\x93\x99\x94Q\xac\xdd4\x07\x9b\x17x8\xaf\x9b\xdd\xadM6\x1fk\xfa \xdd\xdf\xcbp]b|f6n\xd0U\xc0\xee0\x94\xf6!\x93}\xa3\nE\xa5\xc3\x86\xff\x82YO\x0c%\xfe\xeb_T\xa8\x8e\xd3\xe3\x80\x8d\x06\xca\xdf\xf1J5\xf1\x87\xc7g\x17\xe3\xba\xd24\x0b\xa6\xd6\x1bR\x9d_!$\x8c \xc0\xcdH\x16y\xb9\xdd\x9a\x9c!\xc9'SU\x91\x9a\xd1\x8b\xb9n\xb2\xaa\xd15JO\xf0\x08c\xe0`!Y(\xf8K{\xbb\x8d\xdf\x05\xaac\xc7\xbaeo\x1d\xb4\x11\x01r\xc8`'\x02\x92f\xc1^\xc5\xf7H\r\xfd\xd2.A\xb8&\xd7*>\xe8[;\xd5\xa0Z*\x8b^{\xc7\x8bw\x8d\x1c\xbe6\xd4}\x12\xd5+",
            "q": b'\xee\xd3\x7fo\xff\r\x91<}\x9f\xe0\x8f\xc6\x06\xad\x80\x1em\xbe\xa6_\xdd[%Z?\x8b\xe4\xb1\xcd\xea\xec\x9b\xd2\x19\'\x7f\xe8\x80\xf0\xc40A\xcc\xa8} N\xd3\xcbXI\xf6\xb1\xa7\xf7lp\xf3\xaaq\x8d\x80\x83\x10\xc8\xbb)\x06\xc8\x8f\xea\x11\\\xd2\x99C\x95\x7f\xd9\x92w)\xd4\x90\x1dNF\xbd\xf3\xed"\x84\xcej\x81\xfb\xf9\xd2qv\xc9\x152K\xb8\xf7\x10|\xc7\xeb\xdc\xf0\xb9s\x0f\xc0\x0c\xf7T\xb7B\xa4\x05\xbb\x124\xde\xc0\xec\x88\xb5"qI\x91=\x0cso\\R\xc8\x9a\xe3\xff\xc3\xc5\xdb\xb9\xf8\xbdL\xfc\x9c;\xf3\x90\xa1\xa2\xf3\x08\xb5\x07\x19\xceZ@\xea\xe2\x9acO\r\x1f\x1fL\x1c#?\x0b\x0e\x9d\xbd\x97/\xb5\xb0\x90\x84>s\xccs\xa9R\x988\x06\x18\xaf\r\xfa=\xa3\xcf\xa4\x90o\x95\x95\xbd\x1c\xee\xac*\xc2\xad\xeec4THb\xcb{\xc0\x08Nj>?\x16%\x9d\x0eC\x83\xf7\xc8\x80\x14\x91l\x19p\x17\xb8C\xf3\xf4\xb8\x07\xfe\xa6)',
        },
    ),
}
