/*
 *  Videoblaster SE100 driver by Kate Alhola
 *  Copyright Kate Alhola 1995
 *  kate@iki.fi
 *  http://www.iki.fi/~kate
 */


/*
 * Copyright (c) Jim Lowe, 1995, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Jim Lowe
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * 1.0	29-11-96
 *	Initial Release.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>

#include "grabber.h"
#include "Tcl.h"
#include "device-input.h"
#include "module.h"

/*XXX*/
#define NTSC_WIDTH 320
#define NTSC_HEIGHT 240
#define PAL_WIDTH 384
#define PAL_HEIGHT 288
#define CIF_WIDTH 352
#define CIF_HEIGHT 288


#define Vb100_FMT_PAL      1
#define Vb100_FMT_NTSC     2
#define Vb100_FMT_AUTOMODE 3
#define Vb100_INPUT_DEV0   0
#define Vb100_INPUT_DEV1   1
#define Vb100_INPUT_DEV2   2
#define Vb100_INPUT_DEV3   3

#define DEBUG 1

extern int *vmem;
extern "C" int init7195(...);
extern "C" int VMC_SetReg(...);
extern "C" int select_input(...);

class Vb100Grabber : public Grabber {
 public:
	Vb100Grabber(const char* name, const char* format);
	virtual ~Vb100Grabber();
	virtual void start();
	virtual void stop();
 protected:
	virtual int command(int argc, const char*const* argv);
	virtual int capture();
	virtual int grab();
	void format();
	void setsize();

	int video_format_;	/* video input format: NTSC or PAL */
	int dev_;		/* device fd */
	int port_;		/* video input port */
	int coder_format_;	/* 411, 422, or cif */
	u_int basewidth_;	/* Height of frame to be captured */
	u_int baseheight_;	/* Width of frame to be captured */
	u_int decimate_;	/* division of base sizes */
	volatile u_int* pyuv_;	/* pointer to yuv data */
	 
};

static const int	f_411 = 0;	/* coder_format_s */
static const int	f_422 = 1;
static const int	f_cif = 2;

class Vb100Device : public InputDevice {
 public:
	Vb100Device(const char* nickname, const char* devname, int free);
	virtual int command(int argc, const char*const* argv);
 protected:
	const char* name_;
};

static Vb100Device Vb100_device("Vb100","zz",1);

Vb100Device::Vb100Device(const char* nickname, const char *devname, int free):
					InputDevice(nickname), name_(devname)
{
  attributes_ = "\
format {422 411} \
size {large normal small cif} \
port {Port-1 Port-2}";
#ifdef DEBUG
    printf("Vb100Device::Vb100Device name=%s\n", nickname);
#endif /* DEBUG */
}

int Vb100Device::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();

#ifdef DEBUG
    printf("Vb100Device::command argc=%d ", argc);
    int i;
    for (i = 0; i < argc; i++)
	printf("\"%s\" ", argv[i]);
    printf("\n");
#endif

	if ((argc == 3) && (strcmp(argv[1], "open") == 0)) {
		TclObject* o = 0;
		o = new Vb100Grabber(name_, argv[2]);
		if (o != 0)
			tcl.result(o->name());
		return (TCL_OK);
	}
	return (InputDevice::command(argc, argv));
}

Vb100Grabber::Vb100Grabber(const char* name, const char* format)
{
    printf("Vb100Grabber:Vb100Grabber\n");
	coder_format_ = -1;
	if(!strcmp(format, "411")) coder_format_ = f_411;
	if(!strcmp(format, "422")) coder_format_ = f_422;
	if(!strcmp(format, "cif")) coder_format_ = f_cif;
	if(coder_format_ == -1) {
		fprintf(stderr,
			"vic: Vb100Grabber: unsupported format: %s\n",
			format);
		abort();
	};
        init7195();   /* Initialize vbse100 */

        printf("vmem=%08x\n",vmem);

	video_format_ = Vb100_FMT_AUTOMODE;
	decimate_ = 1;
	basewidth_ = CIF_WIDTH;
	baseheight_ = CIF_HEIGHT;
	
}

Vb100Grabber::~Vb100Grabber()
{
}

void Vb100Grabber::setsize()
{

#ifdef DEBUG
    printf("Vb100Grabber:setsize\n");
#endif

	switch(coder_format_) {
	case f_422:
		set_size_422(basewidth_, baseheight_);
		break;
	case f_cif:
	case f_411:
		set_size_411(basewidth_, baseheight_);
		break;
	}

	allocref();	/* allocate reference frame */
}

void Vb100Grabber::format()
{
	int status;

	int w, h;

#ifdef DEBUG
    printf("Vb100Grabber:format\n");
#endif

	video_format_ = Vb100_FMT_PAL;
	w = CIF_WIDTH;
	h = CIF_HEIGHT;
	basewidth_ = w;
	baseheight_ = h;
	setsize();
}


void Vb100Grabber::start()
{
#ifdef DEBUG
    printf("Vb100Grabber:start\n");
#endif	
    format();
	Grabber::start();
}

void Vb100Grabber::stop()
{
#ifdef DEBUG
    printf("Vb100Grabber:stop\n");
#endif
	Grabber::stop();
}

int Vb100Grabber::command(int argc, const char*const* argv)
{
#ifdef DEBUG
    printf("Vb100Grabber::command argc=%d ", argc);
    int i;
    for (i = 0; i < argc; i++)
	printf("\"%s\" ", argv[i]);
    printf("\n");
#endif
	if (argc == 3) {
		if (strcmp(argv[1], "decimate") == 0) {
			int dec = atoi(argv[2]);
			Tcl& tcl = Tcl::instance();
			if (dec <= 0) {
				tcl.resultf("%s: divide by zero", argv[0]);
				return (TCL_ERROR);
			}
			if (dec != decimate_) {
				decimate_ = dec;
				if(running_) {
					stop();
					setsize();
					start();
				}
			}
			return (TCL_OK);	
		} else if (strcmp(argv[1], "port") == 0) {
			int p = port_;
                        if(!strcmp(argv[2], "Port-1")) p = Vb100_INPUT_DEV1;
                        if(!strcmp(argv[2], "Port-2")) p = Vb100_INPUT_DEV2;
			select_input(p);
			return (TCL_OK);	
		} else if (strcmp(argv[1], "format") == 0) {
			if (strcmp(argv[2], "auto") == 0)
				video_format_ = Vb100_FMT_AUTOMODE;
			else if (strcmp(argv[2], "pal") == 0)
				video_format_ = Vb100_FMT_PAL;
			else
				video_format_ = Vb100_FMT_NTSC;
			if (running_)
				format();
			return (TCL_OK);	
		} else if (strcmp(argv[1], "contrast") == 0) {
			contrast(atof(argv[2]));
			return (TCL_OK);	
		}
	} else if (argc == 2) {
		if (strcmp(argv[1], "normalize") == 0) {
#ifdef notdef
			normalize();
#endif
			return (TCL_OK);	
		} else if (strcmp(argv[1], "format") == 0) {
			Tcl& tcl = Tcl::instance();
			switch (video_format_) {

			case Vb100_FMT_AUTOMODE:
				tcl.result("auto");
				break;

			case Vb100_FMT_NTSC:
				tcl.result("ntsc");
				break;

			case Vb100_FMT_PAL:
				tcl.result("pal");
				break;

			default:
				tcl.result("");
				break;
			}
			return (TCL_OK);
			
		}
	}
	return (Grabber::command(argc, argv));
}

int Vb100Grabber::capture()
{
#ifdef DEBUG
/*    printf("Vb100Grabber:capture\n");*/
#endif
    int row,j,bank;
    int		f422 = coder_format_ == f_422;
    int         left=10;
    int 	right = basewidth_+left;
    int         top = 10;
    int         rows  = baseheight_+top;
    u_char* 		lum  = (u_char *)frame_;
    u_char*		uoff = (u_char *)((u_int)lum + (u_int)framesize_);
    u_char*	 	voff = (u_char *)((u_int)uoff + 
					 (u_int)(framesize_>>(f422?1:2)));
    
    

  for (row = top; row < rows; ++row) {
    long   tmp;
    long y, u, v, y1, r, g, b, *yuvptr;
    register int    col;

    for (col = left; col < right; col += 2) {

      j=(row*704)+col;
      bank=j>>13;
      j=j&0x1fff;
      VMC_SetReg(0x19,bank);
      tmp=((unsigned long *)vmem)[j>>1];

/*      v = ((0xff & (tmp >> 24))^ 0x80) - 128;*/
      v = ((0xff & (tmp >> 24))^ 0x80);
      y1 = ((0xff & (tmp >> 16)) );
      if (y < 0) y = 0;

/*      u = ((0xff & (tmp>>8 ))^0x80) - 128;*/
      u = ((0xff & (tmp>>8 ))^0x80);
      y = ((0xff & (tmp)) );
      if (y1 < 0) y1 = 0;
      
      *lum++=y;
      *lum++=y1;
      if((f422==0) && (row &1)) {
	*uoff++=u;
	*voff++=v;
      };

    }
  }
    return 1;
  }

int Vb100Grabber::grab()
{
#ifdef DEBUG
/*    printf("Vb100Grabber:grab\n"); */
#endif
	if (capture() == 0)
		return (0);
	suppress(frame_);
	saveblks(frame_);
	YuvFrame f(media_ts(), frame_, crvec_, outw_, outh_);
	return (target_->consume(&f));
}

